/*
 * Hack-o-Rocket router configuration compiler
 *
 * The specific setting lines are parsed and the global variables
 * are filled here.
 */

#include <sys/types.h>
#include <machine/endian.h>
#include <stdio.h>
#include <strings.h>
#include "config.h"
#include "sdsldef.h"

extern int wan_type;
extern struct wan_config wanconf;
extern struct sdsl_phy_config sdslphy;
extern struct idsl_phy_config idslphy;
extern int dlci, vpi, vci;

extern char *input_filename;
extern int lineno;

set_sdsl(argc, argv)
	char **argv;
{
	int datarate;

	if (wan_type) {
		fprintf(stderr, "%s: line %d: WAN type has already been set\n",
			input_filename, lineno);
		exit(1);
	}

	if (argc != 3) {
		fprintf(stderr,
			"%s: line %d: SDSL keyword requires 3 arguments\n",
			input_filename, lineno);
		exit(1);
	}

	if (!strcasecmp(argv[0], "B"))
		wan_type = WANTYPE_SDSLB;
	else if (!strcasecmp(argv[0], "Nokia"))
		wan_type = WANTYPE_NSDSL;
	else if (!strcasecmp(argv[0], "N1"))
		wan_type = WANTYPE_NSDSL;
	else {
		fprintf(stderr, "%s: line %d: SDSL flavor %s not supported\n",
			input_filename, lineno, argv[0]);
		exit(1);
	}

	if (!strcasecmp(argv[1], "HTU-C"))
		sdslphy.sdsl_terminal_type = _HTUC;
	else if (!strcasecmp(argv[1], "HTU-R"))
		sdslphy.sdsl_terminal_type = _HTUR;
	else if (!strcasecmp(argv[1], "CO"))
		sdslphy.sdsl_terminal_type = _HTUC;
	else if (!strcasecmp(argv[1], "CPE"))
		sdslphy.sdsl_terminal_type = _HTUR;
	else {
		fprintf(stderr,
	"%s: line %d: SDSL terminal type keyword \"%s\" not understood\n",
			input_filename, lineno, argv[1]);
	}

	datarate = atoi(argv[2]);
	if (datarate % 8 || datarate < 144 || datarate > 2320) {
		fprintf(stderr, "%s: line %d: invalid SDSL data rate\n",
			input_filename, lineno);
		exit(1);
	}
	sdslphy.sdsl_data_rate = htons(datarate >> 3);
	sdslphy.sdsl_quat_orient = _SERIAL_SIGNFIRST;
}

set_idsl(argc, argv)
	char **argv;
{
	if (wan_type) {
		fprintf(stderr, "%s: line %d: WAN type has already been set\n",
			input_filename, lineno);
		exit(1);
	}
	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: IDSL keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	wan_type = WANTYPE_IDSL;
	idsl_channels_conf(argv[0]);
}

set_dlci(argc, argv)
	char **argv;
{
	int val;

	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: DLCI keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	if (!is_number_string(argv[0])) {
		fprintf(stderr,
			"%s: line %d: argument \"%s\" is not a valid number\n",
			input_filename, lineno, argv[0]);
		exit(1);
	}
	val = atoi(argv[0]);
	if (val > 1023) {
		fprintf(stderr, "%s: line %d: DLCI value too large\n",
			input_filename, lineno);
		exit(1);
	}
	dlci = val;
}

set_vpi(argc, argv)
	char **argv;
{
	int val;

	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: VPI keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	if (!is_number_string(argv[0])) {
		fprintf(stderr,
			"%s: line %d: argument \"%s\" is not a valid number\n",
			input_filename, lineno, argv[0]);
		exit(1);
	}
	val = atoi(argv[0]);
	if (val > 255) {
		fprintf(stderr, "%s: line %d: VPI value too large\n",
			input_filename, lineno);
		exit(1);
	}
	vpi = val;
}

set_vci(argc, argv)
	char **argv;
{
	int val;

	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: VCI keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	if (!is_number_string(argv[0])) {
		fprintf(stderr,
			"%s: line %d: argument \"%s\" is not a valid number\n",
			input_filename, lineno, argv[0]);
		exit(1);
	}
	val = atoi(argv[0]);
	if (val > 65535) {
		fprintf(stderr, "%s: line %d: VCI value too large\n",
			input_filename, lineno);
		exit(1);
	}
	vci = val;
}

set_quat(argc, argv)
	char **argv;
{
	if (wan_type != WANTYPE_SDSLB) {
		fprintf(stderr,
"%s: line %d: quat orientation setting is only valid for SDSL Flavor B\n",
			input_filename, lineno);
		exit(1);
	}
	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: QUAT keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	if (!strcasecmp(argv[0], "signfirst"))
		sdslphy.sdsl_quat_orient = _SERIAL_SIGNFIRST;
	else if (!strcasecmp(argv[0], "magfirst"))
		sdslphy.sdsl_quat_orient = _SERIAL_MAGFIRST;
	else {
		fprintf(stderr,
			"%s: line %d: invalid quat orientation setting\n",
			input_filename, lineno);
		exit(1);
	}
}

set_q922_addr()
{
	int dlci_upper, dlci_lower;
	u_char *cp;

	dlci_upper = dlci >> 4;
	dlci_lower = dlci & 15;
	cp = (u_char *) &wanconf.wan_q922_addr;
	cp[0] = dlci_upper << 2;
	cp[1] = (dlci_lower << 4) | 0x01;
}
