/*
 * Hack-o-Rocket router configuration compiler
 *
 * IP settings
 */

#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <strings.h>
#include "config.h"
#include "defs.h"

extern struct lan_config lanconf;
extern struct wan_config wanconf;
extern int lanconf_set, wanip_set;
extern u_long add_ips[];
extern struct rt_entry add_routes[];
extern int num_add_ips, num_add_routes;
extern struct rt_entry defroute;
extern struct rt_entry rtable[];
extern int total_routes;

extern char *input_filename;
extern int lineno;

parse_cidr_notation(str, ip, netmask, single_ok)
	char *str;
	u_long *ip, *netmask;
{
	char *cp;
	int prefix;
	u_long mask_accum, mask_bit;

	if (single_ok && (*ip = inet_addr(str)) != INADDR_NONE) {
		*netmask = 0xFFFFFFFF;
		return;
	}
	cp = index(str, '/');
	if (!cp) {
inv:		fprintf(stderr, "%s: line %d: CIDR notation expected\n",
			input_filename, lineno);
		exit(1);
	}
	*cp++ = '\0';
	*ip = inet_addr(str);
	if (*ip == INADDR_NONE)
		goto inv;
	if (!is_number_string(cp))
		goto inv;
	prefix = atoi(cp);
	if (prefix > 32) {
		fprintf(stderr,
		"%s: line %d: CIDR prefix cannot be longer than 32 bits!\n",
			input_filename, lineno);
		exit(1);
	}
	mask_accum = 0;
	for (mask_bit = 0x80000000; prefix--; mask_bit >>= 1)
		mask_accum |= mask_bit;
	*netmask = htonl(mask_accum);
	if ((*ip & *netmask) != *ip) {
		fprintf(stderr,
		"%s: line %d: the host part of CIDR notation must be zero\n",
			input_filename, lineno);
		exit(1);
	}
}

set_lan(argc, argv)
	char **argv;
{
	if (lanconf_set) {
		fprintf(stderr,
			"%s: line %d: LAN configuration has already been set\n",
			input_filename, lineno);
		exit(1);
	}
	if (argc != 2) {
		fprintf(stderr,
			"%s: line %d: LAN keyword requires 2 arguments\n",
			input_filename, lineno);
		exit(1);
	}
	lanconf.lan_ip = inet_addr(argv[0]);
	if (lanconf.lan_ip == INADDR_NONE) {
inv:		fprintf(stderr, "%s: line %d: invalid IP address syntax\n",
			input_filename, lineno);
		exit(1);
	}
	lanconf.lan_netmask = inet_addr(argv[1]);
	if (lanconf.lan_netmask == INADDR_NONE)
		goto inv;
	lanconf_set = 1;
}

set_wanip(argc, argv)
	char **argv;
{
	if (wanip_set) {
		fprintf(stderr,
			"%s: line %d: WAN IP address has already been set\n",
			input_filename, lineno);
		exit(1);
	}
	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: WAN keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	wanconf.wan_ip = inet_addr(argv[0]);
	if (wanconf.wan_ip == INADDR_NONE) {
		fprintf(stderr, "%s: line %d: IP address expected\n",
			input_filename, lineno);
		exit(1);
	}
	wanip_set = 1;
}

set_add_ip(argc, argv)
	char **argv;
{
	u_long newip;

	if (argc != 1) {
		fprintf(stderr,
			"%s: line %d: ADDIP keyword requires 1 argument\n",
			input_filename, lineno);
		exit(1);
	}
	newip = inet_addr(argv[0]);
	if (newip == INADDR_NONE) {
		fprintf(stderr, "%s: line %d: IP address expected\n",
			input_filename, lineno);
		exit(1);
	}
	if (num_add_ips >= MAX_ADD_IPS) {
		fprintf(stderr,
	"Implementation limit exceeded: too many additional IP addresses\n");
		exit(1);
	}
	add_ips[num_add_ips++] = newip;
}

set_add_route(argc, argv)
	char **argv;
{
	struct rt_entry rte;

	if (argc < 2) {
		fprintf(stderr,
		"%s: line %d: ROUTE keyword requires at least 2 arguments\n",
			input_filename, lineno);
		exit(1);
	}
	if (!strcasecmp(argv[0], "default")) {
		rte.rt_dest = 0;
		rte.rt_mask = 0;
	} else
		parse_cidr_notation(argv[0], &rte.rt_dest, &rte.rt_mask, 1);

	if (!strcasecmp(argv[1], "lan")) {
		if (argc != 3) {
argcount:		fprintf(stderr,
				"%s: line %d: wrong number of arguments\n",
				input_filename, lineno);
			exit(1);
		}
		rte.rt_type = ROUTE_TYPE_LANVIA;
		rte.rt_gw = inet_addr(argv[2]);
		if (rte.rt_gw == INADDR_NONE) {
			fprintf(stderr,
				"%s: line %d: gateway IP address expected\n",
				input_filename, lineno);
			exit(1);
		}
	} else if (!strcasecmp(argv[1], "wan")) {
		if (argc != 2)
			goto argcount;
		rte.rt_type = ROUTE_TYPE_WAN;
		rte.rt_gw = 0;
	} else if (!strcasecmp(argv[1], "drop")) {
		if (argc != 2)
			goto argcount;
		rte.rt_type = ROUTE_TYPE_DROP;
		rte.rt_gw = 0;
	} else {
		fprintf(stderr,
		"%s: line %d: route destination \"%s\" not understood\n",
			input_filename, lineno, argv[1]);
		exit(1);
	}

	if (!rte.rt_dest && !rte.rt_mask) {
		bcopy(&rte, &defroute, sizeof(struct rt_entry));
		return;
	}
	if (num_add_routes >= MAX_ADD_ROUTES) {
		fprintf(stderr,
		"Implementation limit exceeded: too many additional routes\n");
		exit(1);
	}
	bcopy(&rte, add_routes + num_add_routes, sizeof(struct rt_entry));
	num_add_routes++;
}

make_routing_table()
{
	struct rt_entry rte;
	int i;

	/* Our own IPs go first */
	rte.rt_dest = lanconf.lan_ip;
	rte.rt_mask = 0xFFFFFFFF;
	rte.rt_type = ROUTE_TYPE_FORME;
	rte.rt_gw = 0;
	add_route(&rte);
	if (wanip_set) {
		rte.rt_dest = wanconf.wan_ip;
		add_route(&rte);
	}
	for (i = 0; i < num_add_ips; i++) {
		rte.rt_dest = add_ips[i];
		add_route(&rte);
	}

	/* Directly attached LAN */
	rte.rt_dest = lanconf.lan_ip & lanconf.lan_netmask;
	rte.rt_mask = lanconf.lan_netmask;
	rte.rt_type = ROUTE_TYPE_LANLOCAL;
	add_route(&rte);

	/* Any additional routes */
	for (i = 0; i < num_add_routes; i++)
		add_route(&add_routes[i]);

	/* Default route ends the table */
	add_route(&defroute);
}

add_route(rts)
	register struct rt_entry *rts;
{
	register struct rt_entry *rtd;

	rtd = rtable + total_routes;
	rtd->rt_dest = rts->rt_dest;
	rtd->rt_mask = rts->rt_mask;
	rtd->rt_type = htonl(rts->rt_type);
	rtd->rt_gw = rts->rt_gw;
	total_routes++;
}
