/*
 * Can not use if the Bit-pump INT_BUG compiler flag is set, the INT_BUG
 * uses the same mircopressor timer
 */
#ifndef INT_BUG

/*
 * PROGRAM NAME:  Bt8960 Single Board
 *
 * VERSION:       1.0
 *
 * FILENAME:      timer.c
 *
 * FILE CREATED:  June 28, 1996
 *
 * LAST MODIFIED: [9-1-98]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1996
 *                Brooktree Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   This module contains the General Purpose Timer functions
 *
 *
 * FUNCTION LIST: _InitGenPurposeTimer()
 *                _EnableGenPurposeTimer()
 *                _DisableGenPurposeTimer()
 *                _GetGenPurposeTimerStatus()
 *                _Timer0_ISR()
 */


/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#ifdef C51
#include <reg51.h>
#endif

#include "typedefs.h"

#include "bitpump\btmain.h"   /* to resolve _NO_OF_LOOPS */

#include "timer.h"

#define TIMER_MSPACE    BP_IDATA

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define TIMER_INTERVAL_LOW      0x00      /* 50.0 mS */
#define TIMER_INTERVAL_HIGH     0x4C


#if 0
#define TIMER_INTERVAL_LOW      0x80      /* 0.998264 mS */
#define TIMER_INTERVAL_HIGH     0xFC
#endif

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/
static void _LoadGenPurposeTimerInterval(void);

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Constants                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

/*
 * Implemented as 2 dim array so can have each timer per bit-pump (channel)
 */
static GEN_PURPOSE_TIMER TIMER_MSPACE gen_timer[_NO_OF_LOOPS][_NO_GEN_PURPOSE_TIMERS];

/* only 1 one-second timer */
static GEN_PURPOSE_CONT_TIMER TIMER_MSPACE gen_cont_timer[_NO_OF_LOOPS][_NO_GEN_PURPOSE_TIMERS];

/*
 * FUNCTION:   _InitGenPurposeTimer
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function initializes the Interrupt #1 so can decrement
 *             the timer counters when the Timer #0 expires.
 *
 *             Timer #0 set in Mode 1 (16 bit).  Timer #1 needs to be set
 *             as baud rate generator (from 'init51.c'), hence the 0x21.
 *             
 * RETURN:     nothing
 *
 * NOTES:      Only applies if INT_BUG is not specified
 *
 * CHANGES:    August 25, 1996    V1.0  :  New
 */
void _InitGenPurposeTimer (void)
{
    BP_U_8BIT bp, timer;

    for ( bp = 0 ; bp < _NO_OF_LOOPS ; bp++ )
    {
        for ( timer = 0 ; timer < _NO_GEN_PURPOSE_TIMERS ; timer++ )
        {
        gen_timer[bp][timer].status.reg = 0;    /* Reset flags */
        }
    }

#ifdef C51
    TMOD = 0x21; /* Timer0 mode1, Timer1 mode2 (baud rate generator) */

    _LoadGenPurposeTimerInterval();

    TR0 = 1; /* Run timer0 */
#endif

    return ;
}

/*
 * FUNCTION:   _LoadGenPurposeTimerInterval
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function loads the micropressor timer counter value.
 *
 *             The counter is set to ~50mS interval.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 28, 1997    V1.0  :  New
 */
void _LoadGenPurposeTimerInterval (void) 
{

#ifdef C51
    TL0 = TIMER_INTERVAL_LOW;  /* Load timer values */
    TH0 = TIMER_INTERVAL_HIGH;
#endif

   return;
}




/*
 * FUNCTION:   _EnableGenPurposeTimer
 *
 * PARAMETERS: bp - which bit-pump
 *             timer - which timer
 *             value - number of 50mS intervals
 *
 * PURPOSE:    This function enables Timer #0 Interrupt and initializes
 *             the counter.
 *
 *             How to determine counter value, i.e. for Pending Deactivation
 *             counter.
 *             40 * 50mS = ~2 second
 *             
 * RETURN:     nothing
 *
 * NOTES:      Only applies if INT_BUG is not specified
 *
 * CHANGES:    August 25, 1996    V1.0  :  New
 */
void _EnableGenPurposeTimer (BP_U_8BIT bp, BP_U_8BIT timer, BP_U_16BIT value)
{
    if ( bp >= _NO_OF_LOOPS )
        {
        return ;
        }

    if ( timer >= _NO_GEN_PURPOSE_TIMERS )
        {
        return ;
        }

    gen_cont_timer[bp][timer].load_value = value;
    gen_cont_timer[bp][timer].elapsed_counter = 0;

    gen_timer[bp][timer].counter_value = value;
    gen_timer[bp][timer].status.bits.state = 1;
    gen_timer[bp][timer].status.bits.complete = 0;

    /*
     * Do some error checking
     */
    if ( value == 0 )
        {
        gen_timer[bp][timer].status.bits.complete = 1;
        }

#ifdef C51
    ET0 = 1;
#endif

    return ;
}



/*
 * FUNCTION:   _ContinuousGenPurposeTimer
 *
 * PARAMETERS: bp - which bit-pump
 *             timer - which timer
 *             state - contiuous ON/OFF
 *
 * PURPOSE:    This function sets the specified timer to continuous or not.
 *
 * RETURN:     nothing
 *
 * NOTES:      Only applies if INT_BUG is not specified
 *
 * CHANGES:    March 4, 1998    New
 */
void _ContinuousGenPurposeTimer (BP_U_8BIT bp, BP_U_8BIT timer, BP_U_8BIT state)
{
    if ( bp >= _NO_OF_LOOPS )
        {
        return ;
        }

    if ( timer >= _NO_GEN_PURPOSE_TIMERS )
        {
        return ;
        }

    if ( state < 0 || state > 1 )
        {
        return ;
        }

    gen_timer[bp][timer].status.bits.continuous = state;
    return ;
}

/*
 * FUNCTION:   _DisableGenPurposeTimer
 *
 * PARAMETERS: bp - which bit-pump
 *             timer - which timer
 *
 * PURPOSE:    This function disables Timer #0 Interrupt.
 *             
 * RETURN:     nothing
 *
 * NOTES:      Only applies if INT_BUG is not specified
 *
 * CHANGES:    August 25, 1996    V1.0  :  New
 */
void _DisableGenPurposeTimer (BP_U_8BIT bp, BP_U_8BIT timer)
{
    BP_U_8BIT i, k;

    if ( bp >= _NO_OF_LOOPS )
        {
        return ;
        }

    if ( timer >= _NO_GEN_PURPOSE_TIMERS )
        {
        return ;
        }

    gen_timer[bp][timer].status.bits.state = 0;
    gen_timer[bp][timer].status.bits.complete = 1;    /* safety net */

    /*
     * If any timer is still enabled, then don't disable interrupt
     */
    for ( k = 0 ; k < _NO_OF_LOOPS ; k++ )
        {
        for ( i = 0 ; i < _NO_GEN_PURPOSE_TIMERS ; i++ )
            {
            if ( gen_timer[k][i].status.bits.state == 1 )
                {
                return ;
                }
            }
        }

#ifdef C51
    /*
     * Disable if all OFF
     */
    ET0 = 0;
#endif

    return ;
}


/*
 * FUNCTION:   _GetGenPurposeTimerStatus
 *
 * PARAMETERS: bp - which bit-pump
 *             timer - which timer
 *
 * PURPOSE:    This function returns the timer status.
 *
 *             Bit #  #define         Description
 *               0    TIMER_STATE     Timer State (0/1) or (OFF/ON)
 *               1    TIMER_COMPLETE  Timer Complete - 1=Yes, 0=No
 *             
 * RETURN:     status
 *             0xFF - invalid 'timer' specified (will force complete)
 *
 * NOTES:      
 *
 * CHANGES:    February 28, 1997    V1.0  :  New
 */
BP_U_8BIT _GetGenPurposeTimerStatus (BP_U_8BIT bp, BP_U_8BIT timer)
{
    if ( bp >= _NO_OF_LOOPS )
        {
        return 0xFF;
        }

    if ( timer >= _NO_GEN_PURPOSE_TIMERS )
        {
        return 0xFF;
        }

    return gen_timer[bp][timer].status.reg;
}


/*
 * FUNCTION:   _GetGenPurposeContCount
 *
 * PARAMETERS: bp - which bit-pump
 *             timer - which timer
 *
 * PURPOSE:    This function returns the continuous timer elapsed count.
 *
 *             
 * RETURN:     status
 *             0xFF - invalid 'timer' specified (will force complete)
 *
 * NOTES:      
 *
 * CHANGES:    February 28, 1997    V1.0  :  New
 */
BP_U_32BIT _GetGenPurposeContCount (BP_U_8BIT bp, BP_U_8BIT timer)
{
    if ( bp >= _NO_OF_LOOPS )
        {
        return 0;
        }

    if ( timer >= _NO_GEN_PURPOSE_TIMERS )
        {
        return 0;
        }

    return gen_cont_timer[bp][timer].elapsed_counter;
}



/*
 * FUNCTION:   _Timer0_ISR
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function gets caltimer every time the Timer #0 overflows.
 *
 *             If during startup, toggles the Startup Timer
 *
 *             If during pending deactivate, toggles the Error Timer
 *             the 'counter_value' is used to keep track of when the
 *             Pending Deactivate expires.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      The interrupt period is set to 50.0mS
 *
 * CHANGES:    August 25, 1996    V1.0  :  New
 *             March 4, 1998      Added Continous Support
 */
#ifdef C51
_Timer0_ISR() interrupt 1 /* Timer0 interrupt */
{
    BP_U_8BIT i, k;

    _LoadGenPurposeTimerInterval();

    for ( k = 0 ; k < _NO_OF_LOOPS ; k++ )
        {
        for ( i = 0 ; i < _NO_GEN_PURPOSE_TIMERS ; i++ )
            {
            if ( gen_timer[k][i].status.bits.state == 1 )
                {
                switch ( gen_timer[k][i].counter_value )
                    {
                    case 0x00:
                        break;
                    case 0x01:
                        if ( gen_timer[k][i].status.bits.continuous )
                            {
                            (gen_cont_timer[k][i].elapsed_counter)++;
                            gen_timer[k][i].counter_value = gen_cont_timer[k][i].load_value;
                            }
                        else
                            {
                            gen_timer[k][i].status.bits.complete = 1;
                            (gen_timer[k][i].counter_value)--;
                            }

                        break;
                    default:
                        (gen_timer[k][i].counter_value)--;
                        break;
                    }   /* end switch */
                }   /* end if timer ON */
            }   /* end for */
        }

}
#endif   /* C51 */


#endif /* not INT_BUG */
