/***********************************************************/
/*    DSL_MAN.C                                            */
/*    (C) Copyright 1998 by Rockwell Semiconductor Systems */
/*                                                         */
/*                                                         */
/* Description:                                            */
/*    This is the dynamic loop control functions.          */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*                                                         */
/* List of Functions found in this module:                 */
/*                                                         */
/*   - _DSLLoopHandler( void );                            */
/*   - _Set_2E1_PairID(BP_U_8BIT bp);                      */
/*   - _Set_2T1_SyncWord(BP_U_8BIT bp);                    */
/*   - _Set_3E1_PairID1(BP_U_8BIT bp);                     */
/*   - _Set_3E1_PairID2(BP_U_8BIT bp);                     */
/*   - _Switch_MasterLoop(BP_U_8BIT loop)                  */
/*   - _Reset_Pid_Validation (BP_U_8BIT b;                 */
/*                                                         */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/* Aug 98 - New: Dean Rasmussen                            */
/* Oct 98 - New: Laura Yuan                                */
/*                                                         */
/***********************************************************/
#include "dsl_incl.h"

#ifdef CHAN_UNIT
#ifndef REPEATER


/* Function: _DSLLoopHandler
 *
 * Usage:   This is a high level loop control function. 
 *          Master loop switch and any necessary loop reversal are done here.
 */
void _DSLLoopHandler (void)
{
    BP_U_8BIT bp, temp;  

    /* If only 1 bit-pump present, no need to check for loop reversal */
    if ( num_bit_pumps == 1) 
        {
        return;
        }

#ifdef SP_API    
    if (sngl_loop_api)
        {
        return;
        }
#endif

    /* good_loop_cnt is a variable which keeps tracking the # of loops in
       normal operation. It is updated in ASM whenever there is transition
       in or out of ACTIVE_TX_RX_STATE.
     */
    switch (dsl_status2.bits.good_loop_cnt)
        {
        case 2:
            /* If rate_index = 2T1 or 2E1, good_loop_cnt = 2 means all the
               loops are in normal operation, nothing needs to be done;
               if rate_index = 3E1, good_loop_cnt = 2 means one of the 3 
               loops is not in normal operation. If it is the master, switch
               the master loop to the next loop in normal operation.
             */
#ifdef CU_3E1
            if ( rate_index == _3E1 )
               {
               temp = _CuFlags._CuMasterLoop;
               if (system_status[temp].bits.activation_state != ACTIVE_TX_RX_STATE)
                  {  
                  /* Switch the master loop to the next available loop */ 
                  _BtControl( _CU_COMMON, _CU_SET_MASTER_LOOP, (temp+1)%3 );
                  }
               }
#endif /* CU_3E1 */
            break;
        case 1:

            /* If rate_index = 2T1 or 2E1, good_loop_cnt = 1 means one of the two
               loops is not in normal operation. Check if it is the master loop.
               If so, switch the master loop to the one that is in normal operation.
             */
            temp = _CuFlags._CuMasterLoop;
            if (system_status[temp].bits.activation_state != ACTIVE_TX_RX_STATE)
               {
               _Switch_MasterLoop(temp);
               }
#ifdef HTUR
            else
               {
            /* If the master loops is not failed and the rate_index = 3E1, check if 
               there is another loop entering normal operation (done by _Set_3E1_
               PairID2() ). If so, 3 loops pair IDs can now be uniquely determined.
             */
               if ( dip_sw.bits.terminal_type == _HTUR )
                  {
#ifdef CU_3E1
                  if (rate_index == _3E1 )
                     {
                     _CuFlags._CuLoopsReversed = 0;
                     _Set_3E1_PairID2(temp);
                     }
#endif /* CU_3E1 */
                  }
               }
#endif /* HTUR */
            break;
        case 0:

            /* 
             *  When all the loops are failed, the first loop comming back 
             *  becomes the master loop.
             *
             *  If rate_index = 2T1 or 2E1, the other loop's pair ID or sync_word
             *  can be uniquely determined. _Set_2E1_PairID() or _Set_2T1_SyncWord()
             *  is called to do this.
             *
             *  If rate_index = 3E1, the other two loops' pair IDs can not be uniquely
             *  determined at this time, _Set_3E1_PairID1() is called to set the mapping
             *  so that the master loop's mapping is correct while the mapping for the
             *  other two loops' are only temporary.
             *
             *  _CuReverseLoops() is called to change the mapping based on pair IDs or
             *  sync_word.
             */

            for (bp = 0; bp < _NO_OF_LOOPS; bp++)
                {
                if(system_status[bp].bits.activation_state == GOTO_ACTIVE_TX_RX_STATE)
                  {
                  _CuFlags._CuLoopsReversed = 0; /*By default assume no loop reversal*/
                  _BtControl( _CU_COMMON, _CU_SET_MASTER_LOOP, bp );
#ifdef HTUR
                    if ( dip_sw.bits.terminal_type == _HTUR )
                        {                 
                        switch (rate_index)
                            {
                            case _2E1:
#ifdef CU_2E1
                                _Set_2E1_PairID(bp);                                
#endif /* CU_2E1 */
                                break;

                            case _2T1:
#ifdef CU_2T1
                                _Set_2T1_SyncWord(bp);
#endif /* CU_2T1 */
                                break;

                            case _3E1:
#ifdef CU_3E1
                                _Set_3E1_PairID1(bp);
#endif /* CU_3E1 */
                                break;
                            } /* end switch on rate_index */    

                        /*
                         * _CuReverseLoops() determines the loop configuration
                         * based on the PID (E1) or sync word select (T1).  
                         * It also writes the tables to the channel unit.
                         */
                        _CuReverseLoops();
                        }  /* end if */
#endif /* HTUR */
                    break;
                    }  /* end if */
                }  /* end for loop */
            break;
        } /* end switch on good_loop_cnt */
}

#ifdef HTUR /* All the pair ID initialization only affects HTU-R side */

#ifdef CU_2E1
/*
 * Function: _Set_2E1_PairID1( BP_U_8BIT bp )
 * Parameter: bp is the master loop.
 *
 * This function sets the pair ID of the slave loop for the HTUR.
 * This PID can be uniquely determined after the master loop has started.  
 * The PID of the master loop for the HTUR is set during startup by 
 * E1_Pairid_Validation().
 *
 * E1_Pairid_Validation() is still called for the slave loop.
 * However, the PID transmitted by the HTUR on slave loop will 
 * be correctly pre-configured by _Set_2E1_PairID1().
 *
 * The PIDs transmitted by both HTUCs are not dynamic.  There are set 
 * one time during startup by _CuSetPid().  The PIDs transmitted by both HTURs 
 * are set to invalid values during startup by _CuSetPidToAllOnes().
 */
void _Set_2E1_PairID( BP_U_8BIT bp )
{
    BP_U_8BIT temp;  
    TX_WR BP_XDATA *tx_wr_ptr;

   /* Get the PID (lower 3 bits in the Z bit register) of the master loop */
   temp = cu_rx_hoh.rzbit_pid[bp].reg & 0x07;

   /* Mask off the PID bits (lower 3 bits) for the slave loop.
    *  If master Loop = 0 or 1, then slave loop = 1 or 0, respectively.
    *   For example, if bp = 0, then (bp+1)%2 = 1.
    */
   cu_rx_hoh.rzbit_pid[(bp+1)%2].reg &= 0xF8;

   /* If PID of the master loop = 1 */
   if ( temp == 0x01 )
      {
      if (bp == 1)
         {
         _CuFlags._CuLoopsReversed = 1;
         }
      /* Set PID of the slave loop to 2 */
      cu_rx_hoh.rzbit_pid[(bp+1)%2].reg |= 0x02;
      }
   else  /* PID of the master loop = 2 */
      {
      if (bp == 0)
         {
         _CuFlags._CuLoopsReversed = 1; 
         }
      /* Set PID of the slave loop to 1 */
      cu_rx_hoh.rzbit_pid[(bp+1)%2].reg |= 0x01;
      }
    /* Update local copy for debugging purposes */
    cu_reg_copy.tzbit_pid[(bp+1)%2].reg = cu_rx_hoh.rzbit_pid[(bp+1)%2].reg;

    /* Set a pointer to the channel unit hardware */
    tx_wr_ptr = get_tx_wr_ptr((bp+1)%2);

    /* Update the channel unit value */
    tx_wr_ptr->tzbit_1 = cu_reg_copy.tzbit_pid[(bp+1)%2].reg;
    return;
}
#endif /* CU_2E1 */

#ifdef CU_3E1
/*
 * Function: _Set_3E1_PairID1( BP_U_8BIT bp )
 * Parameter: bp is the master loop.
 *
 * This function initially sets the pair IDs for all the 3 loops.
 * If the master is not reversed, nothing will be done and the default loop pair ID 
 * will be used. If the master is reversed, it assumes that only two loops are reversed 
 * (including the master) while the 3rd loop's pair ID remains unchanged.
 *
 *                  Case 1      Case 2      Case 3
 * ---------------------------------------------------
 * Master Loop      (1,1)        (1,2)       (1,3) 
 * Non_master       (2,2)        (2,1)       (2,2)
 * Non_master       (3,3)        (3,3)       (3,1)
 *
 * If the master's pair ID matches its physical location, it means the master loop is
 * not reversed. Now since the other two loops haven't reached normal operation yet,
 * the mapping is set assuming that none of the loops are reversed (Case1). If its pair
 * ID doesn't match its physical location, it means that the master loop is switched. 
 * The mapping is then set assuming one loop reversal happened between two of the 3
 * loops and the 3rd loop remains unreversed (Case2 & 3). Note that at this time, since
 * the other two loops haven't reach normal operation yet, the mapping for them may not
 * be correct.
 */
void _Set_3E1_PairID1( BP_U_8BIT bp )
{
    BP_U_8BIT temp;   
    TX_WR BP_XDATA *tx_wr_ptr;

    temp = cu_rx_hoh.rzbit_pid[bp].reg & 0x07;

    /* The default pair ID can be calculated by 0x01<<bp in which "bp" tells the
       bitpump's physical location */
    if ( temp != (0x01<<bp) )
    {
        /* Case 2&3: Loop reversal on master loop */

        /* Clear the receive pair ID of the the loop that switched with master 
           Note: the loop's # can be calculated using the default pair ID:

           Loop # = Default Pair ID / 2;

                       Default Pair ID
           --------------------------- 
           Loop 0           0x01
           Loop 1           0x02
           Lopp 2           0x04
        */
        cu_rx_hoh.rzbit_pid[temp/2].reg &= 0xF8;

        /* Set that loop's pair ID = master's default pair ID 
           Update the rzbit and tzbit in cu_reg_copy */
        cu_rx_hoh.rzbit_pid[temp/2].reg ^= (0x01<<bp);
        cu_reg_copy.tzbit_pid[temp/2].reg = cu_rx_hoh.rzbit_pid[temp/2].reg;
        tx_wr_ptr = get_tx_wr_ptr(temp/2);

        /* Transmit the udpated pair ID */
        tx_wr_ptr->tzbit_1 = cu_reg_copy.tzbit_pid[temp/2].reg;
    }
    return;
}

/*
 * Function: _Set_3E1_PairID2( BP_U_8BIT bp )
 * Parameter: bp is the master loop.
 *
 * This function sets the pair IDs for all the 3 loops after 2 of the 3 loops are
 * in normal operation. At that time, the paid IDs for each loop can be uniquely
 * determined.
 *
 * _CuReverseLoops() is called to change the mapping based on pair IDs.
 */
void _Set_3E1_PairID2( BP_U_8BIT bp )
{
    if (rate_index == _3E1 )
      {
      /* Master Loop = bp = 0, 1 or 2. Check loop 1, 2 or 0 */
      if (system_status[(bp+1)%3].bits.activation_state == GOTO_ACTIVE_TX_RX_STATE)
         {
         /*Check for loop reversal and set a flag used for identifications*/
         if ((cu_rx_hoh.rzbit_pid[(bp+1)%3].reg & 0x07)!= (0x01 << ((bp+1)%3)) )
            {
            _CuFlags._CuLoopsReversed =1;
            }
         
         /* Clear the current pair ID */
         cu_rx_hoh.rzbit_pid[(bp+2)%3].reg &= 0xF8;
       
         /* Make it to the pair ID that is not used by other loops */
         cu_rx_hoh.rzbit_pid[(bp+2)%3].reg ^= ( 0x07 ^ PID_usage);
         _CuReverseLoops();
         }
      else
         {
         /* Master Loop = bp = 0, 1 or 2. Check loop 2, 0 or 1 */
         if ( system_status[(bp+2)%3].bits.activation_state == GOTO_ACTIVE_TX_RX_STATE )
            {
            /*Check for loop reversal and set a flag used for identifications*/
            if ((cu_rx_hoh.rzbit_pid[(bp+2)%3].reg & 0x07) != (0x01 << ((bp+2)%3)) )
               {
               _CuFlags._CuLoopsReversed =1;
               }

            /* Clear the current pair ID */
            cu_rx_hoh.rzbit_pid[(bp+1)%3].reg &= 0xF8;

            /* Make it to the pair ID that is not used by other loops */
            cu_rx_hoh.rzbit_pid[(bp+1)%3].reg ^= ( 0x07 ^ PID_usage);
            _CuReverseLoops();
            }
         }
      }
}
#endif /* CU_3E1 */

#ifdef CU_2T1
void _Set_2T1_SyncWord( BP_U_8BIT bp )
{
    BP_U_8BIT temp;
    RX_RD BP_XDATA *rx_rd_ptr;
    TX_WR BP_XDATA *tx_wr_ptr;
    RSTATUS_1 rstatus_1; 

    switch ( bp )
    {
        case 0:
             rx_rd_ptr = &(cu_rd->rx_rd_loop1);
             tx_wr_ptr = &(cu_wr->tx_wr_loop1);
             break;
        case 1:
             rx_rd_ptr = &(cu_rd->rx_rd_loop2);
             tx_wr_ptr = &(cu_wr->tx_wr_loop2);
             break;
        default:    /* skip if bit-pump isn't present */
             return;
    }
    rstatus_1.reg = rx_rd_ptr->rstatus_1; 
    cu_reg_copy.tcmd_1[bp].bits.sync_sel = rstatus_1.bits.sync_ab;
    _CU_DUMMY_WRITE(cu_reg_copy.tcmd_1[bp].reg);
    tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[bp].reg;
                            
    temp = cu_reg_copy.tcmd_1[bp].bits.sync_sel;
    if ( temp == 0x00 ) /* Detected SYNC_WORD_A*/
      {
      /* Look to see if second loop detected SYNC_WORD_A  */
      if (bp == 1) 
         {
         _CuFlags._CuLoopsReversed = 1;
         }
      cu_reg_copy.tcmd_1[(bp+1)%2].bits.sync_sel = 0x01;
      }
    else /* Detected SYNC_WORD_B */
      {
      /* Look to see if first loop detected SYNC_WORD_B  */
      if (bp == 0) 
         {
         _CuFlags._CuLoopsReversed = 1;
         }
      cu_reg_copy.tcmd_1[(bp+1)%2].bits.sync_sel = 0x00;
      }                           

    switch ( (bp+1)%2 )
      {
      case 0:
         tx_wr_ptr = &(cu_wr->tx_wr_loop1);
         break;
      case 1:
         tx_wr_ptr = &(cu_wr->tx_wr_loop2);
         break;
      default:    /* skip if bit-pump isn't present */
         return;
    }
                        
    _CU_DUMMY_WRITE(cu_reg_copy.tcmd_1[(bp+1)%2].reg);
    tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[(bp+1)%2].reg;
    return;
}
#endif /* CU_2T1 */
#endif /* HTUR */

/*
 * Function:    _Switch_MasterLoop( BP_U_8BIT loop )
 * Parameter:   loop is the current master loop (before switching).
 *              This function switches the master loop to the next available good loop
 *              when it fails.
 * Note:        For 2T1 or 2E1, if there is any channel blocking, _Configure_Channel_Blocking 
 *              will be called after master loop switch.
 *              For 3E1, since there is no such requirement, channel blocking is not handled.
 */
void _Switch_MasterLoop ( BP_U_8BIT loop )
{
    if ( rate_index == _2T1 || rate_index == _2E1 )
        {
        _BtControl( _CU_COMMON, _CU_SET_MASTER_LOOP, (loop+1)%2 );

        /* 
         * Handle any channel blocking
         * timeslot contains previous timeslot usage information
         */
#ifdef CHANNEL_BLOCK
        old_timeslot = 0xFFFFFFFF;
        _Configure_Channel_Blocking();
        old_timeslot = timeslot;
        /* Since _CU_SET_MASTER_LOOP calls _CuWriteMapRouteCombine(),
         * _CuWriteMapRouteCombine() only needs to be called again
         * if _Configure_Channel_Blocking() is called.
         */
        _CuWriteMapRouteCombine();
#endif /*CHANNEL_BLOCK*/
        }

#ifdef CU_3E1
    if ( rate_index == _3E1 )
        {
       _BtControl( _CU_COMMON, _CU_SET_MASTER_LOOP, (loop+1)%3 );
        }
#endif /* CU_3E1 */
}

#endif /* REPEATER */

/*
 * Function: _Reset_Pid_Validation
 *
 * Usage:   This function is used to reset the PID reservation.
 *          The valid bit is set to 1 so that the PID validation procedure won't 
 *          be called until the loop reaches GOTO_PID_VALIDATION_STATE.
 */
void _Reset_Pid_Validation (BP_U_8BIT bp)
{
   if ( cu_reg_copy.cmd_1.bits.e1_mode )
      {
      cu_rx_hoh.rzbit_pid[bp].bits.valid = 1;
      if ( expect_pid[bp] == 0x01 )
         {
         PID_usage &= 0xFE;
         }
      if ( expect_pid[bp] == 0x02 )
         {
         PID_usage &= 0xFD;
         }
      if ( expect_pid[bp] == 0x04 )
         {
         PID_usage &= 0xFB;
         }
      }
}
#endif /* CHAN_UNIT */
