/***********************************************************/
/*    DSL_MAIN.C                                           */
/*    Main program for HDSL EVM                            */
/*    (C) Copyright 1996 by Brooktree Corporation          */
/*                                                         */
/*    This program is copyrighted by Brooktree Corporation */
/*                                                         */
/* Description:                                            */
/*    This is the main routine for the integrated bitpump  */
/*    control software and the user application software.  */
/*                                                         */
/*                                                         */
/* User Modifiable Code:                                   */
/*    Contents of main() routine.                          */
/*                                                         */
/* List of Functions found in this module:                 */
/*                                                         */
/*    main()                                               */
/*    _EnableBitpump()                                     */
/*    _ConfigureBitpump()                                  */
/*    _HandleTestModes()                                   */
/*                                                         */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/* Mar 97 - New: Dean Rasmussen                            */
/*          Bt8970 Support                                 */
/*          Implemented Startup Activation State Diagram.  */
/*                                                         */
/***********************************************************/


#include "dsl_incl.h"

/*---------------------------------------------------------*/
/*  Local Macros                                           */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

/*
 * Dip Switch Mode definitions
 */                           
#define _DIP_SW_NORMAL_OP           0
#define _DIP_SW_ISO_P3              1
#define _DIP_SW_SCR_4LEVEL          2
#define _DIP_SW_BP_INT_ANALOG_LB    3
#define _DIP_SW_BP_NEAR_DIG_LB      4
#define _DIP_SW_CU_PP_LB            5
#define _DIP_SW_LIU_LINE_LB         6
#define _DIP_SW_UNFRAMED            7

#ifdef CHAN_UNIT
#ifdef CU_LED
/*
 * LED definitions
 */
#define LIC_STARTUP_LED     0x08
#define LIC_ERROR_LED       0x01
#endif /* CU_LED */
#endif /* CHAN_UNIT */ 

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/
DIP_SW APPL_SW_MSPACE dip_sw;
#ifdef ZIPSOCKET
PORT3 APPL_SW_MSPACE port_3;
#endif /*ZIPSOCKET*/
BP_U_32BIT APPL_SW_MSPACE last_count[_NO_OF_LOOPS];
SYSTEM_STATUS APPL_SW_MSPACE system_status[_NO_OF_LOOPS];
SYSTEM_FLAGS APPL_SW_MSPACE system_flags[_NO_OF_LOOPS];
DSL_STATUS2 APPL_SW_MSPACE dsl_status2;

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/
#ifdef TDEBUG
BP_U_32BIT APPL_SW_MSPACE startup_count[_NO_OF_LOOPS];
#endif /* TDEBUG */


/*
 * FUNCTION:   main
 *
 * PARAMETERS: none
 *
 * PURPOSE:    
 *             
 * RETURN:     never
 *
 * NOTES:      
 *
 * CHANGES:    
 */
void main (void)
{
    BP_S_8BIT bp;
#ifndef ZIPSOCKET
#ifndef INT_BUG
    GEN_PURPOSE_TIMER_STATUS stat;
#endif
#endif /* ZIPSOCKET */

   
    while ( FOREVER )
        {

        _DSLInitialization();
  
       /*
        * If HTU-C or HTU-R only is compiled, then force terminal type
        * default is to use dip switches.
        */
#if defined (HTUC) && !defined (HTUR)
        dip_sw.bits.terminal_type = _HTUC;
#endif /* HTUC only */

#if defined (HTUR) && !defined (HTUC)
        dip_sw.bits.terminal_type = _HTUR;
#endif /* HTUR only */


        /*
         * Configure EVMs, must perform in this order
         *
         *      - Enable Bitpump
         *      - Init/Config CU EVM
         *      - Configure Bitpump
         *      - Handle Test Modes
         */
        for ( bp = 0 ; bp < _NO_OF_LOOPS ;  bp++ )
            {
            _EnableBitpump(bp);
            }

#ifdef CHAN_UNIT
        _InitChannelUnitEvmBoard();
#endif

        for ( bp = 0 ; bp < _NO_OF_LOOPS ;  bp++ )
            {
            _ConfigureBitpump(bp);
            _HandleTestModes(bp);


#ifdef SER_COM
            /* Disable state machines for UIP mode */
            if ( dip_sw.bits.pc_control )
                {
                system_status[bp].bits.activation_state = SYSTEM_IDLE;
#ifdef CHAN_UNIT
                _CuFlags._CuAutoRestart = 0;
#endif
                }
#endif

            }


#ifndef ZIPSOCKET
#ifndef INT_BUG
        /* wait a second to press far-end reset button
         * timer set @ 50mS increments
         */
        _EnableGenPurposeTimer(0, PENDING_DEACTIVATE_TIMER, 20);

        do
            {
            stat.reg = _GetGenPurposeTimerStatus(0, PENDING_DEACTIVATE_TIMER);
            }
        while ( !stat.bits.complete );

        _DisableGenPurposeTimer(0, PENDING_DEACTIVATE_TIMER);
#endif /* INT_BUG */
#endif /* ZIPSOCKET */

    
        while ( !dsl_status2.bits.system_reset )
            {         
            for ( bp = 0 ; bp < _NO_OF_LOOPS ;  bp++ )
                {
                _BtMain(bp);
                _ActivationStateManager(bp);
                }

#ifdef CHAN_UNIT
            /* This flag indicates stand-alone versus UIP mode.  */
            if ( _CuFlags._CuAutoRestart )
                {
#ifndef REPEATER
        /* Repeater software only supports 1 loop, so loop handler
         * is not needed 
         */
                _DSLLoopHandler();
#endif  /* REPEATER */
                }
               _CuHandleFlags();

#endif /* CHAN_UNIT */


#ifdef ZIPSOCKET
            P1 = dip_sw.port1;
#endif /* ZIPSOCKET */

#ifdef SER_COM
            /*--------------------------------------------------------------*/
            /* Serial communication active: handle received and transmitted */
            /* message buffers.                                             */
            /*--------------------------------------------------------------*/
            _HandleTransmitMessages();
            _HandleReceiveMessages();
#endif /* SER_COM */

            } /* end while (!dsl_status2.bits.system_reset) */ 
        } /* end while FOREVER */

} /* END main() */

/*
 * FUNCTION:   _EnableBitpump
 *
 * PARAMETERS: bp - which bitpump;  _BIT_PUMP0/1/2
 *
 * PURPOSE:    This function Enables the specified bitpump.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _EnableBitpump (BP_U_8BIT bp)
{
#ifdef TDEBUG
    BP_S_8BIT temp;
#endif

    if ( system_status[bp].bits.present == _NOT_PRESENT )
        {
        return ;
        }

    _BtControl(bp, _SYSTEM_CONFIG, _PRESENT);

#ifdef TDEBUG
    _BtStatus(bp, _VERSION, _HW_TYPE_VERSIONS, &temp);
    switch ( temp & 0x07 )
        {
        case BT8952:
            printf("Bt8952 detected.\n");
            break;

        case BT8960:
            printf("Bt8960 detected.\n");
            break;

        case BT8970:
            printf("Bt8970 detected.\n");
            break;

        case BT8973:
            printf("Bt8973 detected.\n");
            break;

        default:
            printf("Unknown bit-pump type detected.\n");
            break;
        }

    _BtStatus(bp, _VERSION, _MAJOR_SW_VERSION, &temp);
    printf("%d: Bitpump SW Version %d.", (int)(bp+1), (int)(temp));
    
    _BtStatus(bp, _VERSION, _MINOR_SW_VERSION, &temp);
    printf("%d\n", (int)(temp));
#endif /* TDEBUG */

    return ;
}


/*
 * FUNCTION:   _ConfigureBitpump
 *
 * PARAMETERS: bp - which bitpump
 *
 * PURPOSE:    This function configures the specified bitpump
 *             by issueing the appropriate API commands.
 *
 *             The symbol rate is determined by the dip switches.
 *             
 * RETURN:     _PASS - Successful
 *             _FAIL - Unable to detect Bt8960 Bit-pump
 *
 * NOTES:      
 *
 * CHANGES:    March 9, 1997    NEW   -   Dean Rasmussen
 */

BP_U_8BIT _ConfigureBitpump (BP_U_8BIT bp)
{
    BP_U_16BIT sym_rate;

    if ( system_status[bp].bits.present == _NOT_PRESENT )
        {
        return _PASS;
        }

#ifdef ZIPSOCKET
    /*---------------------------------------------------------*/
    /*  ZipSocket Module Configuration                         */
    /*---------------------------------------------------------*/
    if ( dsl_status2.bits.cu_present )
        {
        switch ( dip_sw.zip_bits.t1_e1_mode )
            {
            case T1_MODE: /* 2T1 mode */
                sym_rate = 98;
                break;
            case E1_MODE: /* 2E1 mode */
                sym_rate = 146;
                break;
            }
        }
#else /* ZIPSOCKET flag is not defined */
    /*---------------------------------------------------------*/
    /*  EVM Configuration                                      */
    /*---------------------------------------------------------*/

    switch ( dip_sw.bits.symbol_rate )
        {
        case HALF_T1:       /* 784kbps */
            sym_rate = 98;
            break;

        case HALF_E1:       /* 1168kbps */
            sym_rate = 146;
            break;

        case FULL_T1:       /* 2320kbps for 8973, 1552 for 8970 so th ename stayed the same */
            sym_rate = 290; /* 290 is the selection for 2320kbps, 194 is the selection for 1552 */
            break;

        case EIGHT_CHANNEL:
            /* 144kbps (bitpump only)
             * 1552kbps (chanunit)
             */
            if ( dsl_status2.bits.cu_present )
                {
                sym_rate = 194;
                }
            else
                {
                sym_rate = 18;
                }
            break;
        } /* end switch symbol rate */
#endif /* else not ZIPSOCKET */

#ifdef SP_API
    if (sngl_loop_api)
      {
      sym_rate = cust_sym_rate;
      }
#endif /*SP_API*/

    _BtControl(bp, _SYM_RATE_HI, HIGH(sym_rate));
    _BtControl(bp, _SYM_RATE, LOW(sym_rate));
#ifdef TDEBUG
    HTU_PREFIX(bp);
    printf("Data Rate = %dkbps\n", (int)(sym_rate * 8) );
#endif

#ifdef REPEATER   
    if(bp == 0)              /* RegR */
        {
        _BtControl( _BIT_PUMP0, _TERMINAL_TYPE, _HTUR );
        }
    else                  /* RegC */
        {
        _BtControl( _BIT_PUMP1, _TERMINAL_TYPE, _HTUC );                
        _BtControl( _BIT_PUMP1, _REGENERATOR_MODE, ON );                
        }
#else
    /* Set the bitpump variable: user_setup_low.bits.terminal_flag
     * and configure the bitpump registers that depend on the terminal type.
     */
    _BtControl(bp, _TERMINAL_TYPE, dip_sw.bits.terminal_type);
#endif  /* REPEATER */
    _BtControl(bp, _ANALOG_AGC_CONFIG, _SIX_LEVEL_AGC);

    if ( dsl_status2.bits.cu_present )
        {
        /* Setup the bitpump to use an external S1 signal from the channel unit
         * in ACTIVATING_STATE_S1.  The channel unit will generate the S1 
         * (scrambled all ones) data.  The bitpump transmits this data as a four 
         * level signal.
         */ 
        _BtControl(bp, _STARTUP_SEQ_SOURCE, _EXTERNAL);

        /* Turn off the bitpump scramblers.  The channel unit
         * scramblers are later turned on.  The bitpump scramblers
         * are not needed and would also prevent tip/ring reversal
         * detection by the channel unit.
         */
        _BtControl(bp, _TRANSMIT_SCR, _BYPASS);
        _BtControl(bp, _RECEIVE_DESCR, _BYPASS);

        /* The channel unit will do tip/ring reversal in hardware.
         * Turn off the bitpump software to detect tip/ring reversal 
         */
        _BtControl(bp, _AUTO_TIP_RING, OFF);
        
        /* The LOS timeout and activation timers are specified by
         * the HDSL specifications.
         */
        _BtControl(bp, _LOST_TIME_PERIOD, 10);  /* 1 second */
        _BtControl(bp, _ACTIVATION_TIMEOUT, _ACT_TIME_30SEC);
        }
    else
        {
        /* Setup the bitpump to use an internally generated S1 signal */
        _BtControl(bp, _STARTUP_SEQ_SOURCE, _INTERNAL);

        /* Assume that external data will not be scrambled, so need to
         * use the bitpump scrambler and descrambler.
         */
        _BtControl(bp, _TRANSMIT_SCR, _ACTIVATE_SCR);
        _BtControl(bp, _RECEIVE_DESCR, _ACTIVATE_DESCR);

        /* Setup the bitpump to detect tip/ring reversal */
        _BtControl(bp, _AUTO_TIP_RING, ON);
        _BtControl(bp, _LOST_TIME_PERIOD, 40);  /* 4 seconds */

        /* Set the activation timeout as a function of the data rate */
        _BtControl(bp, _ACTIVATION_TIMEOUT, _ACT_TIME_VARIABLE);
        }

#ifdef REPEATER
    if (bp == 0)  /* RegR */
       {
       _BtControl(bp, _FRAMER_FORMAT, _SERIAL_SWAP);
       }
    else /* RegC */
       {
       _BtControl(bp, _FRAMER_FORMAT, _PARALLEL_SLAVE);
       }
#else
    _BtControl(bp, _FRAMER_FORMAT, _SERIAL_SWAP);
#endif /* REPEATER */
 
    /* Assume the other side is a BrookTree (now Conexant) device 
     * This assumption allows a faster startup.
     */
    _BtControl(bp, _BT_OTHER_SIDE, _BT);

#ifdef ZIP_START
    /* Tell the ASM that ZipStartup is enabled.  If a cold startup occurs
     * and the DSP coefficients are saved, then the next startup will be 
     * a Zip start.
     */
    _BtControl(bp, _ZIP_START_CONFIG, zip_system_status[bp].bits.enabled);
#else
    _BtControl(bp, _ZIP_START_CONFIG, 0);

#endif

#ifdef SP_API
    if (sngl_loop_api)
        {
            _BtControl(bp, _ACTIVATION_TIMEOUT, _ACT_TIME_VARIABLE);
        }    
#endif /*SP_API*/

    return _PASS;
} /* end _ConfigureBitpump */

/*
 * FUNCTION:   _HandleTestModes
 *
 * PARAMETERS: bp - which bitpump;  _BIT_PUMP0/1/2
 *
 * PURPOSE:    This function Switches and performs any test modes that
 *             might be selected.
 *             
 * RETURN:     nothing
 *
 * NOTES:      Test Modes Supported
 *
 *             Dip Switches    
 *             3   4   5       Description
 *             -----------------------------------------
 *             0   0   0       Normal Operation
 *             0   0   1       BP Isolated +3
 *             0   1   0       BP Cont 4 Level
 *             0   1   1       BP Analog Loopback
 *             1   0   0       BP Digital Loopback
 *             1   0   1       CU PRA to PRA Loopback (1)
 *             1   1   0       LIU Line Loopback (1)
 *             1   1   1       Normal Operation - Unframed (1 & 2)
 *
 *             Notes:
 *             (1)  Required the Channel Unit Board
 *             (2)  When using the System Unframed Mode, the code continues
 *                  to perfrom startup.  The code does not support
 *                  Unframed Mode along with the other LB modes.
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _HandleTestModes (BP_U_8BIT bp)
{
    if ( system_status[bp].bits.present == _NOT_PRESENT )
        {
        return ;
        }

#ifdef ZIPSOCKET
    return;
#else /* No ZIPSOCKET complier flag */

#ifdef TDEBUG
#ifdef ERLE
    /* When running TDEBUG, UIP is not supported, so can use the 
     * UIP dip switch to run ERLE.
     */
    if ( dip_sw.bits.pc_control )
        {
        system_status[bp].bits.activation_state = BACKGROUND_TESTMODE;
        return ;
        }
#endif /* ERLE */
#endif /* TDEBUG */

    switch ( dip_sw.bits.testmode )
        {
        default:
        case _DIP_SW_NORMAL_OP:

            /*
             * If Channel Unit is present, setup for framed data,
             * otherwise just use default (current) settings
             */

#ifdef CHAN_UNIT
            if ( !dsl_status2.bits.cu_present )
                {
                break;
                }

            _BtControl(_CU_COMMON, _CU_FRAME_FORMAT, _CU_FRAMED_DATA);
#ifdef T1E1_FRAMER
            _BtControl(_FRAMER, _FRAMER_FRAME_FORMAT, _CU_FRAMED_DATA);
#endif /* T1E1_FRAMER */
#endif /* CHAN_UNIT */
            break;

        case _DIP_SW_ISO_P3:
            _BtControl(bp, _TEST_MODE, _ISOLATED_PULSE_PLUS3);
            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;


        case _DIP_SW_SCR_4LEVEL:
            _BtControl(bp, _TEST_MODE, _FOUR_LEVEL_SCR);
            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;


        case _DIP_SW_BP_INT_ANALOG_LB:
            _BtControl(bp, _TEST_MODE, _ISOLATED_ANALOG_LOOPBACK);

#if 0
            /* other options for this dip switch */
            _BtControl(bp, _TEST_MODE, _INTERNAL_ANALOG_LOOPBACK);
            _BtControl(bp, _TEST_MODE, _ANALOG_LOOPBACK);
#endif
            system_status[bp].bits.activation_state = ANALOG_LB_TESTMODE;
            break;


        case _DIP_SW_BP_NEAR_DIG_LB:
            _BtControl(bp, _TEST_MODE, _NEAR_LOOPBACK);

#ifdef CHAN_UNIT
            if ( dsl_status2.bits.cu_present )
                {
                _BtControl(_CU_CHAN1 + bp, _CU_USE_SAME_TAP, _CU_TAP_5);
                _CuConfigureTransmitS1(bp);
                _BtControl(_CU_CHAN1 + bp, _CU_TRANSMIT_PAYLOAD, 0);
                }
#endif /* CHAN_UNIT */

            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;


#ifdef CHAN_UNIT
        case _DIP_SW_CU_PP_LB:
            if ( !dsl_status2.bits.cu_present )
                {
                break;
                }

            _BtControl(_CU_COMMON, _CU_LOOPBACK, _CU_PRA_ON_PRA);
            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;


        case _DIP_SW_LIU_LINE_LB:
            if ( !dsl_status2.bits.cu_present )
                {
                break;
                }

            /*
             * LIU Line Loopback.  Need to set the Channel Unit PRA Loopback on
             * PRA side during LIU loopback so EXCLK will be valid.
             */
            _BtControl(_CU_COMMON, _CU_LOOPBACK, _CU_PRA_ON_PRA);
#ifdef T1E1_FRAMER
            _BtControl(_FRAMER, _LIU_LINE_LOOPBACK, 0);
#endif /* T1E1_FRAMER */

            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;

        case _DIP_SW_UNFRAMED:
            if ( !dsl_status2.bits.cu_present )
                {
                break;
                }

            _BtControl(_CU_COMMON, _CU_FRAME_FORMAT, _CU_UNFRAMED_DATA);
#ifdef T1E1_FRAMER
            _BtControl(_FRAMER, _FRAMER_FRAME_FORMAT, _CU_UNFRAMED_DATA);
#endif /* T1E1_FRAMER */
            break;
#endif /* CHAN_UNIT */

        }   /* end switch testmodes */
    return;
#endif /* ZIPSOCKET */

}   /* end _HandleTestModes() */



