/***********************************************************/
/*    DSL_INIT.C                                           */
/*    (C) Copyright 1998 by Rockwell Semiconductor Systems */
/*                                                         */
/*                                                         */
/* Description:                                            */
/*    This is HDSL initialization functions.               */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*                                                         */
/* List of Functions found in this module:                 */
/*                                                         */
/*    _IsChannelUnitEvmPresent                             */
/*    _InitChannelUnitEvmBoard                             */
/*    _DSLInitialization                                   */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/* Aug 98 - New: Dean Rasmussen                            */
/* Oct 98 - New: Laura Yuan                                */
/*                                                         */
/***********************************************************/
#include "dsl_incl.h"
#ifdef CHAN_UNIT
/*
 * FUNCTION:   _IsChannelUnitEvmPresent
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function checks to see if the Channel Unit EVM Board
 *             is present.
 *
 *             Sequence of events:
 *
 *                  - check if CU EVM Board & Chips are present
 *                  - reset the LEDs
 *
 *             If an error occurs, set the CU_ERROR_LED.
 *             
 * RETURN:     _PRESENT
 *             _NOT_PRESENT
 *
 * NOTES:      
 *
 * CHANGES:    July 8, 1994    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _IsChannelUnitEvmPresent (void)
{
    BP_U_8BIT temp;

    _BtStatus(_CU_COMMON, _CU_PRESENT, 0, (BP_S_8BIT *)&temp);
    dsl_status2.bits.cu_present = temp;

    if ( dsl_status2.bits.cu_present == _NOT_PRESENT )
        {
        return _NOT_PRESENT;
        }

    _BtStatus(_CU_COMMON, _CU_VERSION, 0, (BP_S_8BIT *)&temp);

#ifdef CU_LED
    if ( temp == 0 )
        {
        _BtControl(_FRAMER, _CU_LED_BLOCK_SET, 0xF);
        _BtControl(_CU_COMMON, _CU_LED_BLOCK_SET, 0xF);
        return _NOT_PRESENT;
        }

    /*
     * Clear the LEDs
     */
    set_LED_block(_BP_LED_BLOCK1_2, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_BP_LED_BLOCK3_SP, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_MISC_TERM, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_MISC_TEST_P, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_CU_LED_CHAN1, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_CU_LED_CHAN2, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_CU_LED_CHAN3, _CU_LED_BLOCK_SET, 0x0);
    set_LED_block(_CU_PCM_COMMON, _CU_LED_BLOCK_SET, 0x0);

#endif /* CU_LED */

    return _PRESENT;
}


/*
 * FUNCTION:   _InitChannelUnitEvmBoard
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function initializes the Channel Unit EVM Board.
 *
 *             Sequence of events:
 *
 *                  - determine framer type
 *                  - determine the EVM configuration
 *                  - initialize Channel Unit
 *                  - initialize Framer
 *                  - initialize LIU
 *
 *             If an error occurs, set the CU_ERROR_LED.
 *             
 * RETURN:     _PASS (0) - Successfully Completed Operation
 *             _FAIL (1) - An error occured
 *
 * NOTES:      Only the 2T1 & 2E1 systems are currently supported on
 *             the Bt8970 EVMs.  However, there are experimental hooks
 *             for 3E1, 1T1, & 1E1.
 *
 * CHANGES:    March 9, 1997    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _InitChannelUnitEvmBoard (void)
{
    BP_S_8BIT system_type;
#ifdef T1E1_FRAMER
    BP_S_8BIT framer, frmr_mode; 
#endif /* T1E1_FRAMER */

    if ( dsl_status2.bits.cu_present == _NOT_PRESENT )
        {
        return _PASS;
        }

#ifdef T1E1_FRAMER
    /*
     * Use the _FRAMER_PRESENT command to find out what type of framer
     * is present so we can differentiate between E1/T1
     */

    _BtStatus(_FRAMER, _FRAMER_PRESENT, 0, &framer);
    if ( framer == _NOT_PRESENT )
        {
        _BtControl(_FRAMER, _CU_LED_BLOCK_SET, 0xF);
        return _FAIL;
        }

    switch ( num_bit_pumps )
        {

        case 1:
        case 2:
             if ( dip_sw.bits.symbol_rate == 0 )
                {
                system_type = _2E1;
                frmr_mode = _FRAMER_E1_MODE;
                }
            else if ( dip_sw.bits.symbol_rate == 1 ) 
                {
                system_type = _2T1;
                frmr_mode = _FRAMER_T1_MODE;
                }
            else if ( dip_sw.bits.symbol_rate == 2 ) 
                {
                system_type = _1T1;
                frmr_mode = _FRAMER_T1_MODE;
                }

            else if ( dip_sw.bits.symbol_rate == 3 ) 
                {
                system_type = _1E1;
                frmr_mode = _FRAMER_E1_MODE;
                }
            break;

      
        case 3:
            system_type = _3E1;
            frmr_mode = _FRAMER_E1_MODE;
            break;
        }


#else   /* No T1E1_FRAMER is defined */

#ifdef ZIPSOCKET
    if ( dip_sw.zip_bits.t1_e1_mode == T1_MODE )
        {
        system_type = _2T1;
        }
    else
        {
        system_type = _2E1;
        }
#endif /* ZIPSOCKET */

#endif /* T1E1_FRAMER */

#ifdef TDEBUG
      switch ( system_type )
          {
          case _1T1:
              printf("Channel Unit Configured for 1T1\n");
              break;

          case _1E1:
              printf("Channel Unit Configured for 2E1\n");
              break;

          case _2T1:
              printf("Channel Unit Configured for 2T1\n");
              break;

          case _2E1:
              printf("Channel Unit Configured for 2E1\n");
              break;

          case _3E1:
              printf("Channel Unit Configured for 3E1\n");
              break;
          }
#endif /* TDEBUG */

      

    /*
     * For older LIUs, such as the Bt8069, must Init Channel Unit 
     * first since the LIU requires the EXCLK
     * to be valid during LIU configuration
     *
     * Set the global variable, htu_index.  This variable is used
     * by the channel unit software to differentiate between HTUC and HTUR.
     */
    _BtControl(_CU_COMMON, _CU_TERMINAL_TYPE, dip_sw.bits.terminal_type);

    /*
     * CU Terminal Type must be set before can Configure
     *
     * Call _CU_CONFIGURE to:
     * 1. Set the global variable, rate_index.
     * 2. Set the master loop 2 bits, _CuFlags._CuMasterLoop = bp_position[0] (LIC #1).
     * 3. Call _CuInitChannelUnit();
     */
     
      _BtControl(_CU_COMMON, _CU_CONFIGURE, system_type);

#ifdef T1E1_FRAMER
    /* When Bt8370 present, need to set 6mS multiframe */
    if ( framer == _BT8370 )
        {
        _BtControl( _CU_COMMON, _CU_SET_MFRAME, _6MS_MFRAME );
        }

    _BtControl(_FRAMER, _FRAMER_PCM_MODE, frmr_mode);
    _BtControl(_FRAMER, _FRAMER_CONFIGURE, framer);


#ifdef BT8069_LIU
    if ( frmr_mode == _FRAMER_E1_MODE )
        {
        _BtControl(_FRAMER, _LIU_CONFIGURE, _ELS_E1_PCM30);
        }
    else
        {
        _BtControl(_FRAMER, _LIU_CONFIGURE, _ELS_T1_0_110);
        }
#endif /* BT8069_LIU */



#endif /* T1E1_FRAMER */
    return _PASS;
}

#endif /* CHAN_UNIT */

/*
 * FUNCTION:   _DSLInitialization
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function initializes HDSL system.
 *
 *             The following initialization are done:
 *              -   Software initialization upon power up
 *              -   DIP switch initialization
 *              -   8051 initialization
 *              -   Important global parameter initialization
 *              -   Check the presence of bitpumps and channel unit
 *
 *             If an error occurs, set the CU_ERROR_LED for EVM system.
 *             If an error occurs, set the ALARM LED for ZIPSOCKET system.  
 *
 * CHANGES:    Oct.6, 1998    Laura Yuan
 */
void _DSLInitialization (void)
{
    BP_U_8BIT  bp, temp;

    dsl_status2.reg = 0;

    for ( bp = 0 ; bp < _NO_OF_LOOPS ;  bp++ )
        {
        _BtSwPowerUp(bp); /* Software power-up initialization */

        _MaskBtHomerInt(bp);
        }

#ifdef C51
    _Init8051(); /* Initialize 8051 specific parameters */
#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section BtHomer.1  Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Call user uprocessor initialization routine.             */
    /*                                                          */
    /* Reference:                                               */
    /* "BtHomer Software User's Manual", Section ?.?.1.         */
    /*----------------------------------------------------------*/


    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section BtHomer.1  Ends          <<<*/
    /*----------------------------------------------------------*/
#endif /* C51 */

#ifdef INT_BUG
    _InitVirtualTimers();
#else
    _InitGenPurposeTimer();
#endif

#ifdef CHAN_UNIT
#ifdef C51
    EX1 = 0;        /* Disable EXT1 interrupt pin (Chan Unit Int) */
#endif
    _CuInitAddresses();
#endif

    /*
     * Initialize the Dip Switches
     */
#ifdef C51
    P1 = 0xFF;
    dip_sw.port1 = P1;

#ifdef ZIPSOCKET
    port_3.reg3 = P3;/* SDSL/HDSL Alarm indication Bit*/
    dip_sw.bits.pc_control = STAND_ALONE;
    dip_sw.zip_bits.alarm = ALARM_OFF;
    dip_sw.zip_bits.sync_1 = SYNC_OFF;
    dip_sw.zip_bits.sync_2 = SYNC_OFF;
    dip_sw.zip_bits.test_mode = TEST_OFF; 
    dip_sw.zip_bits.flash_rom = FLASH_OFF;
    P1 = dip_sw.port1; /* update port1 */
#endif /* ZIPSOCKET */

#endif


    /*
     * Must determine which LICs are Present and in which slot
     * so can determine how to initialize the Channel Unit.
     */
#ifdef CHAN_UNIT
    num_bit_pumps = 0;
    timeslot = 0xFFFFFFFF;  /* Use all the timeslots */
    old_timeslot = timeslot;
    PID_usage = 0;
#endif
    
    for ( bp = 0 ; bp < _NO_OF_LOOPS ;  bp++ )
        {
        system_flags[bp].bits.nm_update = 1;
        system_status[bp].reg = 0;


#ifdef ZIP_START
        zip_system_status[bp].reg = 0;
#ifdef HTUC
    rt_zip_system_status[bp] =0;
#endif /*HTUC*/
#endif /*ZIP_START*/

       /*
        * Need to set the 'present' vars for both 'pc_control' & 'reset' code
        */
        _BtStatus(bp, _BIT_PUMP_PRESENT, 0, (BP_S_8BIT *)&temp);
        system_status[bp].bits.present = temp;

#ifdef ZIPSOCKET
      if ( system_status[0].bits.present == _NOT_PRESENT )
         {
         dip_sw.zip_bits.alarm = ALARM_ON;
         }
      if ( system_status[1].bits.present == _NOT_PRESENT )
         {
         if(port_3.bits.hdsl_sdsl)
            {
            dip_sw.zip_bits.alarm = ALARM_ON;
            }
         }
#endif /* ZIPSOCKET */
#ifdef CU_LED
        if ( system_status[bp].bits.present == _NOT_PRESENT )
            {
            cu_led_block[bp].bits.in_sync = SYNC_OFF; 
            cu_led_block[bp].bits.out_of_sync = SYNC_OFF; 
            }
#endif /* CU_LED */

        system_status[bp].bits.activation_state = CONFIGURATION_STATE;

#ifdef REPEATER
       /* In a repeater application, the RegR to COT loop is first
        * activated.  This is required by the HDSL specs and
        * also by the requirement to have a recovered and stable 
        * HCLK clock from the RegR bitpump to drive the RegC bitpump.  
        * When the RegR state manager transitions to ACTIVE_TX_RX_STATE, 
        * it changes the RegC state to INACTIVE_STATE.
        */
        system_status[1].bits.activation_state = SYSTEM_IDLE;
#endif

#ifdef PERF_MONITOR
        available_sec[bp] = 0;
#endif /* PERF_MONITOR */

#ifdef ZIP_START
        zip_system_status[bp].bits.enabled = 1;
#endif


#ifdef CHAN_UNIT
        system_flags[bp].bits.completed = 0;
        bp_position[bp] = -1;   /* default to not present */
        if ( temp ) /* bit-pump present */
            {
            /* setup where the Bit-Pumps are located */
            bp_position[num_bit_pumps] = bp;

            /* 
             * num_bit_pumps is a global variable for the actual number of bitpumps 
             * found in the system.  _NO_OF_LOOPS is a #define for the maximum number 
             * of bitpumps in the system.  Some "for" loops use num_bit_pumps and some
             * "for" loops use _NO_OF_LOOPS.  Some global arrays are sized based on
             * _NO_OF_LOOPS.  To save ROM and RAM, _NO_OF_LOOPS should be set at compile
             *  time to the actual number of bitpumps in the system.
             */
            num_bit_pumps++;
            }
#endif

#ifdef SP_API
        sngl_loop_api = 0;
#endif /*SP_API*/

#ifndef INT_BUG
        _EnableGenPurposeTimer(bp, ONE_SECOND_TIMER, 20);
        _ContinuousGenPurposeTimer(bp, ONE_SECOND_TIMER, 1);
#endif

        }   /* end for bp loop */


#ifdef CHAN_UNIT
    temp = _IsChannelUnitEvmPresent();
#ifdef ZIPSOCKET
    if ( temp == _NOT_PRESENT )
        {
        dip_sw.zip_bits.alarm = ALARM_ON;
        }
    P1 = dip_sw.port1;
#endif /* ZIPSOCKET */
    _CuFlags._CuAutoRestart = 1;
#endif

}
