/***********************************************************/
/*    DSL_ASM.C                                            */
/*    (C) Copyright 1998 by Rockwell Semiconductor Systems */
/*    (C) Copyright 1999 by Conexant Systems               */
/*                                                         */
/*                                                         */
/* Description:                                            */
/*    This is the HDSL activation state manager function.  */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*                                                         */
/* List of Functions found in this module:                 */
/*                                                         */
/*   - _ActivationStateManager                             */
/*                                                         */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/* Aug 98 - New: Dean Rasmussen                            */
/* Oct 98 - New: Laura Yuan                                */
/* Feb 99 - New: Chris Muller                              */
/*                                                         */
/***********************************************************/
#include "dsl_incl.h"


#define WAIT_FOR_LOS_LOST_INTERVAL  30 
#define DECREMENT_ZIP_START_ATTEMPT(bp) \
    if ( system_status[bp].bits.zip_start_attempt <= 1 )\
        system_status[bp].bits.zip_start_attempt = 0;\
    else \
        system_status[bp].bits.zip_start_attempt--;


/*
 * FUNCTION:   _ActivationStateManager
 *
 * PARAMETERS: bp - which bitpump;  _BIT_PUMP0/1/2
 *
 * PURPOSE:    This function manages the system state flow.
 *             It implements a state machine derived from the HDSL standards
 *             (the most recent standard is ITU G.991.1).  This implementation adds
 *             additional states (such as Pair ID Validation State) that are not
 *             shown in the state diagram in the HDSL standard.  Other requirements
 *             require additional states as listed below:
 *
 *                         Requirement                          Added States
 *             =======================================       =================
 *             Standby state for running test modes            SYSTEM_IDLE
 *             and using UIP to control system
 * 
 *             Conexant's particular hardware                  ACTIVATING_STATE_S1
 *             implementation
 *
 *             Maintain synchronization between the            WAIT_FOR_LOST
 *             HTUC and HTUR for restart                       WAIT_FOR_LOS
 *              
 *             PID validation (E1 applications only)           GOTO_PID_VALIDATION_STATE  
 *                                                             PID_VALIDATION_STATE
 *
 *             Delay for DSL_LoopHandler() processing          GOTO_ACTIVE_TX_RX_STATE
 *
 * RETURN:     nothing
 *
 * NOTES:      
 * The HDSL startup can be broken down into these several steps:
 *     Standby:                                     SYSTEM_IDLE 
 *
 *     Configuration and Initialization:            CONFIGURATION_STATE
 *                                                  INACTIVE_STATE
 *
 *     Startup of the bitpump (e.g. Rs8973):        ACTIVATING_STATE    ACTIVATING_STATE_S1
 *                                                 
 *     Startup of the channel unit (e.g. Rs8953B):  ACTIVE_RX_STATE     ACTIVE_TX_STATE
 *                                                  GOTO_PID_VALIDATION_STATE  
 *                                                  PID_VALIDATION_STATE
 *                                                  GOTO_ACTIVE_TX_RX_STATE
 *
 *     Normal operation (both bitpump and channel unit):  ACTIVE_TX_RX_STATE
 *
 *     Error Processing and Shutdown:               PENDING_DEACTIVATED_STATE
 *                                                  DEACTIVATED_STATE
 *                                                  WAIT_FOR_LOST
 *                                                  WAIT_FOR_LOS
 *                                                
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _ActivationStateManager (BP_U_8BIT bp)
{
    status_reg_type bp_status;
    BP_U_32BIT timer_count;
    
            
#ifndef INT_BUG
    GEN_PURPOSE_TIMER_STATUS timer_stat;
#endif
#ifdef CHAN_UNIT
    BP_S_8BIT stage;
    BP_U_8BIT cu_sync_status;
    void _CuConfigureTransmitS1(BP_U_8BIT loop);
    void _CuConfigureBeginStartup(BP_U_8BIT loop);
#endif

#ifdef TDEBUG
#ifdef ERLE
    BP_S_8BIT temp;
    BP_S_8BIT temp2, i;
    BP_S_16BIT slm, felm, value;
    float result;
#endif
#endif
        
    if ( system_status[bp].bits.present == _NOT_PRESENT )
        {
        return ;
        }

    /*
     * Special case, do nothing
     */
    if ( system_status[bp].bits.activation_state == SYSTEM_IDLE )
        {
        return ;
        }

    /* 
     * Obtain the values of bp_status.bits.los and bp_status.bits.nmr_ok. Since
     * a few states need these values, calling the API in one place saves ROM.
     * However, the API call and the noise margin lookup table (in particular)
     * take a relatively long time.  Note that ACTIVE_TX_RX_STATE is the state 
     * for normal operation.  The bitpump status is not checked in ACTIVE_TX_RX_STATE 
     * in applications that use a channel unit.    Therefore, time critical 
     * applications that use a channel unit may want to move and duplicate 
     * this API call to the required states.
     */
    _BtStatus(bp, _STARTUP_STATUS, 0, (BP_S_8BIT *)&(bp_status.status));


#ifdef CHAN_UNIT
    if ( dsl_status2.bits.cu_present )
        {
        _BtStatus(bp + _CU_CHAN1, _CU_SYNC, 0, (BP_S_8BIT *)&cu_sync_status);
    
        /*
         * Stage only used for displaying LEDs
         */
        _BtStatus(bp, _STAGE_NUMBER, 0, &stage);
        }
#endif



    switch ( system_status[bp].bits.activation_state )
        {
        case CONFIGURATION_STATE:
#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("State = Configuration State\n");
#endif

            if ( _ConfigureBitpump(bp) != _PASS )
                {
                system_status[bp].bits.activation_state = SYSTEM_IDLE;
                break;
                }
#ifdef CHAN_UNIT
            _BtControl(_CU_CHAN1 + bp, _CU_USE_SAME_TAP, 0);

            /* Setup the channel unit TCMD_1 for startup.  TCMD_1 is
             * setup to transmit HOH and payload as all ones.  This 
             * command also shuts off the channel unit interrupts.
             */
            _BtControl(_CU_CHAN1 + bp, _CU_FORCE_SCR_ONES, 0);
#endif
            system_status[bp].bits.activation_state = INACTIVE_STATE;
            break;

        case INACTIVE_STATE:

#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("Activating...\n");
#endif 


            /*
             * Configure the Chanel Unit for Scrambled Ones and issue the
             * Bitpump ACTIVATE command
             */

#ifdef CHAN_UNIT
            if ( dsl_status2.bits.cu_present )
               {
               _BtControl(_CU_CHAN1 + bp, _CU_FORCE_SCR_ONES, 0);
               _Reset_Pid_Validation( bp );
               }
#endif /* CHAN_UNIT */

#ifdef TDEBUG
            /* for measuring approx startup times */
            startup_count[bp] = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
#endif /* TDEBUG */

            /*
             * For the HTU-C, assuming the ACTREQ is always TRUE
             */

#ifdef ZIP_START
           /* ZipStartup involves two separate but inter-related processes:
            * 1. _ZipStartValidationManager() determines whether to save the DSP coefficients 
            *    based on the success and startup type (cold or Zip) on the HTUC and HTUR sides
            *    of the current startup.
            * 2. ActivationStateManager() determines whether to attempt a
            *    cold or Zip startup based on the zip_start_attempt variable.
            */
            if ( zip_system_status[bp].bits.enabled )
                {
                zip_system_status[bp].bits.zip_start_state = ZIP_START_VALIDATION_STATE0;
                }
            else
                {
                system_status[bp].bits.zip_start_attempt = 0;
                zip_system_status[bp].bits.zip_start_state = ZIP_START_IDLE;
                }
#endif

            if ( system_status[bp].bits.zip_start_attempt == 0 )
                {
#ifdef TDEBUG
                HTU_PREFIX(bp);
                printf("Cold Startup Attempt...\n");
#endif
                _BtControl(bp, _ACTIVATE, _ACTIVATE_COLD);

                }
#ifdef ZIP_START
            else
                {
#ifdef TDEBUG
                HTU_PREFIX(bp);
                printf("Zip Startup Attempt...\n");
#endif

                _BtControl(bp, _ACTIVATE, _ACTIVATE_ZIP);
                }
#endif
            
            system_status[bp].bits.activation_state = ACTIVATING_STATE;

#ifdef CU_LED
            _Bp_Led_Update(bp, SYNC_OFF);
#endif


#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("State = Activating State\n");
#endif 

            break;

        case ACTIVATING_STATE:

            /*
             * Check for Bitpump Timeouts.
             */
            if ( bp_status.bits.activation_interval )
                {
                system_status[bp].bits.activation_state = DEACTIVATED_STATE;
#ifdef CU_LED
                _Bp_Led_Update(bp, SYNC_OFF); 
#endif
                break;
                }

            if ( !dsl_status2.bits.cu_present )
                {
                if ( bp_status.bits.four_level_indication )
                    {
                    system_status[bp].bits.activation_state = ACTIVATING_STATE_S1;

#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("State = Activating State S1\n");
#endif 

                    }
#ifdef CU_LED
                /*
                 * Set the CU LEDs to the current stage #
                 */
              _BtControl(_CU_CHAN1+bp, _CU_LED_BLOCK_SET, stage>> 2 );


#endif
                }
#ifdef CHAN_UNIT
            else
                {
                /*
                 * Check for Tx 4-Level Indication
                 */
                if ( bp_status.bits.four_level_indication )
                    {
                    /*
                     * Transmit S1:
                     */
                    _CuConfigureTransmitS1(bp);

                    system_status[bp].bits.activation_state = ACTIVATING_STATE_S1;

#ifdef CU_LED
                    _Bp_Led_Update(bp, SYNC_OFF);  
#endif


#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("State = Activating State S1\n");
#endif 

                    }
#ifdef CU_LED
                /*
                 * Set the CU LEDs to the current stage #
                 */
                _BtControl(_CU_CHAN1 + bp, _CU_LED_BLOCK_SET, stage>> 2 );
#endif /* CU_LED */
                }
#endif

            break;


        case ACTIVATING_STATE_S1:

            /*
             * Check for Bitpump Timeout.
             */
            if ( bp_status.bits.activation_interval )
                {
                system_status[bp].bits.activation_state = DEACTIVATED_STATE;    
#ifdef CU_LED
                _Bp_Led_Update(bp, SYNC_OFF);
                _Cu_Led_Update(bp, SYNC_OFF);
#endif

                break;
                }


            /*
             * Need to check for INDC=1 & INDR=1, can employ same
             * algrorithm for both HTU-C & HTU-R.
             *
             * If (NMR O.K. & LOSW=0) ==> Active-Rx State
             *
             * If (Rx RTR Indicator=0) ==> Active-Tx State
             *    - This means that the far-end was able to detect
             *      (NMR O.K. & LOSW=0) and was able to convey that
             *      through the RTR Indicator bit.
             *      
             * Note:  If CU is not present, then only look at NMR O.K. Flag;
             *        therefore can only proceed to Active-Rx State.  It
             *        is not possible to goto Active-Tx State since the
             *        CU is responsible for the RTR Indicator bit.
             */

            if ( !dsl_status2.bits.cu_present )
                {
                if ( bp_status.bits.nmr_ok )
                    {
                    system_status[bp].bits.activation_state = ACTIVE_RX_STATE;
#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("State = Active Rx State\n");
#endif 

                    }
                }
#ifdef CHAN_UNIT
            else 
                {
#ifdef CU_LED
               _BtControl(_CU_CHAN1 + bp, _CU_LED_BLOCK_SET, stage>>2 );
#endif /* CU_LED */
                /* Check if the near side is in sync and NMR is OK */
                if ( cu_sync_status == CU_IN_SYNC && bp_status.bits.nmr_ok )
                    {
                    /*
                     * Need to set RTR indicator bit
                     */
                    _CuSetRtrInd(0, bp);
                    system_status[bp].bits.activation_state = ACTIVE_RX_STATE;
#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("State = Active Rx State\n");
#endif 
                    }

                /* Check if the far side is indicating in sync. */
                if ( cu_rx_hoh.rtr_ind_status[bp].bits.valid )
                    {
                    /*
                     * Turn on 2B1Q data by turning Off Forced Ones payload
                     */
                    _CuForceOnes(OFF, bp);
                    system_status[bp].bits.activation_state = ACTIVE_TX_STATE;
#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("State = Active Tx State\n");
#endif 
                    }

                }
#endif

            break;


        case ACTIVE_RX_STATE:


            /*
             * If the CU is Present, Look for Rx RTR Indicator Bit
             * to determine when to goto Active Tx/Rx State.
             *
             * If the CU is not Present, use the Stage # to determine
             * the current startup state.  Only need to look for 
             * NORMAL_OPERATION condition.
             */
            if ( !dsl_status2.bits.cu_present )
                {
                /*
                 * If the Bitpump Timeouts, Deactivate
                 */
                if ( bp_status.bits.activation_interval )
                    {
                    system_status[bp].bits.activation_state = DEACTIVATED_STATE;
                    break;
                    }


                if ( bp_status.bits.normal_operation )
                    {
                    system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
                    }
                }
#ifdef CHAN_UNIT
            else 
                {
                /*
                 * If the Bitpump Timeouts, continue to Active Tx/Rx State
                 */
                if ( bp_status.bits.activation_interval )
                    {
#ifdef REPEATER
                    system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
#else
                    system_status[bp].bits.activation_state = GOTO_PID_VALIDATION_STATE;
#endif
                    break;
                    }

                if ( cu_rx_hoh.rtr_ind_status[bp].bits.valid )
                    {
#ifdef REPEATER
                    system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
#else
                    system_status[bp].bits.activation_state = GOTO_PID_VALIDATION_STATE;
#endif
                    }
                }
#endif

            break;


#ifdef CHAN_UNIT
        case ACTIVE_TX_STATE:

            /*
             * If the Bitpump Timeouts, continue to Active Tx/Rx State
             */
            if ( bp_status.bits.activation_interval )
                {
#ifdef REPEATER
                system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
#else
                system_status[bp].bits.activation_state = GOTO_PID_VALIDATION_STATE;
#endif
                break;
                }

            /*
             * If the CU is Present, Look for NMR O.K. & LOSW=0
             * to determine when to goto Active Tx/Rx State.
             *
             * Should never get here if the CU is not Present, 
             * however, just goto to next state for precaution.
             */
            if ( !dsl_status2.bits.cu_present )
                {
#ifdef REPEATER
                system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
#else
                system_status[bp].bits.activation_state = GOTO_PID_VALIDATION_STATE;
#endif
                }
            else 
                {
#ifdef CU_LED
                _BtControl(_CU_CHAN1 + bp, _CU_LED_BLOCK_SET, stage>>2 );
#endif /* CU_LED */
                if ( cu_sync_status == CU_IN_SYNC && bp_status.bits.nmr_ok )
                    {
#ifdef REPEATER
                    system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
#else
                    system_status[bp].bits.activation_state = GOTO_PID_VALIDATION_STATE;
#endif
                    }
                }
            break;
#ifndef REPEATER

        case GOTO_PID_VALIDATION_STATE:

            /*
             * This state is a temporary state.  It executes E1 action routines 
             * when either ACTIVE_RX_STATE or ACTIVE_TX_STATE is exited.
             */
            if ( cu_reg_copy.cmd_1.bits.e1_mode )
                {
                /* Clear the expected value used by E1_Pairid_Validation() */
                expect_pid[bp] = 0;

                /* Clear out the 3 PID bits and the validation bit.    Clearing the validation
                 * bit causes E1_Pairid_Validation() to start the PID validation procedure.
                 * E1_Pairid_Validation() is called in the channel unit interrupt.
                 */
                cu_rx_hoh.rzbit_pid[bp].reg = 0; 
#ifdef HTUC
                if ( dip_sw.bits.terminal_type == _HTUC )
                    {
                    /* The HTUC sends out a valid PID */
                    _CuSetPid(bp);

                    /* If the HTUC is connected to a repeater, then restart the
                       activation timer.  The HTUC will wait for the HTUR-to-RegC to
                       startup and send valid PID before proceeding to ACTIVE_TX_RX_STATE.
                     */
                     if ( cu_rx_hoh.rind_lo[bp].bits.hrp == 0 )
                         {
                         _BtControl(bp, _ACTIVATION_TIMER, ON);
                         }
                    }
#endif /* HTUC */

#ifdef HTUR
                if ( dip_sw.bits.terminal_type == _HTUR )
                    {
                    /* 
                     * The HTUR transmits an invalid PID until the validation
                     * procedure is completed.  At that time, it transmits the
                     * PID received (and validated) from the HTUC.
                     */
                    _CuSetPidToAllOnes(bp);
                    }
#endif /* HTUR */
                }
            system_status[bp].bits.activation_state = PID_VALIDATION_STATE;
            break;

        case PID_VALIDATION_STATE:
            /*
             * For E1 applications, this state polls cu_rx_hoh.rzbit_pid[bp].bits.valid
             */
            if ( cu_reg_copy.cmd_1.bits.e1_mode )
                {
                if ( !bp_status.bits.activation_interval )
                    {
                    /* Check if E1_Pairid_Validation() completed PID validation */
                    if ( cu_rx_hoh.rzbit_pid[bp].bits.valid )
                        {
                        system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE;
#ifdef CU_LED
                        _Bp_Led_Update(bp, SYNC_ON);
#endif /* CU_LED */
                        }
                    }
                else
                    {
                    system_status[bp].bits.activation_state = DEACTIVATED_STATE;
                    }
                }
            else
                {
                system_status[bp].bits.activation_state = GOTO_ACTIVE_TX_RX_STATE; 
                }
            break;

#endif /* CHAN_UNIT */

#endif /* not REPEATER */

        case GOTO_ACTIVE_TX_RX_STATE:
            /*
             * This state is a temporary state.  When this state is entered, 
             * _DSLLoopHandler() is processed one time before 
             * dsl_status2.bits.good_loop_cnt is incremented.
             */
            timer_count = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
#ifdef ZIP_START           
#ifdef CU_EOC      
            needs_zip_start_check[bp] = 1; /* enables ZipStart when entering act/TX/RX state */
#endif /*CU_EOC*/
#endif /*ZIP_START*/

#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("Startup Time = ~%ld sec\n", (timer_count - startup_count[bp]) );
            /* mark time 0 of available seconds */
            startup_count[bp] = timer_count;           
#endif
            last_count[bp] = timer_count;

#ifdef PERF_MONITOR   
            InitPMRecord(bp);
#endif /* PERF_MONITOR */

#ifdef ZIP_START

            if ( zip_system_status[bp].bits.enabled )
               {                
                zip_count[bp] = timer_count;
               }
        
#endif 

            /*
             * When the startup has reached the Active Tx/Rx State,
             * need to transmit payload.
             *
             * If the Channel Unit is not present, then
             * transmit external data.
             *
             */

            if ( !dsl_status2.bits.cu_present )
                {
                _BtControl(bp, _TRANSMIT_EXT_DATA, 0);
                }
#ifdef CHAN_UNIT
            else
                {
                _BtControl(_CU_CHAN1 + bp, _CU_TRANSMIT_PAYLOAD, 0);
                }
#endif

#ifdef ZIPSOCKET
            _Sync_Led_Update(bp, SYNC_ON);
#endif /* ZIPSOCKET */
    
#ifdef CU_LED
            _Bp_Led_Update(bp, SYNC_ON);
#endif /* CU_LED */
        
            system_status[bp].bits.activation_state = ACTIVE_TX_RX_STATE;
#ifdef REPEATER
            if (bp==0) /* Reg R */
               {
               /* send invalid PID to HTU-C */
               _CuSetPidToAllOnes(0);

               /* RegR loop is active. Startup RegC loop */
               system_status[1].bits.activation_state = INACTIVE_STATE;
                                                                 
                                                                
               /* Interrupt routine sends z-bits */
               }
            else       /* Reg C */
               {
               _CuRepeaterConfigure(ON); 
               }
 #endif
            /* 
             * Increment the number of good loops.  dsl_status2.bits.good_loop_cnt
             * is monitored by _DSLLoopHandler().
             */
            dsl_status2.bits.good_loop_cnt++;

#ifdef CU_EOC
            eocHandlerState[bp] = EOC_IDLE;
#endif

#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("State = Active Tx/Rx State\n");
#endif 
            break;

        case ACTIVE_TX_RX_STATE:

#ifdef TDEBUG
#ifdef DEBUG_MENU
            /* Process Menu Options */
            if ( keypress(&temp) )
                {
                process_menu_option(bp, temp);
                display_menu();
                }
#endif
#endif

#ifdef ZIP_START
            _ZipStartValidationManager(bp);
#endif

            /*
             * When in Normal Operation, check for:
             *
             *      LOSW (Channel Unit)
             *
             * If the CU is not present, check
             *
             *      LOS or NMR not OK (Bitpump)
             *
             *  goto PENDING_DEACTIVATED_STATE
             */
            if ( !dsl_status2.bits.cu_present )
                {
                /*
                 * When in Normal Operation, check for:
                 *
                 *      LOS or NMR not OK
                 */
                if ( bp_status.bits.los || !bp_status.bits.nmr_ok )
                    {
#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    if ( bp_status.bits.los )
                        printf("LOS Detected\n");

                    if ( !bp_status.bits.nmr_ok )
                        printf("NMR < -5dB Flag Detected\n");

                    HTU_PREFIX(bp);
                    printf("State = Pending Deactivated State\n");
#endif 

#ifndef INT_BUG
                    /* 40 x 50mS = 2 seconds (50mS resolution) */
                    _EnableGenPurposeTimer(bp, PENDING_DEACTIVATE_TIMER, 40);
#endif

#ifdef ZIPSOCKET
                    _Sync_Led_Update(bp, SYNC_OFF);
#endif
                    
                    system_status[bp].bits.activation_state = PENDING_DEACTIVATED_STATE;
                    break;

                    }
                }
#ifdef CHAN_UNIT
            /*
             * Check for Channel Unit LOSW
             */
            else 
            {
                if ( cu_sync_status == CU_OUT_OF_SYNC )
                {

#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("Channel Unit LOSW detected.\n");
                    HTU_PREFIX(bp);
                    printf("State = Pending Deactivated State\n");
#endif
                    
#ifndef INT_BUG
                    /* 40 = 2 seconds (50mS resolution) */
                    _EnableGenPurposeTimer(bp, PENDING_DEACTIVATE_TIMER, 40);
#endif

#ifdef ZIPSOCKET
                    _Sync_Led_Update( bp, SYNC_OFF );
#endif
                    system_status[bp].bits.activation_state = PENDING_DEACTIVATED_STATE; 
                    dsl_status2.bits.good_loop_cnt--;

#ifdef CU_LED
                    _Cu_Led_Update(bp, SYNC_OFF);
                    _Bp_Led_Update(bp, SYNC_OFF);
#endif

                }
            }
#endif /* CHAN_UNIT */


        break;


        case PENDING_DEACTIVATED_STATE:

#ifdef INT_BUG
            /*
             * When INT_BUG is specified, then just go to DEACTIVATED_STATE
             */
            system_status[bp].bits.activation_state = DEACTIVATED_STATE;
#else

            timer_stat.reg = _GetGenPurposeTimerStatus(bp, PENDING_DEACTIVATE_TIMER);

            if ( timer_stat.bits.complete )
            {
                _DisableGenPurposeTimer(bp, PENDING_DEACTIVATE_TIMER);
                system_status[bp].bits.activation_state = DEACTIVATED_STATE;
                
#ifdef REPEATER
#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("Shut RegC & Set To Deactivate\n");
#endif /* TDEBUG */
                
            /* When the RegR or RegC fails, shutdown the RegC and place
             * it into IDLE. This procedure matches the first time startup procedure.
             * The sync word will still be passed through because the regenerator
             * is still in repeater mode because _CuRepeaterConfigure(OFF) has not been 
             * called.  Because the sync word is passed through, a broken link on either 
             * COT-to-RegR or RegC-to-RT side will cause both the RT and COT to go out of sync.
             */
            _SetLoopIdle(1);            /* shutdown the RegC */

            if ( bp == 1 )   /* RegC */
               {
               /* The RegC forces the RegR to go out of sync.  When the RegC-to-RT link
                * is temporarily broken, lab tests show that on a rare occasion 
                * the RegR does not detect the out_of_sync condition.  Only the RegC 
                * reliably detects the out_of_sync condition.   The RegR will turn off
                * the repeater mode when it enters the DEACTIVATED_STATE.
                * The RegC can not shut down the repeater mode (either here or sooner)
                * because the RegR may stay in sync.  The HDSL specifications require
                * that a link failure on one side of the repeater causes a link failure
                * on the other side.   To be conservative, a conditional  
                * "if system_status[0].bits.activation_state == ACTIVE_TX_RX_STATE"
                * could be placed here, but tests indicate that this precautionary
                * conditional is not needed.
                */
               system_status[0].bits.activation_state = DEACTIVATED_STATE;
               }

#endif /* REPEATER */

               break;
            }
                
            if ( !dsl_status2.bits.cu_present )
                {

                /*
                 * If LOS = 0 & NMR OK = 1 - return to Normal Operation
                 */
                if ( !bp_status.bits.los && bp_status.bits.nmr_ok )
                    {

#ifdef ZIP_START
                    if ( zip_system_status[bp].bits.enabled )
                        {
                        /* reset Zip Start counter */
                        zip_count[bp] = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
                        }
#endif


#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("Returning to State = Active Tx/Rx State\n");
#endif
                
                    system_status[bp].bits.activation_state = ACTIVE_TX_RX_STATE;

#ifdef ZIPSOCKET
                    _Sync_Led_Update(bp, SYNC_ON );
#endif
                    }
                
                }
#ifdef CHAN_UNIT
            else
                {
                /*
                 * Must see Channel Unit LOSW for 2 seconds - then DEACTIVATE.
                 *
                 * If LOSW = 0 (Sync Word back) - return to NORMAL
                 */
                if ( cu_sync_status == CU_IN_SYNC )
                    {

                    dsl_status2.bits.good_loop_cnt++;

#ifdef ZIP_START
                    /* reset Zip Start counter */
                    zip_count[bp] = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
#endif

#ifdef ZIPSOCKET
                    _Sync_Led_Update( bp, SYNC_ON );
#endif

#ifdef CU_LED
                    _Cu_Led_Update(bp, SYNC_ON);
                    _Bp_Led_Update(bp, SYNC_ON);
#endif

#ifdef TDEBUG
                    HTU_PREFIX(bp);
                    printf("Returning to State = Active Tx/Rx State\n");
#endif

                    system_status[bp].bits.activation_state = ACTIVE_TX_RX_STATE;
                    break;
                    }
                }
#endif  /* CHAN_UNIT */

#endif  /* else INT_BUG */

            break;


        case DEACTIVATED_STATE:

#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("State = Deactivated State\n");

#if 0
            printf("Press any key to continue...\n");
            getkey();
#endif

#endif 

            /*
             * DEACTIVATE the bitpump.
             *
             * If HTUC, wait for LOST=0  ==> Inactive State
             *
             * If HTUR, wait for LOS=0   ==> Inactive State
             */
            _BtControl(bp, _DEACTIVATE, 0);

#ifdef REPEATER
            if ( bp == 0 )    /* RegR */
               {
               /* When either the RegC or RegR leaves ACTIVATE_TX_RX_STATE, it places
                * the RegC into IDLE.  Theoretically, the RegC can get to this point 
                * with a activation timeout from ACTIVATING_STATE or ACTIVATING_STATE_S1.
                * However, tests indicate that the COT-to-RegR link will timeout before
                * the RegC can timeout.
                */
               _CuRepeaterConfigure(OFF);  /* turn off repeater on both RegR and RegC */
               } 
#endif /* REPEATER */

#ifdef CHAN_UNIT
            if ( dsl_status2.bits.cu_present )
               {
               /* Use an API command to call _CuConfigureBeginStartup()
                * This command shuts off the channel unit interrupts.
                */
               _BtControl(_CU_CHAN1 + bp, _CU_FORCE_SCR_ONES, 0);

               /* Disable calling E1_Pairid_Validation() and un-reserve 
                * the PID numbers for the next statup. 
                */
               _Reset_Pid_Validation( bp );
               }
#endif

#ifdef ZIP_START
            if ( zip_system_status[bp].bits.zip_start_state == ZIP_START_VALIDATION_STATE0 )
                {
                /* Only decrement when startup not successful (ACTIVE_TX_RX_STATE 
                 * is not reached).
                 */
                DECREMENT_ZIP_START_ATTEMPT(bp);
                }
#endif


#ifndef INT_BUG
            last_count[bp] = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
#endif

#ifdef REPEATER
            if ( bp == 1 )    /* RegC */
                {
                system_status[bp].bits.activation_state = WAIT_FOR_LOST;
#ifdef TDEBUG
                HTU_PREFIX(bp);
                printf("State = Wait for LOST\n");
#endif /* TDEBUG */
                }

            if ( bp == 0)     /* RegR */
                {
                system_status[bp].bits.activation_state = WAIT_FOR_LOS;
#ifdef TDEBUG
                HTU_PREFIX(bp);
                printf("State = Wait for LOS\n");
#endif /* TDEBUG */
                }

#else  /* not REPEATER */
#ifdef HTUC
            if ( dip_sw.bits.terminal_type == _HTUC )   
                {
                system_status[bp].bits.activation_state = WAIT_FOR_LOST;
#ifdef TDEBUG
                HTU_PREFIX(bp);
                printf("State = Wait for LOST\n");
#endif /* TDEBUG */
                }
#endif /* HTUC */

#ifdef HTUR
            if ( dip_sw.bits.terminal_type == _HTUR )
                {
                system_status[bp].bits.activation_state = WAIT_FOR_LOS;
#ifdef TDEBUG
                HTU_PREFIX(bp);
                printf("State = Wait for LOS\n");
#endif /* TDEBUG */
                }
#endif /* HTUR */
#endif /* REPEATER */
            break;


#ifdef HTUC
        case WAIT_FOR_LOST:
            if ( bp_status.bits.lost )
                {
                system_status[bp].bits.activation_state = INACTIVE_STATE;
                }

            /* Add a safety net */
            timer_count = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
            if ( (last_count[bp] + WAIT_FOR_LOS_LOST_INTERVAL) <= timer_count )
                {
                system_status[bp].bits.activation_state = INACTIVE_STATE;
                }

            break;
#endif /* HTUC */

#ifdef HTUR
        case WAIT_FOR_LOS:
            if ( bp_status.bits.los )
                {
                system_status[bp].bits.activation_state = INACTIVE_STATE;
                }
            
            /* Add a safety net */
            timer_count = _GetGenPurposeContCount(bp, ONE_SECOND_TIMER);
            if ( (last_count[bp] + WAIT_FOR_LOS_LOST_INTERVAL) <= timer_count )
                {
                system_status[bp].bits.activation_state = INACTIVE_STATE;
                }
            break;
#endif /* HTUR */



         /****************************************/
         /*             TESTMODES                */
         /****************************************/
#ifdef ERLE
         case BACKGROUND_TESTMODE:

#ifdef TDEBUG
            printf("\nBackground Test - Transmitter OFF\n");
#endif

            dip_sw.erle_bits.transmit_state = OFF;
            _BtControl(bp, _ERLE_TEST_MODE, dip_sw.port1);
            
            system_status[bp].bits.activation_state = ERLE_TESTMODE;
            break;

         case ERLE_TESTMODE:

            /*
             * Activation Interval used to indicate error
             */
            if ( bp_status.bits.activation_interval )
                {
                system_status[bp].bits.activation_state = SYSTEM_IDLE;
                break;
                }

              
            /* Wait for Normal Operation */
            if ( !bp_status.bits.normal_operation )
                {
                break;
                }


#ifdef TDEBUG
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_SLM_LOW, &temp);
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_SLM_HIGH, &temp2);
            slm = (BP_S_16BIT)BYTE2WORD(temp2, temp);

            _BtStatus(bp, _ERLE_RESULTS, _ERLE_FELM_LOW, &temp);
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_FELM_HIGH, &temp2);
            felm = (BP_S_16BIT)BYTE2WORD(temp2, temp);

            HTU_PREFIX(bp);
            printf("SLM = %d FELM = %d\n",slm, felm);
#endif
            
            
#ifdef TDEBUG
            printf("\nERLE Test - Transmitter ON, %c Level\n", (dip_sw.erle_bits.transmit_level?'2':'4') );
#endif
            dip_sw.erle_bits.transmit_state = ON;
            _BtControl(bp, _ERLE_TEST_MODE, dip_sw.port1);

            system_status[bp].bits.activation_state = AAGC_TESTMODE;
            break;


         case AAGC_TESTMODE:

            /*
             * Activation Interval used to indicate error
             */
            if ( bp_status.bits.activation_interval )
                {
                system_status[bp].bits.activation_state = SYSTEM_IDLE;
                break;
                }

              
            /* Wait for Normal Operation */
            if ( !bp_status.bits.normal_operation )
                {
                break;
                }

#ifdef TDEBUG
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_SLM_LOW, &temp);
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_SLM_HIGH, &temp2);
            slm = (BP_S_16BIT)BYTE2WORD(temp2, temp);

            _BtStatus(bp, _ERLE_RESULTS, _ERLE_FELM_LOW, &temp);
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_FELM_HIGH, &temp2);
            felm = (BP_S_16BIT)BYTE2WORD(temp2, temp);

            _BtStatus(bp, _ERLE_RESULTS, _ERLE_SLM2_LOW, &temp);
            _BtStatus(bp, _ERLE_RESULTS, _ERLE_SLM2_HIGH, &temp2);
            value = (BP_S_16BIT)BYTE2WORD(temp2, temp);

            HTU_PREFIX(bp);
            printf("SLM = %d FELM = %d SLM/FELM = %d ",slm, felm,(int)(slm/felm));
            result = (float)slm / (float)felm;
            result = 20.0 * log10(result);
            printf("ERLE = %.2f \n", result);
            
            HTU_PREFIX(bp);
            printf("BYPASS HYBRID: SLM2 = %d SLM2/SLM = %d ",value, (value/slm));
            result = (float)value/(float)slm;
            result = 20.0 * log10(result);
            printf("ANALOG ERLE = %.2f \n", result );
#endif

#ifdef TDEBUG
            printf("\nAAGC Check - Transmitter ON, %c Level\n", (dip_sw.erle_bits.transmit_level?'2':'4') );

#endif
            _BtControl(bp, _TEST_MODE, _MEASURE_AAGC);
            system_status[bp].bits.activation_state = AAGC2_TESTMODE;
            break;


         case AAGC2_TESTMODE:

            /*
             * Activation Interval used to indicate error
             */
            if ( bp_status.bits.activation_interval )
                {
                system_status[bp].bits.activation_state = SYSTEM_IDLE;
                break;
                }

              
            /* Wait for Normal Operation */
            if ( !bp_status.bits.normal_operation )
                {
                break;
                }

#ifdef TDEBUG
            for (i = 0 ; i <= AGAIN15DB ; i++)
                {
                HTU_PREFIX(bp);
                printf("AAGC %ddB :", (int)(i * 3) );

                _BtStatus(bp, _AAGC_RESULTS, (i*2), &temp);
                _BtStatus(bp, _AAGC_RESULTS, (i*2) + 1, &temp2);
                slm = (BP_S_16BIT)BYTE2WORD(temp2, temp);

                /* value = SLM @ 0dB */
                if ( i == 0 )
                    {
                    value = slm;
                    }

                printf("SLM = %d ", (int)slm);
                result = (float)slm/(float)value;
                result = 20.0 * log10( result );
                printf("Gain = %.2f\n", result );
                }

            printf("\nTest Complete...\n");

#endif

            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;
#endif /* ERLE */



         case ANALOG_LB_TESTMODE:
            /*
             * Need to turn off scramblers/descramblers
             */
            _BtControl(bp, _STARTUP_SEQ_SOURCE, _INTERNAL);

            _BtControl(bp, _TRANSMIT_SCR, _BYPASS);
            _BtControl(bp, _RECEIVE_DESCR, _BYPASS);
            system_status[bp].bits.activation_state = ANALOG_LB2_TESTMODE;
            break;

         case ANALOG_LB2_TESTMODE:
            /*
             * Activation Interval used to indicate error
             */
            if ( bp_status.bits.activation_interval )
                {
                system_status[bp].bits.activation_state = SYSTEM_IDLE;
                break;
                }

              
            /* Wait for Normal Operation */
            if ( !bp_status.bits.normal_operation )
                {
                break;
                }
     
            _BtControl(bp, _TRANSMIT_EXT_DATA, 0);


#ifdef CHAN_UNIT
            if ( dsl_status2.bits.cu_present )
                {
                _BtControl(_CU_CHAN1 + bp, _CU_USE_SAME_TAP, _CU_TAP_5);
                _CuConfigureTransmitS1(bp);
                _BtControl(_CU_CHAN1 + bp, _CU_TRANSMIT_PAYLOAD, 0);
                }
#endif
            
            system_status[bp].bits.activation_state = SYSTEM_IDLE;
            break;


#ifdef DEBUG_MENU
        case PROCESS_MENU:
            /* Process Menu Options */
            if ( keypress(&temp) )
                {
                process_menu_option(bp, temp);
                display_menu();
                }
            break;
#endif

        default:
            /*
             * Undefined state, reset system
             */
            system_status[bp].bits.activation_state = CONFIGURATION_STATE;
#ifdef TDEBUG
            HTU_PREFIX(bp);
            printf("Unknown State detected.  Reseting System\n");
#endif
            break;

        }  /* end switch system_state */

    return ;
}   /* end _ActivationStateManager */
