/*
 * PROGRAM NAME:  Bt8970 Single Board
 *
 * VERSION:       1.0
 *
 * FILENAME:      dsl_api.c
 *
 * FILE CREATED:  July 9, 1996
 *
 * LAST MODIFIED: [4-23-99]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1996
 *                Brooktree Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   
 *
 *
 * FUNCTION LIST:
 *
 */

#include "dsl_incl.h"

/*---------------------------------------------------------*/
/*  Local Macros                                           */
/*---------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/
#define DSL_SW_VER       20   /* VER 2.0 */
#define DSL_HW_VER       0

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/
BP_U_32BIT APPL_SW_MSPACE timeslot, old_timeslot;
DSL_USER_SETUP APPL_SW_MSPACE dsl_user_setup;
#ifdef CU_EOC
BP_U_8BIT APPL_SW_MSPACE api_dataBuffIndex[_NO_OF_LOOPS],reg_name[_NO_OF_LOOPS];
BP_U_8BIT APPL_SW_MSPACE rd_wr_reg_index[_NO_OF_LOOPS];
#endif /*CU_EOC*/

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/
static BP_U_16BIT APPL_SW_MSPACE interval1_addr[_NO_OF_LOOPS];
static BP_U_8BIT APPL_SW_MSPACE interval2_addr[_NO_OF_LOOPS], interval3_addr[_NO_OF_LOOPS];
static BP_U_8BIT APPL_SW_MSPACE interval2_byte_addr[_NO_OF_LOOPS], interval3_byte_addr[_NO_OF_LOOPS];
static BP_U_8BIT APPL_SW_MSPACE timeslot_number;

#ifdef SP_API
BP_U_8BIT APPL_SW_MSPACE sngl_loop_api;  /* enables single pair API flag*/
BP_U_16BIT APPL_SW_MSPACE cust_sym_rate;

/* Globals set by Single Pair API command */

BP_U_8BIT APPL_SW_MSPACE total_pcm_tslot, total_hdsl_tslot, num_tslot_used; 
BP_U_8BIT APPL_SW_MSPACE fbit_present, derived_mclk; 
BP_U_16BIT APPL_SW_MSPACE MCLK_Freq; 

#ifdef T1E1_FRAMER
BP_U_8BIT _CuInitFramer();
#endif
#endif /* SP_API */

/*---------------------------------------------------------*/
/*  Function Protos                                        */
/*---------------------------------------------------------*/

#ifdef CHAN_UNIT

/*
 * FUNCTION:   _DslControl
 *
 * PARAMETERS: no - destination
 *             opcode - opcode value
 *             parameter - data parmeter value
 *
 * PURPOSE:    This function handles the Control API function calls for the
 *             System Application.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      To 
 *
 * CHANGES:    August 26, 1996    V1.0  :  New
 */
BP_U_8BIT _DslControl (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter)
{
   BP_U_8BIT temp, loop, lb_mode;
   TX_WR BP_XDATA *tx_wr_ptr;
#ifdef CU_EOC
   EOC_STRUCT_DECL_PTR;
#endif /*CU_EOC*/
   
   loop = no - _DSL_CHANNEL0;

#ifdef CU_EOC
   EOC_STRUCT_INIT_PTR(loop);
#endif /*CU_EOC*/
   tx_wr_ptr = get_tx_wr_ptr(loop);

    switch ( opcode )
    {
        case _DSL_SET_ASM_STATE:
            if (parameter == SYSTEM_IDLE)
              {
               _SetLoopIdle(loop);
              }
            else
              {
              system_status[loop].bits.activation_state = parameter;
              }
            break;

        case _DSL_RESET:
            if ( parameter < 0 || parameter > 1 )
                return _FAIL;
            dsl_status2.bits.system_reset = parameter;
            break;

        case _DSL_ASM_ENABLE:
            if ( parameter == 0 )
               {
               for ( temp = 0; temp < _NO_OF_LOOPS; temp++ )
                  {
                  _SetLoopIdle(temp); /*Sets each loop IDLE */
                  }
               }
            else if (parameter == 1)
               {
               dsl_status2.bits.system_reset = 1; /*Resets system*/
               }
            else
               {
                return _FAIL;
               }
            break;

#ifdef PERF_MONITOR
        /*--------------------------------------------------------
         *          Network Management Control APIs
         *--------------------------------------------------------*/
        case _SET_NETMANAGE_STATE:
           if ( parameter < 0 || parameter > 1 )
           {
               return _FAIL;
           }
           system_flags[loop].bits.nm_update = parameter;
           if ( parameter == 0 )
           {
               frame_cnt[loop] = 0;
               system_flags[loop].bits.slipflag = 0;
           }
           break;

        case _INTERVAL1_ADDR_LO:
            /* parameter validity check here */
            interval1_addr[loop] = (BP_U_16BIT)(parameter);
            break;

        case _INTERVAL1_ADDR_HI:
            /* parameter validity check here */
            interval1_addr[loop] = BYTE2WORD( parameter, LOW(interval1_addr[loop]) );
            break;

        case _INTERVAL2_ADDR:
            /* parameter validity check here */
            interval2_addr[loop] = parameter;
            interval2_byte_addr[loop] = 0;
            break;

        case _INTERVAL3_ADDR:
            /* parameter validity check here */
            interval3_addr[loop] = parameter;
            interval3_byte_addr[loop] = 0;
            break;
#endif /* PERF_MONITOR */

        /*--------------------------------------------------------
         *              Diagnostic Control APIs
         *--------------------------------------------------------*/       
        case _DSL_LOOPBACK:

            /* Trace the loopback selection */
            if ( parameter != _LOOPBACKS_OFF )
                dsl_user_setup.bits.loopback_mode = parameter;

            switch ( parameter )
            {
                case _LOOPBACKS_OFF:

                    switch( dsl_user_setup.bits.loopback_mode )
                    {
                        case _CU_PCM_ON_PCM:
                        case _CU_HDSL_ON_PCM:
                            _CuControl( _CU_COMMON, _CU_LOOPBACK, _CU_LOOPBACK_OFF );
                            break;

                        case _BP_ISOLATED_ANALOG_LOOPBACK:
                        case _BP_EXTERNAL_ANALOG_LOOPBACK: 
                            if ( no == _DSL_APPLICATION )
                            {
                                for ( temp = 0; temp < _NO_OF_LOOPS; temp++ )
                                {
                                    _TestMode( temp, _EXIT_TEST_MODE );
                                    system_status[temp].bits.activation_state = CONFIGURATION_STATE;
                                }
                            }
                            else
                            {
                                _TestMode( loop, _EXIT_TEST_MODE );
                                system_status[loop].bits.activation_state = CONFIGURATION_STATE;
                            }
                            _CuControl( _CU_COMMON, _CU_LOOPBACK, _CU_LOOPBACK_OFF );
                            break;
                    }
#ifdef ZIPSOCKET          
                  if ( no == _DSL_APPLICATION )
                     {
                     for ( temp = 0; temp < _NO_OF_LOOPS; temp++ )
                        {
                        if ( system_status[temp].bits.activation_state == ACTIVE_TX_RX_STATE )
                           {
                           _Sync_Led_Update( temp, SYNC_ON );
                           }
                        else
                           {
                           _Sync_Led_Update( temp, SYNC_OFF );
                           }
                        }
                     }
                  else
                     {
                     if ( system_status[loop].bits.activation_state == ACTIVE_TX_RX_STATE )
                        {
                        _Sync_Led_Update( temp, SYNC_ON );
                        }
                     else
                        {
                        _Sync_Led_Update( temp, SYNC_OFF );
                        }
                    }
                    dip_sw.zip_bits.test_mode = TEST_OFF;
#endif /* ZIPSOCKET */
                    break;

                /* Channel unit loopback affects both loops and it is non-destructive */
                case _CU_PCM_ON_PCM:
                case _CU_HDSL_ON_PCM:
                    if ( parameter == _CU_PCM_ON_PCM )
                        lb_mode = _CU_PRA_ON_PRA;
                    else
                        lb_mode = _CU_HDSL_ON_PRA;
#ifdef ZIPSOCKET
                    dip_sw.zip_bits.sync_1 = SYNC_ON;
                    dip_sw.zip_bits.sync_2 = SYNC_ON;
                    dip_sw.zip_bits.test_mode = TEST_ON;
#endif /* ZIPSOCKET */
                    _CuControl( _CU_COMMON, _CU_LOOPBACK, lb_mode );                    
                    break;

                /* Analog loopback is per-loop operation and it is destructive. */
                case _BP_ISOLATED_ANALOG_LOOPBACK:
                case _BP_EXTERNAL_ANALOG_LOOPBACK:
                    if ( parameter == _BP_ISOLATED_ANALOG_LOOPBACK )
                        lb_mode = _ISOLATED_ANALOG_LOOPBACK;
                    else
                        lb_mode = _ANALOG_LOOPBACK;

                    if ( no == _DSL_APPLICATION )
                    {
#ifdef ZIPSOCKET
                        dip_sw.zip_bits.sync_1 = SYNC_ON;
                        dip_sw.zip_bits.sync_2 = SYNC_ON;
#endif /* ZIPSOCKET */
                        for ( temp = 0; temp < _NO_OF_LOOPS; temp++ )
                        {
                            _TestMode( temp, lb_mode );
                            system_status[temp].bits.activation_state = ANALOG_LB_TESTMODE;
                           _Reset_Pid_Validation( temp );
                        }
                    }
                    else
                    {
#ifdef ZIPSOCKET           
                        _Sync_Led_Update( loop, SYNC_ON );
#endif /* ZIPSOCKET */
                        _TestMode( loop, lb_mode );
                        system_status[loop].bits.activation_state = ANALOG_LB_TESTMODE;
                        _Reset_Pid_Validation( loop );
                    }

#ifdef ZIPSOCKET
                    dip_sw.zip_bits.test_mode = TEST_ON;    
#endif /* ZIPSOCKET */
                    break;
            }
            break;

        case _DSL_TESTMODE:  
            
            /* Trace the loopback selection */
            if ( parameter != _BP_TESTMODE_OFF )
                dsl_user_setup.bits.test_mode = parameter;

#ifdef ZIPSOCKET
            dip_sw.zip_bits.test_mode = TEST_ON;
#endif /* ZIPSOCKET */

            switch ( parameter )
            {
                case _BP_ISOLATED_PULSE_PLUS3:
                    _TestMode( loop, _ISOLATED_PULSE_PLUS3 );
                    break;

                case _BP_ISOLATED_PULSE_PLUS1:
                    _TestMode( loop, _ISOLATED_PULSE_PLUS1 );
                    break;

                case _BP_ISOLATED_PULSE_MINUS1:
                    _TestMode( loop, _ISOLATED_PULSE_MINUS1 );
                    break;

                case _BP_ISOLATED_PULSE_MINUS3:
                    _TestMode( loop, _ISOLATED_PULSE_MINUS3 );
                    break;

                case _BP_FOUR_LEVEL_SCR:
                    _TestMode( loop, _FOUR_LEVEL_SCR );
                    break;

                case _BP_TWO_LEVEL_SCR:
                    _TestMode( loop, _TWO_LEVEL_SCR );
                    break;

                case _BP_ERLE_TEST:
           
                    if ( system_status[loop].bits.activation_state == ACTIVE_TX_RX_STATE )
                    {   
                        dsl_status2.bits.good_loop_cnt--;
#ifdef ZIPSOCKET
                        _Sync_Led_Update( loop, SYNC_OFF );
#endif /* ZIPSOCKET */ 
                    }
                    system_status[loop].bits.activation_state = SYSTEM_IDLE;
                    _TestMode( loop, _ERLE_TEST );

                    break;

                case _BP_MEASURE_AAGC:
                    _TestMode( loop, _MEASURE_AAGC );
                    break;

                case _BP_TESTMODE_OFF:
#ifdef ZIPSOCKET
                    dip_sw.zip_bits.test_mode = TEST_OFF;
#endif /* ZIPSOCKET */
                    _TestMode( loop, _EXIT_TEST_MODE );
                    if ( dsl_user_setup.bits.test_mode == _BP_ERLE_TEST )
                     {
                     system_status[loop].bits.activation_state = CONFIGURATION_STATE;
                     _Reset_Pid_Validation( loop );
                     }
                  break;
                  }
               break;

#ifdef CU_EOC
        case _API_DEST:
            eoc_struct_ptr->wrRegData[0x0F][0] = parameter;
            break;

        case _API_OPCODE:
            eoc_struct_ptr->wrRegData[0x0F][1] = parameter;
            break;

        case _API_DATA:
        eoc_struct_ptr->wrRegData[0x0F][2] = parameter;
            break;

        case _API_SEND:          
            if ( dip_sw.bits.terminal_type == _HTUC )
            {
            if(!eocTasks[loop].reg && (eocCtrl[loop].status & EOC_STATUS_AVAILABLE))
               {
               SET( eocTasks[loop].bits.eocWrTask);
               eocWrCommandReg[loop] = EOC_CMD_WRITE_REG_F;
               }
            else
               {
               return _FAIL;
               }
            }
            break;


         /* ------------------------------------------------------
         *         EOC Spec Additions APIs
         *--------------------------------------------------------*/

       case _EOC_REG_SELECT:
            if ( parameter < 0 || parameter > 0x1F)
               {
               return _FAIL;
               }
            api_dataBuffIndex[loop] = 0;/*Sets to 0 so it reads the first byte */
            rd_wr_reg_index[loop] = ((parameter & _RD_WR_BITS) >>4);
            /*write - 0, read - 1*/
            reg_name[loop] = (parameter & 0x0F);

            if(parameter == 0x1D)/* If read register D*/
               {
               needs_zip_start_check[loop] = 0;
               /* Make sure the ZipStart read regD is not used.*/
               /* Forces channel to be open to the user. */
               }
            break;

        case _EOC_REG_SIZE:
            if ( parameter < 0 || parameter > 0x10)
               {
               return _FAIL;
               }

            if(rd_wr_reg_index[loop])
               {
               eoc_struct_ptr->rdRegSize[reg_name[loop]] = parameter;
               }
            else
               {
               eoc_struct_ptr->wrRegSize[reg_name[loop]] = parameter;
               }
            break;

        case _EOC_BYTE_NUM_LOC:
      
         if (rd_wr_reg_index[loop])
            {
            if ( parameter < 0 || parameter > eoc_struct_ptr->rdRegSize[reg_name[loop]])
               {
               return _FAIL;
               }
            }
         else
            {
            if ( parameter < 0 || parameter > eoc_struct_ptr->wrRegSize[reg_name[loop]])
               {
               return _FAIL;
               }
            }
         api_dataBuffIndex[loop] = parameter;
         break;

        case _EOC_WRITE_REG_DATA:
#ifdef HTUC
         if ( dip_sw.bits.terminal_type == _HTUC )
            {
            if ( rd_wr_reg_index[loop] || api_dataBuffIndex[loop] >= eoc_struct_ptr->wrRegSize[reg_name[loop]])
               {
               /* Where HTUC writes message for write commands */
               return _FAIL;
               }
            else 
               {
               eoc_struct_ptr->shadowRegData[reg_name[loop]][api_dataBuffIndex[loop]++] = parameter;
               
               if ((api_dataBuffIndex[loop]) >= eoc_struct_ptr->wrRegSize[reg_name[loop]])
                  {
                  upDateFlags[loop].reg = 1 << reg_name[loop];
                  }
               }
            }
#endif /* HTUC */
#ifdef HTUR
         if ( dip_sw.bits.terminal_type == _HTUR )
            {
            if ( !rd_wr_reg_index[loop] || api_dataBuffIndex[loop] >= eoc_struct_ptr->rdRegSize[reg_name[loop]])
               {
               /* Where HTUR writes status to register for read commands */
               return _FAIL;
               }
            else
               {
               eoc_struct_ptr->shadowRegData[reg_name[loop]][api_dataBuffIndex[loop]++] = parameter;
            
               if (api_dataBuffIndex[loop] >= eoc_struct_ptr->rdRegSize[reg_name[loop]])
                  {
                  upDateFlags[loop].reg = 1 << reg_name[loop];
                  }

               }
            }
#endif /* HTU R */
            break;
     
        case _EOC_SEND_RD_WR:
         if ((parameter > 0x1F) ||(eocTasks[loop].reg) || (!eocCtrl[loop].status & EOC_STATUS_AVAILABLE))
            {
            return _FAIL; /*Since low level EOC is modified, check for not busy */
            }
         else if(parameter & 0x10) /* set the WR task */
            {
            SET(eocTasks[loop].bits.eocRdTask);
            eocRdCommandReg[loop] = ((parameter & 0x0F)|0xE0);/* so 1C = EC this cuts the top of param and adds a (E0) for write */
            }
         else                 /* set the RD task */
            {
            SET(eocTasks[loop].bits.eocWrTask);
            eocWrCommandReg[loop] = ((parameter & 0x0F)|0xD0);/* so 0C = DC this cuts the top of param and adds a (D0) for write*/
            }
         break;

        case _EOC_SET_CONTROL:
         if (eocTasks[loop].reg ||(!eocCtrl[loop].status & EOC_STATUS_AVAILABLE))
            {
            return _FAIL;
            }
         SET(eocTasks[loop].bits.eocCommand);
         eocCommandReg[loop] = parameter;
         break;

        case _EOC_ADD_DEST: /*Only used if send EOC to a repeater*/
         if (parameter > 0x01)
            {
            return _FAIL;
            }
         repeater_off[loop] = parameter;
         break;
#endif /* CU_EOC */

#ifdef CHAN_UNIT
        case _INSERT_CRC6:
         if (parameter > 0x01)
            {
            return _FAIL;
            }
         cu_reg_copy.tcmd_1[loop].bits.icrc_err = parameter;
         tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;
         break;
#endif /*CHAN_UNIT*/

        
        /*--------------------------------------------------------
         *         Fractional T1/E1 Configuration APIs
         *--------------------------------------------------------*/
        case _CB_TIMESLOT_LOCATION:
#ifdef CU_2T1
            if ( rate_index == _2T1 )
            {  
                if ( parameter < 0 || parameter > 23 )
                    return _FAIL;
            }
#endif /* CU_2T1 */

#ifdef CU_2E1
            if ( rate_index == _2E1 )
            {  
                if ( parameter < 0 || parameter > 31 )
                    return _FAIL;
            }
#endif /* CU_2E1 */
            timeslot_number = parameter;
            break;

        case _CB_TIMESLOT_STATE:
            if ( parameter < _TIMESLOT_BLOCKED || parameter > _TIMESLOT_IN_USE )
                return _FAIL;
            if ( parameter == _TIMESLOT_IN_USE )
            {
                /* Make the timeslot in use */
                timeslot |= (long)( (long)0x1 << (long)timeslot_number );
            }
            else
            {
                /* Block the timeslot */              
                timeslot &=  ~(long)( (long)0x1 << (long)timeslot_number );
            }
            break;

        case _CONFIGURE_CHANNEL_BLOCKING:
#ifdef CHANNEL_BLOCK
            _Configure_Channel_Blocking();
            old_timeslot = timeslot;
            /* Write the tables to the channel unit */
            _CuWriteMapRouteCombine();
#else
            return _FAIL;
#endif /*CHANNEL_BLOCK*/
            break;

        case _SET_ALL_TIMESLOTS:
            if ( parameter < 0 || parameter > 1 )
                return _FAIL;
            switch ( parameter )
            {
                case 0:
                    timeslot = 0x0;
                    break;
                case 1:
                    timeslot = 0xFFFFFFFF;
                    break;
            }
            break;

#ifdef SP_API
        /*********************************************/
        /*          Single Loop API commands         */
        /*********************************************/

        case _SP_TOTAL_PCM_TSLOT:
            if (parameter < 1 || parameter > 64)
                {
                return _FAIL;
                }
            total_pcm_tslot = parameter;
            break;

        case _SP_TOTAL_HDSL_TSLOT:
            if (parameter < 1 || parameter > 36)
                {
                return _FAIL;
                }
            total_hdsl_tslot = parameter;
            break;
       
        case _SP_USED_TSLOT:
            if (parameter < 1 || parameter > total_pcm_tslot || parameter > total_hdsl_tslot)
               {
               return _FAIL;
               }
            num_tslot_used = parameter;
            break;

        case _SP_FBIT_PRESENT:
            if (parameter < 0 || parameter > 1)
               {
               return _FAIL;
               }
            fbit_present = parameter;
            break;

        case _SP_DERIVED_MCLK:
            if (parameter < 30 || parameter > 145)
               {
               return _FAIL;
               }
            derived_mclk = parameter;
            break;

        
        case _SP_CONFIGURE:
            
            /* Channel Unit & Bit-pump */
            _ConfigureSinglePair(loop);

#ifdef T1E1_FRAMER
            /* T1/E1 Framer */
            _framer_pcm_mode = !fbit_present;
            _CuInitFramer();
#endif
            break;

#endif /*SP_API */


    }   /* end switch opcode */

    return _PASS;
}


/*
 * FUNCTION:   _DslStatus
 *
 * PARAMETERS: no - Channel Unit Loop or Common
 *             opcode - specified command
 *             parameter - data parameter
 *             *indication - return value
 *
 * PURPOSE:    This function handles the Status API function calls for the
 *             System Application.  The return value is returned
 *             via the *indication pointer.
 *             
 * RETURN:     _PASS / _FAIL
 *
 * NOTES:      
 *
 * CHANGES:    July 25, 1994    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _DslStatus (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter, char *indication)
{
   BP_U_8BIT loop;
#ifdef CU_EOC
   EOC_STRUCT_DECL_PTR;
#endif /*CU_EOC*/

   /* Check if the destination is DSL */
    
   if ( no < _DSL_APPLICATION || no > _DSL_CHANNEL2 )
      {
      return _FAIL;
      }
   loop = no - _DSL_CHANNEL0;
#ifdef CU_EOC
   EOC_STRUCT_INIT_PTR(loop);
#endif /*CU_EOC*/


   switch ( opcode )
      {
      case _DSL_VERSIONS:
         if ( no != _DSL_APPLICATION )
            {
            return _FAIL;
            }
         switch ( parameter )
            {
            case _DSL_SW_VERSION:
               *indication = DSL_SW_VER;
               break;

               
            case _DSL_HW_VERSION:
               *indication = DSL_HW_VER;
               break;
            }
         break;

        case _DSL_STATUS:
            if ( no != _DSL_APPLICATION )
               {
               return _FAIL;
               }
            if ( parameter < 0 || parameter > 1 )
               {
               return _FAIL;
               }
            switch ( parameter )
               {
               case 0:
                  *indication = dip_sw.port1;
                  break;
               case 1:
                  dsl_status2.bits.master_loop = _CuFlags._CuMasterLoop;
                  dsl_status2.bits.loop_reversal = _CuFlags._CuLoopsReversed;
                  *indication = dsl_status2.reg;
                  break;
                  }
               break;

        case _DSL_LOOP_STATUS:
         if ( no == _DSL_APPLICATION )
            return _FAIL;
         *indication = system_status[loop].reg;
         break;

#ifdef ZIP_START
        case _DSL_ZIP_STATUS:
            if ( parameter < _LOCAL || parameter > _FAR_END )
                return _FAIL;
            switch ( parameter )
               {
               case _LOCAL:
                  *indication = zip_system_status[loop].reg;
                  break;
                
               case _FAR_END:
#ifdef CU_EOC
                  if ( dip_sw.bits.terminal_type == _HTUC )
                     {
                     if(!eocTasks[loop].reg && (eocCtrl[loop].status & EOC_STATUS_AVAILABLE))
                        {
                        og_D_Value[loop] = eoc_struct_ptr->rdRegData[0x0D][0]; /*HTUC*/
                        eocRdCommandReg[loop] = EOC_CMD_READ_REG_D;
                        eocTasks[loop].bits.eocRdTask = 1;
                        }
                     }
#endif /* CU_EOC */
            return _FAIL;
            break;
             }
      break;
#endif /* ZIP_START */
    
#ifdef CU_EOC
      /*--------------------------------------------------------
      *          Eoc Read Status APIs
      *--------------------------------------------------------*/

       case _EOC_RCVD_NEWDATA_STATUS: 
         if (!parameter)
            {
            /*    Check if registers 0-7 have new info since last checked */
            *indication = (BP_S_8BIT)(newDataFlags[loop].reg & 0xFF);
            /* Clear new data flags 0-7 */
            newDataFlags[loop].reg = newDataFlags[loop].reg & 0xFF00;
            }
         else if(parameter == 1)
            {
            /*    Check if registers 8-0xF have new info since last checked */
            *indication = (BP_S_8BIT)((newDataFlags[loop].reg &0xFF00)>>8);
            /* Clear new data flags 8-0xF */
            newDataFlags[loop].reg = newDataFlags[loop].reg & 0x00FF;
            }
         else
            {
            return _FAIL;
            }
         break;
      
      case _EOC_READ_REG_DATA:
         
         if (parameter & 0x10)/*COT wants to see the Read Data from RT*/
            {
#ifdef HTUC
            if ( dip_sw.bits.terminal_type == _HTUC )
               {
               if ( api_dataBuffIndex[loop] < eoc_struct_ptr->rdRegSize[parameter & 0xF])
                  {
                  *indication = eoc_struct_ptr->rdRegData[parameter & 0xF][api_dataBuffIndex[loop]++];
                  }
               else
                  {
                  return _FAIL;
                  }
               }
#endif /*HTUC*/
            }
#ifdef HTUR
         else if ( dip_sw.bits.terminal_type == _HTUR )
            {
            if (api_dataBuffIndex[loop] < eoc_struct_ptr->wrRegSize[parameter & 0xF]) 
               {
               *indication = eoc_struct_ptr->wrRegData[parameter & 0xF][api_dataBuffIndex[loop]++];
               }
            else
               {
               return _FAIL;
               }
            }
#endif /*HTUR*/
         break;

       case _EOC_STATUS: 
            if (!parameter)
               {
#ifdef HTUC    
               *indication = eocCtrl[loop].status;
#endif /*HTUC*/
               }
            else if (parameter == 1) 
               {
#ifdef HTUR
               *indication = eocHoldStates[loop];
#endif /*HTUR*/
               }
            else
               {
               return _FAIL;
               }
            break;
#endif /*CU_EOC */

      case _SYSTEM_STATUS_FLAG:
            if ( no != _DSL_APPLICATION )
               {
               return _FAIL;
               }
            *indication = system_flags[loop].reg;  
            break;     

#ifdef PERF_MONITOR
        /*--------------------------------------------------------
         *          Network Management Status APIs
         *--------------------------------------------------------*/

        case _DSL_AVAILABLE_SECONDS:
            if ( parameter < 0 || parameter > 3 )
            {
                return _FAIL;
            }
            switch ( parameter )
            {
                case 0:
                    *indication = LOW( available_sec[loop] );
                    break;
                case 1:
                    *indication = HIGH( available_sec[loop] );
                    break;
                case 2:
                    *indication = HIGHER( available_sec[loop] );
                    break;
                case 3:
                    *indication = HIGHEST( available_sec[loop] );
                    break;
            }
            break;


        case _CRC_ERR_AT_INTERVAL1:
            *indication = crc_interval1[(interval1_addr[loop]+head1[loop])%INTERVAL1][loop];
            interval1_addr[loop]++;
            break;     

        case _CRC_ERR_AT_INTERVAL2:
            switch ( interval2_byte_addr[loop] )
            {
                case 0:
                    *indication = LOW( crc_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
                case 1:
                    *indication = HIGH( crc_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
                case 2:
                    *indication = HIGHER( crc_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
                case 3:
                    *indication = HIGHEST( crc_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
            }
            interval2_byte_addr[loop]++;
            if ( interval2_byte_addr[loop] == 4 )
            {
                interval2_addr[loop]++;
                interval2_byte_addr[loop] = 0;
            }
            break;  

        case _CRC_ERR_AT_INTERVAL3:
            switch ( interval3_byte_addr[loop] )
            {
                case 0:
                    *indication = LOW( crc_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
                case 1:
                    *indication = HIGH( crc_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
                case 2:
                    *indication = HIGHER( crc_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
                case 3:
                    *indication = HIGHEST( crc_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
            }
            interval3_byte_addr[loop]++;
            if ( interval3_byte_addr[loop] == 4 )
            {
                interval3_addr[loop]++;
                interval3_byte_addr[loop] = 0;
            }
            break; 

        case _FEBE_ERR_AT_INTERVAL1:
            *indication = febe_interval1[(interval1_addr[loop]+head1[loop])%INTERVAL1][loop];
            interval1_addr[loop]++;
            break;     

        case _FEBE_ERR_AT_INTERVAL2:
            switch ( interval2_byte_addr[loop] )
            {
                case 0:
                    *indication = LOW( febe_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
                case 1:
                    *indication = HIGH( febe_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
                case 2:
                    *indication = HIGHER( febe_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
                case 3:
                    *indication = HIGHEST( febe_interval2[(interval2_addr[loop]+head2[loop])%INTERVAL2][loop] );
                    break;
            }
            interval2_byte_addr[loop]++;
            if ( interval2_byte_addr[loop] == 4 )
            {
                interval2_addr[loop]++;
                interval2_byte_addr[loop] = 0;
            }
            break;  

        case _FEBE_ERR_AT_INTERVAL3:
            switch ( interval3_byte_addr[no] )
            {
                case 0:
                    *indication = LOW( febe_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
                case 1:
                    *indication = HIGH( febe_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
                case 2:
                    *indication = HIGHER( febe_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
                case 3:
                    *indication = HIGHEST( febe_interval3[(interval3_addr[loop]+head3[loop])%INTERVAL3][loop] );
                    break;
            }
            interval3_byte_addr[loop]++;
            if ( interval3_byte_addr[loop] == 4 )
            {
                interval3_addr[loop]++;
                interval3_byte_addr[loop] = 0;
            }
            break; 
     
       case _LAST_CRC_ERR_INTERVAL1:
            *indication = crc_interval1[tail1[loop]][loop];
            break;

       case _LAST_CRC_ERR_INTERVAL2:
            if ( parameter < 0 || parameter > 3 )
            {
                return _FAIL;
            }
            switch ( parameter )
            {
                case 0:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = LOW( crc_interval2[tail2[loop]][loop] );
                    break;
                case 1:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGH( crc_interval2[tail2[loop]][loop] );
                    break;
                case 2:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHER( crc_interval2[tail2[loop]][loop] );
                    break;
                case 3:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHEST( crc_interval2[tail2[loop]][loop] );
                    break;
            }
            break;

        case _LAST_CRC_ERR_INTERVAL3:
            if ( parameter < 0 || parameter > 3 )
            {
                return _FAIL;
            }
            switch ( parameter )
            {
                case 0:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = LOW( crc_interval3[tail3[loop]][loop] );
                    break;
                case 1:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGH( crc_interval3[tail3[loop]][loop] );
                    break;
                case 2:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHER( crc_interval3[tail3[loop]][loop] );
                    break;
                case 3:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHEST( crc_interval3[tail3[loop]][loop] );
                    break;
            }
            break;

        case _LAST_FEBE_ERR_INTERVAL1:
            *indication = febe_interval1[tail1[loop]][loop];
            break;

       case _LAST_FEBE_ERR_INTERVAL2:
            if ( parameter < 0 || parameter > 3 )
            {
                return _FAIL;
            }
            switch ( parameter )
            {
                case 0:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = LOW( febe_interval2[tail2[loop]][loop] );
                    break;
                case 1:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGH( febe_interval2[tail2[loop]][loop] );
                    break;
                case 2:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHER( febe_interval2[tail2[loop]][loop] );
                    break;
                case 3:
                    if ( interval1_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHEST( febe_interval2[tail2[loop]][loop] );
                    break;
            }
            break;

        case _LAST_FEBE_ERR_INTERVAL3:
            if ( parameter < 0 || parameter > 3 )
            {
                return _FAIL;
            }
            switch ( parameter )
            {
                case 0:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = LOW( febe_interval3[tail3[loop]][loop] );
                    break;
                case 1:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGH( febe_interval3[tail3[loop]][loop] );
                    break;
                case 2:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHER( febe_interval3[tail3[loop]][loop] );
                    break;
                case 3:
                    if ( interval2_count[loop] == 0 )
                        *indication = 0;
                    else
                        *indication = HIGHEST( febe_interval3[tail3[loop]][loop] );
                    break;

            }
            break;
#endif  /* PERF_MONITOR */
        /*--------------------------------------------------------
         *              Diagnostic Status APIs
         *--------------------------------------------------------*/
        case _DSL_ERLE_RESULTS:
            _BitpumpStatus( loop, _ERLE_RESULTS, parameter, indication );
            break;

        case _DSL_AAGC_RESULTS:
            _BitpumpStatus( loop, _AAGC_RESULTS, parameter, indication );
            break;
        
        case _API_RESULT:
#ifdef CU_EOC
         if ( dip_sw.bits.terminal_type == _HTUC )
            {
            if(!eocTasks[loop].reg && (eocCtrl[loop].status & EOC_STATUS_AVAILABLE))
               {
               SET( eocTasks[loop].bits.eocRdTask);
               eocRdCommandReg[loop] = EOC_CMD_READ_REG_F;
               }
            }
#endif /* CU_EOC */
            return _FAIL;
            break;

        /*--------------------------------------------------------
         *              Fractional T1/E1 Status APIs
         *--------------------------------------------------------*/
        case _CB_TIMESLOT_USAGE:
            if ( parameter < 0 || parameter > 3 )
                return _FAIL;
            switch ( parameter )
            {
                case 0:
                    *indication = LOW( timeslot );
                    break;
                case 1:
                    *indication = HIGH( timeslot );
                    break;
                case 2:
                    *indication = HIGHER( timeslot );
                    break;
                case 3:
                    *indication = HIGHEST( timeslot );
                    break;
            }           
            break;

        /*--------------------------------------------------------
         *              Bitpump Status APIs
         *--------------------------------------------------------*/
        case _DSL_FELM:
            _BitpumpStatus( loop, _FELM, parameter, indication );
            break;

        case _DSL_NMR:
            if ( parameter < _LOCAL || parameter > _FAR_END )
                {
                return _FAIL;
                }
            switch ( parameter )
            {
            case _LOCAL:
               _BitpumpStatus( loop, _NMR, parameter, indication );
               break;
            }
            break;
    }
    return _PASS;
}




void _SetLoopIdle (BP_U_8BIT loop)
{
#ifdef CHAN_UNIT
    /* Mask off CU interrupts */
    _CuConfigureBeginStartup(loop);
    _CuUseSameTap(loop, 0);
    
#endif
     _BitpumpControl(loop, _DEACTIVATE, 0);
     if ( system_status[loop].bits.activation_state == ACTIVE_TX_RX_STATE )
        {
        dsl_status2.bits.good_loop_cnt--;
        }
        system_status[loop].bits.activation_state = SYSTEM_IDLE;
}

#ifdef SP_API


/*
 * FUNCTION:   _ConfigureSinglePair
 *
 * PARAMETERS: Loop
 *
 * PURPOSE:    This select multi-rate for the parameter loop.
 *             Sets system Idle, reconfigure bitpump rate and channel unit
 *             rate, then restarts training at new data rate. The 
 *             mapping is the first N PCM and HDSL timeslots.  
 *             
 * RETURN:     
 *
 * NOTES:      
 *
 * CHANGES:    April 15, 1999    NEW   -   Dean Rasmussen
 */

#define MAX_STF_THRESH_A        (BP_U_16BIT)0x155
#define MAX_STF_THRESH_B        (BP_U_16BIT)0x1FF
#define MAX_STF_THRESH_C        (BP_U_16BIT)0x3FF

#define STF_THRESH_A_N          8
#define STF_THRESH_B_N          12
#define STF_THRESH_C_N          24

#define HFCLK_TARGET_FREQ       (BP_U_32BIT)70000  /* 70MHz */

void _ConfigureSinglePair (BP_U_8BIT loop)
{
    BP_U_8BIT APPL_SW_MSPACE MCLK_Multi, NCO_Divider;
    BP_U_8BIT APPL_SW_MSPACE HFCLK_Divider;
    BP_U_8BIT APPL_SW_MSPACE payload_byte_count, register_number, i; 
    BP_S_8BIT APPL_SW_MSPACE bit_number;
    BP_U_16BIT APPL_SW_MSPACE temp_u16, temp2_u16;
    BP_U_16BIT APPL_SW_MSPACE BCLK_Value, TCLK_RCLK_Value;
    BP_U_32BIT APPL_SW_MSPACE HFCLK_Value;

    COMMON_WR BP_XDATA * common_wr_ptr;
   
    common_wr_ptr = &(cu_wr->common_wr_regs);

#ifdef ZIPSOCKET
     _SetLoopIdle(0); 
     _SetLoopIdle(1); 
#else
     _SetLoopIdle(loop);
#endif /*ZIPSOCKET*/

    /* Mask off CU interrupts */
    _CuConfigureBeginStartup(loop);
   
   sngl_loop_api = 1;
   BCLK_Value = ((BP_U_16BIT)(total_hdsl_tslot * 64) + 16);
   cust_sym_rate = BCLK_Value / 8;
#ifdef ZIP_START
   system_status[loop].bits.zip_start_attempt = 0; /*Try COLD start first*/
#endif /*ZIP_START*/


#ifdef MCLK_1024
    derived_mclk = 80;    
    MCLK_Freq = 10240;
#else
   MCLK_Freq = derived_mclk * 128;    
#endif /*MCLK_1024*/
    
    /*
     * Bit-pump should be configured first since the bit-pump will
     * be changing clocks.
     */

    /*
     * PCM Frame Length - set register 1 less than length
     *
     * If 'fbit_present' then need to add '1'
     * Wen 'fbit_present=0' (E1 mode), add 0 bits.
     *
     * Frame Loc Lo is always 2 less than the frame len.
     */
    temp_u16 = ( ( (BP_U_16BIT)8 * (BP_U_16BIT)total_pcm_tslot) + fbit_present - 1);
    common_wr_ptr->frame_len_lo = LOW(temp_u16);
    common_wr_ptr->frame_len_hi = HIGH(temp_u16);
    common_wr_ptr->tframe_loc_lo = LOW(temp_u16 - 2);
    common_wr_ptr->tframe_loc_hi = HIGH(temp_u16 - 2);
    
    /*
     * HSDL frame length
     */
    temp_u16 = 8 * total_hdsl_tslot; 
    common_wr_ptr->hframe_len = LOW(temp_u16);
    common_wr_ptr->hframe_len_hi = HIGH(temp_u16);

    /* PLL multiplier */
    MCLK_Multi = ((BP_U_32BIT)(HFCLK_TARGET_FREQ + derived_mclk * 64))/ ((BP_U_32BIT)(derived_mclk * 128));
    cu_reg_copy.cmd_1.bits.pll_mul = 16 - MCLK_Multi;
    
    /* set fbit_present */
    cu_reg_copy.cmd_1.bits.e1_mode = !fbit_present;

    /*
     * DPLL
     */
    if (total_pcm_tslot < 3)
      {
      cu_reg_copy.cmd_7.bits.nco_scale = 1;
      NCO_Divider = 4;
      }
   else
      {
      cu_reg_copy.cmd_7.bits.nco_scale = 0;
      NCO_Divider = 1;
      }
   common_wr_ptr->cmd_7 = cu_reg_copy.cmd_7.reg;
    
   HFCLK_Value = (BP_U_32BIT)((BP_U_32BIT)MCLK_Freq * (BP_U_32BIT)MCLK_Multi);
   TCLK_RCLK_Value = ((BP_U_16BIT)(8*total_pcm_tslot)+fbit_present)*8;


    /*
     * DPLL Residual
     * 'temp2_u16' contains the Fpcm.  When calculating the DPLL Resid,
     * need to basically add a 1/2 in order for the ROUND to work properly.
     * 'C' doesn't do rounding very well.
     */
    temp2_u16 = NCO_Divider * TCLK_RCLK_Value;

    temp_u16 =  ((65535 * (HFCLK_Value % (2 * temp2_u16)) + temp2_u16) / (2 * temp2_u16) );
    common_wr_ptr->dpll_resid_lo = LOW(temp_u16);
    common_wr_ptr->dpll_resid_hi = HIGH(temp_u16);

    /* DPLL Factor */
    cu_reg_copy.dpll_factor = (257 - (HFCLK_Value/(2 * temp2_u16)));
    common_wr_ptr->dpll_factor = cu_reg_copy.dpll_factor;

    HFCLK_Divider = (2 * ((HFCLK_Value + 12500)/25000));
    cu_reg_copy.cmd_1.bits.pll_div = ((HFCLK_Divider/2) - 1);
    
    /* set PLL mul */
    common_wr_ptr->cmd_1 = cu_reg_copy.cmd_1.reg;

    /*
     * Stuffing
     * If Stuff Threshold C greater than the maximum, set A, B, and C
     * to a predefined value otherwise use formulas.
     */

    /* Calculate 'Stuff C' : N = 24 */
    temp_u16 = (STF_THRESH_C_N * HFCLK_Value)/((BP_U_32BIT) BCLK_Value * HFCLK_Divider);
    if ( temp_u16 > 0x3FF )
        {
        common_wr_ptr->stf_thresh_a_lo = LOW(MAX_STF_THRESH_A);
        common_wr_ptr->stf_thresh_a_hi = HIGH(MAX_STF_THRESH_A);

        common_wr_ptr->stf_thresh_b_lo = LOW(MAX_STF_THRESH_B);
        common_wr_ptr->stf_thresh_b_hi = HIGH(MAX_STF_THRESH_B);

        common_wr_ptr->stf_thresh_c_lo = LOW(MAX_STF_THRESH_C);
        common_wr_ptr->stf_thresh_c_hi = HIGH(MAX_STF_THRESH_C);
        }
    else
        {
        /* Write Stuff C */
        common_wr_ptr->stf_thresh_c_lo = LOW(temp_u16);
        common_wr_ptr->stf_thresh_c_hi = HIGH(temp_u16);

        /* Calculate/Write Stuff A : N = 12 */
        temp_u16 = (STF_THRESH_A_N * HFCLK_Value)/((BP_U_32BIT) BCLK_Value * HFCLK_Divider);
        common_wr_ptr->stf_thresh_a_lo = LOW(temp_u16);
        common_wr_ptr->stf_thresh_a_hi = HIGH(temp_u16);

        /* Calculate/Write Stuff B : N = 12 */
        temp_u16 = (STF_THRESH_B_N * HFCLK_Value)/((BP_U_32BIT) BCLK_Value * HFCLK_Divider);
        common_wr_ptr->stf_thresh_b_lo = LOW(temp_u16);
        common_wr_ptr->stf_thresh_b_hi = HIGH(temp_u16);

        }

    /*
     * RMAP Configuration
     */
    payload_byte_count = 0;
    for ( register_number = 0 ; register_number < 6; register_number++ )
        {
        rmap_table[register_number][loop] = 0;
        for (bit_number = 0; bit_number <6; bit_number++) 
            {
            if ( payload_byte_count < (num_tslot_used + fbit_present) ) 
                {
                rmap_table[register_number][loop] = rmap_table[register_number][loop] << 1;
                rmap_table[register_number][loop] = rmap_table[register_number][loop] | 0x01;
                payload_byte_count++;
                } 
            } /* for payload_byte_count */

        /*
         * If the F-Bit is present, then always need to set the 13th
         * RMAP byte.  This is a feature of the part
         */
        if ( fbit_present ) 
            {
            rmap_table[2][loop] = rmap_table[2][loop] | 1;
            }
        } /* for register_number */

    /*
     * TMAP Configuration
     */
    payload_byte_count = 0;
    for ( register_number = 0 ; register_number < 9 ; register_number++ ) 
        {
        tmap_table[register_number][loop] = 0xAA; 
        for ( bit_number = 0 ; bit_number < 4 ; bit_number++ )
            {
            if ( payload_byte_count < (num_tslot_used + fbit_present) )  
                {
                tmap_table[register_number][loop] = tmap_table[register_number][loop] << 2;
                tmap_table[register_number][loop] = tmap_table[register_number][loop] & 0xFC; 
                }
            payload_byte_count++; 
            }/* for register_number */
        }  /* for bit_number */
        

    /*
     * Route/Combine Tables
     *
     * Only set if occupied, otherwise set to 0x00.
     *
     * route_table = 0x01, 0x04, 0x10 for loop 0, 1, 2
     *             = 1, 4, or 16 so (1 << (loop * 2))
     * combine_table = loop + 1 (1, 2, 3 for loop 0, 1, 2)
     */
    for ( i = 0 ; i < 64 ; i++ )
        {
        if ( i < num_tslot_used )
            {
            route_table[i] = 1 << (loop * 2);
            combine_table[i] = loop + 1;  /* set to 1, 2, or 3 */
            }
        else
            {
            route_table[i] = 0x00;
            combine_table[i] = 0x00; 
            }
        }

    if ( fbit_present )
        {
        route_table[24] = 1 << (loop * 2);
        combine_table[24] = loop + 1; 
        }
    
    /*
     * TFIFO & RFIFO Water Levels
     */
    common_wr_ptr->rfifo_wl_lo = 0xFF; /* works for HDSL 7-15 with PCM 2.048 and 1.544 */
    common_wr_ptr->rfifo_wl_hi = 0x00; /* works for HDSL 7-15 with PCM 2.048 and 1.544*/

    if ( total_hdsl_tslot > 15 )
        {
        common_wr_ptr->rfifo_wl_lo = 0xA0;
        }
        
    if ( total_hdsl_tslot > 30 )
        {
        common_wr_ptr->rfifo_wl_lo = 0x50;
        }

    if ( total_hdsl_tslot < 6 )
        {
        common_wr_ptr->rfifo_wl_hi = 0x01;
        }

    cu_tfifo_wl[loop] = 0x80;

    /*
     * This function writes out the TMAP, RMAP, Route, Combine, and TFIFO
     * Water Level.  It also performs RESETs
     *
     * The common_wr_ptr->rfifo_wl actually writes the 'rfifo_wl' to the part.
     */
    _CuWriteMapRouteCombine();
    system_status[loop].bits.activation_state = CONFIGURATION_STATE;

}  /* end _ConfigureSinglePair() */
#endif /* SP_API */
#endif /* CHAN_UNIT*/
