/*
 * PROGRAM NAME:  Bt8952/Bt8953 EVM System
 *
 * VERSION:       4.3
 *
 * FILENAME:      frmr_api.c
 *
 * FILE CREATED:  June 29, 1994
 *
 * LAST MODIFIED: [5-27-99]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1995
 *                Conexant Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   This file contains the Framer API commands. 
 *
 * FUNCTION LIST:
 *                _FramerControl()
 *                _FramerStatus()
 */

#include "chanunit.h"


#ifdef T1E1_FRAMER

#ifdef BT8069_LIU
/* Maintain a current copy of the LIU */
LIU_CONTROL BP_XDATA current_liu_control;

LIU_CONTROL BP_XDATA * BP_XDATA liu_control;
LIU_STATUS BP_XDATA * BP_XDATA liu_status;
#endif

/*
 * FUNCTION:   _FramerControl
 *
 * PARAMETERS: opcode - spcefied opcode command
 *             parameter - data paramter
 *
 * PURPOSE:    This function processes the specified command
 *             
 * RETURN:     _PASS / _FAIL
 *
 * NOTES:      
 *
 * CHANGES:    June 29, 1994    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _FramerControl (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter)
{
#ifdef BT8510_FRAMER
    BP_S_16BIT i;
#endif

#ifdef BT8069_LIU
    BP_S_16BIT j;
#endif

    switch ( opcode )
        {
        case _FRAMER_OUTPUT_MODE:
            if ( parameter < DISABLE || parameter > ENABLE )
                {
                return _FAIL;
                }

#ifdef BT8370_FRAMER
            if ( parameter == DISABLE)
                {
                fr8370_ptr->program_input_output = 0x00;   /*  018  */
                fr8370_ptr->program_output_en = 0x3F;      /*  019  */
                fr8370_ptr->sys_bus_interface_conf = 0x00; /*  0D0  */
                }
            else
                {
                fr8370_ptr->program_input_output = 0xFE;   /*  018  */
                fr8370_ptr->program_output_en = 0x3A;      /*  019  */

                if (_framer_pcm_mode == _FRAMER_T1_MODE )
                    {
                    fr8370_ptr->sys_bus_interface_conf = 0x47; /*  0D0  */
                    }
                else
                    {
                    fr8370_ptr->sys_bus_interface_conf = 0x46; /*  0D0  */
                    }
                }
#endif /* BT8370_FRAMER */
            break;

        case _FRAMER_PCM_MODE:
            _framer_pcm_mode = parameter;
            break;
      
#ifdef BT8069_LIU
        case _LIU_CONFIGURE:
            current_liu_control.reg = 0;                /* set everything low */
            current_liu_control.bits.cb = 1;            /* set CB */
            current_liu_control.bits.els = parameter;   /* set ELS */

            liu_control->reg = current_liu_control.reg;

            current_liu_control.bits.reset = 1;   /* Set Reset High */
            liu_control->reg = current_liu_control.reg;

            /* Need to hold high for at least 200uS */
            for ( j = 0 ; j < 512 ; j++ )
                {
                current_liu_control.bits.reset = 1; /* keep Reset High */
                }

            current_liu_control.bits.reset = 0;   /* Set Reset low */
            liu_control->reg = current_liu_control.reg;
            break;

        case _LIU_EQ_SELECT:
            current_liu_control.bits.els = parameter;
            liu_control->reg = current_liu_control.reg;
            break;

        case _LIU_MODE_SELECT:
            current_liu_control.bits.mode = parameter;
            liu_control->reg = current_liu_control.reg;
            break;

        case _LIU_LOOP_SELECT:
            current_liu_control.bits.loop = parameter;
            liu_control->reg = current_liu_control.reg;
            break;

        case _LIU_CB_SELECT:
            current_liu_control.bits.cb = parameter;
            liu_control->reg = current_liu_control.reg;
            break;
#endif /* BT8069_LIU */

        case _LIU_LINE_LOOPBACK:
#ifdef BT8370_FRAMER
            if ( _FramerType == _BT8370 )
                {
                if ( parameter >= 1)
                    {
                    fr8370_mode_ptr->lb_conf_reg.lloop = 1;
                    }
                else
                    {
                    fr8370_mode_ptr->lb_conf_reg.lloop = 0;
                    }
                }
#endif /* BT8370_FRAMER */

#ifdef BT8069_LIU
            current_liu_control.bits.mode = 1;
            liu_control->reg = current_liu_control.reg;

            current_liu_control.bits.loop = 1;
            liu_control->reg = current_liu_control.reg;
#endif /* BT8069_LIU */
            break;

        case _FRAMER_CONFIGURE:
            _FramerType = parameter;
            _CuInitFramer();
            break;

        case _FRAMER_FRAME_FORMAT:
            if ( parameter < _CU_FRAMED_DATA || parameter > _CU_UNFRAMED_DATA )
                {
                return _FAIL;
                }

            switch ( _FramerType )
                {
                case _BT8510:
#ifdef BT8510_FRAMER
                    /*
                     * Set Inhibit/Abort Reframing for UnFramed
                     */
                    fr8510_mode_ptr->control_reg1.inhibit_abort_reframing = parameter;
#endif /* BT8510_FRAMER */
                    break;

                case _BT8360:
#ifdef BT8360_FRAMER
                    /*
                     * Disable ESF Mode for UnFramed
                     */
                    fr8360_mode_ptr->configuration.enable_esf_mode = !parameter;
#endif /* BT8360_FRAMER */
                    break;

                case _BT8370:
#ifdef BT8370_FRAMER
                    /*
                     * Disable Recieve Framer
                     */
                    fr8370_mode_ptr->recr_conf.rabort = parameter;
#endif /* BT8370_FRAMER */
                    break;

                }
            break;


        case _FRAMER_TRANSMIT_AIS:
            /* transmit AIS for startup */
            switch ( _FramerType )
                {
#ifdef BT8510_FRAMER
                case _BT8510:
                    if ( parameter == 0 )
                        {
                        /*
                         * At end of Startup, Turn OFF AIS.  In addition,
                         * must first re-initialize the framer.
                         *
                         * The Bt8510 Rev B had a bug.
                         */
                        _CuInitFramer();
                        }
                    fr8510_mode_ptr->control_reg0.transmit_ais = parameter;
                    break;
#endif /* BT8510_FRAMER */

#ifdef BT8360_FRAMER
                case _BT8360:
                    fr8360_mode_ptr->alarm_signal_control.enable_transmit_ais = parameter;
                    break;
#endif /* BT8360_FRAMER */

#ifdef BT8370_FRAMER
                case _BT8370:
                    fr8370_mode_ptr->tran_alm_signal_conf.tais = parameter;
                    break;
#endif /* BT8370_FRAMER */

                }
            break;

        case _FRAMER_LOOPBACK:
            switch ( parameter )
                {
                case _FRAMER_LOOPBACK_OFF:
                    break;

#ifdef BT8510_FRAMER
                case _FRAMER_LINE_PER_CHAN_LB:
                    for ( i = 0 ; i < 16 ; i++ )
                        {
                        fr8510_ram_buffer_ptr->per_channel_rx[i] |= 0x11;
                        }
                    break;

                case _FRAMER_FAR_PER_CHAN_LB:
                    for ( i = 0 ; i < 16 ; i++ )
                        {
                        fr8510_ram_buffer_ptr->per_channel_tx[i] |= 0x11;
                        }
                    break;
#endif /* BT8510_FRAMER */
                }
            break;
#ifdef CU_LED
        case _CU_LED_BLOCK_SET:
        case _CU_LED_BLOCK_ON:
        case _CU_LED_BLOCK_OFF:
            set_LED_block(_FRAMER, opcode, parameter);
            break;
#endif /* CU_LED */

        default:
            /* need to use 'no' */
            no++;
            return _FAIL;
        }

    return _PASS;
}


/*
 * FUNCTION:   _FramerStatus
 *
 * PARAMETERS: opcode - specified command
 *             parameter - data parameter
 *             *indication - return value
 *
 * PURPOSE:    This function queries the Framer/LIU for the
 *             specified opcode command.  The return value is returned
 *             via the *indication pointer.
 *             
 * RETURN:     _PASS / _FAIL
 *
 * NOTES:      
 *
 * CHANGES:    July 25, 1994    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _FramerStatus (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter, BP_S_8BIT *indication)
{
    BP_S_16BIT i;
    BP_U_8BIT BP_XDATA *fr_ptr;
    union
        {
        BP_U_8BIT reg;
        FRAMER_VERSION version;
        } fr_ver;
#ifdef BT8370_FRAMER
    BP_U_8BIT maybe_8370 = 0;
#endif

    switch ( opcode )
        {

        case _FRAMER_PRESENT:

            /*
             * Write something to the SA_BITS to verify at least RAM is
             * there, SA_BITS are at address 0x04 - 0x08
             *
             * Only write to SA_BITS[2-4] (0x06 - 0x08).  SA_BITS[1] maps to
             * SLIP_BUFF_CONTROL_REG on the Bt8360 which has a bit that doesn't
             * get read back.
             */
            fr_ptr = (BP_U_8BIT BP_XDATA *)FRAMER_ADD;
            fr_ptr += 0x04;
            for ( i = 2 ; i < 5 ; i++ )
                {
                *(fr_ptr + i) = 0x11 * (i + 1);
                }

            for ( i = 2 ; i < 5 ; i++ )
                {
                if ( *(fr_ptr + i) != (0x11 * (i+1)) )
                    {
#ifdef BT8370_FRAMER
                    maybe_8370 = 1;
#endif
                    break;                        
                    }
                }

#ifdef BT8370_FRAMER
                if ( maybe_8370 )
                    {
                    fr8370_ptr->ext_dlink_bit = 0x55;
                    fr8370_ptr->version = 0xAA;

                    if (fr8370_ptr->ext_dlink_bit != 0x55)
                        {
                        *indication = _NOT_PRESENT;
                        return _PASS;
                        }
                    fr8370_ptr->ext_dlink_bit = 0xAA;
                    fr8370_ptr->version = 0x55;

                    if (fr8370_ptr->ext_dlink_bit != 0xAA)
                        {
                        *indication = _NOT_PRESENT;
                        return _PASS;
                        }

                    *indication = _BT8370;
                    return _PASS;
                    }
#endif


#if defined(BT8510_FRAMER) || defined(BT8360_FRAMER)
            /*
             * Read the Part # to determine which Framer Type
             * Both Bt8510 & Bt8360 Version Addresses are at reg 0x19
             */
            fr_ptr = (BP_U_8BIT BP_XDATA *)FRAMER_ADD;
            fr_ver.reg = *(fr_ptr + 0x19);
            switch ( fr_ver.version.hardware_part_number )
                {
                case _BT8510:
                    *indication = _BT8510;
                    break;

                case _BT8360:
                    *indication = _BT8360;
                    break;

                default:
                    *indication = _NOT_PRESENT;
                    break;
                }
#endif

            break;


        case _FRAMER_VERSION:
            if ( _FramerType == _NOT_PRESENT )
                {
                *indication = 0;
                }
            else if( _FramerType == _BT8370 )
                {
                /* The Bt8370 Version Addresses is at reg 0x00 */
                fr_ptr = (BP_U_8BIT BP_XDATA *)FRAMER_ADD;
                }
            else
                {
                /* Both Bt8510 & Bt8360 Version Addresses are at reg 0x19 */
                fr_ptr = (BP_U_8BIT BP_XDATA *)FRAMER_ADD;
                fr_ptr += 0x19;
                }

            fr_ver.reg = *fr_ptr;
            *indication = fr_ver.version.hardware_rev_number;
            break;


        default:
            /* need to use 'no' and 'parameter' */
            no++;
            parameter++;
            return _FAIL;
        }

    return _PASS;
}


#endif /* T1E1_FRAMER */

