#ifdef CHAN_UNIT

/*
 * PROGRAM NAME:  Bt8952/Bt8953 EVM System
 *
 * VERSION:       4.3
 *
 * FILENAME:      cu_utils.c
 *
 * FILE CREATED:  November 10, 1994
 *
 * LAST MODIFIED: [4-22-99]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1995
 *                Brooktree Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   This file contains some miscellaneous Channel Unit
 *                functions.
 *
 * FUNCTION LIST:
 *                _CuHandleFlags()
 *                _CuInitAddresses()
 *                _CuWriteMasterCmd5()
 *                _CuDpllOperation()
 *                _CuClearCounters()
 *                _CuHdslToPra()
 *                _CuUseSameTap()
 *                get_tx_wr_ptr()
 *                get_rx_wr_ptr()
 *                get_tx_rd_ptr()
 *                get_rx_rd_ptr()
 *                _CuRepeaterConfigure() 
 */

#include "chanunit.h"
#include "..\dsl_api.h" /* _API_RESULT definition */
BP_U_8BIT BP_XDATA num_bit_pumps;
BP_U_8BIT BP_XDATA bp_position[_NO_OF_LOOPS];
/*Used Pointer to conserve code size ex.  eocStruct[loop].wrRegSize is now eoc_struct_ptr->wrRegSize */

/*
 * FUNCTION:   _CuHandleFlags
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function handles any Flags that may get set
 *             during an interrupt routine.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    November 3, 1994    NEW   -   Dean Rasmussen
 */
void _CuHandleFlags (void)
{
#ifdef CU_EOC
   BP_U_8BIT bp;
   EOC_STRUCT_DECL_PTR;
   BP_U_8BIT i,j;
#ifdef HTUR
   BP_U_8BIT  APPL_SW_MSPACE  dest, opcode, para, temp2;
#endif /* HTUR */

#ifdef HTUC
#ifdef SER_COM
    BP_U_8BIT temp;
#endif /* SER_COM */ 
#endif /* HTUC */
#endif /* CU_EOC */
    if ( !_CuChipVersion.present )
        {
        return;
        }

#ifdef CU_EOC
    for ( bp = 0; bp < _NO_OF_LOOPS; bp++ )
      {
#ifdef HTUC
      if ( htu_index == _HTUC )
         {
#ifdef SER_COM
         if (newDataFlags[bp].bits.NewDataFlagF)
            {
            EOC_STRUCT_INIT_PTR(bp);
            CLEAR(newDataFlags[bp].bits.NewDataFlagF);
            _WriteHostMailBox( 0xFF, 0xFF, 0xFF, 0x55);
            temp = CHECKSUM(_DSL_CHANNEL0+bp|0xF0, _API_RESULT, eoc_struct_ptr->rdRegData[0xF][0]);
         /* NOTE: API commands now are in read and write RegF, previous release returned in RegA*/
            _WriteHostMailBox(_DSL_CHANNEL0+bp|0xF0, _API_RESULT, eoc_struct_ptr->rdRegData[0xF][0], temp);
            }
#endif /* SER_COM */  
         }
#endif /* HTUC */

#ifdef CU_EOC
#ifdef HTUC
/* This will copy the shadow to send reg whenever eoc is not running */  

         if(upDateFlags[bp].reg ) /*if not 0 check register to update with upDateFlag info */ 
            {
            for (i =0; i<16; i++)
               {
               if((upDateFlags[bp].reg & (1 << (i))) && (eocCtrl[bp].status & EOC_STATUS_AVAILABLE))
                  {
                  EOC_STRUCT_INIT_PTR(bp); /*initialize the eoc_struct_ptr to eocStruct[loop]*/
                  for (j=0; j < eoc_struct_ptr->wrRegSize[reg_name[bp]]; j++)
                     {
             /* Updates the WRITE_REGISTERS since HTU-C only writes the WRITE_REGISTERS*/    
                     eoc_struct_ptr->wrRegData[reg_name[bp]][j] = eoc_struct_ptr->shadowRegData[reg_name[bp]][j];
                     }
                  upDateFlags[bp].reg = ((upDateFlags[bp].reg) & (~(1 << i))); /*clear flag*/
                  }
               }
            }
#endif /*HTUC*/
#ifdef HTUR
/* This will copy the shadow to send reg whenever eoc is not running */
         if(upDateFlags[bp].reg)
            {
            for (i =0; i<16; i++)
               {
               if(upDateFlags[bp].reg & (1 << (i)) && rt_open[bp]) /* HTU-R doesnt have */
                          /* eocCtrl[bp].status & EOC_STATUS_AVAILABLE so check rt_open*/
                  {
                  EOC_STRUCT_INIT_PTR(bp); 
                  for (j=0; j < eoc_struct_ptr->rdRegSize[reg_name[bp]]; j++)
                     {
             /* Updates the READ_REGISTERS since HTU-R only writes the READ_REGISTERS*/    
                     eoc_struct_ptr->rdRegData[reg_name[bp]][j] = eoc_struct_ptr->shadowRegData[reg_name[bp]][j];
                     }
                  upDateFlags[bp].reg = (upDateFlags[bp].reg & (~(1 << i)));
                  }
               }
            }

         if(eocLoopReg[bp]& ENABLE_LOOPB)/* ENABLE_LOOPB masks CRC and allows loopbacks since CRC is taken care of in HandleValidRtn[loop]*/
/* eocLoopReg is an higher level version of the HoldStates except extra flags ex. LOOP_OFF, ALOOP_OFF*/
        {
        if (eocLoopReg[bp] & EOC_REQ_LOOP_RT)
           {
            _BtControl( _DSL_CHANNEL0 +bp, _DSL_LOOPBACK, _CU_HDSL_ON_PCM);
            eocLoopReg[bp] &= ~EOC_REQ_LOOP_RT;
           }        
        if (eocLoopReg[bp] & EOC_REQ_ALOOP_RT)
           {
            /*user defined*/    
            eocLoopReg[bp] &= ~EOC_REQ_ALOOP_RT;    
           }        
        if (eocLoopReg[bp] & EOC_REQ_ALOOP_RT_OFF)
           {
                /*user defined*/    
            eocLoopReg[bp] &= ~EOC_REQ_ALOOP_RT_OFF;
           }        
        if (eocLoopReg[bp] & EOC_REQ_LOOP_RT_OFF)
           {
            _BtControl( _DSL_CHANNEL0 +bp, _DSL_LOOPBACK, _LOOPBACKS_OFF);
            eocLoopReg[bp] &= ~EOC_REQ_LOOP_RT_OFF;    
           }        
        }
#endif /* HTUR */
#endif /* CU_EOC */    

#ifdef HTUR

    if ( htu_index == _HTUR )
            {
            if (newDataFlags[bp].bits.NewDataFlagF)
           /* The return data used to be in rdRegData[0xA], now rdRegData[0xF] for sending API */
               {
               EOC_STRUCT_INIT_PTR(bp);          
               CLEAR(newDataFlags[bp].bits.NewDataFlagF);
               dest = eoc_struct_ptr->wrRegData[0xF][0];   
               opcode = eoc_struct_ptr->wrRegData[0xF][1]; 
               para = eoc_struct_ptr->wrRegData[0xF][2];

               if (opcode & 0x80)
                  {
                  _BtStatus( dest, opcode, para, (BP_S_8BIT *)&temp2 );
                  eoc_struct_ptr->rdRegData[0xF][0] = temp2;
                  }
               else
                  {
                  _BtControl( dest, opcode, para );
                  }
               }
            }
#endif /* HTUR */
         }
#endif /* CU_EOC */


#ifdef CU_LED
    /*
     * Turn ON/OFF Leds
     */
    if ( _CuFlags._CuUpdateLeds )
        {
        set_LED_block(_BP_LED_BLOCK1_2, _CU_LED_BLOCK_SET, cu_bp_led_block[0].reg);
        set_LED_block(_BP_LED_BLOCK3_SP, _CU_LED_BLOCK_SET, cu_bp_led_block[1].reg);
        set_LED_block(_CU_PCM_COMMON, _CU_LED_BLOCK_SET, cu_pcm_led_block.reg);
        for ( bp = 0 ; bp < _NO_OF_LOOPS ; bp++ )
            {
            if (system_status[bp].bits.present == 1)
                {
                set_LED_block(_CU_LED_CHAN1 + bp, _CU_LED_BLOCK_SET, cu_led_block[bp].reg);
                }
            }
        }
#endif /* CU_LED */


    return ;
}



/*
 * FUNCTION:   _CuWriteMasterCmd5
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function writes CMD_5 based on the Master Loop
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      CMD_5 : MASTER_SEL, ZBIT_SEL, & STUFF_SEL
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _CuWriteMasterCmd5 (void)
{
   cu_reg_copy.cmd_5.bits.master_sel = _CuFlags._CuMasterLoop+1;
   cu_reg_copy.cmd_5.bits.zbit_sel = _CuFlags._CuMasterLoop+1;
   cu_reg_copy.cmd_5.bits.stuff_sel = _CuFlags._CuMasterLoop+1;

   _CU_DUMMY_WRITE(cu_reg_copy.cmd_5.reg);
   cu_wr->common_wr_regs.cmd_5 = cu_reg_copy.cmd_5.reg;

   return;
}


/*
 * FUNCTION:   _CuDpllOperation
 *
 * PARAMETERS: mode - _CU_DPLL_CLOSED/_CU_DPLL_OPEN (0/1)
 *
 * PURPOSE:    This function sets the DPLL NCO Mode (CMD_5: bit 7)
 *
 *             The DPLL Interrupt Bit is set accordingly.
 *
 *             Must reset DPLL after making changes.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      For OPEN Mode, the DPLL_RST doesn't appear to always
 *             cause the DPLL to program the correct frequency.
 *             It has been experimentally determined that writing
 *             the DPLL Factor register to an incorrect value, then writing
 *             it back causes the DPLL NCO Open mode to work.
 *
 * CHANGES:    December 11, 1995    NEW   -   V4.4
 */
#ifdef C51
#include <intrins.h>
#endif
void _CuDpllOperation (BP_U_8BIT mode)
{
    BP_U_8BIT i;

    cu_reg_copy.cmd_5.bits.dpll_nco = mode;
    _CU_DUMMY_WRITE(cu_reg_copy.cmd_5.reg);
    cu_wr->common_wr_regs.cmd_5 = cu_reg_copy.cmd_5.reg;

    cu_reg_copy.cmd_7.bits.dpll_err_en = (mode == _CU_DPLL_CLOSED)?ON:OFF;
    cu_wr->common_wr_regs.cmd_7 = cu_reg_copy.cmd_7.reg;
    
    if ( mode == _CU_DPLL_OPEN )
        {
        cu_wr->common_wr_regs.dpll_factor = cu_reg_copy.dpll_factor - 2;

        /*
         * Through experimentation, need to wait ~20uS for 2T1 & 3E1
         * systems (less for 2E1).
         */
#ifdef C51
        for ( i = 0 ; i < 25 ; i++ )
            {
            _nop_();
            }
#else

    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section cu_utils.1    Begins     <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Add ~20uS Delay.                                         */
    /*                                                          */
    /* Reference:                                               */
    /*                                                          */
    /*----------------------------------------------------------*/


    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section cu_utils.1    Ends       <<<*/
    /*----------------------------------------------------------*/

         
#endif

        cu_wr->common_wr_regs.dpll_factor = cu_reg_copy.dpll_factor;
        }

    cu_wr->common_wr_regs.dpll_rst = 0;

    return ;
}


/*
 * FUNCTION:   _CuClearCounters
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *             cntr - which counter, as defined in CU_API.H
 *
 * PURPOSE:    This function clears the specified Error Counter(s)
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    November 10, 1994    NEW   -   Dean Rasmussen
 */
void _CuClearCounters (BP_U_8BIT loop, BP_U_8BIT cntr)
{
    if ( cntr == _CU_ALL_COUNTERS)
        {
        error_ctrs[_CU_OUT_OF_SYNC_CTR][loop] = 0;
        error_ctrs[_CU_CRC_ERR_CTR][loop] = 0;
        error_ctrs[_CU_RFIFO_FULL_CTR][loop] = 0;
        error_ctrs[_CU_RFIFO_EMPTY_CTR][loop] = 0;
        error_ctrs[_CU_RFIFO_SLIP_CTR][loop] = 0;
        error_ctrs[_CU_TFIFO_FULL_CTR][loop] = 0;
        error_ctrs[_CU_TFIFO_EMPTY_CTR][loop] = 0;
        error_ctrs[_CU_TFIFO_SLIP_CTR][loop] = 0;
        error_ctrs[_CU_TFIFO_STUFF_CTR][loop] = 0;
        error_ctrs[_CU_DPLL_ERROR_CTR][loop] = 0;
        error_ctrs[_CU_FEBE_ERROR_CTR][loop] = 0;
        error_ctrs[_CU_LOSD_ERROR_CTR][loop] = 0;
        }
    else
        {
        error_ctrs[cntr][loop] = 0;
        }

    return;
}


/*
 * FUNCTION:   _CuHdslToPra
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *             state - ON/OFF
 *
 * PURPOSE:    Puts the specified Chan into HDSL to PRA Loopback
 *             
 * RETURN:     nothing
 *
 * NOTES:      make Tx/Rx use HTU-C SCR/DSCR Tap #5
 *
 * CHANGES:    December 6, 1994    NEW   -   Dean Rasmussen
 */
void _CuHdslToPra (BP_U_8BIT loop, BP_U_8BIT state)
{
    TX_WR *tx_wr_ptr;
    RX_WR *rx_wr_ptr;

    tx_wr_ptr = get_tx_wr_ptr(loop);
    rx_wr_ptr = get_rx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 || rx_wr_ptr == 0 )
        {
        return ;
        }

    cu_reg_copy.rcmd_2[loop].bits.ph_loop = state;

    /*
     * Modify Taps
     */
    if ( state )
        {
        cu_reg_copy.tcmd_2[loop].bits.scr_tap = 0;
        cu_reg_copy.rcmd_2[loop].bits.dscr_tap = 1;
        }
    else
        {
        cu_reg_copy.tcmd_2[loop].bits.scr_tap = htu_index;
        cu_reg_copy.rcmd_2[loop].bits.dscr_tap = htu_index;
        }

    tx_wr_ptr->tcmd_2 = cu_reg_copy.tcmd_2[loop].reg;
    rx_wr_ptr->rcmd_2 = cu_reg_copy.rcmd_2[loop].reg;

    tx_wr_ptr->tfifo_rst = 0x0;
    rx_wr_ptr->rfifo_rst = 0x0;

    return ;
}


/*
 * FUNCTION:   _CuUseSameTap
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *             tap - OFF : return to normal
 *                   _CU_TAP_5 (1)
 *                   _CU_TAP_18 (2)
 *
 * PURPOSE:    make Tx/Rx use HTU-C SCR/DSCR Tap #5 while in normal operation
 *             
 * RETURN:     nothing
 *
 * NOTES:      This allows a Bit-Pump Loopback to be performed
 *
 * CHANGES:    December 6, 1994    NEW   -   Dean Rasmussen
 */
void _CuUseSameTap (BP_U_8BIT loop, BP_U_8BIT tap)
{
    TX_WR *tx_wr_ptr;
    RX_WR *rx_wr_ptr;

    tx_wr_ptr = get_tx_wr_ptr(loop);
    rx_wr_ptr = get_rx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 || rx_wr_ptr == 0 )
        {
        return ;
        }

    if ( tap == _CU_TAP_5 )
        {
        cu_reg_copy.tcmd_2[loop].bits.scr_tap = 0;
        cu_reg_copy.rcmd_2[loop].bits.dscr_tap = 1;
        }
    else if ( tap == _CU_TAP_18 )
        {
        cu_reg_copy.tcmd_2[loop].bits.scr_tap = 1;
        cu_reg_copy.rcmd_2[loop].bits.dscr_tap = 0;
        }
    else
        {
#ifdef REPEATER
        if (loop == 0)         /* RegR */
            {
            cu_reg_copy.tcmd_2[loop].bits.scr_tap = _HTUR;
            cu_reg_copy.rcmd_2[loop].bits.dscr_tap = _HTUR;
            }
        else                   /* RegC */
            {
            cu_reg_copy.tcmd_2[loop].bits.scr_tap = _HTUC;
            cu_reg_copy.rcmd_2[loop].bits.dscr_tap = _HTUC;
            }
#else
        cu_reg_copy.tcmd_2[loop].bits.scr_tap = htu_index;
        cu_reg_copy.rcmd_2[loop].bits.dscr_tap = htu_index;
#endif /* REPEATER */
        }

    tx_wr_ptr->tcmd_2 = cu_reg_copy.tcmd_2[loop].reg;
    rx_wr_ptr->rcmd_2 = cu_reg_copy.rcmd_2[loop].reg;

    tx_wr_ptr->tfifo_rst = 0x0;
    rx_wr_ptr->rfifo_rst = 0x0;

    return ;
}


/*
 * FUNCTION:   get_tx_wr_ptr
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function returns the TX_WR_PTR pointer address
 *             for the specified Channel Unit loop.
 *             
 * RETURN:     pointer to address
 *             0 - invalid LOOP
 *
 * NOTES:      
 *
 * CHANGES:    May 31, 1995    NEW   -   Dean Rasmussen
 */
TX_WR *get_tx_wr_ptr (BP_U_8BIT loop)
{
    switch ( loop )
        {
        case 0:
            return &(cu_wr->tx_wr_loop1);
            break;

        case 1:
            return &(cu_wr->tx_wr_loop2);
            break;

        case 2:
            return &(cu_wr->tx_wr_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return 0;
        }
}

/*
 * FUNCTION:   get_rx_wr_ptr
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function returns the RX_WR_PTR pointer address
 *             for the specified Channel Unit loop.
 *             
 * RETURN:     pointer to address
 *             0 - invalid LOOP
 *
 * NOTES:      
 *
 * CHANGES:    May 31, 1995    NEW   -   Dean Rasmussen
 */
RX_WR *get_rx_wr_ptr (BP_U_8BIT loop)
{
    switch ( loop )
        {
        case 0:
            return &(cu_wr->rx_wr_loop1);
            break;

        case 1:
            return &(cu_wr->rx_wr_loop2);
            break;

        case 2:
            return &(cu_wr->rx_wr_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return 0;
        }
}

/*
 * FUNCTION:   get_tx_rd_ptr
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function returns the TX_RD_PTR pointer address
 *             for the specified Channel Unit loop.
 *             
 * RETURN:     pointer to address
 *             0 - invalid LOOP
 *
 * NOTES:      
 *
 * CHANGES:    May 31, 1995    NEW   -   Dean Rasmussen
 */
TX_RD *get_tx_rd_ptr (BP_U_8BIT loop)
{
    switch ( loop )
        {
        case 0:
            return &(cu_rd->tx_rd_loop1);
            break;

        case 1:
            return &(cu_rd->tx_rd_loop2);
            break;

        case 2:
            return &(cu_rd->tx_rd_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return 0;
        }
}

/*
 * FUNCTION:   get_rx_rd_ptr
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function returns the RX_RD_PTR pointer address
 *             for the specified Channel Unit loop.
 *             
 * RETURN:     pointer to address
 *             0 - invalid LOOP
 *
 * NOTES:      
 *
 * CHANGES:    May 31, 1995    NEW   -   Dean Rasmussen
 */
RX_RD *get_rx_rd_ptr (BP_U_8BIT loop)
{
    switch ( loop )
        {
        case 0:
            return &(cu_rd->rx_rd_loop1);
            break;

        case 1:
            return &(cu_rd->rx_rd_loop2);
            break;

        case 2:
            return &(cu_rd->rx_rd_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return 0;
        }
}


/*
 * FUNCTION:   _CuResetRxFIFO
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function resets the Receiver FIFO for all loops
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    July 11, 1994    NEW   -   Dean Rasmussen
 */
void _CuResetRxFIFO (void)
{
    RX_WR BP_XDATA * rx_wr_ptr;
    BP_U_8BIT loop, loop_cntr;

    for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
        {
        loop = bp_position[loop_cntr];

        rx_wr_ptr = get_rx_wr_ptr(loop);

        if ( rx_wr_ptr == 0 )
            {
            continue;
            }
    
        rx_wr_ptr->rfifo_rst = 0x0;
        ignore_rfifo_err_ctr[loop] = IGNORE_RFIFO_THRESHOLD;
        }   /* end for loop_cntr */

    return;
}

/*
 * FUNCTION:   _CuResetReceiver
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function resets the HDSL channel receive elastic
 *             store FIFOs.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    July 12, 1994    NEW   -   Dean Rasmussen
 */
void _CuResetReceiver (void)
{
    cu_wr->common_wr_regs.rx_rst = 0;
    return;
}


/*
 * FUNCTION:   _CuResetTxFIFO
 *
 * PARAMETERS: none
 *
 * PURPOSE:    Resets the Transmitter FIFO for all loops.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    July 8, 1994    NEW   -   Dean Rasmussen
 */
void _CuResetTxFIFO (void)
{
    TX_WR BP_XDATA * tx_wr_ptr;
    BP_U_8BIT loop, loop_cntr;

    for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
        {
        loop = bp_position[loop_cntr];

        tx_wr_ptr = get_tx_wr_ptr(loop);

        if ( tx_wr_ptr == 0 )
            {
            continue;
            }


        tx_wr_ptr->tfifo_rst = 0x0;
        ignore_tfifo_err_ctr[loop] = IGNORE_TFIFO_THRESHOLD;

        }   /* end for loop_cntr */

    return; 
}

#ifdef REPEATER
/*
 * FUNCTION:   _CuRepeaterConfigure
 *
 * PARAMETERS: none
 *
 * PURPOSE:     Configure tcmd_2 for the repeater mode 
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    December 17, 1998    NEW   -   David Facinelli
 */
void _CuRepeaterConfigure(unsigned char state)
{
    TX_WR BP_XDATA * BP_XDATA tx_wr_ptr;

    unsigned char BP_XDATA loop;
 for (loop = 0; loop < 2; loop++)
    {   
         tx_wr_ptr = get_tx_wr_ptr(loop);
           
        cu_reg_copy.tcmd_2[loop].bits.repeat_en = state;
 
         _CU_DUMMY_WRITE(cu_reg_copy.tcmd_2[loop].reg);
         tx_wr_ptr->tcmd_2 = cu_reg_copy.tcmd_2[loop].reg;
     
     }

    return;
}
#endif /* REPEATER */
#endif /* CHAN_UNIT */


