#ifdef CHAN_UNIT
/*
 * PROGRAM NAME:  Bt8952/Bt8953 EVM System
 *
 * VERSION:       4.3
 *
 * FILENAME:      cu_map.c
 *
 * FILE CREATED:  January 19, 1995
 *
 * LAST MODIFIED: [4-5-99]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1995
 *                Brooktree Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   This file contains functions for the Tx/Rx Mapper and
 *                Router & Combination Tables.
 *
 *
 * FUNCTION LIST:
 *                _CuInitMapper()
 *                _CuWriteMapper()
 *                _CuWriteRouteTable()
 *                _CuWriteCombineTable()
 *                _CuInitRouteTable()
 *                _CuInitCombineTable()
 *                _CuDefaultRouteLoops()
 *                _CuDefaultCombineLoops()
 *                _CuWriteMapRouteCombine
 */

#include "chanunit.h"

/*
 * The 'route_loop[]' & 'combine_loop[]' arrays determine the values
 * to be loaded into the route/combine tables.
 *
 * There are up to 3 elements to these arrays.  The first element determines
 * the appropiate value for the first LIC detected.  The location of the
 * LIC is determined by the 'bp_position[]' array.
 */

BP_U_8BIT BP_XDATA route_loop[_NO_OF_LOOPS];
BP_U_8BIT BP_XDATA combine_loop[_NO_OF_LOOPS];

BP_U_8BIT BP_XDATA tmap_table[_TMAP_LENGTH][_NO_OF_LOOPS];
BP_U_8BIT BP_XDATA rmap_table[_RMAP_LENGTH][_NO_OF_LOOPS];


BP_U_8BIT BP_XDATA cu_tfifo_wl[_NO_OF_LOOPS];

/*
 * The 'route_table[]' & 'combine_table[]' arrays are used to store the
 * route/combine table values.  This allows the user to manipulate a
 * single location.  Call the '_CuWriteRouteTable()' or
 * '_CuWriteCombineTable()' to write these arrays to the part.
 */
BP_U_8BIT BP_XDATA route_table[_ROUTE_COMBINE_LENGTH];
BP_U_8BIT BP_XDATA combine_table[_ROUTE_COMBINE_LENGTH];


/* experiment */
#define NUM_PCMS  32

/*
 * FUNCTION:   _CuInitMapper
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function initializes the HDSL Transmitter/Receiver
 *             Payload Mappers (TMAP & RMAP) global arrays.
 *     
 * CALLED BY:  _CuInitChannelUnit
 * CALLED BY:  _CuReverseLoops
 * 
 * RETURN:     nothing
 *
 * NOTES:      The Unused TMAP codes select from DBANK_1, which is set
 *             to output an All Ones pattern (AIS).
 *
 * CHANGES:    January 17, 1995     V4.0 - New.  Added for Loop Reversal
 */
void _CuInitMapper (void)
{
    BP_U_8BIT loop_cntr, loop;
    
    for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
        {

        loop = bp_position[loop_cntr];

        switch ( rate_index )
            {
#ifdef CU_CUSTOM
            case _CU_CUSTOM:
                tmap_table[0][loop] = 0x00;
                tmap_table[1][loop] = 0x00;
                tmap_table[2][loop] = 0x00;
                tmap_table[3][loop] = 0x00;
                tmap_table[4][loop] = 0x00;
                tmap_table[5][loop] = 0x00;
                tmap_table[6][loop] = 0x00;
                tmap_table[7][loop] = 0x00;
                tmap_table[8][loop] = 0x2A;  /* use DBANK_2 */
                    
                rmap_table[0][loop] = 0x3f;
                rmap_table[1][loop] = 0x3f;
                rmap_table[2][loop] = 0x3f;
                rmap_table[3][loop] = 0x3f;
                rmap_table[4][loop] = 0x3f;
                rmap_table[5][loop] = 0x03;

                cu_tfifo_wl[loop] = 0x50;
                break;
#endif /* CU_CUSTOM */

#ifdef CU_1T1
            case _1T1:
                tmap_table[0][loop] = 0x00;
                tmap_table[1][loop] = 0x00;
                tmap_table[2][loop] = 0x00;
                tmap_table[3][loop] = 0x00;
                tmap_table[4][loop] = 0x00;
                tmap_table[5][loop] = 0x00;
                tmap_table[6][loop] = 0x00;
                tmap_table[7][loop] = 0x00;
                tmap_table[8][loop] = 0x00;           
                    
                /* Use 25 Channels (1 for F-bit) */
                rmap_table[0][loop] = 0x3f;
                rmap_table[1][loop] = 0x3f;
                rmap_table[2][loop] = 0x3f;
                rmap_table[3][loop] = 0x3f;
                rmap_table[4][loop] = 0x01;
                rmap_table[5][loop] = 0x00;

                cu_tfifo_wl[loop] = 0x50;
                break;
#endif /* CU_1T1 */

#ifdef CU_1E1
            case _1E1:

                /***************************************/
                /*  Bt8953A Rev C and RS8953B Support  */
                /***************************************/
               if((_CuChipVersion.major ==_BT8953A &&
                  _CuChipVersion.minor >= _BT8953A_REVC) ||
                  _CuChipVersion.major == _BT8953B)
                  {
                    tmap_table[0][loop] = 0x00;
                    tmap_table[1][loop] = 0x00;
                    tmap_table[2][loop] = 0x00;
                    tmap_table[3][loop] = 0x00;
                    tmap_table[4][loop] = 0x00;
                    tmap_table[5][loop] = 0x00;
                    tmap_table[6][loop] = 0x00;
                    tmap_table[7][loop] = 0x00;
                    tmap_table[8][loop] = 0xAA; /* use DBANK_2 */

                    rmap_table[0][loop] = 0x3F;
                    rmap_table[1][loop] = 0x3F;
                    rmap_table[2][loop] = 0x3F;
                    rmap_table[3][loop] = 0x3F;
                    rmap_table[4][loop] = 0x3F;
                    rmap_table[5][loop] = 0x03;
                  }
#else
                  {
                    /***************************************/
                    /*  Bt8953A Rev A/B and Bt8953 Proto   */
                    /***************************************/

                    /* Transmit 17 Channels on both. */
                    tmap_table[0][loop] = 0x00;    /* First 8 from FIFO */
                    tmap_table[1][loop] = 0x00;
                    tmap_table[2][loop] = 0xAA;    /* Next 4 from DBANK */
                    tmap_table[3][loop] = 0x00;    /* Next 4 from FIFO */
                    tmap_table[4][loop] = 0x00;    /* Next 2 - need to map */

                    rmap_table[0][loop] = 0x3F;
                    rmap_table[1][loop] = 0x03;
                    rmap_table[2][loop] = 0x3F;
                  }

                cu_tfifo_wl[loop] = 0x50;
                break;
#endif /* CU_1E1 */

#ifdef CU_2E1
            case _2E1:
                /* Transmit 17 Channels on both. */
                tmap_table[0][loop] = 0x00;
                tmap_table[1][loop] = 0x00;
                tmap_table[2][loop] = 0x00;
                tmap_table[3][loop] = 0x00;
                tmap_table[4][loop] = 0x08; /* use DBANK_2 */

                /*
                 * Core Frame Map:
                 * Loop 1: TS0,1,3,5,7,9,11,13,15,16,18,20,22,24,26,28,30
                 * Loop 2: TS0,2,4,6,8,10,12,14,16,17,19,21,23,25,27,29,31
                 *
                 * Master Loop Extracts TS0 & TS16.
                 * Other loop only extracts 15 timeslots
                 *
                 * When loop 1 extracts TS16, TS16 is payload byte 10 
                 * When loop 2 extracts TS16, TS16 is payload byte 9
                 */

                /* If the loop is 1 (also called channel 1) */
                if ( (cu_reg_copy.tzbit_pid[loop].reg & 0x7) == 1 ) 
                    {
                    /* If loop 1 is the 2E1 master */
                    if ( bp_position[loop] == _CuFlags._CuMasterLoop )
                        {
                        rmap_table[0][loop] = 0x3F;  /* extract TS0, byte 0   */
                        rmap_table[1][loop] = 0x3F;  /* extract TS16, byte 10 */
                        rmap_table[2][loop] = 0x1F;
                        }
                    else    /* loop 1 is the 2E1 slave */
                        {
                        rmap_table[0][loop] = 0x3E;  /* discard TS0, byte 0   */
                        rmap_table[1][loop] = 0x37;  /* discard TS16, byte 10 */
                        rmap_table[2][loop] = 0x1F;
                        }
                    }
                else        /* the loop is 2 (also called channel 2) */
                    {
                    /* If loop 1 is the 2E1 master */
                    if ( bp_position[loop] == _CuFlags._CuMasterLoop )
                        {
                        rmap_table[0][loop] = 0x3F;  /* extract TS0, byte 0   */
                        rmap_table[1][loop] = 0x3F;  /* extract TS16, byte 9  */
                        rmap_table[2][loop] = 0x1F;
                        }
                    else    /* loop 2 is the 2E1 slave */
                        {
                        rmap_table[0][loop] = 0x3E;  /* discard TS0, byte 0   */
                        rmap_table[1][loop] = 0x3B;  /* discard TS16, byte 9  */
                        rmap_table[2][loop] = 0x1F;
                        }
                    }

                /* These tmap and rmap registers were added to Bt8953A-Rev C 
                 *  and Rs8953B 
                 */
                tmap_table[5][loop] = 0x00;
                tmap_table[6][loop] = 0x00;
                tmap_table[7][loop] = 0x00;
                tmap_table[8][loop] = 0x00;

                rmap_table[3][loop] = 0x00;
                rmap_table[4][loop] = 0x00;
                rmap_table[5][loop] = 0x00;
        
                cu_tfifo_wl[loop] = 0x30;
                break;
#endif /* CU_2E1 */

#ifdef CU_3E1
            case _3E1:
                tmap_table[0][loop] = 0x00;
                tmap_table[1][loop] = 0x00;
                tmap_table[2][loop] = 0x00;
                tmap_table[3][loop] = 0x00;
                tmap_table[4][loop] = 0x00;

                if ( bp_position[loop] == _CuFlags._CuMasterLoop )
                    {
                    /* Use 12 Channels */
                    rmap_table[0][loop] = 0x3F;
                    rmap_table[1][loop] = 0x3F;
                    rmap_table[2][loop] = 0x00;
                    }
                else
                    {
                    /* Use 10 Channels */
                    rmap_table[0][loop] = 0x3E;
                    rmap_table[1][loop] = 0x3E;
                    rmap_table[2][loop] = 0x00;
                    }

                /* The following tmap and rmap registers were added to 
                 * Bt8953A-Rev C and Rs8953B. For earlier vesions of the Bt8953A, 
                 * it is OK to initialize these global variables because
                 * _CuWriteMapper() will not write them to the device. 
                 */
                tmap_table[5][loop] = 0x00;
                tmap_table[6][loop] = 0x00;
                tmap_table[7][loop] = 0x00;
                tmap_table[8][loop] = 0x00;

                rmap_table[3][loop] = 0x00;
                rmap_table[4][loop] = 0x00;
                rmap_table[5][loop] = 0x00;

                cu_tfifo_wl[loop] = 0x30;
                break;
#endif /* CU_3E1 */

#ifdef CU_2T1
            case _2T1:
                tmap_table[0][loop] = 0x00;
                tmap_table[1][loop] = 0x00;
                tmap_table[2][loop] = 0x00;
                tmap_table[3][loop] = 0x00;
                tmap_table[4][loop] = 0x00;

                if ( bp_position[loop] == _CuFlags._CuMasterLoop )
                    {
                    /* Use 13 Channels (1 for F-bit) */
                    rmap_table[0][loop] = 0x3F;
                    rmap_table[1][loop] = 0x3F;
                    rmap_table[2][loop] = 0x01;
                    }
                else
                    {
                    /* Use 12 Channels */
                    rmap_table[0][loop] = 0x3F;
                    rmap_table[1][loop] = 0x3F;
                    rmap_table[2][loop] = 0x00;
                    }

                /* Rev C only */
                tmap_table[5][loop] = 0x00;
                tmap_table[6][loop] = 0x00;
                tmap_table[7][loop] = 0x00;
                tmap_table[8][loop] = 0x00;

                rmap_table[3][loop] = 0x00;
                rmap_table[4][loop] = 0x00;
                rmap_table[5][loop] = 0x00;

               /*
                * The water_level should associate with the payload blocks or 
                * the SYNC_WORD instead of the master loop.
                */
                if ( cu_reg_copy.tcmd_1[bp_position[loop]].bits.sync_sel == 0 )
                    {
                    /* This loop transmit payloads 1-12 */
                    cu_tfifo_wl[loop] = 0x50;
                    }
                else
                    {
                    /* This loop transmit payloads 13-24 */
                    cu_tfifo_wl[loop] = 0x70;
                    }
                break;

#endif /* CU_2T1 */
            }   /* end switch rate */

        }   /* end for loop */

    return;
}



/*
 * FUNCTION:   _CuWriteMapper
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function Writes the HDSL Transmitter/Receiver
 *             Payload Mappers. (TMAP & RMAP)
 *             
 * RETURN:     nothing
 *
 * NOTES:      The global arrays 'tmap_table[]' & 'rmap_table[]' must
 *             be initialized before this function is called.
 *
 * CHANGES:    January 17, 1995     V4.0 - New.  Added for Loop Reversal
 */
void _CuWriteMapper (void)
{
    TX_WR BP_XDATA * tx_wr_ptr;
    RX_WR BP_XDATA * rx_wr_ptr;
    BP_U_8BIT loop_cntr, loop;

    for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
        {
        loop = bp_position[loop_cntr];

        tx_wr_ptr = get_tx_wr_ptr(loop);
        rx_wr_ptr = get_rx_wr_ptr(loop);

        if ( tx_wr_ptr == 0 || rx_wr_ptr == 0 )
            {
            continue;
            }

        tx_wr_ptr->tmap_1 = tmap_table[0][loop];
        tx_wr_ptr->tmap_2 = tmap_table[1][loop];
        tx_wr_ptr->tmap_3 = tmap_table[2][loop];
        tx_wr_ptr->tmap_4 = tmap_table[3][loop];
        tx_wr_ptr->tmap_5 = tmap_table[4][loop];

        rx_wr_ptr->rmap_1 = rmap_table[0][loop];
        rx_wr_ptr->rmap_2 = rmap_table[1][loop];
        rx_wr_ptr->rmap_3 = rmap_table[2][loop];

        /***************************************/
        /*  Bt8953A Rev C and RS8953B Support  */
        /***************************************/

        
        tx_wr_ptr->tmap_6 = tmap_table[5][loop];
        tx_wr_ptr->tmap_7 = tmap_table[6][loop];
        tx_wr_ptr->tmap_8 = tmap_table[7][loop];
        tx_wr_ptr->tmap_9 = tmap_table[8][loop];
            
        rx_wr_ptr->rmap_4 = rmap_table[3][loop];
        rx_wr_ptr->rmap_5 = rmap_table[4][loop];
        rx_wr_ptr->rmap_6 = rmap_table[5][loop];
        
        tx_wr_ptr->tfifo_wl = cu_tfifo_wl[loop];

        /* Reset the Tx/Rx FIFOs */
        tx_wr_ptr->tfifo_rst = 0x0;
        rx_wr_ptr->rfifo_rst = 0x0;

        ignore_rfifo_err_ctr[loop] = IGNORE_RFIFO_THRESHOLD;
        ignore_tfifo_err_ctr[loop] = IGNORE_TFIFO_THRESHOLD;
        }   /* end for loop_cntr */

    return;
}

/*
 * FUNCTION:   _CuWriteRouteTable
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function Writes the Route Table.  The global
 *             array 'route_table[]' contains the 64 Route values.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    January 19, 1995    NEW   -   Dean Rasmussen
 */
void _CuWriteRouteTable (void)
{
    COMMON_WR BP_XDATA * common_wr_ptr;
    BP_U_8BIT i;

    common_wr_ptr = &(cu_wr->common_wr_regs);

    cu_reg_copy.cmd_3.bits.route_en = 1;

    /* The channel unit uses a one byte latch register when writing to the 
     * route and combine tables.  When writing to these tables, the old data
     * in the latch may initially be strobed to a table entry before the new
     * data is completely latched.  To prevent this occurence, the 
     * _CU_DUMMY_WRITE macro pre-loads the new data by writing to a dummy 
     * address.  After the macro executes, the desired address is written
     * without any data transients occurring because the new data was 
     * pre-loaded into the latch buffer.
     */  
    _CU_DUMMY_WRITE(cu_reg_copy.cmd_3.reg);
    common_wr_ptr->cmd_3 = cu_reg_copy.cmd_3.reg;

    for ( i = 0 ; i < _ROUTE_COMBINE_LENGTH ; i++)
        {
        _CU_DUMMY_WRITE(route_table[i]);
        common_wr_ptr->route_tbl = route_table[i];
        }

    cu_reg_copy.cmd_3.bits.route_en = 0;
    _CU_DUMMY_WRITE(cu_reg_copy.cmd_3.reg);
    common_wr_ptr->cmd_3 = cu_reg_copy.cmd_3.reg;
    return;
}


/*
 * FUNCTION:   _CuWriteCombineTable
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function Writes the Comine Table.  The global
 *             array 'combine_table[]' contains the 64 combine values.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    January 19, 1995    NEW   -   Dean Rasmussen
 */
void _CuWriteCombineTable (void)
{
    BP_U_8BIT i;
    COMMON_WR BP_XDATA * common_wr_ptr;
   
    common_wr_ptr = &(cu_wr->common_wr_regs);

    cu_reg_copy.cmd_3.bits.comb_en = 1;

    /* See comment in _CuWriteRouteTable for _CU_DUMMY_WRITE */
    _CU_DUMMY_WRITE(cu_reg_copy.cmd_3.reg);
    common_wr_ptr->cmd_3 = cu_reg_copy.cmd_3.reg;

    for ( i = 0 ; i < _ROUTE_COMBINE_LENGTH ; i++)
        {
        _CU_DUMMY_WRITE(combine_table[i]);
        common_wr_ptr->combine_tbl = combine_table[i];
        }
    
    cu_reg_copy.cmd_3.bits.comb_en = 0;
    _CU_DUMMY_WRITE(cu_reg_copy.cmd_3.reg);
    common_wr_ptr->cmd_3 = cu_reg_copy.cmd_3.reg;

    return;
}


/*
 * FUNCTION:   _CuInitRouteTable
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function initializes the Route Table.  The global
 *             array 'route_loop[]' determines how to route.
 *             
 * RETURN:     nothing
 *
 * NOTES:      This function only fills the 'route_table' for standard
 *             2E1, 3E1, & 2T1 modes.
 *
 *             For custom route tables, modify the 'route_table' individually
 *             then call the '_CuWriteRoute_table()'
 *
 * CHANGES:    July 12, 1994    NEW   -   Dean Rasmussen
 *             January 17, 1995     V4.0 - Modified for Loop Reversal
 */
void _CuInitRouteTable (void)
{
    BP_U_8BIT i;

    for ( i=0 ; i < _ROUTE_COMBINE_LENGTH ; i++)
        {
        route_table[i] = 0;
        }

    switch ( rate_index )
        {
        /* PCM4 Experiment */
#ifdef CU_CUSTOM
        case _CU_CUSTOM:

            /* Unused are already discarded */
            for ( i = 0 ; i < NUM_PCMS ; i++ )
                {
                route_table[i] = route_loop[0];
                }
            break;
#endif /* CU_CUSTOM */

#ifdef CU_1T1
        case _1T1:
            /*
             * Core Frame Map:
             *
             * Loop 1: All on same loop, includes F-bit
             */
            for ( i=0 ; i < 25 ; i++)
                {
                route_table[i] = route_loop[0];
                }
            break;
#endif /* CU_1T1 */

#ifdef CU_1E1
        case _1E1:
            /*
             * Core Frame Map:
             *
             * Loop 1: All on same loop
             */
            for ( i=0 ; i < 32 ; i++)
                {
                route_table[i] = route_loop[0];
                }
            break;
#endif /* CU_1E1 */

#ifdef CU_2E1
        case _2E1:

            /*
             * Core Frame Map:
             *
             * Loop 1: TS0,1,3,5,7,9,11,13,15,16,18,20,22,24,26,28,30
             * Loop 2: TS0,2,4,6,8,10,12,14,16,17,19,21,23,25,27,29,31
             *
             */
            route_table[0] = route_loop[0] | route_loop[1];
            route_table[1] = route_loop[0];
            route_table[2] = route_loop[1];
            route_table[3] = route_loop[0];
            route_table[4] = route_loop[1];
            route_table[5] = route_loop[0];
            route_table[6] = route_loop[1];
            route_table[7] = route_loop[0];
            route_table[8] = route_loop[1];
            route_table[9] = route_loop[0];
            route_table[10] = route_loop[1];
            route_table[11] = route_loop[0];
            route_table[12] = route_loop[1];
            route_table[13] = route_loop[0];
            route_table[14] = route_loop[1];
            route_table[15] = route_loop[0];
            route_table[16] = route_loop[0] | route_loop[1];
            route_table[17] = route_loop[1];
            route_table[18] = route_loop[0];
            route_table[19] = route_loop[1];
            route_table[20] = route_loop[0];
            route_table[21] = route_loop[1];
            route_table[22] = route_loop[0];
            route_table[23] = route_loop[1];
            route_table[24] = route_loop[0];
            route_table[25] = route_loop[1];
            route_table[26] = route_loop[0];
            route_table[27] = route_loop[1];
            route_table[28] = route_loop[0];
            route_table[29] = route_loop[1];
            route_table[30] = route_loop[0];
            route_table[31] = route_loop[1];
            break;
#endif /* CU_2E1 */

#ifdef CU_3E1
        case _3E1:

            /*
             * Core Frame Map:
             *
             * Loop 1: TS0,1,4,7,10,13,16,17,20,23,26,29
             * Loop 2: TS0,2,5,8,11,14,16,18,21,24,27,30
             * Loop 3: TS0,3,6,9,12,15,16,19,22,25,28,31
             */

            route_table[0] = route_loop[0]| route_loop[1] | route_loop[2];
            route_table[1] = route_loop[0];
            route_table[2] = route_loop[1];
            route_table[3] = route_loop[2];
            route_table[4] = route_loop[0];
            route_table[5] = route_loop[1];
            route_table[6] = route_loop[2];
            route_table[7] = route_loop[0];
            route_table[8] = route_loop[1];
            route_table[9] = route_loop[2];
            route_table[10] = route_loop[0];
            route_table[11] = route_loop[1];
            route_table[12] = route_loop[2];
            route_table[13] = route_loop[0];
            route_table[14] = route_loop[1];
            route_table[15] = route_loop[2];
            route_table[16] = route_loop[0] | route_loop[1] | route_loop[2];
            route_table[17] = route_loop[0];
            route_table[18] = route_loop[1];
            route_table[19] = route_loop[2];
            route_table[20] = route_loop[0];
            route_table[21] = route_loop[1];
            route_table[22] = route_loop[2];
            route_table[23] = route_loop[0];
            route_table[24] = route_loop[1];
            route_table[25] = route_loop[2];
            route_table[26] = route_loop[0];
            route_table[27] = route_loop[1];
            route_table[28] = route_loop[2];
            route_table[29] = route_loop[0];
            route_table[30] = route_loop[1];
            route_table[31] = route_loop[2];
            break;
#endif /* CU_3E1 */

#ifdef CU_2T1
        case _2T1:

            /*
             * Core Frame Map:
             *
             * Loop 1: TS0,1,2,3,4,5,6,7,8,9,10,11,F
             * Loop 2: TS12,13,14,15,16,17,18,19,20,21,22,23,F
             *
             * map the F-bit to both loops (Timeslot 24)
             */

            route_table[0] = route_loop[0];
            route_table[1] = route_loop[0];
            route_table[2] = route_loop[0];
            route_table[3] = route_loop[0];
            route_table[4] = route_loop[0];
            route_table[5] = route_loop[0];
            route_table[6] = route_loop[0];
            route_table[7] = route_loop[0];
            route_table[8] = route_loop[0];
            route_table[9] = route_loop[0];
            route_table[10] = route_loop[0];
            route_table[11] = route_loop[0];
            route_table[12] = route_loop[1];
            route_table[13] = route_loop[1];
            route_table[14] = route_loop[1];
            route_table[15] = route_loop[1];
            route_table[16] = route_loop[1];
            route_table[17] = route_loop[1];
            route_table[18] = route_loop[1];
            route_table[19] = route_loop[1];
            route_table[20] = route_loop[1];
            route_table[21] = route_loop[1];
            route_table[22] = route_loop[1];
            route_table[23] = route_loop[1];
            route_table[24] = route_loop[0] | route_loop[1];
            break;
#endif /* CU_2T1 */
        }

    return;
}


/*
 * FUNCTION:   _CuInitCombineTable
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function initializes the Combination Table  The global
 *             array 'combine_loop[]' determines how to combine.
 *             
 * RETURN:     nothing
 *
 * NOTES:      This function only fills the 'combine_table' for standard
 *             2E1, 3E1, & 2T1 modes.
 *
 *             For custom combine tables, modify the 'combine_table'
 *             individually then call the '_CuWriteCombine_table()'
 *
 * CHANGES:    July 12, 1994    NEW   -   Dean Rasmussen
 *             January 17, 1995     V4.0 - Modified for Loop Reversal
 */
void _CuInitCombineTable (void)
{
    BP_U_8BIT i;

    for ( i = 0 ; i < _ROUTE_COMBINE_LENGTH ; i++)
        {
        combine_table[i] = 0;
        }

    switch ( rate_index )
        {
#ifdef CU_CUSTOM
        case _CU_CUSTOM:

            for ( i=0 ; i < NUM_PCMS ; i++ )
                {
                combine_table[i] = combine_loop[0];
                }
            break;
#endif /* CU_CUSTOM */

#ifdef CU_1T1
        case _1T1:
            /*
             * Core Frame Map:
             *
             * Loop 1: All on same loop, includes F-bit
             */
            for ( i=0 ; i < 25 ; i++)
                {
                combine_table[i] = combine_loop[0];
                }
            break;
#endif /* CU_1T1 */

#ifdef CU_1E1
        case _1E1:
            /*
             * Loop 1: All on same loop
             */
            for ( i=0 ; i < 32 ; i++)
                {
                combine_table[i] = combine_loop[0];
                }
            break;
#endif /* CU_1E1 */

#ifdef CU_2E1
        case _2E1:
            /*
             * Core Frame Map:
             *
             * Loop 1: TS0,1,3,5,7,9,11,13,15,16,18,20,22,24,26,28,30
             * Loop 2: TS2,4,6,8,10,12,14,17,19,21,23,25,27,29,31
             *
             * TS0 & TS16 are extracted from Master Loop.
             */
            combine_table[0] = _CuFlags._CuMasterLoop + 1;
            combine_table[1] = combine_loop[0];
            combine_table[2] = combine_loop[1];
            combine_table[3] = combine_loop[0];
            combine_table[4] = combine_loop[1];
            combine_table[5] = combine_loop[0];
            combine_table[6] = combine_loop[1];
            combine_table[7] = combine_loop[0];
            combine_table[8] = combine_loop[1];
            combine_table[9] = combine_loop[0];
            combine_table[10] = combine_loop[1];
            combine_table[11] = combine_loop[0];
            combine_table[12] = combine_loop[1];
            combine_table[13] = combine_loop[0];
            combine_table[14] = combine_loop[1];
            combine_table[15] = combine_loop[0];
            combine_table[16] = _CuFlags._CuMasterLoop + 1;
            combine_table[17] = combine_loop[1];
            combine_table[18] = combine_loop[0];
            combine_table[19] = combine_loop[1];
            combine_table[20] = combine_loop[0];
            combine_table[21] = combine_loop[1];
            combine_table[22] = combine_loop[0];
            combine_table[23] = combine_loop[1];
            combine_table[24] = combine_loop[0];
            combine_table[25] = combine_loop[1];
            combine_table[26] = combine_loop[0];
            combine_table[27] = combine_loop[1];
            combine_table[28] = combine_loop[0];
            combine_table[29] = combine_loop[1];
            combine_table[30] = combine_loop[0];
            combine_table[31] = combine_loop[1];
            break;
#endif /* CU_2E1 */

#ifdef CU_3E1
        case _3E1:
            /*
             * Core Frame Map:
             *
             * Loop 1: TS0,1,4,7,10,13,16,17,20,23,26,29
             * Loop 2: TS2,5,8,11,14,18,21,24,27,30
             * Loop 3: TS3,6,9,12,15,19,22,25,28,31
             *
             * TS0 & TS16 are extracted from Master Loop.
             */

            combine_table[0] = _CuFlags._CuMasterLoop + 1;
            combine_table[1] = combine_loop[0];
            combine_table[2] = combine_loop[1];
            combine_table[3] = combine_loop[2];
            combine_table[4] = combine_loop[0];
            combine_table[5] = combine_loop[1];
            combine_table[6] = combine_loop[2];
            combine_table[7] = combine_loop[0];
            combine_table[8] = combine_loop[1];
            combine_table[9] = combine_loop[2];
            combine_table[10] = combine_loop[0];
            combine_table[11] = combine_loop[1];
            combine_table[12] = combine_loop[2];
            combine_table[13] = combine_loop[0];
            combine_table[14] = combine_loop[1];
            combine_table[15] = combine_loop[2];
            combine_table[16] = _CuFlags._CuMasterLoop + 1;
            combine_table[17] = combine_loop[0];
            combine_table[18] = combine_loop[1];
            combine_table[19] = combine_loop[2];
            combine_table[20] = combine_loop[0];
            combine_table[21] = combine_loop[1];
            combine_table[22] = combine_loop[2];
            combine_table[23] = combine_loop[0];
            combine_table[24] = combine_loop[1];
            combine_table[25] = combine_loop[2];
            combine_table[26] = combine_loop[0];
            combine_table[27] = combine_loop[1];
            combine_table[28] = combine_loop[2];
            combine_table[29] = combine_loop[0];
            combine_table[30] = combine_loop[1];
            combine_table[31] = combine_loop[2];
            break;
#endif /* CU_3E1 */

#ifdef CU_2T1
        case _2T1:

            /*
             * Core Frame Map:
             *
             * Loop 1: TS0,1,2,3,4,5,6,7,8,9,10,11,F
             * Loop 2: TS12,13,14,15,16,17,18,19,20,21,22,23
             *
             * extract the F-bit from loop 1 (Timeslot 24)
             */
            combine_table[0] = combine_loop[0];
            combine_table[1] = combine_loop[0];
            combine_table[2] = combine_loop[0];
            combine_table[3] = combine_loop[0];
            combine_table[4] = combine_loop[0];
            combine_table[5] = combine_loop[0];
            combine_table[6] = combine_loop[0];
            combine_table[7] = combine_loop[0];
            combine_table[8] = combine_loop[0];
            combine_table[9] = combine_loop[0];
            combine_table[10] = combine_loop[0];
            combine_table[11] = combine_loop[0];
            combine_table[12] = combine_loop[1];
            combine_table[13] = combine_loop[1];
            combine_table[14] = combine_loop[1];
            combine_table[15] = combine_loop[1];
            combine_table[16] = combine_loop[1];
            combine_table[17] = combine_loop[1];
            combine_table[18] = combine_loop[1];
            combine_table[19] = combine_loop[1];
            combine_table[20] = combine_loop[1];
            combine_table[21] = combine_loop[1];
            combine_table[22] = combine_loop[1];
            combine_table[23] = combine_loop[1];

            /* really needs to be bp_position[master_loop] - change later */
            combine_table[24] = _CuFlags._CuMasterLoop + 1;
            break;
#endif /* CU_2T1 */
        }
   
    return;
}


/*
 * FUNCTION:   _CuDefaultRouteLoops
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function sets the Default 'route_loop[]' values.
 *             The default routes the Channel Unit Channels 1,2,3 to
 *             the LIC 1,2,3 respecitively.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    January 16, 1995    NEW   -   Ver 4.0
 */
void _CuDefaultRouteLoops (void)
{
    BP_U_8BIT loop_cntr;

    for ( loop_cntr = 0 ; loop_cntr < _NO_OF_LOOPS; loop_cntr++ )
        {
        switch ( bp_position[loop_cntr] )
            {
            case 0:
                route_loop[loop_cntr] = 0x01;
                break;

            case 1:
                route_loop[loop_cntr] = 0x04;
                break;

            case 2:
                route_loop[loop_cntr] = 0x10;
                break;

            default:
                route_loop[loop_cntr] = 0x00;
                break;

            }   /* end switch */
        }   /* end for loop_cntr */

    return;
}


/*
 * FUNCTION:   _CuDefaultCombineLoops
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function sets the Default 'combine_loop[]' values.
 *             The default combines the Channel Unit Channels 1,2,3 from
 *             the LIC 1,2,3 respecitively.
 *             
 * RETURN:     nothing
 *
 * NOTES:      Unused loops will be set to 0, because 'bp_position[]'
 *             is set to -1 if a loop is not present.
 *
 * CHANGES:    January 16, 1995    NEW   -   Ver 4.0
 */
void _CuDefaultCombineLoops (void)
{
    BP_U_8BIT loop_cntr;

    for ( loop_cntr = 0 ; loop_cntr < _NO_OF_LOOPS; loop_cntr++ )
        {
        combine_loop[loop_cntr] = bp_position[loop_cntr] + 1;
        }

    return;
}

/*
 * FUNCTION:   _CuWriteMapRouteCombine
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function Writes the Channel Unit Map/Route/Combine
 *             Tables and then resets the Rx Receiver & Tx/Rx FIFOs.
 *             
 * RETURN:     nothing
 *
 * NOTES:      The Route/Table array values must be set before this
 *             function is called.
 *
 * CHANGES:    January 20, 1995    NEW   -   V 4.0 : Added Route/Combine Tables
 */
void _CuWriteMapRouteCombine (void)
{
    _CuWriteMapper();

    _CuWriteRouteTable();
    _CuWriteCombineTable();

    _CuResetReceiver();

    _CuResetTxFIFO();
    _CuResetRxFIFO();

    return ;
}


#endif /* CHAN_UNIT */
