/***********************************************************/
/*    CU_LOOP.C                                            */
/*    (C) Copyright 1998 by Rockwell Semiconductor Systems */
/*                                                         */
/*                                                         */
/* Description:                                            */
/*    These are the channel unit functions for loop control*/
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*                                                         */
/* List of Functions found in this module:                 */
/*                                                         */
/*      - _CuReverseLoops(void);                           */
/*      - _CuSetMasterLoop(BP_U_8BIT loop);                */
/*      - _Configure_Channel_Blocking(void);               */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/* Aug 98 - New: Dean Rasmussen                            */
/* Oct 98 - New: Laura Yuan                                */
/*                                                         */
/***********************************************************/
#ifdef CHAN_UNIT

#include "chanunit.h"

/*
 * FUNCTION:   _CuReverseLoops
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function swaps the loops based on the ZBits detected.
 *             Need to modify the Route/Combine Tables accordingly.
 *             The '_CuFlags._CuMasterLoop' is set accordingly.
 *             
 *             It performs functions similiar to the HDSL startup routine:
 *             _CuInitChannelUnit().  One difference is that _CuInitChannelUnit() 
 *             calls  _CuDefaultRouteLoops() and  _CuDefaultCombineLoops() to setup 
 *             default values for route_loop[] and combine_loop[].  _CuReverseLoops()
 *             assumes that the PID or Sync Selects have been correctly configured.
 *
 *             Identical to  _CuInitChannelUnit(), it calls these routines:
 *                _CuInitMapper();
 *                _CuInitRouteTable();
 *                _CuInitCombineTable();
 *                _CuWriteMapRouteCombine();
 *
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    January 20, 1995    NEW   -   V 4.0
 */
void _CuReverseLoops (void)
{
    BP_U_8BIT loop_cntr, loop;
    BP_U_8BIT route_index[] = {0x01, 0x04, 0x10};

    switch ( rate_index )
        {
#ifdef CU_2T1
        case _2T1:
           /* 
            * There are two loop reversal cases; one is the real loop reversal
            * The other is the case when you reverse the loop back.
            */
         if ( cu_reg_copy.tcmd_1[0].bits.sync_sel == 1 )
            {
            /* 
             * This is loop 0 but transmitting SYNC_WORD_B
             * So this is a real loop reversal.
             */
            route_loop[0] = route_index[ bp_position[1] ];
            combine_loop[0] = bp_position[1] + 1;
            }
         else
            {
            /*
            * This is loop 0 and transmitting SYNC_WORD_A
            * So this is the case when the loop is reversed back
            */
            route_loop[0] = route_index[ bp_position[0] ];
            combine_loop[0] = bp_position[0] + 1;
            }

         if ( cu_reg_copy.tcmd_1[1].bits.sync_sel == 0 )
            {
            route_loop[1] = route_index[ bp_position[0] ];
            combine_loop[1] = bp_position[0] + 1;
            }
         else
            {
            route_loop[1] = route_index[ bp_position[1] ];
            combine_loop[1] = bp_position[1] + 1;
            }
         break;

#endif /* CU_2T1 */

        case _2E1:
        case _3E1:
        case _CU_CUSTOM:

         /*
          * Generic algorithm to handle 2E1 and 3E1
          */
          for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
            {
            loop = bp_position[loop_cntr];

            switch ( cu_rx_hoh.rzbit_pid[loop_cntr].reg & 0x7 )
                  {
                  case 1:
                      route_loop[0] = route_index[loop];
                      combine_loop[0] = loop + 1;
                      break;

                  case 2:
                      route_loop[1] = route_index[loop];
                      combine_loop[1] = loop + 1;
                      break;

                  case 4:
                      route_loop[2] = route_index[loop];
                      combine_loop[2] = loop + 1;
                      break;

                  default:
                      break;
                  }   /* end switch zbits */
              }   /* end for loop_cntr */
            break;
        }   /* end switch rate_index */

    /*
     * Re-init the Route/Combine Array Tables based on
     * the swapped loops.
     */

    _CuInitMapper();
    _CuInitRouteTable();
    _CuInitCombineTable();

#ifdef CHANNEL_BLOCK
    old_timeslot = 0xFFFFFFFF;
    _Configure_Channel_Blocking();
    old_timeslot = timeslot;
#endif /*CHANNEL_BLOCK*/

    /* _CuWriteMapRouteCombine() needs to be outside the 
     * #ifdef CHANNEL_BLOCK block because other routines, such as 
     * _CuInitMapper(), require that the channel unit registers be 
     * re-written.
     */
    _CuWriteMapRouteCombine();

    return ;
}


/*
 * FUNCTION:   _CuSetMasterLoop
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function makes the specified loop the master loop.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      The following is done:
 *
 *             - Set _CuFlags._CuMasterLoop
 *             - Set CMD_5
 *
 *               2T1
 *               - Modify 'combine_table[24]' to select from master loop
 *               - Modify RMAP Tables
 *                   - If master loop - include 13th byte (F-Bit)
 *                   - If Not master loop - discard 13th byte (F-Bit)
 *
 *               2E1 & 3E1
 *               - Modify Combine Table & RMAP so master loop extracts
 *                  timeslot 0 & 16 and non master loop ignores these
 *                  timeslots.
 *
 *             - Write the Combine/RMAP Tables
 *
 *
 * CHANGES:    September 29, 1995    NEW   -   Dean Rasmussen
 */
void _CuSetMasterLoop (BP_U_8BIT loop)
{
    _CuFlags._CuMasterLoop = loop;
    _CuWriteMasterCmd5();

    switch ( rate_index )
        {
#ifdef CU_2T1
        case _2T1:

            combine_table[24] = loop + 1;
            if ( _CuFlags._CuMasterLoop == bp_position[0] )
               {
               rmap_table[2][0] = 0x01;
               rmap_table[2][1] = 0x00;
               }
            else
               {
               rmap_table[2][0] = 0x00;
               rmap_table[2][1] = 0x01;
               }
            break;

#endif /* CU_2T1 */

        case _2E1:
        case _CU_CUSTOM:
            
            combine_table[0] = loop + 1;
            combine_table[16] = loop + 1;

            if ( _CuFlags._CuMasterLoop == bp_position[0] )
               {
               if ( (cu_reg_copy.tzbit_pid[bp_position[0]].reg & 0x7 ) == 1 )
                  {
                  rmap_table[0][0] = 0x3F;
                  rmap_table[1][0] = 0x3F;
                  rmap_table[0][1] = 0x3E;
                  rmap_table[1][1] = 0x3B;
                  }
               else
                  {
                  rmap_table[0][0] = 0x3F;
                  rmap_table[1][0] = 0x3F;
                  rmap_table[0][1] = 0x3E;
                  rmap_table[1][1] = 0x37;
                  }
               }
            else
               {
               if ( (cu_reg_copy.tzbit_pid[bp_position[1]].reg & 0x7 ) == 2 )
                  {
                  rmap_table[0][0] = 0x3E;
                  rmap_table[1][0] = 0x37;
                  rmap_table[0][1] = 0x3F;
                  rmap_table[1][1] = 0x3F;
                  }
               else
                  {
                  rmap_table[0][0] = 0x3E;
                  rmap_table[1][0] = 0x3B;
                  rmap_table[0][1] = 0x3F;
                  rmap_table[1][1] = 0x3F;
                  }
               }
            break;

#ifdef CU_3E1
        case _3E1:

            combine_table[0] = loop + 1;
            combine_table[16] = loop + 1;

            /*
             * Note:  This doesn't completely handle the case when loops are reversed.
             */
            if ( _CuFlags._CuMasterLoop == bp_position[0] )
               {
               rmap_table[0][0] = 0x3F;
               rmap_table[1][0] = 0x3F;
               
               rmap_table[0][1] = 0x3E;
               rmap_table[1][1] = 0x3E;

               rmap_table[0][2] = 0x3E;
               rmap_table[1][2] = 0x3E;
               }
            else if ( _CuFlags._CuMasterLoop == bp_position[1] )
               {
               rmap_table[0][0] = 0x3E;
               rmap_table[1][0] = 0x3E;
               
               rmap_table[0][1] = 0x3F;
               rmap_table[1][1] = 0x3F;

               rmap_table[0][2] = 0x3E;
               rmap_table[1][2] = 0x3E;
               }
            else
               {
               rmap_table[0][0] = 0x3E;
               rmap_table[1][0] = 0x3E;
               
               rmap_table[0][1] = 0x3E;
               rmap_table[1][1] = 0x3E;

               rmap_table[0][2] = 0x3F;
               rmap_table[1][2] = 0x3F;
               }

            break;
#endif /* CU_3E1 */
        }

    _CuWriteMapRouteCombine();

    return;
}

#ifdef CHANNEL_BLOCK  

/* 
 * FUNCTION:    _Configure_Channel_Blocking
 *
 * PARAMETER:   None
 *
 * PURPOSE:     This function is used to configure the fractional T1 or E1 
 *              operation based on the timeslot selection. The function will
 *              modify:
 *              (1) Transmit routing table
 *              (2) Receive combine table
 *              (3) Transmit payload mapping register
 *              (4) Receive payload mapping register
 *
 * E1 Application: 
 *
 * Payload:      17    16  15  14  13  12  11  10  9   8   7   6   5   4   3   2   1   0
 *             -------------------------------------------------------------------------
 * Loop1:  TS  DBANK   30  28  26  24  22  20  18  16  15  13  11  9   7   5   3   1   0
 * Loop2:  TS  DBANK   31  29  27  25  23  21  19  17  16  14  12  10  8   6   4   2   0
 *
 *                                                                     Full E1
 *              tmap[0]:    B3      B2      B1      B0
 *                          00      00      00      00                  0x00
 *              tmap[1]:    B7      B6      B5      B4                  
 *                          00      00      00      00                  0x00
 *              tmap[2]:    B11     B10     B9      B8
 *                          00      00      00      00                  0x00
 *              tmap[3]:    B15     B14     B13     B12
 *                          00      00      00      00                  0x00
 *              tmap[4]:                    B17     B16
 *                                          01      00                  0x04
 *
 *  Loop1:                                                             Full E1
 *              rmap[0]:    x   x   B5  B4  B3  B2  B1  B0      
 *                          -   -   1   1   1   1   1   1               0x3f
 *              rmap[1]:    x   x   B11 B10 B9  B8  B7  B6
 *                          -   -   1   1   1   1   1   1               0x3f
 *              rmap[2]:    x   x   B17 B16 B15 B14 B13 B12
 *                          -   -   0   1   1   1   1   1               0x1f
 *
 *  Loop2:                                                             Full E1
 *              rmap[0]:    x   x   B5  B4  B3  B2  B1  B0      
 *                          -   -   1   1   1   1   1   0               0x3e
 *              rmap[1]:    x   x   B11 B10 B9  B8  B7  B6
 *                          -   -   1   1   1   0   1   1               0x3b
 *              rmap[2]:    x   x   B17 B16 B15 B14 B13 B12
 *                          -   -   0   1   1   1   1   1               0x1f
 *  -------------------------------------------------------------------------------------
 *
 *  T1 Application: 
 *
 *  Payload:    12  11  10  9   8   7   6   5   4   3   2   1   0
 *              -------------------------------------------------
 *  Loop1:  TS  F   11  10  9   8   7   6   5   4   3   2   1   0
 *  Loop2:  TS  F   23  22  21  20  19  18  17  16  15  14  13  12  
 *
 *                                                                     Full T1
 *              tmap[0]:    B3      B2      B1      B0
 *                          00      00      00      00                  0x00
 *              tmap[1]:    B7      B6      B5      B4                  
 *                          00      00      00      00                  0x00
 *              tmap[2]:    B11     B10     B9      B8
 *                          00      00      00      00                  0x00              
 *              tmap[3]:    B15     B14     B13     B12                  
 *                          00      00      00      00                  0x00
 *              tmap[4]:    B19     B18     B17     B16
 *                          00      00      00      00                  0x00    
 *
 *  Loop1:                                                             Full T1
 *              rmap[0]:    x   x   B5  B4  B3  B2  B1  B0      
 *                          -   -   1   1   1   1   1   1               0x3f
 *              rmap[1]:    x   x   B11 B10 B9  B8  B7  B6
 *                          -   -   1   1   1   1   1   1               0x3f
 *              rmap[2]:    x   x   B17 B16 B15 B14 B13 B12
 *                          -   -   0   0   0   0   0   1               0x01
 *
 *  Loop2:                                                             Full T1
 *              rmap[0]:    x   x   B5  B4  B3  B2  B1  B0      
 *                          -   -   1   1   1   1   1   1               0x3f
 *              rmap[1]:    x   x   B11 B10 B9  B8  B7  B6
 *                          -   -   1   1   1   1   1   1               0x3f
 *              rmap[2]:    x   x   B17 B16 B15 B14 B13 B12
 *                          -   -   0   0   0   0   0   0               0x00
 *
 * NOTE:        This assumes that loop 0 is the master loop.
 */
void _Configure_Channel_Blocking ()
 {
#if defined (CU_2E1) || defined (CU_2T1)
     BP_U_8BIT i, pay_loc, channel, reg_loc, bit_loc; 
#endif 

#ifdef CU_2T1
     if ( rate_index == _2T1 )
      {
      old_timeslot ^= timeslot;
      for ( i = 0; i < 24; i ++ )
         {
         if ( i < 12 )
            {
            pay_loc = i;
            channel = cu_reg_copy.tcmd_1[bp_position[0]].bits.sync_sel;
            }
         else
            {
            pay_loc = i - 12;
            channel = cu_reg_copy.tcmd_1[bp_position[1]].bits.sync_sel;
            }

         if ( old_timeslot & ( (BP_U_32BIT)( (BP_U_32BIT)0x1 << (BP_U_32BIT)i ) ) )
            {
            if ( !( timeslot & ( (BP_U_32BIT)( (BP_U_32BIT)0x1 << (BP_U_32BIT)i ) ) ) )
               {
              /* 
               * Modify the corresponding entry in the transmit routing table. 
               * Discard the data in this timeslot, it won't go to TFIFO.
               */
               route_table[i] = 0x00;

              /* 
               * Modify the corresponging entry in the receive combine table.
               * Fill the timeslot with data from DBANK1.
               */
               combine_table[i] = 0x10;

              /* 
               * Modify the corresponding bits in the transmit payload mapping 
               */
               reg_loc = pay_loc/4;
               bit_loc = ( pay_loc - reg_loc * 4 ) * 2;
               tmap_table[reg_loc][channel] |= ( 0x01 << bit_loc );

              /* 
               * Modify the corresponding bits in the transmit payload mapping 
               */
               reg_loc = pay_loc/6;
               bit_loc = pay_loc - reg_loc * 6;
               rmap_table[reg_loc][channel] &= ~( 0x01 << bit_loc );
               }
            else
               {
               route_table[i] = 0x01 << (channel*2);
               combine_table[i] = channel + 1;

               reg_loc = pay_loc/4;
               bit_loc = ( pay_loc - reg_loc * 4 ) * 2;
               tmap_table[reg_loc][channel] &= ~( 0x01 << bit_loc );
                    
               reg_loc = pay_loc/6;
               bit_loc = pay_loc - reg_loc * 6;
               rmap_table[reg_loc][channel] |= ( 0x01 << bit_loc );
               }
            }
         }
      }
#endif /* CU_2T1 */

#ifdef CU_2E1
     if ( rate_index == _2E1 )
      {
      /* This shows the changed timeslot from last time */
      old_timeslot ^= timeslot;
      for ( i = 0; i < 32 ;i ++ )
         {
         if ( i < 16 )
            {
            pay_loc = ( i + 1 ) / 2;
            if ( ( cu_reg_copy.tzbit_pid[bp_position[0]].reg & 0x07 ) == 1 )
               {
               channel = ( i + 1 ) % 2;
               }
            else
               {
               channel = i % 2;
               }
            }
         else
            {
            pay_loc = i / 2 + 1;
            if ( ( cu_reg_copy.tzbit_pid[bp_position[0]].reg & 0x07 ) == 1 )
               {
               channel = i % 2;
               }
            else
               {
               channel = ( i + 1 ) % 2;
               }
            }

             if ( old_timeslot & ( (BP_U_32BIT)( (BP_U_32BIT)0x1 << (BP_U_32BIT)i ) ) )
               {
               if ( !( timeslot & ( (BP_U_32BIT)( (BP_U_32BIT)0x1 << (BP_U_32BIT)i ) ) ) )
                  {
                  /* 
                   * Modify the corresponding entry in the transmit routing table. 
                   * Discard the data in this timeslot, it won't go to TFIFO.
                   */
                  route_table[i] = 0x00;

                  /*
                   * Modify the corresponging entry in the receive combine table.
                   * Fill the timeslot with data from DBANK1.
                   */
                  combine_table[i] = 0x10;

                  /* 
                   * Modify the corresponding bits in the transmit & receive payload mapping 
                   */
                   switch ( i )
                     {
                  
                     case 0:
                        /* Timeslot0 is mapped to both loops */
                        tmap_table[0][0] |= 0x01;           /* From DBANK1 */
                        tmap_table[0][1] |= 0x01;
                        /* Only the master loop extracts TS0 */
                        rmap_table[0][_CuFlags._CuMasterLoop] &= 0xFE;
                        break;

                       
                     case 16:
                        /* Timeslot16 is mapped to both loops */
                        if ( ( cu_reg_copy.tzbit_pid[bp_position[0]].reg & 0x07 ) == 1 )
                           {
                           tmap_table[2][0] |= 0x04;
                           tmap_table[2][1] |= 0x01;
                           if ( _CuFlags._CuMasterLoop == 0 )
                              {
                              rmap_table[1][0] &= 0xF7;
                              }
                           else
                              {
                              rmap_table[1][1] &= 0xFB;
                              }
                           }
                        else
                           {
                           tmap_table[2][0] |= 0x01;
                           tmap_table[2][1] |= 0x04;
                           if ( _CuFlags._CuMasterLoop == 0 )
                              {
                              rmap_table[1][0] &= 0xFB;
                              }
                           else
                              {
                              rmap_table[1][1] &= 0xF7;
                              }
                           }
                        break;
                       
                        default:
                           /* TMAP */
                           reg_loc = pay_loc/4;
                           bit_loc = ( pay_loc - reg_loc * 4 ) * 2;
                           tmap_table[reg_loc][channel] |= ( 0x01 << bit_loc );
                           /* RMAP */
                           reg_loc = pay_loc/6;
                           bit_loc = ( pay_loc - reg_loc * 6 );
                           rmap_table[reg_loc][channel] &= ~( 0x01 << bit_loc );
                           break;
                           }/* End of Switch */
                        }
                     else /* It is changed back to be in use */
                        {
                    switch ( i )
                     {
                     case 0:
                        route_table[i] = 0x05;
                        combine_table[i] = 0x01 << _CuFlags._CuMasterLoop;
                        /* Timeslot0 is mapped to both loops */
                        tmap_table[0][0] &= 0xFC;
                        tmap_table[0][1] &= 0xFC;
                        if ( _CuFlags._CuMasterLoop == 0 )
                           {
                           rmap_table[0][0] |= 0x01;   
                           }
                         else
                           {
                           rmap_table[0][1] |= 0x01;
                           }
                        break;

                        
                     case 16:
                        route_table[i] = 0x05;
                        combine_table[i] = 0x01 << _CuFlags._CuMasterLoop;
                        /* Timeslot16 is mapped to both loops */
                        if (( cu_reg_copy.tzbit_pid[bp_position[0]].reg & 0x07 ) == 1 )
                           {
                           tmap_table[2][0] &= 0xF3;   
                           tmap_table[2][1] &= 0xFC;
                           if ( _CuFlags._CuMasterLoop == 0 )
                              {
                              rmap_table[1][0] |= 0x08;
                              }
                           else
                              {
                              rmap_table[1][1] |= 0x04;
                              }
                           }
                        else
                           {
                           tmap_table[2][0] &= 0xFC;   
                           tmap_table[2][1] &= 0xF3;
                           if ( _CuFlags._CuMasterLoop == 0 )
                              {
                              rmap_table[1][0] |= 0x04;
                              }
                           else
                              {
                              rmap_table[1][1] |= 0x08;
                              }
                           }
                        break;
                        default:
                           route_table[i] = 0x01 << (channel*2);
                           combine_table[i] = 0x01 << channel;
                           /* TMAP */
                           
                           reg_loc = pay_loc/4;
                           bit_loc = ( pay_loc - reg_loc * 4 ) * 2;
                           tmap_table[reg_loc][channel] &= ~( 0x01 << bit_loc );
                           /* RMAP */
                           reg_loc = pay_loc/6;
                           bit_loc = pay_loc - reg_loc * 6;
                           rmap_table[reg_loc][channel] |= ( 0x01 << bit_loc );
                           break;
                           }/* End of Switch */
                        }
                     }
                  }
               }
#endif /* CU_2E1 */

            /* _CuWriteMapRouteCombine() is called if required
             * by routines that call _Configure_Channel_Blocking ()
             */
            }

#endif /*CHANNEL_BLOCK*/
#endif /* CHAN_UNIT */
