
#ifdef CHAN_UNIT
/*
 * PROGRAM NAME:  Bt8952/Bt8953 EVM System
 *
 * VERSION:       4.3
 *
 * FILENAME:      cu_int.c
 *
 * FILE CREATED:  August 3, 1994
 *
 * LAST MODIFIED: [4-27-99]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1995
 *                Brooktree Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   This file contains the Channel Unit/Framer Interrupt
 *                functions.
 *
 * COMMENTS:      The interrupt routine polls each of the 3 channels/loops
 *                within the channel unit to determine which channel/loop
 *                caused the interrupt.
 *
 *                The DPLL error is the only Rx error that is enabled. 
 *                It is enabled in CMD_7.  A DPLL error causes an interrupt, sets 
 *                RX_ERR in IRR (0x1F), and sets the DPLL_ERR in ERR_STATUS (0x3C).
 *
 *                RFIFO and TFIFO errors do not cause interrupts because they
 *                are not enabled in RCMD_2 and TCMD_1, respectively.
 *                Specific TFIFO and RFIFO errors are checked in the interrupt
 *                routines by polling STATUS_1 for RFIFO errors and
 *                STATUS_3 for TFIFO errors.  Since they are not enabled,
 *                the IRR register can not be polled as an intermediate
 *                step to check for general Rx or Tx errors.  Also, the DPLL
 *                error also sets the RX_ERR in the IRR, so the individual error 
 *                bits for DPLL and RFIFO errors would still need to be checked. 
 *
 *                In the Rx/Tx Interrupt functions, the Rx/Tx FIFO gets
 *                reset whenever an error occurs.  Resetting the FIFO causes
 *                3 HDSL Frames to be lost (corrupted) which would cause
 *                another Rx/Tx FIFO error, so the first 3 passes into
 *                the Rx/Tx Interrupt function are Ignored after resetting
 *                the Rx/Tx FIFO.
 *
 *                Indicator bits are active low (0)
 *
 * FUNCTION LIST:
 *                _CuInterruptHandler()
 *                _CuTxInterrupt()
 *                _CuRxInterrupt()
 *                _CuDpllInterrupt()
 *                _CuFramerInterrupt()
 */


#include "chanunit.h"

#define IGNORE_LOOP_SYNC_THRESHOLD      3

void _CuTxInterrupt( BP_U_8BIT loop );
void _CuRxInterrupt( BP_U_8BIT loop );
void _CuDpllInterrupt();
void _CuResetRxFIFO_2(void);
void E1_Pairid_Validation( BP_U_8BIT loop );

#ifdef T1E1_FRAMER
    void _CuFramerInterrupt(void);
#endif /* T1E1_FRAMER */

BP_U_16BIT BP_XDATA error_ctrs[_CU_NUM_ERROR_CTRS][_NO_OF_LOOPS];

BP_U_8BIT BP_XDATA cu_system_sync_state = CU_ACQUIRING_SYNC;

BP_U_8BIT BP_XDATA loop_sync_state[_NO_OF_LOOPS];

BP_U_8BIT BP_XDATA ignore_rfifo_err_ctr[_NO_OF_LOOPS];
BP_U_8BIT BP_XDATA ignore_tfifo_err_ctr[_NO_OF_LOOPS];
BP_U_8BIT BP_XDATA ignore_loop_sync_ctr[_NO_OF_LOOPS];

BP_U_8BIT BP_XDATA crc_err_detected[_NO_OF_LOOPS];

#define IGNORE_DPLL_CTR_THRESHOLD   250 /* 1.5s */
BP_U_8BIT BP_XDATA ignore_dpll_ctr;

BP_U_8BIT BP_XDATA PID_usage;
BP_U_8BIT BP_XDATA expect_pid[_NO_OF_LOOPS];
BP_U_8BIT BP_XDATA cu_dpll_state_man;

/*
 * FUNCTION:   _CuInterruptHandler
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function determines what caused the interrupt and
 *             handles the event appropiately.
 *             
 * RETURN:     nothing
 *
 * NOTES:      The 'is_master_loop' flag only allows the DPLL & Framer
 *             Interrupt functions to occur every 6mS.  
 *
 * CHANGES:    July 12, 1994    NEW   -   Dean Rasmussen
 */
#ifdef C51
/* Using 8051,C51 compiler,Int vector at 0003H */
void _CuInterruptHandler (void) interrupt 2
#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section cu_int.1    Begins       <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Write the interrupt handler routine header.              */
    /* This routine should be entered whenever Channel Unit     */
    /* interrupt is activated.                                  */
    /*                                                          */
    /* Reference:                                               */
    /*                                                          */
    /*----------------------------------------------------------*/
void _CuInterruptHandler (void)
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section cu_int.1    Ends         <<<*/
    /*----------------------------------------------------------*/
#endif
{
    COMMON_RD BP_XDATA *common_rd_ptr;
    COMMON_WR BP_XDATA *common_wr_ptr;
#ifndef REPEATER
    ERR_STATUS err_stat;
    BP_U_8BIT loop_cntr;
#endif
    IRR irr, imr;
    BP_U_8BIT is_master_loop;

    common_rd_ptr = &(cu_rd->common_rd_regs);
    common_wr_ptr = &(cu_wr->common_wr_regs);

    irr.reg = common_rd_ptr->irr;
    imr.reg = common_rd_ptr->imr;

    /* 
     * Clear a local flag that indicates the loop is 
     * the master loop.  This flag may be set and checked
     * later in this routine.
     */

    is_master_loop = 0;

    /*----------------------------------------------------------*/
    /*                   Tx Interrupt Checks                    */
    /*----------------------------------------------------------*/

    /* For each loop:
     * 1. Check the IRR for an active interrupt AND 
     *    Check the IMR if an enabled interrupt.
     * 2. If active and enabled, then call _CuTxInterrupt to process 
     *    the interrupt.
     * 3. Clear the interrupt by clearing a bit in ICR.
     */
      
    if ( irr.bits.tx1 && !imr.bits.tx1 )
        {
        _CuTxInterrupt(0);
        common_wr_ptr->icr = 0xFE;
        }

    if ( irr.bits.tx2 && !imr.bits.tx2 )
        {
        _CuTxInterrupt(1);
        common_wr_ptr->icr = 0xFD;
        }

    if ( irr.bits.tx3 && !imr.bits.tx3 )
        {
        _CuTxInterrupt(2);
        common_wr_ptr->icr = 0xFB;
        }

    /*----------------------------------------------------------*/
    /*                   Rx Interrupt Checks                    */
    /*----------------------------------------------------------*/

    /* For each loop:
     * 1. Check the IRR for an active interrupt AND 
     *    Check the IMR if an enabled interrupt.
     * 2. If active and enabled, then call _CuRxInterrupt to process 
     *    the interrupt.
     * 3. Clear the interrupt by clearing a bit in ICR.
     * 4. Check if the loop is the master loop.
     * 5. If the master loop, set a flag when checking for errors
     *    such as DPLL errors in the active and enabled, then call _CuRxInterrupt to process 
     *    the interrupt.
     */

    if ( irr.bits.rx1 && !imr.bits.rx1 )
        {
        _CuRxInterrupt(0);
        common_wr_ptr->icr = 0xF7;
        if ( _CuFlags._CuMasterLoop == 0 )
            {
            is_master_loop = 1;
            }
        }

    if ( irr.bits.rx2 && !imr.bits.rx2 )
        {
        _CuRxInterrupt(1);
        common_wr_ptr->icr = 0xEF;
        if ( _CuFlags._CuMasterLoop == 1 )
            {
            is_master_loop = 1;
            }
        }

    if ( irr.bits.rx3 && !imr.bits.rx3 )
        {
        _CuRxInterrupt(2);
        common_wr_ptr->icr = 0xDF;
        if ( _CuFlags._CuMasterLoop == 2 )
            {
            is_master_loop = 1;
            }
        }

#ifdef REPEATER
    return;
#else
    /*----------------------------------------------------------*/
    /*                 System SYNC Status Check                 */
    /*----------------------------------------------------------*/
    /* Older implementation.  Still in the code to maintain
     * backward compatibility with older application software.
     * 
     * Determine a global sync status by looping through all
     * of the channel unit loops.  The API command:
     *  _BtStatus(_CU_COMMON, _CU_SYNC, 0, &cu_sync_status);
     * reads back this global status.
     * The newer software has separate ASMs for each loop.
     * It uses the API command for a single channel:
     *  _BtStatus(bp + _CU_CHAN1, _CU_SYNC, 0, &cu_sync_status);
      * This API command directly reads the channel unit register 
     */

    /* cu_system_sync_state is global variable that the
     * ActivationStateManager() reads to determine the sync
     * status of the channel unit.  First assume that the
     * system is In Sync, then check of the loops for any
     * out of sync loops.
     */
    cu_system_sync_state = CU_IN_SYNC;

    for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
        {
        /* loop_sync_state[] is a global array that is updated
         * in _CuRxInterrupt().
         */
        if ( loop_sync_state[ bp_position[loop_cntr] ] != CU_IN_SYNC )
            {
            cu_system_sync_state = CU_OUT_OF_SYNC;
            }
        }

    /*----------------------------------------------------------*/
    /*                 DPLL Interrupt Checks                    */
    /*----------------------------------------------------------*/

        /*
         * Bt8953A , RS8953B
         */

        if ( irr.bits.tx_err && !imr.bits.tx_err )
            {
            common_wr_ptr->icr = 0xBF;
            }

        if ( irr.bits.rx_err && !imr.bits.rx_err )
            {
            /*
             * Look at the ERR_STAT Reg to determine what caused
             * the RX_ERR interrupt
             */
            err_stat.reg = common_rd_ptr->err_status;

            /* dpll_err_en is set by the API command, _CU_TRANSMIT_PAYLOAD,
             * when the GOTO_ACTIVE_TX_RX_STATE is exited.
             */
            if ( err_stat.bits.dpll_err && cu_reg_copy.cmd_7.bits.dpll_err_en )
                {
                /* Reset the DPLL state machine to initiate error processing */
                cu_dpll_state_man = DPLL_MAX_GAIN_STATE;

                /* Call the DPLL state machine */
                _CuDpllInterrupt();
                }

            common_wr_ptr->icr = 0x7F;
            }
        else 
            {
            /*
             * Only handle DPLL state machine processing for the
             * Master Loop Rx Interrupt
             */
            if ( is_master_loop )
                {
                /* Call the DPLL state machine. If the DPLL state machine is 
                 * NOT in IDLE_GAIN_STATE, it will perform DPLL processing.
                 */
                _CuDpllInterrupt();
                }
            }

    /*----------------------------------------------------------*/
    /*                 Framer Status Check                      */
    /*----------------------------------------------------------*/

#ifdef T1E1_FRAMER
    /*
     * Only handle Framer Interrupts on Master Loop Rx Int
     */
    if ( is_master_loop )
        {
        _CuFramerInterrupt();
        }
#endif /* T1E1_FRAMER */

    return;

#endif /* REPEATER */
}


/*
 * FUNCTION:   _CuTxInterrupt
 *
 * PARAMETERS: loop - specified loop
 *
 * PURPOSE:    This function handles the Transmitter Interrupts
 *             
 * RETURN:     nothing
 *
 * NOTES:      see comments in File Description about FIFOs
 *
 * CHANGES:    July 11, 1994    NEW   -   Dean Rasmussen
 */
void _CuTxInterrupt (BP_U_8BIT loop)
{
    TX_RD BP_XDATA *tx_rd_ptr;
    TX_WR BP_XDATA *tx_wr_ptr;
#ifndef REPEATER
    TSTATUS_1 status_1;
    BP_U_8BIT reset_tx_flag;
#endif  /* REPEATER */

    switch ( loop )
        {
        case 0:
            tx_rd_ptr = &(cu_rd->tx_rd_loop1);
            tx_wr_ptr = &(cu_wr->tx_wr_loop1);
            break;

        case 1:
            tx_rd_ptr = &(cu_rd->tx_rd_loop2);
            tx_wr_ptr = &(cu_wr->tx_wr_loop2);
            break;

        case 2:
            tx_rd_ptr = &(cu_rd->tx_rd_loop3);
            tx_wr_ptr = &(cu_wr->tx_wr_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return;
        }
    /*----------------------------------------------------------*/
    /*               Indicator (IND) Bits - Set/Clear           */
    /*----------------------------------------------------------*/

    /* Individual bits are set and cleared in the global bytes:
     * cu_reg_copy.tind_lo[loop] and cu_reg_copy.tind_lo[loop] 
     * These bytes are initialized in _CuInitHdslLoops().
     * After the bits are set and cleared, then the global byte
     * is copied into the channel unit TIND register.
     */

    /*---------------------------------------------*/
    /*     CRC Error Processing                    */
    /*     If CRC Error:                           */
    /*        If device is not a repeater:         */
    /*           FEBE is cleared                   */
    /*        else device is a repeater:           */
    /*           RRBE is cleared for RegR          */
    /*           RCBE is cleared for RegC          */
    /*     else no CRC Error:                      */
    /*        If device is not a repeater:         */
    /*           FEBE is set                       */
    /*        else device is a repeater:           */
    /*           RRBE is set for RegR              */
    /*           RCBE is set for RegC              */
    /*                                             */
    /*     The indicator bits use negative         */
    /*     logic.  If a CRC error is detected,     */
    /*     the associated indicator bit is cleared */
    /*---------------------------------------------*/

    /*
     * _CuRxInterrupt()    detects a CRC error and sets crc_err_detected[]
     * when an error occurs.   This flag is cleared later in this
     * function.
     */
    if ( crc_err_detected[loop] )
        {
#ifndef REPEATER
        cu_reg_copy.tind_lo[loop].bits.febe = 0;
#else
        /* When the RegR detects a CRC error, it
         * transmits rrbe on both loops 
         */
        if (loop == 0)     /* RegR */
          {
            /* According to the HDSL standards, the rrbe bits
           * are cleared by the repeater "towards the LTU and NTU"
           */
          cu_reg_copy.tind_lo[0].bits.rrbe = 0;
          cu_reg_copy.tind_lo[1].bits.rrbe = 0;
          }

        /* When the RegC detects a CRC error, it
         * transmits rcbe on both loops 
         */
        else             /* RegC */
          {
            /* According to the HDSL standards, the rcbe bits
           * are cleared by the repeater "towards the LTU and NTU"
           */
          cu_reg_copy.tind_lo[0].bits.rcbe = 0;
          cu_reg_copy.tind_lo[1].bits.rcbe = 0;
          }
#endif  /* REPEATER */
        }
    else
        {
#ifndef REPEATER
        cu_reg_copy.tind_lo[loop].bits.febe = 1;
#else
        /* See comments above when the rrbe and rcbe bits are set */
      if (loop == 0)     /* RegR */
         {
         cu_reg_copy.tind_lo[0].bits.rrbe = 1;
         cu_reg_copy.tind_lo[1].bits.rrbe = 1;
         }
      else             /* RegC */
         {
         cu_reg_copy.tind_lo[0].bits.rcbe = 1;
         cu_reg_copy.tind_lo[1].bits.rcbe = 1;
         }
#endif  /* REPEATER */
        }

    /* clear the flag that _CuRxInterrupt() sets */
    crc_err_detected[loop] = 0;

    /*--------------------------------------------------------*/
    /*      LOSD Processing for Repeater                      */
    /*      FEBE Processing for Repeater                      */
    /*                                                        */
    /* In Section 5.6.6.1.1, the HDSL standards state:        */
    /* "the REG-specific overhead bits (eoc, rrbe, rega,      */
    /*  hrp, crc) are active, all other overhead bits,        */
    /*  are transferred transparently".                       */
    /*                                                        */
    /* Therefore, the LOSD and FEBE bits are passed through   */
    /* in repeater mode                                       */
    /*                                                        */
    /*--------------------------------------------------------*/
#ifdef REPEATER
    /* Determine whether device is RegR or RegC */
    if (loop == 0)  /* RegR */
       {
       /* Check if LOSD was received */
       if ( cu_rx_hoh.rind_lo[0].bits.losd )
          {
          /* Set and pass through LOSD to the RegC */
          cu_reg_copy.tind_lo[1].bits.losd = 1;
          }
       else
          {
          /* Clear and pass through LOSD to the RegC */
          cu_reg_copy.tind_lo[1].bits.losd = 0;
          }

       /* Check if FEBE was received */
       if ( cu_rx_hoh.rind_lo[0].bits.febe )
          {
          /* Set and pass through FEBE to the RegC */
          cu_reg_copy.tind_lo[1].bits.febe = 1;
          }
       else
          {
          /* Clear and pass through FEBE to the RegC */
          cu_reg_copy.tind_lo[1].bits.febe = 0;
          }
       }
    else  /* RegC */
       {
       /* Check if LOSD was received */
       if ( cu_rx_hoh.rind_lo[1].bits.losd )
          {
          /* Set and pass through LOSD to the RegR */
          cu_reg_copy.tind_lo[0].bits.losd = 1;
          }
       else
          {
          /* Clear and pass through LOSD to the RegR */
          cu_reg_copy.tind_lo[0].bits.losd = 0;
          }

       /* Check if FEBE was received */
       if ( cu_rx_hoh.rind_lo[1].bits.febe )
          {
          /* Set and pass through FEBE to the RegR */
          cu_reg_copy.tind_lo[0].bits.febe = 1;
          }
       else
          {
          /* Clear and pass through FEBE to the RegR */
          cu_reg_copy.tind_lo[0].bits.febe = 0;
          }
       }
#else
    /*----------------------------------------*/
    /*      LOSD Processing for Non-Repeater  */
    /*----------------------------------------*/
#ifdef CU_LED
    /*
     * If E1/T1 LOS & OOF detected, set the LOSD Indicator bit
     */
    if ( cu_pcm_led_block.bits.t1e1_err )
        {
        cu_reg_copy.tind_lo[loop].bits.losd = 0;
        }
    else
        {
        cu_reg_copy.tind_lo[loop].bits.losd = 1;
        }
#endif /* CU_LED   */
#endif /* REPEATER */

    /*----------------------------------------------------------*/
    /*               Indicator (IND) Bits - Final Output        */
    /*----------------------------------------------------------*/

    /* Write the TIND register */
    tx_wr_ptr->tind_lo = cu_reg_copy.tind_lo[loop].reg;

#ifdef REPEATER
    /*----------------------------------------*/
    /*          Z-BIT Processing              */
    /*----------------------------------------*/

    /* send the z-bit out on the RegC (loop 1).
     * These values are set equal to the values received by
     * the RegR (loop 0).  The RT connected
     * to the RegC will require a valid PID before
     * it proceeds to Activate TxRx State.
     * When the Channel Unit enters repeater mode, the
     * z-bits are automatically passed-through so this
     * code is not required but there is no harm in still
     * executing it.
       */
    if (loop == 1)     /* RegC */
        {
        /* set the loop 1 tx value = loop 0 rx value */
        tx_wr_ptr->tzbit_1 = cu_rx_hoh.rzbit_pid[0].reg;
        }
#endif

    /*----------------------------------------------------------*/
    /*                     EOC Processing                       */
    /*----------------------------------------------------------*/

#ifdef CU_EOC
#ifdef REPEATER
if (loop == 0)     /* RegR */
   {
   /* For the RegR, do not transmit the received data from 
      the RT if the    repeater is processing an EOC message.
      Repeater EOC processing occurs on loop 0, so check 
      EOC_STATUS_RUN on loop 0.  */
   if (!(eocCtrl[0].status & EOC_STATUS_RUN))
      {
      /* Transmit on loop 0, the EOC data received on loop 1 */
      tx_wr_ptr->teoc_lo = GET_LBYTE( eocReceiveMessage[1] );
      tx_wr_ptr->teoc_hi = GET_HBYTE( eocReceiveMessage[1] );
      }
   else
      {
      /* Transmit on loop 0, the EOC data processsed by
         the repeater and written into eocSendMessage[0] */
      tx_wr_ptr->teoc_lo = GET_LBYTE( eocSendMessage[0] );
      tx_wr_ptr->teoc_hi = GET_HBYTE( eocSendMessage[0] );
      }
   }
else                /* Reg C */
   {
   /* On the RegC, pass through the EOC data. 
    * Transmit on loop 1, the EOC data received on loop 0.
    * According to ETSI, the repeater is transparent to all 
    * messages in the direction LTU to NTU
    */
    tx_wr_ptr->teoc_lo = GET_LBYTE( eocReceiveMessage[0] );
    tx_wr_ptr->teoc_hi = GET_HBYTE( eocReceiveMessage[0] );
   }
#else
    /* 
     * Update TEOC register with eocSendMessage.
     */
    tx_wr_ptr->teoc_lo = GET_LBYTE( eocSendMessage[loop] );
    tx_wr_ptr->teoc_hi = GET_HBYTE( eocSendMessage[loop] );
#endif /* REPEATER */
#endif /* CU_EOC */
    
#ifdef REPEATER
    /* During normal operation, ignore FIFO errors
     * At initialization, _CuResetTxFIFO() is called 
     */
    return;
#else
    /*----------------------------------------------------------*/
    /*      Transmit Status 1 (STATUS_3) Processing             */
    /*----------------------------------------------------------*/

    /*
     * See COMMENTS in File Header
     */
    if ( ignore_tfifo_err_ctr[loop] > 0)
        {
        ignore_tfifo_err_ctr[loop]--;
        return ;
        }

    /* assume there is no error */
    reset_tx_flag = 0;

    /* read the status_1 byte from the channel unit*/
    status_1.reg = tx_rd_ptr->tstatus_1;
   
    /* error checking */
    if ( status_1.bits.tfifo_full )             /* FIFO FULL */
        {
        error_ctrs[_CU_TFIFO_FULL_CTR][loop]++;
        reset_tx_flag = 1;
        }
   
    if ( status_1.bits.tfifo_empty )            /* FIFO EMPTY */
        {
        error_ctrs[_CU_TFIFO_EMPTY_CTR][loop]++;
        reset_tx_flag = 1;
        }

    if ( status_1.bits.tfifo_slip )             /* FIFO SLIP */
        {
        error_ctrs[_CU_TFIFO_SLIP_CTR][loop]++;
        reset_tx_flag = 1;
        }

    if ( status_1.bits.stuff_err )              /* STUFF_ERR */
        {
        error_ctrs[_CU_TFIFO_STUFF_CTR][loop]++;
        reset_tx_flag = 1;
        }

    /* error processing */
    if ( reset_tx_flag )
        {
        tx_wr_ptr->tfifo_rst = 0x0;
        ignore_tfifo_err_ctr[loop] = IGNORE_TFIFO_THRESHOLD;
        }

#ifdef CU_LED
    cu_led_block[loop].bits.tfifo_err = reset_tx_flag;
#endif

    return;
#endif /* REPEATER */
}


/*
 * FUNCTION:   _CuRxInterrupt
 *
 * PARAMETERS: loop - specified loop
 *
 * PURPOSE:    This function handles the Receiver Interrupts
 *             
 * RETURN:     nothing
 *
 * NOTES:      see comments in File Description
 *
 * CHANGES:    July 11, 1994    NEW   -   Dean Rasmussen
 */
void _CuRxInterrupt (BP_U_8BIT loop)
{
    RX_RD BP_XDATA *rx_rd_ptr;
    RX_WR BP_XDATA *rx_wr_ptr;
    RSTATUS_2 rstatus_2;
#ifndef REPEATER
    RSTATUS_1 rstatus_1;
    BP_U_8BIT reset_rx_flag;
#endif
    switch ( loop )
        {
        case 0:
            rx_rd_ptr = &(cu_rd->rx_rd_loop1);
            rx_wr_ptr = &(cu_wr->rx_wr_loop1);
            break;

        case 1:
            rx_rd_ptr = &(cu_rd->rx_rd_loop2);
            rx_wr_ptr = &(cu_wr->rx_wr_loop2);
            break;

        case 2:
            rx_rd_ptr = &(cu_rd->rx_rd_loop3);
            rx_wr_ptr = &(cu_wr->rx_wr_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return;
        }
 
    /*----------------------------------------------------------*/
    /*               Indicator (IND) Bits - Set/Clear           */
    /*----------------------------------------------------------*/

    /* read the IND values into global variables */
    cu_rx_hoh.rind_lo[loop].reg = rx_rd_ptr->rind_lo;
    cu_rx_hoh.rind_hi[loop].reg = rx_rd_ptr->rind_hi;

    /*-------------------------------------*/
    /*     FEBE and LOSD Processing        */
    /*-------------------------------------*/

    if ( cu_rx_hoh.rind_lo[loop].bits.febe == 0 )
        {
        error_ctrs[_CU_FEBE_ERROR_CTR][loop]++;
        }

    if ( cu_rx_hoh.rind_lo[loop].bits.losd == 0 )
        {
        error_ctrs[_CU_LOSD_ERROR_CTR][loop]++;
        }

    /*-------------------------------------*/
    /*     RTR Processing                  */
    /*-------------------------------------*/

    /*
     * Check RTR (Ready to Receive).  Must be valid for 6 consective frames.
     */
    if ( cu_rx_hoh.rind_hi[loop].bits.rtr == 0 )
        {
        if ( !cu_rx_hoh.rtr_ind_status[loop].bits.valid )
            {
            cu_rx_hoh.rtr_ind_status[loop].bits.counter++;
            if ( cu_rx_hoh.rtr_ind_status[loop].bits.counter >= RTR_IND_THRESHOLD )
                {
                cu_rx_hoh.rtr_ind_status[loop].bits.valid = 1;
                }
            }
        }
    else
        {
        cu_rx_hoh.rtr_ind_status[loop].reg = 0;
        }

    /*----------------------------------------------------------*/
    /*               Performance Monitoring                     */
    /*----------------------------------------------------------*/

#ifdef PERF_MONITOR
    if ( system_flags[loop].bits.nm_update )
    {
        frame_cnt[loop]++;
        if (( frame_cnt[loop] + system_flags[loop].bits.slipflag ) == 167 )
        {
            frame_cnt[loop] = 0;
            system_flags[loop].bits.slipflag = (system_flags[loop].bits.slipflag + 1) % 2;
            running_seconds[loop]++;
            available_sec[loop]++;
            UpdatePMRecord(loop);
            _CuClearCounters( loop, _CU_CRC_ERR_CTR );
            _CuClearCounters( loop, _CU_FEBE_ERROR_CTR );           
        }
    }
#endif /* PERF_MONITOR */
    

    /*----------------------------------------------------------*/
    /*                     EOC Processing                       */
    /*----------------------------------------------------------*/

#ifdef CU_EOC
    /* 
     * get received eoc messages from REOC register 
     *     and call EOC handler routines.
     */
    SET_LBYTE(eocReceiveMessage[loop], rx_rd_ptr->reoc_lo);
    SET_HBYTE(eocReceiveMessage[loop], rx_rd_ptr->reoc_hi);

#ifdef REPEATER
   if (loop == 0)     /* RegR */
      {
      /* The RegR acts like an RT. Pass the loop number 
       * as a constant, and not off the stack.
       * Using a constant saves 1 byte
       */
      EocSlave(0);
      EocTaskHandler_RT(0);
      }
#else  /* No REPEATER */
    if ( htu_index == _HTUC )
    {
#ifdef HTUC
       EocMaster(loop);
       EocTaskHandler_CO(loop);
#endif /* HTUC */
    }
    else
    {
#ifdef HTUR
       EocSlave(loop);
       EocTaskHandler_RT(loop);
#endif /* HTUR */
    }
#endif /* REPEATER */
#endif /* CU_EOC */

    /*----------------------------------------*/
    /*          Z-BIT Processing              */
    /*----------------------------------------*/
#ifdef REPEATER

    if (loop == 0)  /* RegR */
       /*  
        * Read the z-bit received on the RegR (loop 0).
        * These values are sent out by _CuTxInterrupt 
        * on the RegC output (loop 1).  The RT connected
        * to the RegC requires a valid PID before
        * it proceeds to Activate TxRx State.
          *
        * Use the cu_rx_hoh.rzbit_pid[0].reg byte
        * In non-repeater applications, this byte is 
        * used by routines such as E1_Pairid_Validation().
        * The byte contains extra bits for PID processing.
        */
        {
        cu_rx_hoh.rzbit_pid[0].reg = rx_rd_ptr->rzbit_1;
        }
#else
    if ( cu_reg_copy.cmd_1.bits.e1_mode )
        {
        E1_Pairid_Validation( loop );
        }
#endif

    /* CU Sync Status and RFIFO Error Processing Overview:
     *
     *       Event                                            Action(s)
     * --------------------                               ---------------
     * 1. state = CU_OUT_OF_SYNC                 set ignore_loop_sync_ctr[]
     *                                           return;
     * 2. ignore_rfifo_err_ctr[] != 0            decrement ignore_rfifo_err_ctr[]
     *                                           return;
     * 3. Check for RFIFO errors                 set reset_rx_flag
     * 4. CU_IN_SYNC &&                          decrement ignore_loop_sync_ctr[]
     *    ignore_loop_sync_ctr[] != 0            return;
     * 5. CU_IN_SYNC &&                               
     *    ignore_loop_sync_ctr[] == 0            set reset_rx_flag
     */

    /*-------------------------------------*/
    /*   CU Losing Sync and CU Out if Sync */
    /*-------------------------------------*/

    rstatus_2.reg = rx_rd_ptr->rstatus_2;

    if ( rstatus_2.bits.sync_state == CU_LOSING_SYNC )
        {
        error_ctrs[_CU_OUT_OF_SYNC_CTR][loop]++;

#ifdef CU_LED
        cu_led_block[loop].bits.in_sync = 0;
        cu_led_block[loop].bits.out_of_sync = 1;
#endif
        }

    /*
     * Event 1. state = CU_OUT_OF_SYNC
     *
     * Must look for Out-of-Sync before 'ignore...ctr' check.
     */
    if ( rstatus_2.bits.sync_state == CU_OUT_OF_SYNC )
        {
        /* Check the previous value to check for a transition. 
         * Only increment once when the CU goes out of sync.
         */
        if ( loop_sync_state[loop] != CU_OUT_OF_SYNC )
            {
            error_ctrs[_CU_OUT_OF_SYNC_CTR][loop]++;
            }

        /* Update the previous value */
        loop_sync_state[loop] = CU_OUT_OF_SYNC;

#ifndef REPEATER
        /* Set a counter to ignore RFIFO errors after going
         * back into sync.
         */
        ignore_loop_sync_ctr[loop] = IGNORE_LOOP_SYNC_THRESHOLD;
#endif

        /*
         * don't deal with any other errors, FIFOs will get reset
         * when Receiver goes back in SYNC
         */
        return;
        }

#ifdef REPEATER
    if ( rstatus_2.bits.crc_err )       /* CRC_ERR */
        {
        error_ctrs[_CU_CRC_ERR_CTR][loop]++;
        crc_err_detected[loop] = 1;
        }

    /* In sync now, so update the previous value */
    if ( rstatus_2.bits.sync_state == CU_IN_SYNC ) 
        {
        loop_sync_state[loop] = CU_IN_SYNC;
        }

    return;
#else
    /*----------------------------------------------------------*/
    /*        Receive Status 1 (STATUS_1) Processing            */
    /*  (see the COMMENTS section at the top of this file)      */
    /*----------------------------------------------------------*/

    /* Event 2. ignore_rfifo_err_ctr[] != 0
     * reset_rx_flag processing sets ignore_rfifo_err_ctr[]
     */
    if ( ignore_rfifo_err_ctr[loop] > 0)
        {
        ignore_rfifo_err_ctr[loop]--;
        return ;
        }

    /* assume there is no error */
    reset_rx_flag = 0;

    /*
     * Event 3.  Check for RFIFO errors
     * If an error occurs:
     * 1. increment a counter in error_ctrs[] for debugging purposes
     * 2. set the reset_rx_flag for additional processing.
     */
    rstatus_1.reg = rx_rd_ptr->rstatus_1; 
   
    if ( rstatus_1.bits.rfifo_full )    /* RX_FIFO_FULL */
        {
        error_ctrs[_CU_RFIFO_FULL_CTR][loop]++;
        reset_rx_flag = 1;
        }

    if ( rstatus_1.bits.rfifo_empty )   /* RX_FIFO_EMPTY */
        {
        error_ctrs[_CU_RFIFO_EMPTY_CTR][loop]++;
        reset_rx_flag = 1;
        }

    if ( rstatus_1.bits.rfifo_slip )    /* RX_FIFO_SLIP */
        {
        error_ctrs[_CU_RFIFO_SLIP_CTR][loop]++;
        reset_rx_flag = 1;
        }

    /*----------------------------------------------------------*/
    /*        Receive Status 2 (STATUS_2) Processing            */
    /*----------------------------------------------------------*/

    /*------------------*/
    /*   CRC Check      */
    /*------------------*/
    if ( rstatus_2.bits.crc_err )       /* CRC_ERR */
        {
        error_ctrs[_CU_CRC_ERR_CTR][loop]++;
        crc_err_detected[loop] = 1;
        }

    /*---------------*/
    /* CU_IN_SYNC    */
    /*---------------*/
    if ( rstatus_2.bits.sync_state == CU_IN_SYNC )
        {
#ifdef CU_LED
        cu_led_block[loop].bits.in_sync = 1;
        cu_led_block[loop].bits.out_of_sync = 0;
#endif

        /*
         * Event 4. CU_IN_SYNC && ignore_loop_sync_ctr[] != 0
         * CU is out of sync or not in sync and stable 
         */
        if ( ignore_loop_sync_ctr[loop] > 0 )
            {
            ignore_loop_sync_ctr[loop]--;
            return;  /* don't deal with any RFIFO errors */
            }
        /*
         * Event 5. CU_IN_SYNC && ignore_loop_sync_ctr[] == 0
         * CU is in sync and stable.
         */
        else
            {
            /* Only reset the RFIFO one time by checking
             * for a transition from out of sync to in sync
             */
            if ( loop_sync_state[loop] != CU_IN_SYNC )
                {
                /* reset the RFIFO one time */
                reset_rx_flag = 1;
                }

            /* update the previous value */
            loop_sync_state[loop] = CU_IN_SYNC;
            }
        }

    /* Receiver error processing
     * reset_rx_flag is set above when either:
     * 1. the channel unit achieves a stable in sync condition
     * 2. a RFIFO error occurs and ignore_rfifo_err_ctr[] = 0
     */
    if ( reset_rx_flag )
        {
        if ( _CuFlags._CuMasterLoop == loop )
            {
            /* reset the PCM receiver */
            cu_wr->common_wr_regs.rx_rst = 0;

            /* issue RFIFO resets for all channels */
            _CuResetRxFIFO_2();
            }
        else
            {
            /* issue a RFIFO reset for one channel */
            rx_wr_ptr->rfifo_rst = 0x0;
            ignore_rfifo_err_ctr[loop] = IGNORE_RFIFO_THRESHOLD;
            }

        }

#ifdef CU_LED
    cu_led_block[loop].bits.rfifo_err = reset_rx_flag;
#endif /* CU_LED */

    return;
#endif /* REPEATER */
}


#ifndef REPEATER
/*
 * FUNCTION:   _CuDpllInterrupt
 *
 * PARAMETERS: mode : 1 - Dpll Interrupt detected
 *                    0 - No Dpll Interrupt detected
 *
 * PURPOSE:    This function handles any DPLL Interrupts
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      If a DPLL Interrupt error occurs, set the DPLL Phase/Gain
 *             registers to search for a wider band (MAX_DPLL_GAIN_VALUE).
 *             Use the DPLL Phase Error to determine when DPLL is stable.
 *
 *             When the DPLL captures a frequency, set the DPLL Phase/Gain
 *             register to narrowest gain value (MIN_DPLL_GAIN_VALUE).
 *             Reset the Rx FIFO when the Phase/Gain reaches 0.
 *
 *             The MIN_DPLL_GAIN_VALUE, MAX_DPLL_GAIN_VALUE, and
 *             IGNORE_DPLL_CTR_THRESHOLD can be adjusted for optimization
 *             for each specific application.
 *
 *             Phase = DPLL_GAIN.DC_GAIN2 (Integration Coefficient)
 *             Gain = DPLL_GAIN.DC_GAIN1 (Proportional Gain)
 *
 * CHANGES:    November 18, 1994    NEW   -   Dean Rasmussen
 *             October 3, 1998      Implement as state machine
 */
void _CuDpllInterrupt ()
{     
    COMMON_WR BP_XDATA *common_wr_ptr;
    COMMON_RD BP_XDATA *common_rd_ptr;

    common_wr_ptr = &(cu_wr->common_wr_regs);
    common_rd_ptr = &(cu_rd->common_rd_regs);

   
    switch ( cu_dpll_state_man )
        {
        case DPLL_IDLE_STATE:
            break;
        
        case DPLL_MAX_GAIN_STATE:
#ifdef CU_LED
            cu_pcm_led_block.bits.dpll_err = 1;
#endif /* CU_LED */

            error_ctrs[_CU_DPLL_ERROR_CTR][0]++;
            cu_reg_copy.dpll_gain.reg = MAX_DPLL_GAIN_VALUE;
            common_wr_ptr->dpll_gain = cu_reg_copy.dpll_gain.reg;
            cu_wr->common_wr_regs.dpll_rst = 0;

            ignore_dpll_ctr = DPLL_FRAME_WAIT;
            cu_dpll_state_man = DPLL_MED_GAIN_STATE;
            break;

        case DPLL_MED_GAIN_STATE:
            if ( ignore_dpll_ctr == 0 )
                {
                ignore_dpll_ctr = DPLL_FRAME_WAIT;
                cu_reg_copy.dpll_gain.reg = MED_DPLL_GAIN_VALUE;
                common_wr_ptr->dpll_gain = cu_reg_copy.dpll_gain.reg;
                cu_dpll_state_man = DPLL_MIN_GAIN_STATE;
                }
            else
                {
                if ((abs(common_rd_ptr->dpll_phs_err)) < 40 )
                    {
                    ignore_dpll_ctr--;
                    }
                else
                    {
                    ignore_dpll_ctr = DPLL_FRAME_WAIT;
                    }
                }
            break;

        case DPLL_MIN_GAIN_STATE:
            if (ignore_dpll_ctr == 0)
                {
                cu_reg_copy.dpll_gain.reg = MIN_DPLL_GAIN_VALUE;
                common_wr_ptr->dpll_gain = cu_reg_copy.dpll_gain.reg;
#ifdef CU_LED
                cu_pcm_led_block.bits.dpll_err = 0;
#endif /* CU_LED */

                cu_wr->common_wr_regs.rx_rst = 0;
                _CuResetRxFIFO_2();
                cu_dpll_state_man = DPLL_IDLE_STATE;
                }
            else
                {
                if ((abs(common_rd_ptr->dpll_phs_err)) < 20 )
                    {
                    ignore_dpll_ctr--;
                    }
                else
                    {
                    ignore_dpll_ctr = DPLL_FRAME_WAIT; 
                    }
                }
            break;
        }   /* end switch */
}

/*
 * FUNCTION:   _CuResetRxFIFO_2
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function resets the Receiver FIFO for all loops
 *             
 * RETURN:     nothing
 *
 * NOTES:      Duplicate of _CuResetRxFIFO(), can't call function
 *             from multiple segments (no re-entrant code).
 *
 * CHANGES:    March 13, 1997    NEW   -   Dean Rasmussen
 */
void _CuResetRxFIFO_2 (void)
{
    RX_WR BP_XDATA * rx_wr_ptr;
    BP_U_8BIT loop, loop_cntr;

    for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
        {
        loop = bp_position[loop_cntr];

        switch ( loop )
            {
            case 0:
                rx_wr_ptr = &(cu_wr->rx_wr_loop1);
                break;

            case 1:
                rx_wr_ptr = &(cu_wr->rx_wr_loop2);
                break;

            case 2:
                rx_wr_ptr = &(cu_wr->rx_wr_loop3);
                break;

            default:    /* skip if bit-pump isn't present */
                rx_wr_ptr = 0;
            }

        if ( rx_wr_ptr == 0 )
            {
            continue;
            }
    
        rx_wr_ptr->rfifo_rst = 0x0;
        ignore_rfifo_err_ctr[loop] = IGNORE_RFIFO_THRESHOLD;
        }   /* end for loop_cntr */

    return;
}

#ifdef T1E1_FRAMER
/*
 * FUNCTION:   _CuFramerInterrupt
 *
 * PARAMETERS: none
 *
 * PURPOSE:    This function polls the Framer Status registers.
 *             The LEDs are updately accordingly.
 *             
 * RETURN:     none
 *
 * NOTES:      This function gets called once every 6mS so LOSD error
 *             reporting can be generated.
 *
 * CHANGES:    July 11, 1994    NEW   -   Dean Rasmussen
 */
void _CuFramerInterrupt (void)
{
#ifdef BT8510_FRAMER
    FR8510_STATUS_0 status_0;
#endif

#ifdef BT8360_FRAMER
    FR8360_ALARM_STATUS alarm_status;
#endif

#ifdef BT8370_FRAMER
    FR8370_ALARM_1_INT_STATUS BP_XDATA alm_1_status;
#endif

    switch ( _FramerType )
        {

#ifdef BT8510_FRAMER
        case _BT8510:
            status_0 = fr8510_mode_ptr->status_0;
#ifdef CU_LED
            cu_pcm_led_block.bits.t1e1_sync = status_0.in_frame;
            cu_pcm_led_block.bits.t1e1_lof = status_0.loss_of_signal;
            cu_pcm_led_block.bits.t1e1_err = status_0.ais_signal_detect;
#endif /* CU_LED */
            break;
#endif /* BT8510_FRAMER */

#ifdef BT8360_FRAMER
        case _BT8360:
            alarm_status = fr8360_mode_ptr->alarm_status;
#ifdef CU_LED
            cu_pcm_led_block.bits.t1e1_sync = alarm_status.out_of_frame;
            cu_pcm_led_block.bits.t1e1_lof = alarm_status.loss_of_signal;
            cu_pcm_led_block.bits.t1e1_err = alarm_status.alarm_indication_signal;
#endif /* CU_LED */
#endif /* BT8360_FRAMER */
    
#ifdef BT8370_FRAMER
        case _BT8370:
            alm_1_status = fr8370_mode_ptr->alm_1_status;

#ifdef CU_LED
            cu_pcm_led_block.bits.t1e1_sync = ~alm_1_status.rlof;
            cu_pcm_led_block.bits.t1e1_lof = alm_1_status.rlof;
            cu_pcm_led_block.bits.t1e1_err = alm_1_status.rlos;
#endif /* CU_LED */
           break;
#endif

        default:
            return ;

        }   /* end switch framer */

    return;
}
#endif /* T1E1_FRAMER */

/*
 * Function: E1_Pairid_Validation
 *
 *           This function is used to validate the E1 pair id at the startup.
 *           A valid pair id should be unique and appear consistantly for
 *           6 frames.
 */
void E1_Pairid_Validation( BP_U_8BIT loop )
{
    BP_U_8BIT temp;
    RX_RD BP_XDATA *rx_rd_ptr;
    TX_WR BP_XDATA *tx_wr_ptr;

    switch ( loop )
    {
        case 0:
            rx_rd_ptr = &(cu_rd->rx_rd_loop1);
            tx_wr_ptr = &(cu_wr->tx_wr_loop1);
            break;

        case 1:
            rx_rd_ptr = &(cu_rd->rx_rd_loop2);
            tx_wr_ptr = &(cu_wr->tx_wr_loop2);
            break;

        case 2:
            rx_rd_ptr = &(cu_rd->rx_rd_loop3);
            tx_wr_ptr = &(cu_wr->tx_wr_loop3);
            break;

        default:    /* skip if bit-pump isn't present */
            return;
    }

    temp = rx_rd_ptr->rzbit_1;

    /* set/clear individual bits in cu_rx_hoh.rzbit_pid[] to match
       the 3 LSBs within the received z-bit byte      */
    cu_rx_hoh.rzbit_pid[loop].bits.pair_num1 = (temp & 0x01);
    cu_rx_hoh.rzbit_pid[loop].bits.pair_num2 = ((temp & 0x02) >> 1);
    cu_rx_hoh.rzbit_pid[loop].bits.pair_num3 = ((temp & 0x04) >> 2);

    /* check if the valid process has completed */
    if (  !cu_rx_hoh.rzbit_pid[loop].bits.valid )
    {
        /* 
         * Check if received PID is a valid pattern.
         * If it is, mark it on PID_usage.
         */

        /* temp holds the received PID */
        temp = cu_rx_hoh.rzbit_pid[loop].reg & 0x07;

        if ( (temp == 0x01) || (temp == 0x02) || ( temp == 0x04 ) )
        {
           /* verification setup not completed */
           if ( !cu_rx_hoh.rzbit_pid[loop].bits.start )
           {                             
              /* PID has not been reserved yet */
              if ( !(PID_usage & temp) )
              {
                 if ( htu_index == _HTUC )
                 {
                    /* If received PID = sent PID */
                    if ( temp == (0x01<<loop) )
                    {
                       /* reserve the PID number being used */
                       PID_usage ^= temp;

                       /* bypass the verification setup on next pass */
                       cu_rx_hoh.rzbit_pid[loop].bits.start = 1;

                       /* hold the first value received */
                       expect_pid[loop] = temp;
                    }
                 }
                 else   /* HTUR */
                 {
                    PID_usage ^= temp;
                    cu_rx_hoh.rzbit_pid[loop].bits.start = 1;
                    expect_pid[loop] = temp;
                 }
              }
            }
          }        
            
          /* verification setup has been completed above */
          if ( cu_rx_hoh.rzbit_pid[loop].bits.start )
          {
                   /* first value received = new value received */
                 if ( expect_pid[loop] == temp )
             {
                      /* increment the wait-for-6 counter */
                     cu_rx_hoh.rzbit_pid[loop].bits.counter++;

                     /* value received for 6 times */
                     if ( cu_rx_hoh.rzbit_pid[loop].bits.counter >= 6 )
                     {
                        /* tell the ASM that PID verification has passed */
                        cu_rx_hoh.rzbit_pid[loop].bits.valid = 1;
#ifdef HTUR
                    if ( htu_index == _HTUR )
                    {
                        /* set the transmit copy = received value */
                        cu_reg_copy.tzbit_pid[loop].reg = cu_rx_hoh.rzbit_pid[loop].reg;

                        /* set the transmit value = received value */
                        tx_wr_ptr->tzbit_1 = cu_reg_copy.tzbit_pid[loop].reg;
                    }
#endif /* HTUR */
                 }
             }
             else
             {
                 /* PID appears discontinuously */
                 cu_rx_hoh.rzbit_pid[loop].bits.start = 0;
                     cu_rx_hoh.rzbit_pid[loop].bits.counter = 0;
                     
                 if ( expect_pid[loop] == 0x01 )
                    PID_usage &= 0xFE;

                 if ( expect_pid[loop] == 0x02 )
                    PID_usage &= 0xFD;
                 
                 if ( expect_pid[loop] == 0x04 )
                    PID_usage &= 0xFB;

                 expect_pid[loop] = 0;
             } /* end else expect_pid */
         } /* end if start */
    }/* end of valid */ 
}/* End of function E1_Pairid_Validation */

#endif /* REPEATER    */
#endif /* CHAN_UNIT */


