#include "chanunit.h"
#ifdef CU_EOC
  BP_U_8BIT       BP_XDATA    eocHoldStates[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    eocReceiveBuffer[_NO_OF_LOOPS][16];
  BP_U_8BIT       BP_XDATA    dataRecCtr[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    dataBuffSize[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    dataBuffIndex[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    eocRecvCommand[_NO_OF_LOOPS]; 
  BP_U_8BIT       BP_IDATA    eocHandlerState[_NO_OF_LOOPS];
  BP_U_16BIT      BP_XDATA    eocSendMessage[_NO_OF_LOOPS]; 
  BP_U_16BIT      BP_XDATA    eocReceiveMessage[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    responseDelay[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    awaitResponse[_NO_OF_LOOPS];

  EOCTASKS        BP_XDATA    eocTasks[_NO_OF_LOOPS];
  EOC_STRUCT      BP_XDATA    eocStruct[_NO_OF_LOOPS];  
  UPDATE_FLAG     BP_XDATA    upDateFlags[_NO_OF_LOOPS];
  NEW_DATA_FLAG   BP_XDATA    newDataFlags[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    og_D_Value[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    needs_zip_start_check[_NO_OF_LOOPS]; 
  BP_U_8BIT       BP_XDATA    rt_open[_NO_OF_LOOPS];    
  BP_U_8BIT       BP_XDATA    eocCommandReg[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    eocWrCommandReg[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    eocRdCommandReg[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    RdWrUpdate[_NO_OF_LOOPS];  
  BP_U_8BIT       BP_XDATA    repeater_off[_NO_OF_LOOPS];  
  Bt8953_EOC_CTRL BP_XDATA    eocCtrl[_NO_OF_LOOPS];
  BP_U_8BIT                   loop;
  BP_U_8BIT       BP_XDATA    eocLoopReg[_NO_OF_LOOPS];
  BP_U_8BIT       BP_XDATA    cu_eoc_rd_nmr[_NO_OF_LOOPS];

#define EOC_CTRL_DECL_PTR          Bt8953_EOC_CTRL BP_XDATA *ctrl_ptr
#define EOC_CTRL_INIT_PTR(loop)    ctrl_ptr = &(eocCtrl[loop])

EOC_CTRL_DECL_PTR; 
/*Used Pointer to conserve code size ex.1 eocStruct[loop].wrRegSize is now eoc_struct_ptr->wrRegSize*/
/*                      ex.2 eocCtrl[loop].status is now ctrl_ptr->status */

TX_WR BP_XDATA * tx_wr_ptr; /* for CRC-ins */

#if defined (HTUC) && !defined (REPEATER)
/*-------------------------------------------------------------------------
             void    EocMaster( void )
         ==========================================
 
  written:    Gert Schmalfuss
 
  usage:
         Handles the EOC protocol for the COT side according ETR-152
         specification.
 
  application notes:
         Adapt modifications in the HDSL standard here but nothing else!
         
  revision history:
  -------------------------------------------------------------------------*/

void EocMaster ( BP_U_8BIT loop )
{

   BP_U_8BIT eocLow, eocHigh, eocHeader;
   EOC_CTRL_DECL_PTR;
   EOC_STRUCT_DECL_PTR;

   EOC_CTRL_INIT_PTR(loop); 
   EOC_STRUCT_INIT_PTR(loop);
       

    /***************************************************
                        EVERY_STATE:
     ***************************************************/
   /*Used to set eocHandlerState to IDLE if !ACTIVE_TX_RX_STATE*/

   /* eocReceiveMessage is updated each framer receive interrupt    
       Extract the low and high byte for future use.                */


   eocLow = GET_LBYTE( eocReceiveMessage[loop] );
   eocHigh = GET_HBYTE ( eocReceiveMessage[loop] );

   switch( eocHandlerState[loop] )
      {
        /****************************************************/
      case                EOC_IDLE:
        /****************************************************/
        
         if ( system_status[loop].bits.activation_state == ACTIVE_TX_RX_STATE )
            {
            /* : Initialization */ 
            ctrl_ptr->status = EOC_STATUS_AVAILABLE;
            eocHandlerState[loop] = EOC_CMD_READY;
            ctrl_ptr->command = 0;
            eocSendMessage[loop] = 0;
            CLEAR( eocTasks[loop].reg );
            system_flags[loop].bits.zip_status_ready = 0;
            }
         else
            ctrl_ptr->status = 0;
         return;

        /****************************************************/
      case                EOC_UTC_STATE:
        /****************************************************/
       
         /* Set EOC_STATUS_ERROR.  This bit is checked by EocTaskHandler()
          * and any application tasks suh as Zip_ZipStartValidationManager()
          * which may wish to try an EOC re-transmission upon an error.
          */
         ctrl_ptr->status |= EOC_STATUS_ERROR;

         /* - handle error condition automatically                    */
         if(ctrl_ptr->status & EOC_STATUS_HOLD)
            ctrl_ptr->command = EOC_CMD_HOLD;
         else
            ctrl_ptr->command = EOC_CMD_RTN;
         eocHandlerState[loop] = EOC_CMD_READY;
         return;

        /****************************************************/
      case                EOC_CMD_READY:
        /****************************************************/
     
         if(!(ctrl_ptr->status & EOC_STATUS_RUN))
            /* nothing to do -> leave                                */
            return;

         if((ctrl_ptr->command & 0xF0) == (EOC_CMD_WRITE_REG_A & 0xF0))
            {
            /* Check if eocCtrl.command is EOC_CMD_WRITE         */    
            /* handle WRITE request                                            */
            /* --------------------                                            */
            /*   Be aware that eocCtrl[loop].dataSizeCtr,                    */
            /*   eocCtrl[loop].dataBuffIndex and eocCtrl[loop].eocSlave are    */
            /*   correct set by task that requests EOC handling !            */
            /*   Furthermore the eocData has to be prepared before!            */
            ctrl_ptr->oddEvenCtr = EOC_ODD_INDICATOR;
            eocHandlerState[loop] = EOC_WRITE_REQUEST;
            }
         else if((ctrl_ptr->command & 0xF0) == (EOC_CMD_READ_REG_A & 0xF0))
            {
            /* Check if eocCtrl.command is EOC_CMD_READ         */    
            /* handle READ request                                    */
            /* -------------------                                    */
            /*   See note in WRITE request block above !            */
            ctrl_ptr->oddEvenCtr = EOC_ODD_INDICATOR;
            eocHandlerState[loop] = EOC_READ_REQUEST;
            }
         else
            {
            /* Check if eocCtrl.command is EOC_CMD          */    
            /* handle command request                           */
            /* ----------------------                           */
            /*   Be aware that ctrl_ptr->eocSlave is            */
            /*   correct set by task that requests EOC handling !    */
            /* eocHandlerState[loop]( EOC_CMD_REQUEST ); */
            eocHandlerState[loop] = EOC_CMD_REQUEST;
            }

         awaitResponse[loop] = ctrl_ptr->command;
         eocHeader = SEND_EOC_COMMAND | ctrl_ptr->eocSlave; 
         eocSendMessage[loop] = EocSendWord(awaitResponse[loop], eocHeader);
         responseDelay[loop] = 0;
         return;

        /****************************************************/
      case                EOC_CMD_REQUEST:
        /****************************************************/
        
         /* expect 1st response to transmitted command        */
         if((eocLow & EOC_ADDRESS_MASK) != ctrl_ptr->eocSlave)
            /* Do nothing */ ;
         else if((eocLow & EOC_MESSAGE_INDICATOR) &&
            (awaitResponse[loop] == EocGetData(eocLow, eocHigh )))
            {
            responseDelay[loop] = 0;
            eocHandlerState[loop] = EOC_CMD_RESPONS_1;
            return;
            }
         if(++responseDelay[loop] > WAIT_RESPONSE_TIME)
            {
            if(awaitResponse[loop] != EOC_CMD_END_OF_DATA)
               {
               ctrl_ptr->status |= EOC_STATUS_ERROR;
               }
            ctrl_ptr->status &= ~EOC_STATUS_RUN;
             
            /* Break here if command is not acknowledged.        */
            /* Don't go to UTC_STATE. The missing response may   */
            /* also be a RTN- or HOLD-response that is issued    */
            /* during UTC handling.                              */
            eocHandlerState[loop] = EOC_CMD_READY;
            }
         return;

        /****************************************************/
      case                EOC_READ_REQUEST:
        /****************************************************/
      
         /* expect 1st EOC_READ_DATA_REG response            */
         if((eocLow & EOC_ADDRESS_MASK) != ctrl_ptr->eocSlave)
            ;
         else if((eocLow & EOC_MESSAGE_INDICATOR) &&
            (awaitResponse[loop] == EocGetData(eocLow, eocHigh )))
            {
            responseDelay[loop] = 0;
            /* slave responds with 1st echo, so READ operation    */
            /* seems to be possible                                */
            eocHandlerState[loop] = EOC_RCMD_ACK_1;
            return;
            }
         if(++responseDelay[loop] > WAIT_RESPONSE_TIME)
            {
            /* slave doesn't respond or doesn't allow for        */
            /* the required READ operation                        */
            /* Go to UTC_STATE and finish with RTN or HOLD.        */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_DATA_READ_STATE:
        /****************************************************/
        
         /* expect 1st NEXT_ACK Response                        */
         if((eocLow & EOC_ADDRESS_MASK) != ctrl_ptr->eocSlave)
            ;
         else if((eocLow & EOC_MESSAGE_INDICATOR) &&
            (awaitResponse[loop] == EocGetData(eocLow, eocHigh)))
            {
            responseDelay[loop] = 0;
            /* slave responds to NEXT_BYTE command            */
            eocHandlerState[loop] = EOC_NEXT_ACK_1;
            return;
            }
         if(++responseDelay[loop] > WAIT_RESPONSE_TIME)
            {
            /* slave doesn't respond or doesn't allow for    */
            /* contiuous READ operation                        */
            /* Go to UTC_STATE and finish with RTN or HOLD.    */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_DATA_RECEIVE_STATE:
        /****************************************************/
      
         /* expect 1st data byte                                */
         if((eocLow & EOC_ADDRESS_MASK) != ctrl_ptr->eocSlave)
            ;
         else if(!(eocLow & EOC_MESSAGE_INDICATOR))
            {
            /* use awaitResponse[loop] as temporary data buffer    */
            awaitResponse[loop] = EocGetData(eocLow, eocHigh);
            responseDelay[loop] = 0;
            eocHandlerState[loop] = EOC_BYTE_READ_1;
            return;
            }
         else if(++responseDelay[loop] >  WAIT_RESPONSE_TIME)
            {
            /* slave doesn't respond with data                */
            /* Go to UTC_STATE and finish with RTN or HOLD.    */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_WRITE_REQUEST:
        /****************************************************/
      
         /* expect 1st EOC_WRITE_DATA_REG response            */
         if((eocLow & EOC_ADDRESS_MASK) != ctrl_ptr->eocSlave)
            return;
         else if((eocLow & EOC_MESSAGE_INDICATOR) &&
            (awaitResponse[loop] == EocGetData(eocLow, eocHigh)))
            {
            /* slave responds with 1st echo, so WRITE operation    */
            /* seems to be possible                                */
            responseDelay[loop] = 0;
            eocHandlerState[loop] = EOC_WCMD_ACK_1;
            return;
            }
         if(++responseDelay[loop] > WAIT_RESPONSE_TIME)
            {
            /* slave doesn't respond or doesn't allow for        */
            /* the required WRITE operation                        */
            /* Go to UTC_STATE and finish with RTN or HOLD.        */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_DATA_WRITE_STATE:
        /****************************************************/
      
         /* expect 1st DATA response                            */
         if((eocLow & EOC_ADDRESS_MASK) != ctrl_ptr->eocSlave)
            ;
         else if(!(eocLow & EOC_MESSAGE_INDICATOR) &&
            (awaitResponse[loop] == EocGetData(eocLow, eocHigh)))
            {
            /* slave has acknowledged byte 1st time            */
            responseDelay[loop] = 0;
            eocHandlerState[loop] = EOC_BYTE_WRITE_1;
            return;
            }
         if(++responseDelay[loop] > WAIT_RESPONSE_TIME)
            {
            /* slave doesn't respond or accept written data */
            /* Go to UTC_STATE and finish with RTN or HOLD.    */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_CMD_RESPONS_1:
        /****************************************************/
      
         /* expect 2nd response to previously transmitted command    */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            eocHandlerState[loop] = EOC_CMD_RESPONS_2;
         else
            eocHandlerState[loop] = EOC_CMD_REQUEST;
         return;

        /****************************************************/
      case                EOC_CMD_RESPONS_2:
        /****************************************************/
      
         /* expect 3rd response to previously transmitted command    */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            {
            /* EOC_STATUS_HOLD tracks whether to complete any subsequent
             * data read message sequences with EOC_CMD_RTN or EOC_CMD_HOLD.
             * If any command other than EOC_CMD_RTN or EOC_CMD_END_OF_DATA
             * is sent, then assume that the command is a latching (hold)
             * command.  EOC_CMD_RTN is the only 
             */
            if(awaitResponse[loop] == EOC_CMD_RTN)
               ctrl_ptr->status &= ~EOC_STATUS_HOLD;
            else if(awaitResponse[loop] != EOC_CMD_END_OF_DATA)
               ctrl_ptr->status |= EOC_STATUS_HOLD;
            ctrl_ptr->status &= ~EOC_STATUS_RUN;
            eocHandlerState[loop] = EOC_CMD_READY;
            }
         else
            eocHandlerState[loop] = EOC_CMD_REQUEST;
         return;

        /****************************************************/
      case                EOC_RCMD_ACK_1:
        /****************************************************/
      
         /* expect 2nd EOC_READ_DATA_REG response            */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            eocHandlerState[loop] = EOC_RCMD_ACK_2;
         else
            eocHandlerState[loop] = EOC_READ_REQUEST;
         return;

        /****************************************************/
      case                EOC_RCMD_ACK_2:
        /****************************************************/
      
         /* expect 3rd EOC_READ_DATA_REG response            */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            {
            /* Read operation is supported by slave, so continue    */
            /* with NEXT_BYTE command and wait for data                */
            responseDelay[loop] = 0;
            awaitResponse[loop] = EOC_CMD_NEXT_BYTE;
            eocHeader = SEND_EOC_COMMAND | READ_ODD_DATA | ctrl_ptr->eocSlave;
            eocSendMessage[loop] = EocSendWord(EOC_CMD_NEXT_BYTE, eocHeader);
            eocHandlerState[loop] = EOC_DATA_READ_STATE;
            }
         else
            eocHandlerState[loop] = EOC_READ_REQUEST;
         return;

        /****************************************************/
      case                EOC_NEXT_ACK_1:
        /****************************************************/
      
         if(!(eocLow & EOC_MESSAGE_INDICATOR))
            {
            /* receive 1st data byte and expect it twice again    */
            /* use awaitResponse[loop] as temporary data buffer        */
            awaitResponse[loop] = EocGetData(eocLow, eocHigh);
            responseDelay[loop] = 0;
            eocHandlerState[loop] = EOC_BYTE_READ_1;
            }
         else if(EocGetData(eocLow, eocHigh) == EOC_CMD_NEXT_BYTE)
            eocHandlerState[loop] = EOC_NEXT_ACK_2;

         else if(++responseDelay[loop] > WAIT_RESPONSE_TIME)
            {
            /* Go to UTC_STATE and finish with RTN or HOLD.    */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_NEXT_ACK_2:
        /****************************************************/
      
         if(!(eocLow & EOC_MESSAGE_INDICATOR))
            {
            /* receive 1st data byte and expect it twice again    */
            /* use awaitResponse[loop] as temporary data buffer        */
            awaitResponse[loop] = EocGetData(eocLow, eocHigh);
            responseDelay[loop] = 0;
            eocHandlerState[loop] = EOC_BYTE_READ_1;
            }
         else if(EocGetData(eocLow, eocHigh) == EOC_CMD_NEXT_BYTE)
            {
            eocHandlerState[loop] = EOC_DATA_RECEIVE_STATE;
            }
         else if(++responseDelay[loop] >  WAIT_RESPONSE_TIME)
            {
            /* Go to UTC_STATE and finish with RTN or HOLD.    */
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
      case                EOC_BYTE_READ_1:
        /****************************************************/
      
         /* expect 2nd EOC_READ_DATA_REG response (per byte)    */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            {
            /* receive data byte and expect it once again    */
            eocHandlerState[loop] = EOC_BYTE_READ_2;
            }
         else
            /* Old code:                                       */
            /* eocHandlerState[loop] = EOC_DATA_RECEIVE_STATE;  */
            /* New code:                                       */     
            /* The data has changed since the first data echo */
            /* Assume that the data is bogus and not valid    */
            /* Do NOT try to receive the new data             */
            {
            eocHandlerState[loop] = EOC_UTC_STATE;
            }
         return;

        /****************************************************/
       case                EOC_BYTE_READ_2:
        /****************************************************/
      
         /* expect 3rd EOC_READ_DATA_REG response (per byte)    */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            {
            /* remember: awaitResponse[loop] contains data byte  */
            eoc_struct_ptr->rdRegData[eocRdCommandReg[loop]&0xF][ctrl_ptr->dataBuffIndex++] = awaitResponse[loop];
            responseDelay[loop] = 0;
            if(--ctrl_ptr->dataSizeCtr != 0)
               {
               /* read one more byte        */
               awaitResponse[loop] = EOC_CMD_NEXT_BYTE;
               /* toggle ODD/EVEN bit        */
               ctrl_ptr->oddEvenCtr ^= EOC_ODD_INDICATOR;
               eocHeader = ctrl_ptr->oddEvenCtr | SEND_EOC_COMMAND | ctrl_ptr->eocSlave;
               eocSendMessage[loop] = EocSendWord(EOC_CMD_NEXT_BYTE, eocHeader);
               eocHandlerState[loop] = EOC_DATA_READ_STATE;
               }
            else
               {
               /* last byte has been red    */
               if(ctrl_ptr->status & EOC_STATUS_HOLD)
                  ctrl_ptr->command = EOC_CMD_HOLD;
               else
                  ctrl_ptr->command = EOC_CMD_RTN;
               eocHandlerState[loop] = EOC_CMD_READY;
               }
            }
         else
            /* Old code:                                       */
            /* eocHandlerState[loop] = EOC_DATA_RECEIVE_STATE;  */
            /* New code:                                       */     
            /* The data has changed since the first data echo */
            /* Assume that the data is bogus and not valid    */
            /* Do NOT try to receive the new data             */
            eocHandlerState[loop] = EOC_UTC_STATE;
         return;

        /****************************************************/
      case                EOC_WCMD_ACK_1:
        /****************************************************/
      
         /* expect 2nd EOC_WRITE_DATA_REG response (per byte)*/    
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            eocHandlerState[loop] = EOC_WCMD_ACK_2;
         else
            eocHandlerState[loop] = EOC_WRITE_REQUEST;
         return;

        /****************************************************/
      case                EOC_WCMD_ACK_2:
        /****************************************************/
      
         /* expect 3rd EOC_WRITE_DATA_REG response (per byte)*/ 
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            {
            /* slave is ready to accept data, so send it out*/ 
            responseDelay[loop] = 0;
            /* use awaitResponse[loop] as temporary data buffer    */
                      
            awaitResponse[loop] = eoc_struct_ptr->wrRegData[eocWrCommandReg[loop]&0xF][ctrl_ptr->dataBuffIndex++];

            eocHeader = SEND_ODD_DATA | ctrl_ptr->eocSlave;
            eocSendMessage[loop] = EocSendWord(awaitResponse[loop], eocHeader);
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
            }
         else
            eocHandlerState[loop] = EOC_WRITE_REQUEST;
         return;

        /****************************************************/
      case                EOC_BYTE_WRITE_1:
        /****************************************************/
      
         /* expect 2nd DATA response (per byte)                */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            eocHandlerState[loop] = EOC_BYTE_WRITE_2;
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_BYTE_WRITE_2:
        /****************************************************/
         /* expect 3rd DATA response (per byte)                */
         if(awaitResponse[loop] == EocGetData(eocLow, eocHigh))
            {
            responseDelay[loop] = 0;
            if(--ctrl_ptr->dataSizeCtr != 0)
               {
               /* write one more byte        */
               /* toggle ODD/EVEN bit        */
               ctrl_ptr->oddEvenCtr ^= EOC_ODD_INDICATOR;
               eocHeader = SEND_EVEN_DATA | ctrl_ptr->oddEvenCtr | ctrl_ptr->eocSlave;
               awaitResponse[loop] = eoc_struct_ptr->wrRegData[eocWrCommandReg[loop]&0xF][ctrl_ptr->dataBuffIndex++];
               eocSendMessage[loop] = EocSendWord(awaitResponse[loop], eocHeader);
               eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
               }
            else
               {
               /* last byte has been written                    */
               /* finish transmission with END_OF_DATA command    */
               awaitResponse[loop] = ctrl_ptr->command = EOC_CMD_END_OF_DATA;
               eocHeader = SEND_EOC_COMMAND | ctrl_ptr->eocSlave;
               eocSendMessage[loop] = EocSendWord(awaitResponse[loop], eocHeader);
               responseDelay[loop] = 0;
               eocHandlerState[loop] = EOC_CMD_REQUEST;
               }
            }
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;
         }
      }

/*--------------------------------------------------------------------
             void    EocTaskHandler_CO( void )
         ============================================
 
  written:    Gert Schmalfuss
 
  usage:
         Serves the interface to the EOC master by preparing the
         EOC command to send and by setting buffer index and buffer
         counters for data transmission. Checks if the EOC master is
         available or ready and notifies application side or performs
         application specific actions after end of EOC communication.
 
  application notes:
         Implementation of additional EOC features should be done here.
 
         See description of EOC-Master interface in software doc
         to figure out how to do this.
 
  revision history:
  --------------------------------------------------------------------*/

void EocTaskHandler_CO ( BP_U_8BIT loop )
{

   EOC_CTRL_DECL_PTR;
   EOC_STRUCT_DECL_PTR;

   EOC_CTRL_INIT_PTR(loop); 
   EOC_STRUCT_INIT_PTR(loop);

  
   /* if an EOC process is already in progress, then don't restart */
   if( !eocTasks[loop].bits.eocTaskRun )
      {
   
      if(!eocTasks[loop].reg || !(ctrl_ptr->status & EOC_STATUS_AVAILABLE))
      /* nothing to do or DSL not ready                    */
      return;

      if(ctrl_ptr->status & EOC_STATUS_BUSY)
      /* EOC Handler is reserved by other task            */
      return;
    
      /* reserve EOC Handler                                    */
      ctrl_ptr->status |= EOC_STATUS_BUSY;
   
      /* check out, what task to perform and prepare action   */

      if ( eocTasks[loop].bits.eocCommand)
         {
         ctrl_ptr->command = eocCommandReg[loop];
         }

      if ( eocTasks[loop].bits.eocWrTask)
         {
         ctrl_ptr->command = eocWrCommandReg[loop]; /*eocWrCommandReg is set by dsl_api.c*/
         ctrl_ptr->dataBuffIndex = 0;
         ctrl_ptr->dataSizeCtr = eoc_struct_ptr->wrRegSize[eocWrCommandReg[loop]&0xF];
         }

      if ( eocTasks[loop].bits.eocRdTask)
         {
         ctrl_ptr->command = eocRdCommandReg[loop];
         ctrl_ptr->dataBuffIndex = 0;
         ctrl_ptr->dataSizeCtr = eoc_struct_ptr->rdRegSize[eocRdCommandReg[loop]&0xF];

         if(needs_zip_start_check[loop] && eocRdCommandReg[loop] == EOC_CMD_READ_REG_D )
            {
            ctrl_ptr->dataSizeCtr = 1; /* For ZipStart to work the reg must be 1 byte */
            }
         }

      /* Initialized to access RT, API allows user configurability EOC_REPEATER_MODE */
     if (repeater_off[loop])
         {    
           ctrl_ptr->eocSlave = EOC_ADDRESS_RT;
         }
     else
         {
           ctrl_ptr->eocSlave = EOC_ADDRESS_REP;
         }
    /* Set the EOC Status byte. 
     *
     * Don't change EOC_STATUS_AVAILABLE which was previously set
     * by EocMaster() after it completes initialization and leaves
     * the EOC_IDLE state.
     *        
     * EOC_STATUS_RUN tells the EocMaster() that a command needs
     * to be sent.  The flag is checked in EOC_CMD_READY which is
     * normal EOC master state.  This flag is cleared when EOC
     * EocMaster() completes the message tranmission sequence
     * either successfully or unsuccessfully.  Iif it is unsuccessful, 
     * then EocMaster() sets EOC_STATUS_ERROR.
     *
     * The following statement clears out the EOC_STATUS_ERROR
     * bit.  This bit would have been set by EocMaster() and checked 
     * by EocTaskHandler() or other routines such as Zip_ZipStartValidationManageron the previous
     * transmission
     * 
     */
      ctrl_ptr->status = EOC_STATUS_AVAILABLE | EOC_STATUS_RUN;

      /* Don't start-up the EOC process on the next pass
       * eocTaskRun is checked at the top of this if statement 
       */
      SET( eocTasks[loop].bits.eocTaskRun );
      return;
      }

   if(ctrl_ptr->status & EOC_STATUS_RUN)
      /* EOC-Handler is not ready yet    */
      return;

 
#ifdef SER_COM
   /* task has been finished ..        */
   if(!(ctrl_ptr->status & EOC_STATUS_ERROR))
      {
      /* .. successful                */
      /* no response if eocRegF should be written or else answer to   */
      /* previously received and still unacknowledged host request    */    
      /* Used for special API over EOC commands                       */
      /*
       * Notify the generic EOC status request's result is ready.
       */

      if (eocTasks[loop].bits.eocRdTask)
         {
         if (eocRdCommandReg[loop] == EOC_CMD_READ_REG_F)
            {
            newDataFlags[loop].reg |= (BP_S_16BIT)READ_REGF_FLAG;
            }
         else if (needs_zip_start_check[loop]&&(eocRdCommandReg[loop] == EOC_CMD_READ_REG_D))
            {
            system_flags[loop].bits.zip_status_ready = 1;
            }
         else
            {
            newDataFlags[loop].reg |= (1 <<(eocRdCommandReg[loop]&0xF));
            }
         }
      }
#endif

   /* release EOC Handler    */
   ctrl_ptr->status &= ~EOC_STATUS_BUSY;
   /* finish action        */
   CLEAR( eocTasks[loop].reg );
}
#endif /* HTUC */ 

#ifdef HTUR

/*----------------------------------------------------------------------
 
             void    HandleValidRtn[loop]( void )
         ============================================
 
  written:    Gert Schmalfuss
 
  usage:
         Function that is called if a RTN (return to normal) command
         is received from the COT. If there are pending so called 
         'latched' commands there are done some steps to bring back
         the RT to the 'normal' state.
 
  application notes:
         Insert here more 'latched' commands if new features are
         implemented.
 
  revision history:
  -----------------------------------------------------------------------*/

void HandleValidRtn ( BP_U_8BIT loop )
{
   TX_WR BP_XDATA * tx_wr_ptr;       /* for insert CRC */
   tx_wr_ptr = get_tx_wr_ptr(loop);  /* for insert CRC */

   if(eocHoldStates[loop] & EOC_REQ_CCRC_RT)
      {
      cu_reg_copy.tcmd_1[loop].bits.icrc_err = 0;
      tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;
      eocLoopReg[loop] &=~EOC_REQ_CCRC_RT;
      }
   if(eocHoldStates[loop] & EOC_REQ_LOOP_RT) 
      {
      eocLoopReg[loop] |= EOC_REQ_LOOP_RT_OFF;/* need a flag in CuHandleFlags to turn off loopback in CuHandleFlags*/
      }
   if(eocHoldStates[loop] & EOC_REQ_ALOOP_RT)         
      {
      eocLoopReg[loop] |= EOC_REQ_ALOOP_RT_OFF;/* need a flag in CuHandleFlags to turn off loopback in CuHandleFlags*/
      }
   if(eocHoldStates[loop] & EOC_REQ_CCRC_CO)
      {
#ifdef REPEATER
      cu_reg_copy.tcmd_1[loop].bits.icrc_err = 0;
      tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;
      eocLoopReg[loop] &=~EOC_REQ_CCRC_CO;
#endif /*REPEATER turns off CRC at CO for regen */
      }
   CLEAR(eocHoldStates[loop]);
   eocSendMessage[loop] = EocSendWord(EOC_CMD_RTN, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
   eocHandlerState[loop] = EOC_ECHO_RESPONSE;
   }


/*--------------------------------------------------------------------
             void    HandleValidHold[loop]( void )
         =============================================
 
  written:    Gert Schmalfuss
 
  usage:
         Function that is called if a HOLD command is received from 
         the COT. If there are pending so called 'latched' commands
         there RT has to respond with an HOLD echo or else it has to
         respond with RTN.
 
  application notes:
         If 'eocHoldStates[loop]' is also used on extentions related to the
         EOC commands there should be no modifications necessary.
         
  revision history:
  --------------------------------------------------------------------*/

void HandleValidHold ( BP_U_8BIT loop )
{
   if( eocHoldStates[loop] )
      eocSendMessage[loop] = EocSendWord(EOC_CMD_HOLD, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
   else
      eocSendMessage[loop] = EocSendWord(EOC_CMD_RTN, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
   eocHandlerState[loop] = EOC_ECHO_RESPONSE;
}


/*------------------------------------------------------------------------
             void    EocSlave( void )
         ==========================================
 
  written:    Gert Schmalfuss
 
  usage:
         Handles the EOC protocol for the RT side according ETR-152
         specification.
         For a detailed description see VPG4-EVM software manual and
         HDSL specification ETR-152!
 
  application notes:
         Adapt modifications in the HDSL standard here but nothing else!
         
  revision history:
  -----------------------------------------------------------------------*/

void EocSlave ( BP_U_8BIT loop )
{
   BP_U_8BIT      eocLoByte, eocHiByte;
   BP_U_8BIT      currEocCmd, i, j;
   EOC_STRUCT_DECL_PTR;

   /*  ignore data and commands with unknown EOC-address  */
   eocLoByte = GET_LBYTE( eocReceiveMessage[loop] );
   eocHiByte = GET_HBYTE( eocReceiveMessage[loop] );
   EOC_STRUCT_INIT_PTR(loop);

   if ( system_status[loop].bits.activation_state != ACTIVE_TX_RX_STATE )
      {
#ifdef ZIP_START
      needs_zip_start_check[loop] = 1;/* if line goes down reset ZipStart enable for this loop*/
#endif /*ZIP_START*/
      }

  
   if((eocLoByte & EOC_ADDRESS_MASK) != EOC_OWN_ADDRESS)
      {
#ifndef REPEATER  /* NTU (RT) */
      /* As defined by the HDSL specifications, such as TS-101-135, Section 5.5.1:
       * "The NTU if not properly addressed, shall insert the 'Hold State' 
       *  message with the NTU-address "00" in the direction NTU -> LTU."
       */
      eocSendMessage[loop] = EocSendWord(EOC_CMD_HOLD, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
#endif
      return;
      }
 
   switch( eocHandlerState[loop] )
      {
        /****************************************************/
      case                EOC_IDLE:
        /****************************************************/
      
         /*           Framer is not synchronized                */   
         if ( system_status[loop].bits.activation_state == ACTIVE_TX_RX_STATE )
            {
            /* Even though eocRecvCommand[loop] is set to zero, if the COT is not reset, then 
             * the COT's eocSendMessage[] will not be reset to zero.  If the COT is not reset,
             * then the slave will process the previous command stored in the COT eocSendMessage[]
             * buffer.
             */
            eocRecvCommand[loop] = 0;
            eocHoldStates[loop] = 0;
            eocSendMessage[loop] = EocSendWord(EOC_CMD_RTN, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
            eocHandlerState[loop] = EOC_ECHO_RESPONSE;
            system_flags[loop].bits.co_check_zip = 0;
            }
         return;

        /****************************************************/
      case                EOC_UTC_STATE:
        /****************************************************/
       
         eocRecvCommand[loop] = EocGetData(eocLoByte, eocHiByte);
         if(eocRecvCommand[loop] == EOC_CMD_RTN)
            eocHandlerState[loop] = EOC_UTC_RTN_1;
         else if(eocRecvCommand[loop] == EOC_CMD_HOLD)
            eocHandlerState[loop] = EOC_UTC_HOLD_1;
         return;

        /*****************************************************/
      case                EOC_ECHO_RESPONSE:
        /*****************************************************/
        /*      Ready for receiption of a new command        */
        /* This state is the normal opeation state.  The EOC */
        /* slave stays here while waiting for a new command  */
        /*****************************************************/
#ifdef REPEATER
         /* Clear a flag for EOC processing in progress.  TxInterrupt()
          * checks this flag to determine whether to transmit repeater
          * or HTUR data to the COT.  According to the HDSL specs,
          * the repeater normally pass through the HTUR except when
          * it receives a command and is processing it.
          * Rather than clear this flag when an operation completes
          * in many states, it is cleared here in one place.
          * 
          */
         eocCtrl[loop].status &= ~EOC_STATUS_RUN;
#endif
         /* rt_open[] means that the RT EOC is open or available.
          * A true value means the read registers can be be updated
          * in the background without corrupting the transmitted data.
          * _CuHandleFlags() performs the background processing. 
          */
         SET(rt_open[loop]);
         if(eocLoByte & EOC_MESSAGE_INDICATOR)
            {
            currEocCmd = EocGetData(eocLoByte, eocHiByte);
            if(currEocCmd != eocRecvCommand[loop])
               {
               eocRecvCommand[loop] = currEocCmd;
               eocHandlerState[loop] = EOC_CMD_RESPONS_1;
               /*
               * EocSlave should responde to a received command 
               * with its echo.
               */
               eocSendMessage[loop] = eocReceiveMessage[loop];    

               /* See comment several lines above where this flag is cleared */
               eocCtrl[loop].status |= EOC_STATUS_RUN;
               }    
            }
         return;

        /****************************************************/
      case                EOC_CMD_RESPONS_1:
        /****************************************************/
      
         /* expect previously received command for 2nd time  */
         if(eocRecvCommand[loop] != EocGetData(eocLoByte, eocHiByte))
            {
            eocRecvCommand[loop] = 0;
            eocHandlerState[loop] = EOC_ECHO_RESPONSE;
            }
         else
            eocHandlerState[loop] = EOC_CMD_RESPONS_2;
         return;

        /****************************************************/
      case                EOC_CMD_RESPONS_2:
        /****************************************************/
       
         /* expect previously received command for 3rd time  */
         if(eocRecvCommand[loop] != EocGetData(eocLoByte, eocHiByte))
            {
            eocRecvCommand[loop] = 0;
            eocHandlerState[loop] = EOC_ECHO_RESPONSE;
            }
         else
            {
            /* Initially checks for Read or Write */

            switch(eocRecvCommand[loop]&0xF0)
               {
               case EOC_CMD_READ_REG_0:
                  /* Set a task flag to indicate the read register task has started.
                   * This flag is read by EocTaskHandler_RT to take appropriate action. 
                   * In this case, it is a semphore that to block any updates to the 
                   * read registers during the EOC read sequence.
                   */
                  SET( eocTasks[loop].bits.eocRdTask);
                  eocRdCommandReg[loop] = eocRecvCommand[loop];
                  dataBuffSize[loop] = eoc_struct_ptr->rdRegSize[eocRecvCommand[loop]&0xF];
                  dataBuffIndex[loop] = 0; /* 0 for READ */
                  if( eocRecvCommand[loop] == EOC_CMD_READ_REG_D)
                     {
                     /*       ZipStart Application specific       */
#ifdef ZIP_START                  
                     if (needs_zip_start_check[loop])
                        {
                        dataBuffSize[loop] = 1;/* ZipStart requires the buffSize to be 1 */

                        /* Save original value of regD to a variable */
                        og_D_Value[loop] = eoc_struct_ptr->rdRegData[0xD][0];

                        /* Load ZipStart info into regD */
                        eoc_struct_ptr->rdRegData[0xD][0]  = _bp_vars[loop].bp_flags.reg;
 
                        /* Signal ZipValidManager() that the COT read the zip status */
                        system_flags[loop].bits.co_check_zip = 1;
                        }
#endif /* ZIP_START */
                     } 

                    if( eocRecvCommand[loop] == EOC_CMD_READ_REG_E)
                        {
                        dataBuffSize[loop] = 1; /* 1 is standard NMR size */
                        eoc_struct_ptr->rdRegData[0xE][0] = cu_eoc_rd_nmr[loop];
                        }
             
                  eocHandlerState[loop] = EOC_DATA_READ_STATE;
                  break;
        
           
               case EOC_CMD_WRITE_REG_0:
                  eocWrCommandReg[loop] = eocRecvCommand[loop];
                  dataBuffSize[loop] = eoc_struct_ptr->wrRegSize[eocRecvCommand[loop]&0xF];
 
                  /* Set index to 15 for WRITE command when BuffSize is 16 */
                  dataBuffIndex[loop] = (eoc_struct_ptr->wrRegSize[eocRecvCommand[loop]&0xF] -1); 
                  eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
                  break;

               default:
               /* interprete received command */
               /* Next checks checks for any Valid commands, ex. loopback, ins-crc */
                  switch(eocRecvCommand[loop])
                     {
                      /* Return to normal and get rid of any hold states*/
                     case EOC_CMD_RTN:
                        HandleValidRtn(loop);
                        return;
           
                     case EOC_CMD_LOOP_NTU:  /* Application loopback at NTU */
                        eocHoldStates[loop] |= EOC_REQ_LOOP_RT;
                        /* Notify EocTaskHandler.
                         *Commands have individual task bits, so they can have
                         * multiple calls at once, e.g., loopback and CRC-ins at same time
                         */
                        SET( eocTasks[loop].bits.eocLoopRt );    
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;  
                        break;
          
                     case EOC_CMD_HOLD:
                        HandleValidHold(loop);
                        return;
      
                     case EOC_CMD_ALOOP_REG:       /* 0x19 */
                        eocHoldStates[loop] |= EOC_REQ_ALOOP_RT;
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.AgLb_regC );
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;
                        break;

                     case EOC_CMD_NTU_CCRC_REQ:    /* 0x20 */
                        eocHoldStates[loop] |= EOC_REQ_CCRC_RT; 
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.RqCorCrc_R_regR );
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;
                        break;
#ifdef REPEATER 
                     case EOC_CMD_REGC_CCRC_REQ:   /* 0x22 */
                        eocHoldStates[loop] |= EOC_REQ_CCRC_CO;
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.RqCorCrc_regC );
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;
                        break;
#endif /*REPEATER*/

                     case EOC_CMD_NTU_CCRC_END:    /* 0x28 */
                        eocHoldStates[loop] &= ~EOC_REQ_CCRC_RT; 
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.RqEndCrc_R_regR );
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;
                        break;

#ifdef REPEATER 
                     case EOC_CMD_REGC_CCRC_END:   /* 0x29 */
                        eocHoldStates[loop] &= ~EOC_REQ_CCRC_CO; 
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.RqEndCrc_regC );
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;    
                        break;
#endif /*REPEATER*/

                       case EOC_CMD_NTU_CCRC_IND:    /* 0x3F */
                        eocHoldStates[loop] |= EOC_REQ_NOT_CCRC_RT; 
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.NtfCorCrc_R_regR);
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;    
                        break;

#ifdef REPEATER
                     case EOC_CMD_REGC_CCRC_IND: /* 0x50 */
                        eocHoldStates[loop] |= EOC_REQ_NOT_CCRC_CO; 
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.NtfCorCrc_regC);
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;
                        break;

#endif /*REPEATER*/
                     case EOC_CMD_NTU_CRC_OK:    /* 0x5F */
                        eocHoldStates[loop] &= ~EOC_REQ_NOT_CCRC_RT; 
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.NtfEndCrc_R_regR);
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;        
                          break;

#ifdef REPEATER
                     case EOC_CMD_REGC_CRC_OK:   /* 0x60 */
                        eocHoldStates[loop] &= ~EOC_REQ_NOT_CCRC_CO;/* notify system that regC CRC stopped    */
                        /* Notify EocTaskHandler */
                        SET( eocTasks[loop].bits.NtfEndCrc_regC);
                        eocHandlerState[loop] = EOC_ECHO_RESPONSE;        
                        break;    
#endif /*REPEATER*/
                     default:
                        /* send UTC */
                        eocSendMessage[loop] = EocSendWord(EOC_CMD_UTC, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
                        eocHandlerState[loop] = EOC_UTC_STATE;
                     return;
                     }   /* end switch "eocRecvCommand[loop]" */

               }  /* end switch "eocRecvCommand[loop]&0xF0" */

            /* Reset Data Receive Counter if new Action starts */
            dataRecCtr[loop] = 0;
            }   /* end else */
         return;

        /****************************************************/
      case                EOC_DATA_READ_STATE:
        /****************************************************/
      
         /*            expect 1st NEXT_BYTE command            */
         if(!(eocLoByte & EOC_MESSAGE_INDICATOR))
            return;

         eocRecvCommand[loop] = EocGetData(eocLoByte, eocHiByte);
         if(eocRecvCommand[loop] == EOC_CMD_NEXT_BYTE)
            {
            /*    test for valid ODD / EVEN-Bit  */
            if(dataRecCtr[loop] & 0x1)
               {
               if(eocLoByte & EOC_ODD_INDICATOR)
                  return;
               }
            else if(!(eocLoByte & EOC_ODD_INDICATOR))
               return;

            if(dataRecCtr[loop] <  dataBuffSize[loop])
               {
               /* acknowledge message if transfer is not yet
               complete ..
               */
               eocSendMessage[loop] = eocReceiveMessage[loop];
               eocHandlerState[loop] = EOC_NEXT_READ_1;
               }
            else
               {
               /* .. or else send back END_OF_DATA message */
               eocSendMessage[loop] = EocSendWord(EOC_CMD_END_OF_DATA, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
               }
            }
         else if(eocRecvCommand[loop] == EOC_CMD_RTN)
            eocHandlerState[loop] = EOC_READ_RTN_1;
         else if(eocRecvCommand[loop] == EOC_CMD_HOLD)
            eocHandlerState[loop] = EOC_READ_HOLD_1;
         return;

        /****************************************************/
      case                EOC_DATA_WRITE_STATE:
        /****************************************************/
      
         /* expect next data byte for 1st time or END_OF_DATA command */
         eocRecvCommand[loop] = EocGetData(eocLoByte, eocHiByte);
         if(!(eocLoByte & EOC_MESSAGE_INDICATOR))
            {
            /* test for valid ODD / EVEN-Bit */
            if(dataRecCtr[loop] & 0x1)
               {
               if(eocLoByte & EOC_ODD_INDICATOR)
                  return;
               }
            else if(!(eocLoByte & EOC_ODD_INDICATOR))
               return;

            if(dataRecCtr[loop] > dataBuffSize[loop])
               {
               /* return UTC if no more data allowed */
               eocSendMessage[loop] = EocSendWord(EOC_CMD_UTC, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
               eocHandlerState[loop] = EOC_UTC_STATE;
               }
            else
               {
               /* send echo to HTUC */
               eocHandlerState[loop] = EOC_NEXT_WRITE_1;
               eocSendMessage[loop] = eocReceiveMessage[loop];
               }
            }
         else if(eocRecvCommand[loop] == EOC_CMD_END_OF_DATA)
            eocHandlerState[loop] = EOC_WRITE_END_1;
         else if(eocRecvCommand[loop] == EOC_CMD_RTN)
            eocHandlerState[loop] = EOC_WRITE_RTN_1;
         else if(eocRecvCommand[loop] == EOC_CMD_HOLD)
            eocHandlerState[loop] = EOC_WRITE_HOLD_1;
         return;

        /****************************************************/
      case                EOC_NEXT_READ_1:
        /****************************************************/
      
         /*           expect 2nd NEXT_BYTE command           */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_NEXT_BYTE)
            eocHandlerState[loop] = EOC_NEXT_READ_2;
         else
            eocHandlerState[loop] = EOC_DATA_READ_STATE;
         return;

        /****************************************************/
      case                EOC_NEXT_READ_2:
        /****************************************************/
      
         /*           expect 3rd NEXT_BYTE command           */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_NEXT_BYTE)
            {
            /* transmit requested data byte */
            dataRecCtr[loop]++;

            if(dataRecCtr[loop] & 0x1)
               eocSendMessage[loop] = EocSendWord(eoc_struct_ptr->rdRegData[eocRdCommandReg[loop]&0xF/*reg_name[loop]*/][dataBuffIndex[loop]++], SEND_ODD_DATA | EOC_OWN_ADDRESS);
            else
               eocSendMessage[loop] = EocSendWord(eoc_struct_ptr->rdRegData[eocRdCommandReg[loop]&0xF/*reg_name[loop]*/][dataBuffIndex[loop]++], SEND_EVEN_DATA | EOC_OWN_ADDRESS);
            }
         eocHandlerState[loop] = EOC_DATA_READ_STATE;
         return;

        /****************************************************/
      case                EOC_NEXT_WRITE_1:
        /****************************************************/
      
         /*          expect next data byte 2nd time            */
         if(eocRecvCommand[loop] == EocGetData(eocLoByte, eocHiByte))
            eocHandlerState[loop] = EOC_NEXT_WRITE_2;
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_NEXT_WRITE_2:
        /****************************************************/
     
         /*          expect next data byte 3rd time          */
         if(eocRecvCommand[loop] == EocGetData(eocLoByte, eocHiByte))
            {
            /* Don't be confused: in this case eocRecvCommand[loop]
            contains the reveived data byte
            */
            eocReceiveBuffer[loop][dataRecCtr[loop]++] = eocRecvCommand[loop];
            }
         eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_WRITE_END_1:
        /****************************************************/
      
         /* expect 2nd END_OF_DATA command after WRITE-operation */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_END_OF_DATA)
            eocHandlerState[loop] = EOC_WRITE_END_2;
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_WRITE_END_2:
        /****************************************************/

         /* expect 3rd END_OF_DATA command after WRITE-operation */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_END_OF_DATA)
            {
            eocSendMessage[loop] = EocSendWord(EOC_CMD_END_OF_DATA, SEND_EOC_COMMAND | EOC_OWN_ADDRESS);
            /* received bytes are valid -> copy */
            i = dataBuffIndex[loop];
            for(j=dataRecCtr[loop]; j;)
               {
               eoc_struct_ptr->wrRegData[eocWrCommandReg[loop]&0xF][i--] = eocReceiveBuffer[loop][(--j)];
               }

            /* Set a task flag to indicate the write register task has completed.
             * This flag is read by EocTaskHandler_RT to take appropriate action 
             * In this case, it will tell the application that new data has arrived.
             */
            SET( eocTasks[loop].bits.eocWrTask);
            eocHandlerState[loop] = EOC_ECHO_RESPONSE;
            }
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_UTC_RTN_1:
        /****************************************************/
      
         /* expect 2nd RTN command after invalid command / operation */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_RTN)
            eocHandlerState[loop] = EOC_UTC_RTN_2;
         else
            eocHandlerState[loop] = EOC_UTC_STATE;
         return;

        /****************************************************/
      case                EOC_UTC_RTN_2:
        /****************************************************/
     
         /* expect 3rd RTN command after invalid command / operation */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_RTN)
            HandleValidRtn(loop);
         else
            eocHandlerState[loop] = EOC_UTC_STATE;
         return;

        /****************************************************/
      case                EOC_UTC_HOLD_1:
        /****************************************************/
      
         /* expect 2nd HOLD command after invalid command / operation */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_HOLD)
            eocHandlerState[loop] = EOC_UTC_HOLD_2;
         else
            eocHandlerState[loop] = EOC_UTC_STATE;
         return;

        /****************************************************/
      case                EOC_UTC_HOLD_2:
        /****************************************************/
      
         /* expect 3rd HOLD command after invalid command / operation */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_HOLD)
            HandleValidHold(loop);
         else
            eocHandlerState[loop] = EOC_UTC_STATE;
         return;

        /****************************************************/
      case                EOC_WRITE_RTN_1:
        /****************************************************/
      
         /* expect 2nd RTN command after WRITE operation     */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_RTN)
            eocHandlerState[loop] = EOC_WRITE_RTN_2;
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_WRITE_RTN_2:
        /****************************************************/
     
         /* expect 3rd RTN command after WRITE operation     */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_RTN)
            HandleValidRtn(loop);
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_WRITE_HOLD_1:
        /****************************************************/
      
         /* expect 2nd HOLD command after WRITE operation    */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_HOLD)
            eocHandlerState[loop] = EOC_WRITE_HOLD_2;
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_WRITE_HOLD_2:
        /****************************************************/
       
         /* expect 3rd HOLD command after WRITE operation    */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_HOLD)
            HandleValidHold(loop);
         else
            eocHandlerState[loop] = EOC_DATA_WRITE_STATE;
         return;

        /****************************************************/
      case                EOC_READ_RTN_1:
        /****************************************************/

         /* expect 2nd RTN command after READ operation        */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_RTN)
            eocHandlerState[loop] = EOC_READ_RTN_2;
         else
            eocHandlerState[loop] = EOC_DATA_READ_STATE;
         return;

        /********************************************************/
      case                EOC_READ_RTN_2:
        /****************************************************/
      
         /* expect 3rd RTN command after READ operation        */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_RTN)
            {
            HandleValidRtn(loop);
#ifdef ZIP_START
            if((needs_zip_start_check[loop]) && (eocRdCommandReg[loop] == EOC_CMD_READ_REG_D))
               {
               CLEAR(needs_zip_start_check[loop]);
               eoc_struct_ptr->rdRegData[0xD][0] = og_D_Value[loop];  /*HTUR replaces original D0 value after ZipStart*/
               }
#endif /* ZIP_START */
            }
         else
            eocHandlerState[loop] = EOC_DATA_READ_STATE;
         return;

        /****************************************************/
      case                EOC_READ_HOLD_1:
        /****************************************************/
    
         /* expect 2nd HOLD command after READ operation        */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_HOLD)
            eocHandlerState[loop] = EOC_READ_HOLD_2;
         else
            eocHandlerState[loop] = EOC_DATA_READ_STATE;
         return;

        /****************************************************/
      case                EOC_READ_HOLD_2:
        /****************************************************/
      
         /* expect 3rd HOLD command after READ operation        */
         if(EocGetData(eocLoByte, eocHiByte) == EOC_CMD_HOLD)
            HandleValidHold(loop);
         else
            eocHandlerState[loop] = EOC_DATA_READ_STATE;
        return;
      }
} 
/*--------------------------------------------------------------------
             void    EocTaskHandler_RT( void )
         ============================================
 
  written:    Gert Schmalfuss
 
  usage:
         Responds to EOC commands from the COT (Master) side either
         by notifying application side or performing application 
         specific actions.
 
  application notes:
         Implementation of additional EOC features should be done here.
 
  revision history:
  ---------------------------------------------------------------------*/
void EocTaskHandler_RT ( BP_U_8BIT loop )
{
   EOC_STRUCT_DECL_PTR;    
   TX_WR BP_XDATA * tx_wr_ptr;     /*for CRC insert*/
   EOC_STRUCT_INIT_PTR(loop);
   tx_wr_ptr = get_tx_wr_ptr(loop);/*for CRC insert*/
  
                                    
   if (eocTasks[loop].reg)
      {
      eocLoopReg[loop] = eocHoldStates[loop];
      CLEAR(rt_open[loop]);

  if (eocTasks[loop].bits.eocWrTask)
      {
      newDataFlags[loop].reg |= (newDataFlags[loop].reg = 1 <<(eocWrCommandReg[loop]&0xF));
      }

/* Shift is done to load newDataFlags variable bit corresponding to the proper reg_name */    
/* the eocLoopReg = eocholdState takes care of loop, aloop, NtfCorCrc_R_regR,*/
/* so they don't need to be updated */

      if(eocTasks[loop].bits.RqCorCrc_R_regR)
         {
         cu_reg_copy.tcmd_1[loop].bits.icrc_err = 1;
         tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;
         }
#ifdef REPEATER 
      /* loop 0 = RegR.  The RegR performs EOC processing.
       * For the "Reguest Corrupted CRC Reg-C command. The
       * RegC must send corrupted CRCs towards the RT.
       */
      if(eocTasks[0].bits.RqCorCrc_regC)
         {
         cu_reg_copy.tcmd_1[1].bits.icrc_err = 1;
         
         /* change the hardware pointer to loop 1 */
         tx_wr_ptr = get_tx_wr_ptr(1);
         tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[1].reg;
         }
#endif /*REPEATER*/

      if(eocTasks[loop].bits.RqEndCrc_R_regR)
         {
         cu_reg_copy.tcmd_1[loop].bits.icrc_err = 0;
         tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;
         }
#ifdef REPEATER 
      if(eocTasks[0].bits.RqEndCrc_regC)
         {
         cu_reg_copy.tcmd_1[1].bits.icrc_err = 0;
         /* change the hardware pointer to loop 1 */
         tx_wr_ptr = get_tx_wr_ptr(1);
         tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[1].reg;
         }
#endif /*REPEATER*/  
      }
   /* finish EOC action */
   CLEAR( eocTasks[loop].reg );
}
#endif /* HTUR */

/*---------------------------------------------------------------------------                                        
  The following two functions are used to shift bits to or from the positions
  within the TEOC/REOC-registers of the framer. This is to ensure that the
  bits correspond to the recommendations of the ETR-152 specification.
  The table below shows how the bitpositions are arranged within different
  data areas resp. registers
    
        ETR-152                T(R)EOC_LO        T(R)EOC_HI        command/data
   eoc1  (slave-adress 0)        0
   eoc2  (slave-adress 1)        1
   eoc3  (command/data)          2
   eoc4  (odd / even)            3
   eoc5  (unused)                4    
   eoc6  (cmd/data-MSB)          5                                7
   eoc7  (cmd/data-Bit 6)        6                                6
   eoc8  (cmd/data-Bit 5)        7                                5
   eoc9  (cmd/data-Bit 4)                        0                4
   eoc10 (cmd/data-Bit 3)                        1                3
   eoc11 (cmd/data-Bit 2)                        2                2
   eoc12 (cmd/data-Bit 1)                        3                1
   eoc13 (cmd/data-LSB)                          4                0
   ---------------------------------------------------------------------------*/
/*
              BYTE    EocGetData( BYTE lByte, BYTE hByte )
         ==========================================================
 
  written:    Gert Schmalfuss
 
  usage:
         To strip the header from the REOC_LO register and to
         concatenate and adjust the bits in REOC_LO and REOC_HI to
         a BYTE value that can be handled within EOC handler and
         other application specific code.
 
  parameters:
         - lByte: content of REOC_LO (eoc1 .. eoc8)
         - hByte: content of REOC_HI (eoc9 .. eoc13)
 
  returns:
         - byte that contains the received command/data with the bits
           at correct positions
 
  application notes:
         No changes are necessary as long as the assignment of the bits
         within framers EOC_LO and EOC_HI registers remains unchanged.
         
  revision history:
*/ 

BP_U_8BIT EocGetData ( BP_U_8BIT lByte, BP_U_8BIT hByte )
{
   BP_U_8BIT      tmp;
  
   /* lByte: 567hhhhh,  hByte: ---01234 */

   tmp = (lByte & 0xE0) | (hByte & 0x1F);    /* tmp  : 56701234 */

   lByte = tmp & 0x44;                        /* lByte: -6---2-- */
   lByte |= ((tmp<<2) & 0x88);                /* lByte: 76--32-- */
   lByte |= ((tmp>>2) & 0x22);                /* lByte: 765-321- */
   tmp = (tmp/16) | (tmp*16);                /* tmp  : 12345670 */
   lByte |= (tmp & 0x11);                    /* lByte: 76543210 */

   return(lByte);

}

/*  ---------------------------------------------------------------------
            WORD    EocSendWord( BYTE command, BYTE header )
        ============================================================

    written:    Gert Schmalfuss

    usage:
        To split the command- or data-byte and shift its bits to
        positions that meet the requirements of ETR-152 if transmitted
        using the framers TEOC_LO and TEOC_HI registers

    parameters:
        - command: command/data byte that has to be splitted and
               shifted
        - header:  byte containing the header bits eoc1 .. eoc5 
               in its 5 LSBs (LSB = eoc1)
 
    returns:
        - word that fits to TEOC_LO and TEOC_HI with the correct
          bit position of the command/data and header.
          Note that the byte positions within the word are turned. That
          means the high-byte has to be written to TEOC_LO and the
          low-byte to TEOC_HI.

    application notes:
        See application notes of function "EocGetData()" above!
        
    revision history:
   -----------------------------------------------------------------------*/

BP_U_16BIT EocSendWord ( BP_U_8BIT command, BP_U_8BIT header )
 
{
   BP_U_8BIT          tmp;
   BP_U_16BIT      retVal;
  
   /*  cmd  : 76543210 */
   tmp = command & 0x44;                            /* tmp  : -6---2--        */
   tmp |= ((command<<2) & 0x88);                    /* tmp  : 56--12--        */
   tmp |= ((command>>2) & 0x22);                    /* tmp  : 567-123-        */
   command = (command/16) | (command*16);        /* cmd  : 32107654        */
   tmp |= (command & 0x11);                        /* tmp  : 56701234        */

   retVal = ((tmp << 8) | tmp) & 0x1FE0;            /* ---01234 567-----    */
   SET_LBYTE(retVal,(GET_LBYTE(retVal)|header));    /* ---01234 567hhhhh    */

   return( retVal );
}

#endif /* CU_EOC */

