/***********************************************************/
/*    CU_ASM.C                                             */
/*    (C) Copyright 1998 by Rockwell Semiconductor Systems */
/*                                                         */
/*                                                         */
/* Description:                                            */
/*    These are the channel unit functions called by ASM.  */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*                                                         */
/* List of Functions found in this module:                 */
/*                                                         */
/*      - _CuConfigureBeginStartup(BP_U_8BIT loop);        */
/*      - _CuConfigureTransmitS1(BP_U_8BIT loop);          */
/*      - _CuConfigureStartupComplete(BP_U_8BIT loop);     */
/*      - _CuForceOnes(BP_U_8BIT state, BP_U_8BIT loop);   */
/*      - _CuHohEn(BP_U_8BIT state, BP_U_8BIT loop);       */
/*      - _CuSetRtrInd(BP_U_8BIT state, BP_U_8BIT loop);   */
/*      - _CuSetPid( BP_U_8BIT loop );                     */
/*      - _CuSetPidToAllOnes( BP_U_8BIT loop );            */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/* Aug 98 - New: Dean Rasmussen                            */
/* Oct 98 - New: Laura Yuan                                */
/*                                                         */
/***********************************************************/
#ifdef CHAN_UNIT

#include "chanunit.h"
/* 
 * FUNCTION:   _CuConfigureBeginStartup
 *
 * PARAMETERS: loop - which loop;  0,1,2
 *
 * PURPOSE:    This function configures the specified Channel Unit
 *             for startup.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      The following is done:
 *
 *                  - HDSL Overhead and ZBits to all ones
 *                  - Payload to all ones
 *                  - Disable Tx/Rx Error Ints
 *                  - Disable Tx/Rx 6mS Ints
 *                  - Disable DPLL Err Int (if Master Loop)
 *                  - Open DPLL NCO (if Master Loop)
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */

void _CuConfigureBeginStartup (BP_U_8BIT loop)
{
    COMMON_WR BP_XDATA * common_wr_ptr;
    TX_WR BP_XDATA *tx_wr_ptr;
    RX_WR BP_XDATA *rx_wr_ptr;

    common_wr_ptr = &(cu_wr->common_wr_regs);
    tx_wr_ptr = get_tx_wr_ptr(loop);
    rx_wr_ptr = get_rx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 || rx_wr_ptr == 0 )
        {
        return;
        }

      _CuSetRtrInd(1, loop);
    
#ifdef CU_EOC
      eocCtrl[loop].status = 0;
      /*Forces the EOC channel to NOT AVAILABLE when link is down*/
#endif /*CU_EOC*/
      
    cu_reg_copy.tcmd_1[loop].bits.tx_err_en = OFF;
    cu_reg_copy.rcmd_2[loop].bits.rx_err_en = OFF;
    rx_wr_ptr->rcmd_2 = cu_reg_copy.rcmd_2[loop].reg;

    cu_reg_copy.tcmd_1[loop].bits.force_one = ON;
    cu_reg_copy.tcmd_1[loop].bits.hoh_en = OFF;
    tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;
    
    /*
     * Mask 6mS Interrupts
     */
    switch ( loop )
        {
        case 0:
            cu_reg_copy.imr.bits.tx1 = 1;
            cu_reg_copy.imr.bits.rx1 = 1;
            break;

        case 1:
            cu_reg_copy.imr.bits.tx2 = 1;
            cu_reg_copy.imr.bits.rx2 = 1;
            break;

        case 2:
            cu_reg_copy.imr.bits.tx3 = 1;
            cu_reg_copy.imr.bits.rx3 = 1;
            break;
        }

    common_wr_ptr->imr = cu_reg_copy.imr.reg;

    if ( loop == _CuFlags._CuMasterLoop )
        {
        _CuDpllOperation(_CU_DPLL_OPEN);
        }

    return ;

} /* end _CuConfigureBeginStartup */



/*
 * FUNCTION:   _CuConfigureTransmitS1
 *
 * PARAMETERS: loop - which loop;  0,1,2
 *
 * PURPOSE:    This function configures the specified Channel Unit
 *             for S1 Transmission.
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      The following is done:
 *
 *                  - HDSL Overhead Normal
 *                  - Enable Tx/Rx 6mS Ints (used for SYNC & INDC/R)
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _CuConfigureTransmitS1 (BP_U_8BIT loop)
{
    _CuHohEn(ON, loop);
    
    /*
     * UnMask 6mS Interrupts
     */
    switch ( loop )
        {
        case 0:
            cu_reg_copy.imr.bits.tx1 = 0;
            cu_reg_copy.imr.bits.rx1 = 0;
            break;

        case 1:
            cu_reg_copy.imr.bits.tx2 = 0;
            cu_reg_copy.imr.bits.rx2 = 0;
            break;

        case 2:
            cu_reg_copy.imr.bits.tx3 = 0;
            cu_reg_copy.imr.bits.rx3 = 0;
            break;
        }

    cu_wr->common_wr_regs.imr = cu_reg_copy.imr.reg;

    return ;

} /* end _CuConfigureTransmitS1 */



/*
 * FUNCTION:   _CuConfigureStartupComplete
 *
 * PARAMETERS: loop - which loop;  0,1,2
 *
 * PURPOSE:    This function configures the specified Channel Unit
 *             for startup.
 *             
 *             The following is done:
 *
 *                  - HDSL Overhead Normal
 *                  - Set Z-Bit PID (E1)
 *                  - Payload normal
 *                  - Enable Tx/Rx Error Ints (should already be done)
 *                  - Enable DPLL Err Int (if Master Loop)
 *                  - Close DPLL NCO (if Master Loop)
 *                  - Reset Rx Receiver (if Master Loop)
 *                  - Reset Tx/Rx FIFOs
 *
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      Depending on the Activation State Diagram Flow,
 *             the RTR & Force Ones Off may have already be performed.
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _CuConfigureStartupComplete (BP_U_8BIT loop)
{
    TX_WR BP_XDATA *tx_wr_ptr;
    RX_WR BP_XDATA *rx_wr_ptr;

    tx_wr_ptr = get_tx_wr_ptr(loop);
    rx_wr_ptr = get_rx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 || rx_wr_ptr == 0 )
        {
        return;
        }

    _CuSetRtrInd(0, loop);

    cu_reg_copy.tcmd_1[loop].bits.tx_err_en = ON;
    cu_reg_copy.rcmd_2[loop].bits.rx_err_en = ON;
    rx_wr_ptr->rcmd_2 = cu_reg_copy.rcmd_2[loop].reg;

    cu_reg_copy.tcmd_1[loop].bits.force_one = OFF;
    cu_reg_copy.tcmd_1[loop].bits.hoh_en = ON;
    tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;


    if ( loop == _CuFlags._CuMasterLoop )
        {
        _CuDpllOperation(_CU_DPLL_CLOSED);
        _CuResetReceiver();
        }

    _CuClearCounters(loop, _CU_ALL_COUNTERS);

    tx_wr_ptr->tfifo_rst = 0x0;
    ignore_tfifo_err_ctr[loop] = IGNORE_TFIFO_THRESHOLD;

    rx_wr_ptr->rfifo_rst = 0x0;
    ignore_rfifo_err_ctr[loop] = IGNORE_RFIFO_THRESHOLD;

    return;

} /* end _CuConfigureStartupComplete */

/*
 * FUNCTION:   _CuForceOnes
 *
 * PARAMETERS: state - ON/OFF (All Ones/Normal payload)
 *             loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function programs the HDSL Tx Force Ones state.
 *
 * RETURN:     nothing
 *
 * NOTES:      See TCMD_1 (Addr 0x06)
 *
 * CHANGES:    October 2, 1995      NEW   -   V5.0
 */
void _CuForceOnes (BP_U_8BIT state, BP_U_8BIT loop)
{
    TX_WR BP_XDATA *tx_wr_ptr;

    tx_wr_ptr = get_tx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 )
        {
        return;
        }

    cu_reg_copy.tcmd_1[loop].bits.force_one = state;
    _CU_DUMMY_WRITE(cu_reg_copy.tcmd_1[loop].reg);
    tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;

    return ;
}


/*
 * FUNCTION:   _CuHohEn
 *
 * PARAMETERS: state - ON/OFF (All Ones/Normal payload)
 *             loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function programs the HDSL Tx HOH_EN state.
 *             
 * RETURN:     nothing
 *
 * NOTES:      See TCMD_1 (0x06)
 *
 * CHANGES:    October 2, 1995      NEW   -   V5.0
 */
void _CuHohEn (BP_U_8BIT state, BP_U_8BIT loop)
{
    TX_WR BP_XDATA *tx_wr_ptr;

    tx_wr_ptr = get_tx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 )
        {
        return;
        }

    cu_reg_copy.tcmd_1[loop].bits.hoh_en = state;
    _CU_DUMMY_WRITE(cu_reg_copy.tcmd_1[loop].reg);
    tx_wr_ptr->tcmd_1 = cu_reg_copy.tcmd_1[loop].reg;

    return ;
}


/*
 * FUNCTION:   _CuSetRtrInd
 *
 * PARAMETERS: state - ON/OFF
 *             loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function sets the Tx RTR Indicator Bit.
 *             
 * RETURN:     nothing
 *
 * NOTES:      
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _CuSetRtrInd (BP_U_8BIT state, BP_U_8BIT loop)
{
    TX_WR BP_XDATA *tx_wr_ptr;

    tx_wr_ptr = get_tx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 )
        {
        return;
        }

    cu_reg_copy.tind_hi[loop].bits.rtr = state;
    tx_wr_ptr->tind_hi = cu_reg_copy.tind_hi[loop].reg;

    return;

} /* end _CuSetRtrInd */


/*
 * FUNCTION:   _CuSetPid
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function sets the Z-Bit Loop Identifier (PID).
 *
 *             The HTU-C (LTU) defines the PID
 *             The HTU-R (LTU) uses what it detects from the HTU-C
 *             
 * RETURN:     nothing
 *
 * NOTES:      Only valid for E1 Applications
 *
 *             This gets complicated because the Channel Unit has
 *             3 loops and the Bitpumps could be connected to any
 *             combination.
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _CuSetPid (BP_U_8BIT loop)
{
    TX_WR BP_XDATA *tx_wr_ptr;
    BP_U_8BIT loop_ctr;

    if ( cu_reg_copy.cmd_1.bits.e1_mode == 0 )
        {
        return;
        }

    tx_wr_ptr = get_tx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 )
        {
        return;
        }

    /*
     * ZBits 1-3 : indicate which loop (PID)
     * ZBits 4-8 : unused, should already be all ones
     */
    for ( loop_ctr = 0 ; loop_ctr < _NO_OF_LOOPS ; loop_ctr++ )
        {
        if ( bp_position[loop_ctr] == loop )
            {
            switch ( loop_ctr )
                {
                case 0: /* first LIC found */
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num1 = 1;
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num2 = 0;
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num3 = 0;
                    break;

                case 1: /* second LIC found */
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num1 = 0;
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num2 = 1;
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num3 = 0;
                    break;

                case 2: /* third LIC found */
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num1 = 0;
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num2 = 0;
                    cu_reg_copy.tzbit_pid[loop].bits.pair_num3 = 1;
                    break;

                }
            tx_wr_ptr->tzbit_1 = cu_reg_copy.tzbit_pid[loop].reg;
            }
        }

    return;

} /* end _CuSetPid */

#ifdef HTUR
/*
 * FUNCTION:   _CuSetPidToAllOnes
 *
 * PARAMETERS: loop - specified loop (0,1,2)
 *
 * PURPOSE:    This function sets the Z-Bit Loop Identifier (PID).
 *
 *             The HTU-C (LTU) defines the PID
 *             The HTU-R (LTU) uses what it detects from the HTU-C
 *             
 * RETURN:     nothing
 *
 * NOTES:      Only valid for E1 Applications
 *
 *             This gets complicated because the Channel Unit has
 *             3 loops and the Bitpumps could be connected to any
 *             combination.
 *
 * CHANGES:    September 28, 1995    NEW   -   Dean Rasmussen
 */
void _CuSetPidToAllOnes (BP_U_8BIT loop)
{
    TX_WR BP_XDATA *tx_wr_ptr;
    
    if ( cu_reg_copy.cmd_1.bits.e1_mode == 0 )
        {
        return;
        }

    tx_wr_ptr = get_tx_wr_ptr(loop);

    if ( tx_wr_ptr == 0 )
        {
        return;
        }

    cu_reg_copy.tzbit_pid[loop].bits.pair_num1 = 1;
    cu_reg_copy.tzbit_pid[loop].bits.pair_num2 = 1;
    cu_reg_copy.tzbit_pid[loop].bits.pair_num3 = 1;
    
    tx_wr_ptr->tzbit_1 = cu_reg_copy.tzbit_pid[loop].reg;
    return;

} /* end _CuSetPidToAllOnes */
#endif /* HTUR */
#endif /* CHAN_UNIT */
