#ifdef CHAN_UNIT

/*
 * PROGRAM NAME:  Bt8952/Bt8953 EVM System
 *
 * VERSION:       4.3
 *
 * FILENAME:      cu_api.c
 *
 * FILE CREATED:  June 29, 1994
 *
 * LAST MODIFIED: [5-27-99]
 *
 * DEVELOPED BY:  Dean Rasmussen
 *                (C) Copyright 1995
 *                Brooktree Inc.
 *                San Diego, CA
 *
 * DESCRIPTION:   This file contains the Channel Unit API commands.
 *
 * FUNCTION LIST:
 *                _CuControl()
 *                _CuStatus()
 */

#include "chanunit.h"

#define _CU_SW_VERSION_NUM  62   /* divide by 10 */

/*
 * Globals
 */
CU_CHIP_VERSION BP_XDATA _CuChipVersion = {0,0,0};

/*
 * Local Globals
 */
static BP_U_8BIT BP_XDATA * BP_XDATA write_reg_ptr;
static BP_U_16BIT BP_XDATA write_reg_ptr_lwr;
static BP_U_16BIT BP_XDATA write_reg_ptr_upr;

static BP_U_16BIT BP_XDATA write_reg_offset = 1;
static BP_U_16BIT BP_XDATA write_reg_offset_lwr = 1;
static BP_U_16BIT BP_XDATA write_reg_offset_upr = 0;

static BP_U_8BIT BP_XDATA combine_address = 0;
static BP_U_8BIT BP_XDATA route_address = 0;

static BP_U_8BIT BP_XDATA cu_loopback = _CU_LOOPBACK_OFF;

/*
 * Extern Globals
 */
extern BP_U_8BIT BP_XDATA _CuFrameFormat;
extern BP_U_8BIT BP_XDATA cu_system_sync_state;


/*
 * FUNCTION:   _CuControl
 *
 * PARAMETERS: no - Channel Unit Loop or Common
 *             opcode - spcefied opcode command
 *             parameter - data paramter
 *
 * PURPOSE:    This function processes the specified command
 *             
 * RETURN:     _PASS / _FAIL
 *
 * NOTES:      
 *
 * CHANGES:    June 29, 1994    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _CuControl (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter)
{
    COMMON_WR BP_XDATA *common_wr_ptr;
    TX_WR BP_XDATA *tx_wr_ptr;
    RX_WR BP_XDATA *rx_wr_ptr;
    BP_U_8BIT loop_cntr, loop;

    loop = no - _CU_CHAN1;

    /*
     * Grab PTR to proper loop
     */
    tx_wr_ptr = get_tx_wr_ptr(loop);
    rx_wr_ptr = get_rx_wr_ptr(loop);

    /* Always set Common Ptr */
    common_wr_ptr = &(cu_wr->common_wr_regs);

    switch ( opcode )
        {
        case _CU_TERMINAL_TYPE:
            if ( parameter < _HTUC || parameter > _HTUR )
                {
                return _FAIL;
                }

            htu_index = parameter;
            break;

        case _CU_FRAME_FORMAT:

            if ( parameter < _CU_FRAMED_DATA || parameter > _CU_UNFRAMED_DATA )
                {
                return _FAIL;
                }

            _CuFlags._CuFrameFormat = parameter;
            
            cu_reg_copy.cmd_2.bits.pcm_float = _CuFlags._CuFrameFormat;
            _CU_DUMMY_WRITE(cu_reg_copy.cmd_2.reg);
            common_wr_ptr->cmd_2 = cu_reg_copy.cmd_2.reg;
            break;

        case _CU_CONFIGURE:
            if ( parameter < _2T1 || parameter > _1E1) 
                {
                return _FAIL;
                }
#ifdef SP_API 
               if(parameter == _1T1 ||parameter == _1E1)
                  {
                  rate_index = _CU_CUSTOM;   /*Since 1E1 and 1T1 were removed from the CuInit Function*/
                  _CuInitChannelUnit();/*force it to initialze all in CU_CUSTOM then reconfigure*/

                  if (parameter == _1T1)
                     {
                     total_pcm_tslot = 24;
                     total_hdsl_tslot = 24;
                     num_tslot_used = 24;
                     fbit_present = 1;
                     }
                  else 
                     {
                     total_pcm_tslot = 32;
                     total_hdsl_tslot = 36;
                     num_tslot_used = 32;
                     fbit_present = 0;
                     }
                  _ConfigureSinglePair(0); /*Only works on Loop 0*/
                  break;
                  }
#endif /*SP_API*/
         rate_index = parameter;

            _CuFlags._CuMasterLoop = bp_position[0];
            _CuInitChannelUnit();
            break;

        case _CU_TRANSMIT_PAYLOAD:

            if ( no == _CU_COMMON )
                {
                /*
                 * Provided to be backward compatible w/ UIP
                 */
                for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
                    {
                    _CuConfigureTransmitS1(bp_position[loop_cntr]);
                    _CuConfigureStartupComplete(bp_position[loop_cntr]);
                    }
                }
            else
                {
                /*
                 * Do specified loop only
                 */
                _CuConfigureStartupComplete(loop);
                }

            break;

        case _CU_FORCE_SCR_ONES:  /* This behaves like an ACTIVATE command */

            if ( no == _CU_COMMON )
                {
                /*
                 * Provided to be backward compatible w/ UIP
                 */
                for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
                    {
                    _CuConfigureBeginStartup(bp_position[loop_cntr]);
                    }
                }
            else
                {
                /*
                 * Do specified loop only
                 */
                _CuConfigureBeginStartup(loop);
                }

            break;

        case _CU_RESET_RX_FIFO:
            if ( no == _CU_COMMON )
                {
                _CuResetRxFIFO();
                }
            else
                {
                rx_wr_ptr->rfifo_rst = 0x0;
                ignore_rfifo_err_ctr[loop] = IGNORE_RFIFO_THRESHOLD;
                }
            break;

        case _CU_RESET_TX_FIFO:
            if ( no == _CU_COMMON )
                {
                _CuResetTxFIFO();
                }
            else
                {
                tx_wr_ptr->tfifo_rst = 0x0;
                ignore_tfifo_err_ctr[loop] = IGNORE_TFIFO_THRESHOLD;
                }
            break;

        case _CU_TX_EOC_LO:
        case _CU_TX_EOC_HI:
            /* do nothing, provided for backwards compatibility */
            break;

        case _CU_LOOPBACK:

            switch ( parameter )
                {
                case _CU_LOOPBACK_OFF:
                    switch ( cu_loopback )
                        {
                        case _CU_PRA_ON_PRA:
                            cu_reg_copy.cmd_2.bits.pp_loop = 0;
                            cu_reg_copy.cmd_2.bits.rclk_sel = 0;
                            common_wr_ptr->cmd_2 = cu_reg_copy.cmd_2.reg;
                            break;

                        case _CU_HDSL_ON_PRA:
                            cu_reg_copy.cmd_2.bits.hp_loop = 0;
                            cu_reg_copy.cmd_2.bits.tclk_sel = 0;
                            common_wr_ptr->cmd_2 = cu_reg_copy.cmd_2.reg;
                            break;

                        case _CU_PRA_ON_HDSL:
                            if ( no == _CU_COMMON )
                                {
                                for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
                                    {
                                    _CuHdslToPra(bp_position[loop_cntr], OFF);
                                    }
                                }
                            else
                                {
                                /*
                                 * Do specified loop only
                                 */
                                _CuHdslToPra(loop, OFF);
                                }

                            break;

                        }
                    break;

                case _CU_PRA_ON_PRA:
                    cu_reg_copy.cmd_2.bits.pp_loop = 1;
                    cu_reg_copy.cmd_2.bits.rclk_sel = 3;
                    common_wr_ptr->cmd_2 = cu_reg_copy.cmd_2.reg;
                    break;

                case _CU_HDSL_ON_PRA:
                    cu_reg_copy.cmd_2.bits.hp_loop = 1;
                    cu_reg_copy.cmd_2.bits.tclk_sel = 2;
                    common_wr_ptr->cmd_2 = cu_reg_copy.cmd_2.reg;
                    break;

                case _CU_PRA_ON_HDSL:
                    if ( no == _CU_COMMON )
                        {
                        for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
                            {
                            _CuHdslToPra(bp_position[loop_cntr], ON);
                            }
                        }
                    else
                        {
                        /*
                         * Do specified loop only
                         */
                        _CuHdslToPra(loop, ON);
                        }
                    break;
                }   /* switch parameter */

            /*
             * Keep track of current loopback mode so can turn off properly
             */
            cu_loopback = parameter;

            break;

        case _CU_CLEAR_ERROR_COUNTERS:
            if ( no == _CU_COMMON )
                {
                for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
                    {
                    _CuClearCounters(bp_position[loop_cntr], _CU_ALL_COUNTERS);
                    }
                }
            else
                {
                _CuClearCounters(loop, parameter);
                }

            break;

        case _CU_USE_SAME_TAP:
            if ( no == _CU_COMMON )
                {
                for ( loop_cntr = 0 ; loop_cntr < num_bit_pumps ; loop_cntr++ )
                    {
                    _CuUseSameTap(bp_position[loop_cntr], parameter);
                    }
                }
            else
                {
                _CuUseSameTap(loop, parameter);
                }
            break;

        case _CU_REVERSE_LOOPS:
            _CuReverseLoops();
            break;

        case _CU_TFIFO_WL:
            if ( no != _CU_COMMON )
                {
                tx_wr_ptr->tfifo_wl = parameter;
                tx_wr_ptr->tfifo_rst = 0x0;
                ignore_tfifo_err_ctr[loop] = IGNORE_TFIFO_THRESHOLD;
                }
            break;

        case _CU_RFIFO_WL_LO:
            if ( no == _CU_COMMON )
                {
                common_wr_ptr->rfifo_wl_lo = parameter;
                _CuResetReceiver();
                _CuResetRxFIFO();
                }
            break;

        case _CU_RFIFO_WL_HI:
            if ( no == _CU_COMMON )
                {
                common_wr_ptr->rfifo_wl_hi = parameter;
                _CuResetReceiver();
                _CuResetRxFIFO();
                }
            break;

        case _CU_COMBINE_VALUE:
            if ( combine_address < _ROUTE_COMBINE_LENGTH )
                {
                combine_table[combine_address] = parameter;
                combine_address++;
                }
            break;

        case _CU_COMBINE_WRITE:
            _CuWriteCombineTable();
            break;

        case _CU_COMBINE_ADDR:
            combine_address = parameter;
            break;

        case _CU_ROUTE_VALUE:
            if ( route_address < _ROUTE_COMBINE_LENGTH )
                {
                route_table[route_address] = parameter;
                route_address++;
                }
            break;

        case _CU_ROUTE_WRITE:
            _CuWriteRouteTable();
            break;

        case _CU_ROUTE_ADDR:
            route_address = parameter;
            break;

        case _CU_DBANK_1:
            common_wr_ptr->dbank_1 = parameter;
            break;

        case _CU_DBANK_2:
            common_wr_ptr->dbank_2 = parameter;
            break;

        case _CU_DBANK_3:
            common_wr_ptr->dbank_3 = parameter;
            break;

        case _CU_FILL_PATT:
            common_wr_ptr->fill_patt = parameter;
            break;

        case _CU_TMAP1_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[0][loop] = parameter;
                }
            break;

        case _CU_TMAP2_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[1][loop] = parameter;
                }
            break;

        case _CU_TMAP3_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[2][loop] = parameter;
                }
            break;

        case _CU_TMAP4_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[3][loop] = parameter;
                }
            break;

        case _CU_TMAP5_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[4][loop] = parameter;
                }
            break;

        case _CU_TMAP6_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[5][loop] = parameter;
                }
            break;

        case _CU_TMAP7_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[6][loop] = parameter;
                }
            break;

        case _CU_TMAP8_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[7][loop] = parameter;
                }
            break;

        case _CU_TMAP9_VALUE:
            if ( no != _CU_COMMON )
                {
                tmap_table[8][loop] = parameter;
                }
            break;


        case _CU_WRITE_TMAP:
            _CuWriteMapper();
            break;

        case _CU_RMAP1_VALUE:
            if ( no != _CU_COMMON )
                {
                rmap_table[0][loop] = parameter;
                }
            break;

        case _CU_RMAP2_VALUE:
            if ( no != _CU_COMMON )
                {
                rmap_table[1][loop] = parameter;
                }
            break;

        case _CU_RMAP3_VALUE:
            if ( no != _CU_COMMON )
                {
                rmap_table[2][loop] = parameter;
                }
            break;

        case _CU_RMAP4_VALUE:
            if ( no != _CU_COMMON )
                {
                rmap_table[3][loop] = parameter;
                }
            break;

        case _CU_RMAP5_VALUE:
            if ( no != _CU_COMMON )
                {
                rmap_table[4][loop] = parameter;
                }
            break;

        case _CU_RMAP6_VALUE:
            if ( no != _CU_COMMON )
                {
                rmap_table[5][loop] = parameter;
                }
            break;

        case _CU_WRITE_RMAP:
            _CuWriteMapper();
            break;

        case _CU_BER_START:
            common_wr_ptr->ber_rst = 0;
            break;

        case _CU_BER_CONFIGURE:

            if ( _CuChipVersion.major != _BT8953 )
                {
                cu_reg_copy.cmd_6.bits.ber_sel = parameter & 0x03;
                _CU_DUMMY_WRITE(cu_reg_copy.cmd_6.reg);
                common_wr_ptr->cmd_6 = cu_reg_copy.cmd_6.reg;
                }

            cu_reg_copy.cmd_3.reg &= 0x83;
            cu_reg_copy.cmd_3.reg |= (parameter & 0x7C);

            _CU_DUMMY_WRITE(cu_reg_copy.cmd_3.reg);
            common_wr_ptr->cmd_3 = cu_reg_copy.cmd_3.reg;

            common_wr_ptr->prbs_rst = 0;
            break;

        case _CU_SET_MASTER_LOOP:
            
            if ( parameter < 0 || parameter >= _NO_OF_LOOPS )
                {
                break;
                }

            _CuSetMasterLoop(parameter);
            break;

        case _CU_DPLL_MODE:
            
            if ( parameter < _CU_DPLL_CLOSED || parameter > _CU_DPLL_OPEN )
                {
                break;
                }

            _CuDpllOperation(parameter);
            break;

        case _CU_AUTO_RESTART:
            _CuFlags._CuAutoRestart = parameter;
            break;

        case _CU_SET_MFRAME:
            /*
             * parameter is valid from 0x00 to 0x2F ( 0 to 47 ).
             * parameter = MF_LEN;
             * TMF_LOC = MF_LEN - 1;
             * ( MF_CNT + 1 ) * ( MF_LEN + 1 ) = 48;
             */
            if ( parameter < 0 || parameter > _6MS_MFRAME )
                {
                break;
                }
               if (parameter == 0)
                  {
                  common_wr_ptr->tmf_loc = 0x00;
                  }
               else
                  {
                  common_wr_ptr->tmf_loc = parameter - 1;
                  }
               common_wr_ptr->mf_len = parameter;
               common_wr_ptr->mf_cnt = (48 / ( parameter + 1 )) - 1;
               break;

        case _CU_RESET_PCM:
            _CuResetReceiver();
            _CuResetTxFIFO();
            _CuResetRxFIFO();
            break;

#ifdef REPEATER
        case _CU_REPEATER_CONFIGURE:
            if ( no == _CU_COMMON )
               {
                if (parameter == 0)
                   {
                    _CuRepeaterConfigure(OFF);
                   }
                else
                   {
                    _CuRepeaterConfigure(ON);
                   }
                }
           break;
#endif /* REPEATER */

#ifdef CU_LED
        case _CU_WRITE_MISC_OUTPUT:
            set_LED_block(_MISC_OUTPUT_REG, _CU_LED_BLOCK_SET, parameter);
            break;

        case _CU_LED_BLOCK_SET:
        case _CU_LED_BLOCK_ON:
        case _CU_LED_BLOCK_OFF:
            set_LED_block(no, opcode, parameter);
            break;

        case _CU_UPDATE_LEDS:
            _CuFlags._CuUpdateLeds = parameter;
            break;
#endif /* CU_LED */

        case _CU_WRITE_REG_ADDR_LWR:
            write_reg_ptr_lwr = ( BP_U_16BIT )parameter & 0xFF;    
            write_reg_ptr =(BP_U_8BIT BP_XDATA *)write_reg_ptr_upr + write_reg_ptr_lwr;
            break;

        case _CU_WRITE_REG_ADDR_UPR:
            write_reg_ptr_upr = (parameter << 8) & 0xFF00;
            write_reg_ptr =(BP_U_8BIT BP_XDATA *)write_reg_ptr_upr + write_reg_ptr_lwr;
            break;

        case _CU_WRITE_REG_DATA:
            *write_reg_ptr = parameter;
            break;

        case _CU_WRITE_REG_DATA_INC:
            *write_reg_ptr = parameter;
            write_reg_ptr += write_reg_offset;
            break;

        case _CU_WRITE_REG_OFFSET_LWR:
            write_reg_offset_lwr = parameter & 0xFF;
            write_reg_offset = write_reg_offset_upr + write_reg_offset_lwr;
            break;

        case _CU_WRITE_REG_OFFSET_UPR:
            write_reg_offset_upr = (parameter << 8) & 0xFF00;
            write_reg_offset = write_reg_offset_upr + write_reg_offset_lwr;
            break;

        case _CU_WRITE_IND_LO:
            cu_reg_copy.tind_lo[loop].reg = parameter;
            tx_wr_ptr->tind_lo = cu_reg_copy.tind_lo[loop].reg;
            break;

        case _CU_WRITE_IND_HI:
            cu_reg_copy.tind_hi[loop].reg = parameter;
            tx_wr_ptr->tind_hi = cu_reg_copy.tind_hi[loop].reg;
            break;

        default:
            return _FAIL;

        }   /* end switch opcode */
    return _PASS;
}


/*
 * FUNCTION:   _CuStatus
 *
 * PARAMETERS: no - Channel Unit Loop or Common
 *             opcode - specified command
 *             parameter - data parameter
 *             *indication - return value
 *
 * PURPOSE:    This function queries the Channel Unit for the
 *             specified opcode command.  The return value is returned
 *             via the *indication pointer.
 *             
 * RETURN:     _PASS / _FAIL
 *
 * NOTES:      
 *
 * CHANGES:    July 25, 1994    NEW   -   Dean Rasmussen
 */
BP_U_8BIT _CuStatus (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter, BP_S_8BIT *indication)
{
    COMMON_RD BP_XDATA *common_rd_ptr;
    COMMON_WR BP_XDATA *common_wr_ptr;
    TX_RD BP_XDATA *tx_rd_ptr;
    RX_RD BP_XDATA *rx_rd_ptr;
    RSTATUS_2 rstatus_2;
    BP_U_8BIT loop;

    loop = no - _CU_CHAN1;

    /*
     * Grab PTR to proper loop
     */
    tx_rd_ptr = get_tx_rd_ptr(loop);
    rx_rd_ptr = get_rx_rd_ptr(loop);

    /* Always set Common Ptr */
    common_rd_ptr = &(cu_rd->common_rd_regs);
    common_wr_ptr = &(cu_wr->common_wr_regs);

    switch ( opcode )
        {
        case _CU_PRESENT:
            /*
             * The IMR register is the only Write/Readable register.
             *
             * Bit 7 is not used in Bt8953.
             */

            common_wr_ptr->cmd_4 = 0x04;

            _DisableCuInterrupt();

            _CuChipVersion.major = 0;
            _CuChipVersion.minor = 0;
            _CuChipVersion.present = 0;

            common_wr_ptr->imr = 0x55;
            if ( (common_rd_ptr->imr & 0x7F) != 0x55 )
                {
                *indication = _NOT_PRESENT;
                break;
                }

            common_wr_ptr->imr = 0x1A;
            if ( (common_rd_ptr->imr & 0x7F) != 0x1A )
                {
                *indication = _NOT_PRESENT;
                break;
                }
            
            /* Clear & disable all CU interrupts */
            cu_reg_copy.imr.reg = 0xFF; /* Mask all interrupts */
            common_wr_ptr->imr = 0xFF;
            common_wr_ptr->icr = 0;

            _EnableCuInterrupt();

            /* CU is present */
            *indication = _PRESENT;

            /*
             * Read Chip Version # (2 Major & 3 Minor bits)
             *
             * Only valid on Loop #3
             *
             * RSTATUS_1 7:6 denotes Major Version #
             * RIND_HI 7:5 denotes Minor Version #
             */
            rx_rd_ptr = &(cu_rd->rx_rd_loop3);

            _CuChipVersion.major = (rx_rd_ptr->rstatus_1 >> 6) & 0x3;
            _CuChipVersion.minor = (rx_rd_ptr->rind_hi >> 5) & 0x7;
            _CuChipVersion.present = 1;

            _CuFlags._CuUpdateLeds = 1;
            break;

        case _CU_SYNC:

            /*
             * return concept:
             *
             *      _CU_COMMON: 1 = all are IN_SYNC
             *                  0 = all are not IN_SYNC
             *
             *      PER CHANNEL: return SYNC STATE,
             *              0 - CU_OUT_OF_SYNC
             *              1 - CU_ACQUIRING_SYNC
             *              2 - CU_IN_SYNC
             *              3 - CU_LOSING_SYNC
             */
            if ( no == _CU_COMMON )
                {
                *indication = cu_system_sync_state;
                }
            else
                {
                rstatus_2.reg = rx_rd_ptr->rstatus_2;
                *indication = rstatus_2.bits.sync_state;
                }

            break;

        case _CU_RX_EOC_HI:
        case _CU_RX_EOC_LO:
            /* do nothing, provided for backwards compatibility */
            break;

        case _CU_READ_DPLL:
    if ( no == _CU_COMMON )
                *indication = cu_dpll_state_man;
          else
                return _FAIL;            
             break;

        case _CU_VERSION:
            *indication = (_CuChipVersion.major << 3) | _CuChipVersion.minor;
            break;

        case _CU_SW_VERSION:
            *indication = _CU_SW_VERSION_NUM;
            break;

        case _CU_ERROR_COUNTERS_LO:
            if ( parameter == _CU_DPLL_ERROR_CTR )
                {
                *indication = LOW(error_ctrs[parameter][0]);
                return _PASS;
                }

            if ( no == _CU_COMMON )
                {
                *indication = 0;
                return _PASS;
                }

            if ( parameter >= 0 && parameter < _CU_NUM_ERROR_CTRS )
                {
                *indication = LOW(error_ctrs[parameter][loop]);
                }
            break;

        case _CU_ERROR_COUNTERS_HI:
            if ( parameter == _CU_DPLL_ERROR_CTR )
                {
                *indication = HIGH(error_ctrs[parameter][0]);
                return _PASS;
                }
            
            if ( no == _CU_COMMON )
                {
                *indication = 0;
                return _PASS;
                }

            if ( parameter >= 0 && parameter < _CU_NUM_ERROR_CTRS )
                {
                *indication = HIGH(error_ctrs[parameter][loop]);
                }
            break;

        case _CU_LOOPS_REVERSED:
            *indication = _CuFlags._CuLoopsReversed;
            break;

        case _CU_BER_STATUS:
            *indication = common_rd_ptr->ber_status;
            break;

        case _CU_MEASURE_BER:
            *indication = common_rd_ptr->ber_meter;
            break;

        case _CU_READ_COMBINE:
            if ( parameter >= _ROUTE_COMBINE_LENGTH )
                {
                return _FAIL;
                }
            *indication = (BP_S_8BIT)(combine_table[parameter]);
            break;

        case _CU_READ_ROUTE:
            if ( parameter >= _ROUTE_COMBINE_LENGTH )
                {
                return _FAIL;
                }
            *indication = (BP_S_8BIT)(route_table[parameter]);
            break;

        case _CU_READ_TMAP:
            if ( parameter >= 5 )
                {
                return _FAIL;
                }
            *indication = tmap_table[parameter][loop];
            break;

        case _CU_READ_RMAP:
            if ( parameter >= 3 )
                {
                return _FAIL;
                }
            *indication = rmap_table[parameter][loop];
            break;

        case _CU_READ_REG:
            *indication = *(write_reg_ptr);
            switch ( parameter )
                {
                case _CU_READ_REG_INC:
                    write_reg_ptr += write_reg_offset;
                    break;

                case _CU_READ_REG_DEC:
                    write_reg_ptr -= write_reg_offset;
                    break;

                case _CU_READ_REG_HOLD:
                default:
                    /* don't adjust ptr */
                    break;
                }
            break;

        case _CU_READ_IND_LO:
            *indication = cu_rx_hoh.rind_lo[loop].reg;
            break;

        case _CU_READ_IND_HI:
            *indication = cu_rx_hoh.rind_hi[loop].reg;
            break;


        default:
            return _FAIL;
        }

    return _PASS;
}


#endif /* CHAN_UNIT */

