/**************************************************************/
/*    UTIL.C                                                  */
/*    Bitpump utility routines.                               */
/*    (C) Copyright 1993 by Rockwell Corporation              */
/*                                                            */
/*    This program is copyrighted by Rockwell Corporation     */
/*                                                            */
/* Description:                                               */
/*    This file contains several utility routines:            */
/*                                                            */
/* Notes:                                                     */
/*                                                            */
/* User Modifiable Code:                                      */
/*    None                                                    */
/*                                                            */
/* List of functions included in this module:                 */
/*  _BtReset()                                                */
/*  _BtInitialize()                                           */
/*  _ClearBitpumpIntSource()                                  */
/*  _DcCancel()                                               */
/*  _BtLoadMicroCode()                                        */
/*  _LookUpTable()                                            */
/*  _NormlizeMeter()                                          */
/*  _RdByte()                                                 */
/*  _WrByte()                                                 */
/*  _WrWord()                                                 */
/*  _ReadNmr()                                                */
/*  _ScaleByGain()                                            */
/*  _ReadAccessByteRAM()                                      */
/*  _WriteAccessByteRAM()                                     */
/*  _WrScratchPad()                                           */
/*  _RdScratchPad()                                           */
/*                                                            */
/* Programmer:                                                */
/*     Dean Rasmussen             March - 1998                */
/*     Iris Shuker                21-Oct-1993                 */
/*                                                            */
/* Revision History:                                          */
/*                                                            */
/**************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define MASK_ALL                  0xFF

#define NL_EC_DELAY_INIT_VALUE        3
#define PKD_DELAY_INIT_VALUE          4
#define PHASE_OFFSET_INIT_VALUE       0         /* No phase offset */
#define VCXO_FREQ_INIT_VALUE          0         /* Nominal frequency */
#define CURSOR_LEVEL_INIT_VALUE       1024
#define NOISE_HISTOGRAM_TH_INIT_VALUE 512
#define SCR_SYNC_TH_INIT_VALUE        0
#define SNR_ALARM_INIT_VALUE          2       /* 580 per sample = 10dB SNR */
#define EP_PAUSE_INIT_VALUE           0xffff    /* No EP Pause */


#define MICROCODE_LENGTH            64

#define PDM_VALUE   15000

#define PLL_STABILIZE_DELAY     (BP_U_16BIT)200

/* Continuous Real Time Filter Settings */
#define _1168K_FILTER     0
#define _784K_FILTER      1
#define _2320K_FILTER     2
#define _400K_FILTER      3

/* Setup _HandleTempEnv() to peform temperature/environment processing every
 * 3 meter interrupts.  These interrupts occur every 32,768 symbols. A value of 
 * 3 meter intervals compensates for a maximum temperature variation of 10
 * 10 degrees C/hour.  See the meter interval versus max. temperature variation 
 * table in the Software Integration chapter in the Zipwire Software User's Guide 
 * for more information.
 */
#define DEFAULT_TE_METER_INTERVALS  3

/*---------------------------------------------------------*/
/*  Local Macros                                           */
/*---------------------------------------------------------*/

#define NOISE_MARGIN_DB(x)       ((x) - 32)

/*
 * Default user setup
 * Frame Format = SERIAL          <user setup high = 0x02>
 */
#define DEFAULT_USER_SETUP   WR_WORD(no, USER_SETUP, 0x00, 0x02);


#define SYM_RATE_BT8960(symbol_rate) \
if ( symbol_rate <= 27 ) /* 0kHz - 110kHz */\
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, TWO_CHANNEL_60);\
    else if ( symbol_rate <= 43 ) /* 110kHz - 176kHz */\
        BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, FOUR_CHANNEL_60);\
        else if ( symbol_rate <= 59 ) /* 176kHz - 242kHz */\
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, SIX_CHANNEL_60);\
            else /* Above 242kHz */\
                BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, EIGHT_CHANNEL_60);

#define SYM_RATE_BT8970(symbol_rate) \
if (symbol_rate <= 80) /* 0KHz - 328KHz */\
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, EIGHT_CHANNEL_70);\
    else if (symbol_rate <= 118) /* 328KHz - 484KHz */\
        BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, HALF_T1_70);\
        else if ( symbol_rate <= 167 ) /* 484KHz - 684KHz */\
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, FULL_T1_70);   /* was HALF_E1_70 */\
            else /* Above 684KHz */\
                BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, FULL_T1_70);

#define SYM_RATE_BT8973(symbol_rate) \
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, clk_freq, (symbol_rate & 0x300) >> 8 );\
    BP_WRITE_REG(bp_ptr, clock_freq, (symbol_rate & 0xFF) );

/*---------------------------------------------------------*/
/*  Global Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

static void _ClearBitpumpIntSource(BP_U_8BIT no);

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Constants                                              */
/*---------------------------------------------------------*/

extern BP_TABLE BP_CONSTANT _noise_margin[TABLE_LENGTH];

/*
 * Based on 13.5dB transmitted power.  Calibrated to match 150k Sine Wave
 * Attenuation
 */
BP_TABLE BP_CONSTANT _far_end_signal[_FELM_NUM_DATA_RATES][TABLE_LENGTH] = {
    /* 144kbps */
{   5708, 5356, 5025, 4714, 4421, 4147, 3890, 3649, 3423, 3212, 3014,
    2830, 2658, 2498, 2348, 2209, 2079, 1958, 1845, 1741, 1643, 1552,
    1467, 1388, 1314, 1245, 1181, 1120, 1063, 1009, 958, 909, 863, 820,
    778, 737, 699, 661, 625, 590, 556, 523, 491, 459, 429, 399, 371,
    344, 317, 292, 269, 247, 226, 208, 192, 179, 168, 161, 156, 141, 131,
    120, 110, 100},
    /* 272kbps */
{   6314, 5909, 5527, 5167, 4828, 4509, 4209, 3928, 3664, 3417, 3185,
    2969, 2767, 2578, 2402, 2238, 2086, 1944, 1813, 1691, 1577, 1472,
    1375, 1285, 1202, 1125, 1054, 988, 926, 870, 817, 768, 723, 680,
    640, 603, 568, 535, 504, 475, 446, 420, 394, 369, 346, 323, 302,
    281, 261, 242, 224, 207, 191, 177, 164, 152, 142, 134, 128, 125, 124,
    109, 102, 97},
    /* 400kbps */
{   6080, 5653, 5252, 4876, 4524, 4194, 3887, 3600, 3332, 3083,
    2852, 2637, 2439, 2255, 2084, 1928, 1783, 1650, 1528, 1415, 1312, 1218,
    1132, 1053, 980, 914, 854, 799, 748, 702, 659, 620, 583, 550,
    518, 489, 461, 435, 410, 386, 363, 341, 319, 298, 278, 258, 238, 220,
    201, 184, 167, 151, 136, 123, 111, 101, 93, 87, 84, 84, 74, 69, 63, 59},
    /* 528kbps */
{   6159, 5704, 5278, 4879, 4507, 4160, 3837, 3537, 3258, 2999, 2760,
    2539, 2335, 2147, 1975, 1816, 1671, 1538, 1417, 1306, 1206, 1114,
    1031, 956, 887, 825, 769, 718, 672, 630, 592, 556, 524, 494,
    466, 440, 416, 393, 370, 349, 328, 307, 288, 268, 249, 229, 211, 192,
    174, 157, 140, 125, 110, 97, 85, 76, 68, 64, 62, 56, 52, 47, 43, 40},
    /* 784kbps */
{   6132, 5647, 5194, 4773, 4380, 4016, 3678, 3366, 3077, 2811,
    2566, 2341, 2134, 1946, 1774, 1617, 1475, 1346, 1229, 1124, 1029, 944,
    868, 800, 739, 684, 636, 593, 554, 519, 488, 460, 434, 411, 389,
    369, 350, 331, 313, 296, 278, 261, 244, 226, 209, 191, 174, 156, 139, 122,
    106, 91, 76, 64, 53, 44, 37, 34, 27, 21, 15, 9, 2, 1},
    /* 1168kbps */
{   5912, 5414, 4951, 4520, 4122, 3753, 3413, 3099, 2811, 2546, 2304,
    2083, 1882, 1699, 1534, 1384, 1250, 1129, 1021, 925, 840, 764, 697,
    639, 587, 542, 503, 468, 438, 412, 389, 368, 350, 333, 318, 303, 289,
    276, 262, 249, 235, 221, 206, 191, 175, 159, 143, 127, 110, 94, 78,
    63, 49, 36, 26, 17, 12, 10, 8, 6, 4, 2, 0, 0},
    /* 1552kbps */
{   5682, 5173, 4701, 4265, 3862, 3492, 3151, 2839, 2554, 2294,
    2058, 1844, 1650, 1476, 1320, 1180, 1056, 946, 849, 764, 689, 624,
    568, 520, 478, 443, 413, 388, 366, 348, 332, 318, 306, 295, 284, 274,
    264, 253, 242, 231, 218, 205, 190, 175, 159, 142, 124, 106, 88, 70,
    53, 37, 22, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* 2320kbps */
{   5116, 4628, 4177, 3762, 3381, 3031, 2712, 2421, 2156, 1916,
    1699, 1505, 1330, 1175, 1036, 914, 807, 713, 632, 562, 502, 451,
    409, 373, 344, 320, 301, 285, 273, 263, 255, 249, 243, 238, 233, 228,
    222, 215, 207, 198, 188, 177, 164, 149, 134, 118, 100, 83, 65, 47, 29,
    12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
}; /* end _far_end_signal[][] */




    /* Microcode */
static BP_S_32BIT BP_CONSTANT micro_code[MICROCODE_LENGTH] =
{0x79bbc23c,
0x87db80bc,
0x8997841c,
0xfff84014,
0x898fc60c,
0xfd470178,
0x897f060c,
0xfd350128,
0x896fc60c,
0xfd24c128,
0x895fc60c,
0xfd148128,
0x84ffc60c,
0xfd044128,
0x84efc60c,
0xfc740128,
0x84dfc60c,
0xfc61c0e4,
0x84cfc60c,
0xfc5180e0,
0x84bfc60c,
0xfc4140e0,
0x84afc60c,
0xfc3100e0,
0x849fc60c,
0xfc20c0e0,
0x848fc60c,
0xfc1080e0,
0x1cefc60c,
0xfc0040e0,
0x18d3c600,
0xfda000e0,
0x14c38600,
0xfdd68204,
0x10b34600,
0x87f7c200,
0x0ca30600,
0xfff84214,
0x0892c600,
0xffffc20c,
0x04828600,
0x73efc20c,
0x01c24600,
0xfff80218,
0x51820600,
0xffffc20c,
0x4d764600,
0xfff741b0,
0x49660600,
0xffffc20c,
0x4555c600,
0x87ffc20c,
0x42158600,
0xfff84214,
0x7a056600,
0xffffc20c,
0xffffc60c,
0xfdafc1f4,
0xfdbfc20c,
0xb2d6827c,
0xfff6c0ac,
0xffc89c0c,
0xfff8500c,
0xfff78a0c};


/***********************************************************/
/*    _BtReset()                                           */
/*    Reset Bitpump to default setup, load micro-code.     */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _BtReset(no);                                 */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _BtReset (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;


#ifdef TDEBUG
    PREFIX;
    printf("Reset system \n");
#endif

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    /*------------------------------------------------*/
    /* Reset All Bitpump registers to default values  */
    /*------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, serial_monitor_source, hclk_freq, _HCLK_DEFAULT);

    BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON);

    _BitpumpSetSymbolRate(no, _bp_vars[no].symbol_rate);
    _BtLoadMicroCode(no);

    BP_WRITE_REG(bp_ptr, adc_control, RESET); /* Reset ADC Gain Control register, default value: fixed gain, no AGC */

    if ( GET_BITPUMP_TYPE() == BT8960 )
        {
        BP_WRITE_REG(bp_ptr, clock_freq, 0xC0);  /* 'reserved10' for Bt8960 */
        BP_WRITE_REG(bp_ptr, test_reg35, 0x60);
        }
    else if ( GET_BITPUMP_TYPE() == BT8970 )
        {
        WRITE_REG_35(0x00, 0x43, 0x00);
        BP_WRITE_REG(bp_ptr, clock_freq, 0x0C);  /* 'reserved10' for Bt8970 */
        }
    else
        {
        /* Bt8973 */
        /*-------------------------------------*/
        /* Pulse Shaping Filter (Bt8973 Only)  */
        /*-------------------------------------*/
         
        if ( _bp_vars[no].symbol_rate < 50 )
         {
         if (BP_READ_BIT(bp_mode_ptr, global_modes,hw_revision) >=0x02)
            {
            BP_WRITE_BIT(bp_mode_ptr, adc_control, cont_time, _400K_FILTER);
            }
         else
            {
            BP_WRITE_BIT(bp_mode_ptr, adc_control, cont_time, _784K_FILTER);
            }
         BP_WRITE_BIT(bp_mode_ptr, adc_control, switch_cap_pole, 0);
         BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, imp_short, 0);
         }
        else if ( _bp_vars[no].symbol_rate < 100 )
            { /* Less than 800kbps */\
            BP_WRITE_BIT(bp_mode_ptr, adc_control, cont_time, _784K_FILTER);
            BP_WRITE_BIT(bp_mode_ptr, adc_control, switch_cap_pole, 0);
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, imp_short, 0);
            }
        else if ( _bp_vars[no].symbol_rate < 150 )
            { /* 800KHz - 1200KHz */
            BP_WRITE_BIT(bp_mode_ptr, adc_control, cont_time, _1168K_FILTER);
            BP_WRITE_BIT(bp_mode_ptr, adc_control, switch_cap_pole, 0);
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, imp_short, 0);
            }
        else
            { /* Above 1200KHz */
            BP_WRITE_BIT(bp_mode_ptr, adc_control, cont_time, _2320K_FILTER);

            /* Turn on switched cap pole which works for RS8973, Rev. B and thereafter.
             * Switched cap pole does not work on Rev. A, see Non-conformance #2. 
             */
            if ( BP_READ_BIT(bp_mode_ptr, global_modes, hw_revision) != 0x00 )
                {
                BP_WRITE_BIT(bp_mode_ptr, adc_control, switch_cap_pole, 1);
                }
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, imp_short, 0);
            }
     
        WRITE_REG_35(0x00, 0x42, 0x00);
        }     

    BP_WRITE_REG(bp_ptr, cu_interface_modes, _SERIAL); /* Set CU Interface default value: Normal operationn and serial interface with clock outputs */

    BP_WRITE_BIT(bp_mode_ptr, misc_test, reg_clk_en, 0);

    _InitTxGain(no);
    
    BP_WRITE_REG(bp_ptr, peak_detector_delay, PKD_DELAY_INIT_VALUE); /* Set peak detector delay */

    WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE); /* Reset Bitpump operational stage */

    WR_WORD(no, PARAMETERS, RESET, T_LOSS_DEFAULT_VALUE);

    DEFAULT_USER_SETUP;

    _BtInitialize(no);
    _SetMeterTimer(no, ALT_METER);

#ifdef ZIP_START
    ZIP_START_MODE = OFF;
#endif

} /* END _BtReset() */

/***********************************************************/
/*    _BtInitialize()                                      */
/*    Initialize Bitpump to startup initialization state.  */
/*    User defined parameters are NOT lost.                */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _BtInitialize(no);                            */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _BtInitialize (BP_U_8BIT no)
{

    DECLARE_PTR;
    DECLARE_MODE_PTR;

    user_setup_low_type user_setup_low;
    user_setup_high_type user_setup_high;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/    
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

#ifdef TDEBUG
    PREFIX;
    printf("Init regs \n");
#endif

    /*----------------------------------------------------------------*/
    /* Initialize Bitpump registers for startup state.                */
    /*----------------------------------------------------------------*/

    /*----------------------------------------------------------------*/
    /* User setup register.                                           */
    /*----------------------------------------------------------------*/
    RD_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Read user setup */
    user_setup_high.bits.test_mode = RESET; /* Reset test mode bits */
    WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */

    /*----------------------------------------------------------------*/
    /* A/D sampling phase + Impulse Shortening (Bt8973)               */
    /*----------------------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0);

    /*----------------------------------------------------------------*/
    /* Linear EC                                                      */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, linear_ec_modes, RESET); /* Reset linear EC modes register */
    SET_RESET(bp_mode_ptr, linear_ec_modes, zero_coefficients); /* Clear linear EC coefficients */
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, HIGHEST_GAIN); /* EC at highest adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, enable_dc_tap, ON); /* Enale adaptive DC cancellation */

    /*----------------------------------------------------------------*/
    /* NL EC                                                          */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, nonlinear_ec_modes, RESET); /* Reset NL EC modes register */
    SET_RESET(bp_mode_ptr, nonlinear_ec_modes, zero_coefficients); /* Clear NL EC coefficients */
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF);
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* Non linear EC at high adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, symbol_delay, NL_EC_DELAY_INIT_VALUE); /* Set NL EC symbol delay */

    /*----------------------------------------------------------------*/
    /* DFE                                                            */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, dfe_modes, RESET); /* Reset DFE modes register */
    SET_RESET(bp_mode_ptr, dfe_modes, zero_coefficients); /* Clear DFE coefficients */
    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, HIGH_GAIN); /* DFE at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* Transmitter                                                    */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, transmitter_modes, RESET); /* Reset transmitter modes register */
    if ( GET_BITPUMP_TYPE() <= BT8970 )
        {
        /* Fixes potential Bt8960/70 scrambler dead space issue */
        BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, 0x04);
        }
    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, ON); /* Set transmitter OFF */
    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, 0x00);

    if (user_setup_low.bits.terminal_flag == _HTUR) /* HTUR terminal */
        BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, htur_lfsr, _HTUR);
    if (user_setup_low.bits.seq_source) /* Internal startup sequence source */
        BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES); /* Internal 2-level scrambled 1's */
    else /* External startup sequence source */
        BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_TWO_LEVEL_DATA); /* External 2-leveled data */

    /*----------------------------------------------------------------*/
    /* PLL & Timing                                                   */
    /*----------------------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON);
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, negate_symbols, OFF); /* Don't Invert PD symbols */
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHEST_GAIN); /* PLL at highest adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, phase_detector_gain, HIGH_GAIN); /* Phase detector at high gain */
    SET_WORD(bp_ptr, pll_phase_offset_low, pll_phase_offset_high, PHASE_OFFSET_INIT_VALUE); /* Init Pll offset register */
    SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_FREQ_INIT_VALUE); /* Init fixed frequency value */

    /*----------------------------------------------------------------*/
    /* Detector                                                       */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, detector_modes, RESET); /* Reset detector mode register */
    if (user_setup_low.bits.terminal_flag == _HTUR) /* HTUR terminal */
        BP_WRITE_BIT(bp_mode_ptr, detector_modes, htur_lfsr, _HTUR);
    if (user_setup_low.bits.rx_descr) /* Activate receive descrambler */
        {
        BP_WRITE_BIT(bp_mode_ptr, detector_modes, descr_on, ON);
        BP_WRITE_BIT(bp_mode_ptr, detector_modes, output_mux_control, SELECT_DESCRAMBLER_OUTPUT);
        } /* END-IF activate receive descrambler */
    SET_WORD(bp_ptr, cursor_level_low, cursor_level_high, CURSOR_LEVEL_INIT_VALUE); /* Set slicer cursor level init value */
    SET_WORD(bp_ptr, noise_histogram_th_low, noise_histogram_th_high, NOISE_HISTOGRAM_TH_INIT_VALUE);
    SET_WORD(bp_ptr, ep_pause_th_low, ep_pause_th_high, EP_PAUSE_INIT_VALUE);
    BP_WRITE_REG(bp_ptr, scr_sync_th, SCR_SYNC_TH_INIT_VALUE); /* Init scrambler syncronization threshold */
    SET_WORD(bp_ptr, snr_alarm_th_low, snr_alarm_th_high, SNR_ALARM_INIT_VALUE); /* Set snr alarm threshold init value */

    /*----------------------------------------------------------------*/
    /* AGC & DC Offset                                                */
    /*----------------------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, 0);  /* loopback control */

    _SetAdcControlAgain(no, AGAIN9DB);
    SET_WORD(bp_ptr, dc_offset_low, dc_offset_high, 0);

    /*----------------------------------------------------------------*/
    /* DAGC                                                           */
    /*----------------------------------------------------------------*/
    SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, DAGC_TARGET_INIT_VALUE);
    BP_WRITE_REG(bp_ptr, dagc_modes, RESET); /* Reset DAGC mode register */
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, HIGH_GAIN); /* DAGC at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* FFE                                                            */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, ffe_modes, RESET); /* Reset FFE mode register */

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, ON); /* Clear FFE coefficients */
    DELAY2SYMBOL;
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, OFF);

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, HIGH_GAIN); /* FFE at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* EP                                                             */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, ep_modes, RESET); /* Reset EP mode register */
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, ON); /* Adapt EP coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, zero_coefficients, ON); /* Clear EP coefficients */
    DELAY2SYMBOL;
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, OFF); /* Freeze EP coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, zero_coefficients, OFF);
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_gain, HIGH_GAIN); /* EP at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* Interrupts                                                     */
    /*----------------------------------------------------------------*/
    _ClearBitpumpIntSource(no);

    /*----------------------------------------------------------------*/
    /* Status register (los active on initialize)                     */
    /*----------------------------------------------------------------*/
    WR_BYTE(no, STATUS, INIT_STATUS_REG); /* Reset status register */

    /*----------------------------------------------------------------*/
    /* Meter timer, FELM High/Low thresholds and DC Cancellation      */
    /*----------------------------------------------------------------*/
    SET_WORD(bp_ptr, dc_offset_low, dc_offset_high, 0);

    /*----------------------------------------------------------------*/
    /* Temperature/Environment Stage = IDLE on intialize              */
    /*----------------------------------------------------------------*/
    WR_WORD(no, STAGE2, TEMP_ENV_IDLE, 0);

    WR_WORD(no, FELM_THRESHOLD, LOW_FELM_ALARM_TH, HIGH_FELM_ALARM_TH);

    _bp_vars[no].bp_flags.bits.temp_env_adapt = TEMP_ENV_NOT_ADAPT;

    _bp_vars[no].te_num_meter_intervals = DEFAULT_TE_METER_INTERVALS;

#ifdef BER_METER
    _bp_vars[no].bp_flags.bits.ber_meter_state = 0;
    _bp_vars[no].ber_meter.bit_errors = 0;
    _bp_vars[no].ber_meter.meter_intervals = 0;
#endif

#ifdef ZIP_START
    zip_start_global_var[no].update_state = ZIP_UPDATE_IDLE;
#endif

} /* END _BtInitialize() */



/***********************************************************/
/*    _BitpumpSetSymbolRate()                              */
/*    For the Bt8960/70, sets the appropriate PLL Mode     */
/*    Range.  For the Bt8973, sets the CLAD register.      */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_16BIT symbol_rate              */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _ClearBitpumpIntSource(no);                   */
/*                                                         */
/* Programmer:                                             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _BitpumpSetSymbolRate (BP_U_8BIT no, BP_U_16BIT symbol_rate)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_U_8BIT temp;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/    
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    if ( GET_BITPUMP_TYPE() == BT8960 )
        {
        SYM_RATE_BT8960(symbol_rate);
        }
    else if ( GET_BITPUMP_TYPE() == BT8970 )
        {
        SYM_RATE_BT8970(symbol_rate);
        }
    else
        {
        SYM_RATE_BT8973(symbol_rate);
        }

    /*
     * Determine FELM Lookup Table Index based on Symbol Rate
     */
    temp = (BP_U_8BIT)(symbol_rate >> 3); /* divide by 8 */
    if ( temp < 3 )
        {
        _bp_vars[no].felm_lookup_index = 0;  /* < 192kbps */
        }
    else if ( temp < 5 )
        {
        _bp_vars[no].felm_lookup_index = 1;  /* < 320kbps */
        }
    else if ( temp < 7 )
        {
        _bp_vars[no].felm_lookup_index = 2;  /* < 448kbps */
        }
    else if ( temp < 10 )
        {
        _bp_vars[no].felm_lookup_index = 3;  /* < 640kbps */
        }
    else if ( temp < 15 )
        {
        _bp_vars[no].felm_lookup_index = 4;  /* < 960kbps */
        }
    else if ( temp < 21 )
        {
        _bp_vars[no].felm_lookup_index = 5;  /* < 1344kbps */
        }
    else if ( temp < 30 )
        {
        _bp_vars[no].felm_lookup_index = 6;  /* < 1920kbps */
        }
    else 
        {
        _bp_vars[no].felm_lookup_index = 7;  /* > 1920kbps */
        }

    _WaitMicroSecond(PLL_STABILIZE_DELAY);
    return ;
}



/***********************************************************/
/*    _ClearBitpumpIntSource()                             */
/*    Clears all Bitpump interrupt sources.                */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _ClearBitpumpIntSource(no);                   */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _ClearBitpumpIntSource (BP_U_8BIT no)
{

    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/    
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    BP_WRITE_REG(bp_ptr, mask_low_reg, MASK_ALL); /* Mask all interrupts */
    BP_WRITE_REG(bp_ptr, mask_high_reg, MASK_ALL); /* Mask all interrupts */

    BP_WRITE_REG(bp_ptr, timer_enable, RESET); /* Disable all timers */
    BP_WRITE_REG(bp_ptr, timer_continous, RESET); /* Reset timer continous mode register */

    BP_WRITE_REG(bp_ptr, timer_source, RESET); /* Reset all interrupt sources */
    BP_WRITE_REG(bp_ptr, irq_source, RESET); /* Reset all interrupt sources */
    int_ptr -> bytes.timer_source = RESET; /* Reset timer source interrupts flags */
    int_ptr -> bytes.irq_source = RESET; /* Reset irq source interrupt flags */

    BP_WRITE_REG(bp_ptr, timer_enable, 0xFF);  /* Enable all timers */
    BP_WRITE_REG(bp_ptr, mask_low_reg, 0x10);  /* Unmask all timers except Meter Timer */
    BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON); /* METER timer at continous mode */

#ifdef INT_BUG
    _ResetTimers(no);
#endif

} /* END ClearBitpumpIntSource */

/***********************************************************/
/*    _DcCancel()                                          */
/*    Measures average DC level at A/D input, and          */
/*    compensates by subtracting this constant value from  */
/*    all input samples.                                   */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _DcCancel(no);                                */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _DcCancel (BP_U_8BIT no)
{

    DECLARE_PTR;

    BP_U_8BIT meter;
    BP_S_16BIT value;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;

    NORM(meter) /* Calculate meter normalization factor */

    /*----------------------------------------------------*/
    /* Read DC meter value                                */
    /*----------------------------------------------------*/
    READ_METER_REG(dc_meter_low, dc_meter_high, value, meter);


    /*----------------------------------------------------------*/
    /* Set DC offset register to compensate for measured offset */
    /*----------------------------------------------------------*/
    SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , value); /* Set DC offset register */
#ifdef TDEBUG
    PREFIX;
    printf("DC Offset %d \n", (int)value);
#endif

} /* END _DcCancel */


/***********************************************************/
/*    _InitTxGain()                                        */
/*    Initilizes transmit gain if it wasn't                */   
/*    set in production. Compensates gain for              */
/*    slower data rates                                    */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*                                                         */
/* Programmer:                                             */
/*     Chris Muller              09-April-1999             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitTxGain (BP_U_8BIT no)
{
    DECLARE_MODE_PTR;
    BP_S_8BIT cal_tx_gain;

    INIT_BP_MODE_PTR;

    /* Read Calibrated Tx Gain */
    cal_tx_gain = BP_READ_BIT(bp_mode_ptr, tx_calibrate, tx_gain);


    if ( ( GET_BITPUMP_TYPE() == BT8970 ) &&
         ( BP_READ_BIT(bp_mode_ptr, global_modes, hw_revision) == 0x02 ) &&
         ( cal_tx_gain == 0 ) )
        {

        /* This sets any parts that missed production testing.
         * If the device is a Bt8970-12 and it has not been calibrated by
         * the factory.  A un-tested part will have a '0' as the Tx Gain.
         * To compensate, set the tx_gain to a value of 0x03.
         * This value is based on test data.
         */ 

        /* force new Tx Gain to 0x03 */
        cal_tx_gain = 0x03;
        
        /*
         * Modification of band_gap value.  First set the band_gap value.
         * The value is 0x02 for all Bt8970 devices
         */ 
        BP_WRITE_BIT(bp_mode_ptr, tx_gain, band_gap_high, 0x00);
        BP_WRITE_BIT(bp_mode_ptr, tx_gain, band_gap_low,  0x02);
        
        /*
         * Second, change the default value of band_gap_test_mode to 1.
         * This bit controls which register sets the band_gap
         * (if 1, then tx_gain, else tx_calibrate)
         */ 
        BP_WRITE_BIT(bp_mode_ptr, misc_test, band_gap_test_mode, ON); 
        }
    else if ( GET_BITPUMP_TYPE() == BT8973 ) 
        {

        /* 
         * Due to the use of a low inductance transformer, the transmit gain
         * should be increased at slower data rates. The following algorithm
         * checks the gain value and raises it by the appropriate level.
         * See BP4.2 Release Notes.
         */

        /* Convert 4-bit signed to 8-bit signed value */
        if (cal_tx_gain & 0x08)
            {
            cal_tx_gain |=0xF0;
            }

        if(_bp_vars[no].symbol_rate <= 18) /* <= 144kbps */
            {
            cal_tx_gain += 5;
            }
        else if(_bp_vars[no].symbol_rate <= 26) /* <= 208kbps */
            {
            cal_tx_gain += 3;
            }
        else if(_bp_vars[no].symbol_rate <= 42) /* <= 336kbps */
            {
            cal_tx_gain += 2;
            }
        else if(_bp_vars[no].symbol_rate <= 66) /* <= 528kbps */
            {
            cal_tx_gain += 1;
            }
        
        /* TX-Gain register can be a maximum of 7 */
        if ( cal_tx_gain > 7 )
            {
            cal_tx_gain = 7;
            }
        }

    /*
     * Write the cal_tx_gain to the Tx Gain register.
     * The cal_tx_gain will be either be the adjusted (for untested Bt8970s
     * and low data rate RS8973s) or the default for tested Bt8970s
     * and Bt8960s.
     */
    BP_WRITE_BIT(bp_mode_ptr, tx_gain, tx_gain, cal_tx_gain);

} /* End _InitTxGain() */






/***********************************************************/
/*    _BtLoadMicroCode()                                   */
/*    Program BrHomer micro-code                           */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _BtLoadMicroCode(no);                         */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _BtLoadMicroCode (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT i;

    INIT_BP_PTR;

#ifdef TDEBUG
    PREFIX;
    printf("Load MicroCode \n");
#endif

    for ( i = 0 ; i < MICROCODE_LENGTH ; i++ ) /* Write EQ micro code */
        {
        BP_WRITE_REG(bp_ptr, access_data_byte0, LOW(micro_code[i]));
        BP_WRITE_REG(bp_ptr, access_data_byte1, HIGH(micro_code[i]));
        BP_WRITE_REG(bp_ptr, access_data_byte2, HIGHER(micro_code[i]));
        BP_WRITE_REG(bp_ptr, access_data_byte3, HIGHEST(micro_code[i]));
        BP_WRITE_REG(bp_ptr, eq_microcode_add_write, i);
        DELAY2SYMBOL;
        } /* END-FOR write EQ micro code */

} /* END _BtLoadMicroCode() */

/***********************************************************/
/*    _LookUpTable()                                       */
/*    Look for a known value in a table. There are two     */
/*    types of tables: NLM table and FELM table. The       */
/*    returned value is the corresponding index.           */
/*                                                         */
/*    returns: BP_U_8BIT table index                       */
/*                                                         */
/*    Input variables: BP_TABLE *table_ptr                 */
/*                     BP_S_16BIT value                    */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           noise_margin = _LookUpTable(_nlm_table,       */
/*                                  nlm_meter);            */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Nov-1993              */
/*                                                         */
/* revision history:                                       */
/*   Function is completely changed to increase the look-  */
/*   up-speed. The original linear search algorithm is     */
/*   replaced by a logarithmic algorithm. This way the     */
/*   result is found after 1 up to maximal 7 loops.        */
/*   Measuring results have shown that "LookUpTable()" is  */
/*   now ten times faster. For a 11.0592 MHz 8032 this is: */
/*     more than 6.6 ms before -> less than 750 s now     */
/*                                                         */
/*   Note that using an logarithmic algorithm assumes a    */
/*   sorted list to work properly!                         */
/*                                                         */
/*   This function does a comparison on the high byte then */
/*   checks the low byte.                                  */
/*                                                         */
/*   For non-C51, use a simple 16-bit comparison.          */
/*                                                         */
/***********************************************************/

#ifdef C51
#define GET_LBYTE(x)    (BP_U_8BIT)(x)
#endif

BP_S_8BIT _LookUpTable (BP_TABLE *table_ptr, BP_S_16BIT value)
{
#ifdef C51
    BP_U_8BIT lwr_index, upr_index, index;
    BP_S_8BIT result;
    BP_S_8BIT BP_CONSTANT *tPtr;

    lwr_index = 0;
    upr_index = TABLE_LENGTH - 1;
    do
        {
        index = (upr_index + lwr_index) / 2;
        tPtr = (BP_S_8BIT BP_CONSTANT *)table_ptr + (2 * index);
        result = *tPtr - *((BP_S_8BIT *)(&value));
        if( !result )
            {
            result = *(++tPtr);
            if( GET_LBYTE(value) == (BP_U_8BIT)result )
                return( index );
            if( GET_LBYTE(value) > (BP_U_8BIT)result )
                upr_index = index;
            else
                lwr_index = index + 1;
            }   /* end if !result */
        else if( result < 0 )
            upr_index = index;
        else
            lwr_index = index + 1;
        } while( (upr_index - lwr_index) > 1 );

    if ( lwr_index == 0 )
        {
        return 0;
        }

    if( value >= table_ptr[lwr_index] )
        {
        return (lwr_index);
        }
    return (upr_index);
#else
    /* non-C51 */
    BP_U_8BIT lwr_index, upr_index, index;

    lwr_index = 0;
    upr_index = TABLE_LENGTH - 1;
    do
        {
        index = (upr_index + lwr_index) / 2;

        if ( table_ptr[index] == value )
            {
            return index;
            }
        else if ( table_ptr[index] < value )
            {
            upr_index = index;
            }
        else 
            {
            lwr_index = index;
            }

        } while( (upr_index - lwr_index) > 1 );

    if ( lwr_index == 0 )
        {
        return 0;
        }

    if( value >= table_ptr[lwr_index] )
        {
        return (lwr_index);
        }
    return (upr_index);
#endif

} /* END _LookUpTable */


/***********************************************************/
/*    _NormlizeMeter()                                     */
/*    Calculate the normalization factors for various      */
/*    meters/                                              */
/*                                                         */
/*    returns: BP_U_8BIT normalization factor              */
/*                                                         */
/*    Input variables: BP_S_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           meter = _NormlizeMeter(no);                   */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Nov-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _NormlizeMeter (BP_U_8BIT no)
{

BP_U_8BIT x;
DECLARE_PTR;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;


    x = BP_READ_REG(bp_ptr, meter_high); 
    switch(x) 
    { 
        case 0x1: 
            x = 8; 
            break; 
        case 0x2: 
            x = 7; 
            break; 
        case 0x4: 
            x = 6; 
            break; 
        case 0x8: 
            x = 5; 
            break; 
        case 0x10: 
            x = 4; 
            break; 
        case 0x20: 
            x = 3; 
            break; 
        case 0x40: 
            x = 2; 
            break; 
        case 0x80: 
            x = 1; 
            break; 
    } /* END-SWITCH */
    
    return (x);

} /* END NormlizeMeter() */


/***********************************************************/
/*    _RdByte()                                            */
/*    Read byte from EQ ram                                */   
/*                                                         */
/*    returns: BP_S_8BIT coefficient value                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_8BIT address                   */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           value = _RdByte(no, STAGE);                   */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Nov-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_S_8BIT _RdByte (BP_U_8BIT no, BP_U_8BIT address)
{
    _ReadAccessByteRAM(no, EQ_ACCESS_RAM, address);
    return ( _bp_vars[no].access_data_byte[0] );

} /* End _RdByte() */



/***********************************************************/
/*    _WrByte()                                            */
/*    Write byte to EQ ram                                 */   
/*                                                         */
/*    returns: none                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_8BIT address                   */
/*                     BP_S_8BIT value                     */     
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           value = _WrByte(no, STAGE, WAIT_FOR_SIGNAL);  */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Nov-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _WrByte (BP_U_8BIT no, BP_U_8BIT address, BP_S_8BIT value)
{
#ifdef NO_INDIRECT_RAM_VARS
    /*
     * Write bit-pump variables to external RAM, not EQ RAM
     */
    if ( address >= EQ_RAM_FIRST_VAR && address <= EQ_RAM_LAST_VAR )
        {
        _bp_vars[no].eq_ram_vars[address - EQ_RAM_FIRST_VAR][0] = value;
        }

#else
    DECLARE_PTR;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;

    BP_WRITE_REG(bp_ptr, access_data_byte0, value); 
    BP_WRITE_REG(bp_ptr, eq_add_write, (address)); 
    DELAY2SYMBOL;
#endif

} /* End _WrByte() */

/***********************************************************/
/*    _WrWord()                                            */
/*    Write Word to EQ ram                                 */   
/*                                                         */
/*    returns: none                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_8BIT address                   */
/*                     BP_S_8BIT low_value                 */     
/*                     BP_S_8BIT high_value                */     
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Nov-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _WrWord (BP_U_8BIT no, BP_U_8BIT address, BP_S_8BIT low_value, BP_S_8BIT high_value)
{

    DECLARE_PTR;

    INIT_BP_PTR;

#ifdef NO_INDIRECT_RAM_VARS
    /*
     * Write bit-pump variables to external RAM, not EQ RAM
     */
    if ( address >= EQ_RAM_FIRST_VAR && address <= EQ_RAM_LAST_VAR )
        {
        _bp_vars[no].eq_ram_vars[address - EQ_RAM_FIRST_VAR][0] = low_value;
        _bp_vars[no].eq_ram_vars[address - EQ_RAM_FIRST_VAR][1] = high_value;
        return;
        }

#endif

    BP_WRITE_REG(bp_ptr, access_data_byte0, low_value);
    BP_WRITE_REG(bp_ptr, access_data_byte1, high_value);
    BP_WRITE_REG(bp_ptr, eq_add_write, address);
    DELAY2SYMBOL;

} /* End _WrWord() */



/***********************************************************/
/*    _ReadNmr()                                           */
/*    Read Noise meter value                               */
/*                                                         */
/*    returns: BP_U_8BIT NMR value (dB)                    */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           value = _ReadNmr(no);                         */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                20-Feb-1996              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

BP_S_8BIT _ReadNmr (BP_U_8BIT no)
{
    DECLARE_PTR;
    status_reg_type status_reg;
    user_setup_low_type user_setup_low;
    BP_S_16BIT value, p_value;
    BP_S_8BIT temp;
    BP_U_8BIT meter;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;

    RD_BYTE(no, STATUS, status_reg.status);
    RD_WORD(no, USER_SETUP, user_setup_low.setup, temp);

    NORM(meter); /* Normlize to meter timer value */
    READ_METER_REG(nlm_low, nlm_high, value, meter);

    if (status_reg.bits.los) /* LOS of signal */
      value = 1000;
    else
      {
      if (user_setup_low.bits.terminal_flag == _HTUC)
        {
        if (_IsFirstEyeOpen(no) == _FAIL)
          value = 1000;
        } /* End if HTUC */
      else /* HTUR */
        {
        /* Check to see if PLL is free running */
        READ_METER_REG(vcxo_frequency_low, vcxo_frequency_high, p_value, 0);
        if ( p_value >= MAX_PLL_VALUE || p_value <= -MAX_PLL_VALUE )
            {
            value = 1000;
            }


        } /* End else HTUR */
      } /* End else */
    temp = _LookUpTable((BP_TABLE *)&_noise_margin, value);
    temp = NOISE_MARGIN_DB(temp);

#ifdef CHAN_UNIT
#ifdef CU_EOC
    /*
     * The Channel Unit EOC Register E in the ETSI standards is defined
     * to return the NMR.  Since the _ReadNmr() is continuously called
     * by the _STARTUP_STATUS API in the ASM(), we can just set the NMR
     * result to a global variable.  The EOC then just returns the global
     * variable.
     */
    {
    extern BP_U_8BIT       BP_XDATA    cu_eoc_rd_nmr[_NO_OF_LOOPS];
    cu_eoc_rd_nmr[no] = temp;
    }
#endif
#endif

    
    return (temp);
    
} /* End _ReadNmr() */



/***********************************************************/
/*    _ScaleByGain()                                       */
/*    Scale value by analog gain value.                    */
/*                                                         */
/*    returns: scaled value                                */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_S_16BIT value                    */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           value = _ScaleByGain(no);                     */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                20-Feb-1996              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_S_16BIT _ScaleByGain (BP_U_8BIT no, BP_S_16BIT value)
{
    BP_U_8BIT gain;

    gain = _GetAdcControlAgain(no);
    switch (gain)
        {
        case AGAIN0DB: /* Really -3dB */
            /* value+value/4+value/8+value/32 = 1.40625*value (1.4142*value) */
            value += (value>>2) + (value>>3) + (value>>5);
            break;
        case AGAIN3DB:  /* Really 0dB */
            /* No change */
            break;
        case AGAIN6DB:  /* Really 3dB */ 
            /* value/2+value/8+value/16+value/32 = 0.71875*value (0.707*value) */
            value = (value >> 1) + (value >> 3) + (value >> 4) +\
            (value >> 5);
            break;
        case AGAIN9DB:  /* Really 6dB */
            /* 0.5*value */
            value >>= 1;
            break;
        case AGAIN12DB:  /* Really 9dB */
            /* 0.5*0.71875*value = 0.359375*value (0.3548*value) */
            value >>= 1;
            value = (value >> 1) + (value >> 3) + (value >> 4) +\
            (value >> 5);
            break;
        case AGAIN15DB:  /* Really 12dB */
            /* 0.25*value */
            value >>= 2;
            break;
        }
    return (value);

} /* End _ScaleByGain() */



/****************************************************************/
/*    _ReadAccessByteRAM()                                      */
/*    This function reads the Reads the specified Access        */
/*    Data Byte RAM locations.  The bit-pump access data        */
/*    byte registers are copied to the                          */
/*    'global_access_data_byte' array.                          */
/*                                                              */
/*    In the Bt8970 Rev C, need to disable the Bitpump          */
/*    Interrupts during this operation.                         */
/*                                                              */
/*    Notes:                                                    */
/*                                                              */
/*    No error checking is done.                                */
/*                                                              */
/*    It is up to the calling function to determine how         */
/*    many bytes are accessible.  Otherwise, unpredicable       */
/*    behavior could result.                                    */
/*                                                              */
/*    returns: none                                             */
/*                                                              */
/*    Input variables: BP_U_8BIT no                             */
/*                     BP_U_8BIT ram                            */
/*                     BP_U_8BIT address                        */
/*                                                              */
/*    Output variables: 'global_access_data_byte'               */
/*                                                              */
/*    example:                                                  */
/*          _ReadAccessByteRAM(no, EQ_ACCESS_RAM, STATUS);      */
/*                                                              */
/* Programmer:                                                  */
/*     Dean Rasmussen             09-Sep-1996                   */
/*                                                              */
/* revision history:                                            */
/*                                                              */
/****************************************************************/
void _ReadAccessByteRAM (BP_U_8BIT no, BP_U_8BIT ram, BP_U_8BIT address)
{
    DECLARE_PTR;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;

#ifdef NO_INDIRECT_RAM_VARS
    if ( ram == EQ_ACCESS_RAM )
        {
        /* 2-bytes */
        if ( address >= EQ_RAM_FIRST_VAR && address <= EQ_RAM_LAST_VAR )
            {
            _bp_vars[no].access_data_byte[0] = _bp_vars[no].eq_ram_vars[address - EQ_RAM_FIRST_VAR][0];
            _bp_vars[no].access_data_byte[1] = _bp_vars[no].eq_ram_vars[address - EQ_RAM_FIRST_VAR][1];
            return;
            }
        }
    else if ( ram == SCRATCH_ACCESS_RAM )
        {
        /* only 1-byte wide */
        _bp_vars[no].access_data_byte[0] = _bp_vars[no].scratch_pad_vars[address];
        return;
        }
#endif

    _DisableBitpumpInterrupt();

    switch ( ram )
        {
        case LEC_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, linear_ec_tap_select_read, address);
            break;

        case NLEC_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, nonlinear_ec_tap_select_read, address);
            break;

        case DFE_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, dfe_tap_select_read, address);
            break;

        case SCRATCH_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, scratch_pad_tap_select_read, address);
            break;

        case EQ_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, eq_add_read, address);
            break;

#if 0
          /* Never Used */
        case MICRO_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, eq_microcode_add_read, address);
            break;
#endif

        }

    DELAY2SYMBOL;
    _bp_vars[no].access_data_byte[0] = BP_READ_REG(bp_ptr, access_data_byte0);
    _bp_vars[no].access_data_byte[1] = BP_READ_REG(bp_ptr, access_data_byte1);
    _bp_vars[no].access_data_byte[2] = BP_READ_REG(bp_ptr, access_data_byte2);
    _bp_vars[no].access_data_byte[3] = BP_READ_REG(bp_ptr, access_data_byte3);

    _EnableBitpumpInterrupt();

    return ;

} /* End _ReadAccessByteRAM() */


#if 0
/* Never Used, keep around just in case */

/****************************************************************/
/*    _WriteAccessByteRAM()                                     */
/*    This function writes the specified Access                 */
/*    Data Byte RAM locations.  The bit-pump access data        */
/*    byte registers are copied from the                        */
/*    'global_access_data_byte' array.                          */
/*                                                              */
/*    In the Bt8970 Rev C, need to disable the Bitpump          */
/*    Interrupts during this operation.                         */
/*                                                              */
/*    Notes:                                                    */
/*                                                              */
/*    No error checking is done.                                */
/*                                                              */
/*    It is up to the calling function to determine how         */
/*    many bytes are accessible.  Otherwise, unpredicable       */
/*    behavior could result.                                    */
/*                                                              */
/*    returns: none                                             */
/*                                                              */
/*    Input variables: BP_U_8BIT no                             */
/*                     BP_U_8BIT ram                            */
/*                     BP_U_8BIT address                        */
/*                     'global_access_data_byte'                */ 
/*                                                              */
/*    Output variables: none.                                   */
/*                                                              */
/*    example:                                                  */
/*          _WriteAccessByteRAM(no, EQ_ACCESS_RAM, STATUS);     */
/*                                                              */
/* Programmer:                                                  */
/*     Dean Rasmussen             29-May-1997                   */
/*                                                              */
/* revision history:                                            */
/*                                                              */
/****************************************************************/
void _WriteAccessByteRAM  (BP_U_8BIT no, BP_U_8BIT ram, BP_U_8BIT address)
{
    DECLARE_PTR;

    /*------------------------------------------------*/
    /* Initiate bitpump pointers                      */
    /*------------------------------------------------*/
    INIT_BP_PTR;

    _DisableBitpumpInterrupt();

    BP_WRITE_REG(bp_ptr, access_data_byte0, _bp_vars[no].access_data_byte[0]);
    BP_WRITE_REG(bp_ptr, access_data_byte1, _bp_vars[no].access_data_byte[1]);
    BP_WRITE_REG(bp_ptr, access_data_byte2, _bp_vars[no].access_data_byte[2]);
    BP_WRITE_REG(bp_ptr, access_data_byte3, _bp_vars[no].access_data_byte[3]);

    switch ( ram )
        {
        case LEC_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, linear_ec_tap_select_write, address);
            break;

        case NLEC_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, nonlinear_ec_tap_select_write, address);
            break;

        case DFE_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, dfe_tap_select_write, address);
            break;

        case SCRATCH_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, scratch_pad_tap_select_write, address);
            break;

        case EQ_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, eq_add_write, address);
            break;

        case MICRO_ACCESS_RAM:
            BP_WRITE_REG(bp_ptr, eq_microcode_add_write, address);
            break;
        }

    DELAY2SYMBOL;

    _EnableBitpumpInterrupt();

    return ;

} /* End _WriteAccessByteRAM() */
#endif


/***********************************************************/
/*    _WrScratchPad()                                      */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                     BP_U_8BIT index;                    */
/*                     BP_S_16BIT value;                   */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _WrScratchPad (BP_U_8BIT no, BP_U_8BIT index, BP_S_16BIT value)
{
#ifdef NO_INDIRECT_RAM_VARS
    _bp_vars[no].scratch_pad_vars[ ((index) << 1) ] = LOW(value);
    _bp_vars[no].scratch_pad_vars[ ((index) << 1) + 1 ] = HIGH(value);
#else
    DECLARE_PTR;
 
    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;

    BP_WRITE_REG(bp_ptr, access_data_byte0, LOW(value)); 
    BP_WRITE_REG(bp_ptr, scratch_pad_tap_select_write, (index) << 1); 
    DELAY2SYMBOL; 

    BP_WRITE_REG(bp_ptr, access_data_byte0, HIGH(value)); 
    BP_WRITE_REG(bp_ptr, scratch_pad_tap_select_write, ((index) << 1) + 1); 
    DELAY2SYMBOL; 
#endif
} /* END _WrScratchPad() */


/***********************************************************/
/*    _RdScratchPad()                                      */
/*                                                         */
/*    returns: BP_S_16BIT value                            */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                     BP_U_8BIT index;                    */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_S_16BIT _RdScratchPad (BP_U_8BIT no, BP_U_8BIT index)
{
    BP_S_16BIT value;
    BP_S_8BIT low_byte, high_byte;

#ifdef NO_INDIRECT_RAM_VARS
    low_byte = _bp_vars[no].scratch_pad_vars[ ((index) << 1) ];
    high_byte = _bp_vars[no].scratch_pad_vars[ ((index) << 1) + 1 ];
#else
    _ReadAccessByteRAM(no, SCRATCH_ACCESS_RAM, (BP_U_8BIT)((index) << 1));
    low_byte = _bp_vars[no].access_data_byte[0];
    
    _ReadAccessByteRAM(no, SCRATCH_ACCESS_RAM, (BP_U_8BIT)((index << 1) + 1));
    high_byte = _bp_vars[no].access_data_byte[0];
#endif

    (value) = BYTE2WORD(high_byte, low_byte);
    
    return (value);

} /* END _RdScratchPad() */
