/***********************************************************/
/*    USER.C                                               */
/*    User definitions.                                    */
/*    (C) Copyright 1993 by Rockwell Corporation           */
/*                                                         */
/*    This program is copyrighted by Rockwell Corporation  */
/*                                                         */
/* Description:                                            */
/*     This file contains definitions of the absolute      */
/*     addresses of Bitpump chips, and routine for         */
/*     initializing the Bitpump pointers to these          */
/*     addresses.                                          */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*     Bitpumps absolute addresses.                        */
/*     _BitpumpPointersInit() contents.                    */
/*                                                         */
/* List of functions included in this module:              */
/*  _BtHomerPointersInit()                                 */
/*  _Delay2Symbols()                                       */
/*  _EnableUserInterrupts()                                */
/*  _DisableUserInterrupts()                               */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             March - 1998             */
/*     Iris Shuker                15-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/***********************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.1     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Declare Bitpumps base addresses.                         */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.1            */
    /*----------------------------------------------------------*/

static BP_U_16BIT BP_CONSTANT bit_pump_addr[] =
{
#ifdef ZIPSOCKET
    0x8000,
    0xA000
#else
    0xF000,
    0xD000,
    0xE000,
    0xF000, /* Not implemented on EVMs */
    0xF000, /* Not implemented on EVMs */
    0xF000  /* Not implemented on EVMs */
#endif /* ZIPSOCKET */
};

    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.1     Ends          <<<*/
    /*----------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/

DECLARE_PTR_ARRAY;

    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.2     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Calibtare Noise Margin table according to user noise     */
    /* source.                                                  */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.2            */
    /*----------------------------------------------------------*/

    /* Noise Margin look up table */
BP_TABLE BP_CONSTANT _noise_margin[TABLE_LENGTH] =

#if 0
/*
 * The following table is calibrated for Consultronics DLS200 ANSI ISDN
 * Loop 4 with DSL-Next noise.
 */
/* Cell no: */
/*     0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  */
{    880, 834, 789, 748, 708, 669, 633, 599, 566, 535, 506, 478, 452,
/*    13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  */
     427, 404, 381, 360, 341, 322, 304, 288, 272, 258, 244, 231, 219,
/*    26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  */
     208, 198, 188, 179, 169, 163, 155, 147, 141, 135, 130, 124, 120,
/*    39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  */
     115, 111, 107, 103,  99,  97,  93,  91,  89,  85,  83,  82,  79,
/*    52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63        */
      77,  77,  74,  72,  72,  70,  69,  69,  67,  66,  66,  65 };
#endif

/*
 * The following table is calibrated for W&G ILS200 ETSI Loop 2 with shaped
 * noise with low crest factor
 */
/* Cell no: */
/*     0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14 */
   { 999, 999, 999, 999, 999, 999, 959, 905, 855, 807, 762, 719, 679, 641, 605, \
/*    15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29 */
     571, 539, 509, 481, 454, 428, 404, 382, 366, 351, 324, 304, 290, 272, 258, \
/*    30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44 */
     244, 232, 218, 208, 198, 188, 178, 170, 160, 154, 146, 138, 132, 126, 120, \
/*    45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,*/
     110, 104, 100,  94,  90,  86,  84,  80,  76,  72,  68,  66,  64,  62,  60, \
/*    60,  61,  62,  63*/ 
      58,  56,  52,  50};

    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.2     Ends          <<<*/
    /*----------------------------------------------------------*/



/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

void _BtHomerPointersInit (BP_U_8BIT no)
{

#ifdef C51

#ifdef PDATA_MODE
    _bit_pump[no] = (transceiver BP_VOLATILE BP_PDATA * )(HIGH(bit_pump_addr[no]));
#elif defined XDATA_MODE
    _bit_pump[no] = (transceiver BP_VOLATILE BP_XDATA * )(bit_pump_addr[no]);
#endif

#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.3     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Bitpump pointers initialization for non 8031 environment.*/
    /* Executing this routine should initialize the Bitpumps    */
    /* pointer array values based on the 'bit_pump_addr[]' array*/
    /* the casting details are probably compiler specific.      */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.3            */
    /*----------------------------------------------------------*/

    _bit_pump[no] = (transceiver BP_VOLATILE * )(bit_pump_addr[no]);


    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.3     Ends          <<<*/
    /*----------------------------------------------------------*/
#endif

} /* END _BtHomerPointersInit() */

#ifdef ADD_DELAY

#ifdef C51
#include <intrins.h>
#endif

void _Delay2Symbols (BP_U_8BIT no)
{
#ifdef C51

    /*
     * Need uProc delay when running at slower Symbol Rates
     *
     * The 'if(...)' adds ~25uS of delay on Intel 8032 @ 11.0592MHz
     *
     * At 160kbps: 2 symbols ~= 25uS (which is sym_rate = 20)
     * Need to add delay for < 160kbps (i.e. 144kbps)
     *
     * 5 NOPs add ~5uS more delay which totals ~30uS, which supports ~128kbps
     */
    if ( _bp_vars[no].symbol_rate <= 20 )
        {
        _nop_();
        _nop_();
        _nop_();
        _nop_();
        _nop_();
        }   /* end if <= 65 */

#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.4     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.4            */
    /*----------------------------------------------------------*/



    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.4     Ends          <<<*/
    /*----------------------------------------------------------*/
#endif

} /* END _Delay2Symbols() */
#endif /* ADD_DELAY */


/***********************************************************/
/*    _SetActvationInterval()                              */
/*    This function sets the Activation Interval Timeout   */
/*    value into SUT4 based on the _ACTIVATION_TIMEOUT     */
/*    setting.                                             */
/*                                                         */
/*    returns: none                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             10-Jun-1998              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _SetActivationInterval (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    user_param_low_type user_param_low;
    BP_U_16BIT temp_u16, sym_rate;
    BP_U_8BIT temp;

    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    sym_rate = COMP_SYM_RATE();

    RD_WORD(no, PARAMETERS, user_param_low.reg, temp);
    
    if ( user_param_low.bits.activation_timeout == _ACT_TIME_30SEC )
        {
        /* (sym_rate * 32) - (sym_rate * 1) = sym_rate * 31 :  31 Sec Timer */ 
        temp_u16 = (((sym_rate << 5) - sym_rate) << 2);
        }
    else if ( user_param_low.bits.activation_timeout == _ACT_TIME_52SEC )
        {
        /* (sym_rate * 32) + (sym_rate * 16) + (sym_rate * 4) = sym_rate * 52 :  52 Sec Timer */ 
        temp_u16 = (((sym_rate << 5) + (sym_rate << 4) + (sym_rate << 2) ) << 2);
        }
    else
        {
        /* SUT4 = (48 * sym_rate + 8192) */
        temp_u16 = (sym_rate << 5) + (sym_rate << 4) + 8192;
        }

    SET_WORD(bp_ptr, sut4_low, sut4_high, temp_u16);
#ifdef INT_BUG
    _RestartVirtualTimer(no, _SUT4, temp_u16);
#endif

    RESTART(bp_mode_ptr, sut4);

}


#ifdef HTUR
void _EnableUserInterrupts (void)
{

#ifdef C51
#ifdef SER_COM
        ES = 1;
#endif
#ifdef TDEBUG
        ES = 0;
#endif
#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.5     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Write the contents of routine EnableUserInterrupts().    */
    /* Calling this routine should enable all user interrupts.  */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.5.           */
    /*----------------------------------------------------------*/


    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.5     Ends          <<<*/
    /*----------------------------------------------------------*/
#endif

} /* END _EnableUserInterupts() */



void _DisableUserInterrupts (void)
{

#ifdef C51
        ES = 0;
#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.6     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Write the contents of routine DisableUserInterrupts().   */
    /* Calling this routine should disable all user interrupts  */
    /* excluding Bitpump interrupt.                             */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.6.           */
    /*----------------------------------------------------------*/


    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.6     Ends          <<<*/
    /*----------------------------------------------------------*/
#endif

} /* END _DisableUserInterupts() */
#endif


/*
 * FUNCTION:   _WaitMicroSecond
 *
 * PARAMETERS: delay - number of microseconds to delay; range = 0 to 65535uS
 *
 * PURPOSE:    This function creates a delay of the specied number of
 *             microseconds.
 *             
 * RETURN:     none
 *
 * NOTES:      For bit-pump PLL to stabilze, need ~200uS (200 is passed in).
 *
 * CHANGES:    July 2, 1998    V1.0  :  New
 */
void _WaitMicroSecond (BP_U_16BIT delay)
{
    BP_U_16BIT i;

    for ( i = 0 ; i < delay ; i++ )
        {
#ifdef C51
#ifdef ADD_DELAY
        /*
         * On Intel 8032 running @ 11.0592MHz, each pass through the for loop
         * is ~20uS.  So increment 'i' by 20, which is effectively creates a
         * divide by 20.  In other words, this will only be looped
         * 'delay / 20' times.
         *
         * For example, if wanted to wait 200uS, then need to perform
         * the 'for()' loop only 10 times.  (200uS / 20uS = 10)
         */
        _nop_();
        i += 20;
#endif /*ADD_DELAY*/
#else
    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.7     Begins        <<<*/
    /*----------------------------------------------------------*/
    /* Modification Description:                                */
    /* Create a delay of at least 1 micro-second.               */
    /*                                                          */
    /* Reference:                                               */
    /* "Bt8970 Software User's Guide", Section ?.?.7.           */
    /*----------------------------------------------------------*/


    /*----------------------------------------------------------*/
    /*>>>   User Modifiable Section user.7     Ends          <<<*/
    /*----------------------------------------------------------*/
#endif

        }
}


#ifdef TDEBUG
#ifdef PRINTF_NOT_SUPPORTED

#include <stdarg.h>
#include <string.h>

char BP_XDATA printf_buf[256];
va_list BP_XDATA arglist;

/*
 * FUNCTION:   my_printf
 *
 * PARAMETERS: see printf()
 *
 * PURPOSE:    Use this function in systems that don't support the
 *             printf() command.  This function uses the 'vsprintf()'
 *             function to copy the formatted string into 'printf_buf[]'.
 *             The 'put()' function then just outputs the string.
 *
 *             The 'put()' function should then be replaced by whatever
 *             equivalent function is used in the desired system, i.e.
 *             outp() or output(), ...
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    April 20, 1998    V1.0  :  New
 */
int my_printf (char *format, ...)
{
    int rValue;
    char *ptr;

    va_start(arglist, format);

    rValue = vsprintf(printf_buf, format, arglist);

    va_end(arglist);

    /* Replace all new-line '\n' with a blank space. 'puts' adds its
     * own new-line
     */
    do
        {
        ptr = strchr(printf_buf, '\n');
        if ( ptr != NULL )
            {
            ptr[0] = ' ';
            }
        }
    while ( ptr != NULL );

    puts(printf_buf);

    return rValue;
}

#endif /* PRINTF_NOT_SUPPORTED */
#endif /* TDEBUG */
