/***********************************************************/
/*    TESTMODE.C                                           */
/*    (C) Copyright 1993 by Rockwell Corporation           */
/*                                                         */
/*    This program is copyrighted by Rockwell Corporation  */
/*                                                         */
/* Description:                                            */
/*    Performs test modes:                                 */
/*    loopbacks, isolated transmit pulse, scrambed 1's     */
/*    VCXO control voltage test.                           */
/*    Performs Bitpump full SelfTest.                      */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*    None                                                 */
/*                                                         */
/* List of functions included in this module:              */
/*  _TestMode()                                            */
/*  _SelfTest()                                            */
/*  _HandleTestMode()                                      */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             March - 1998             */
/*     Iris Shuker                31-August-1993           */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/***********************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"


/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define CHECK_VALUE      0xAAAA
#define CHECK_BYTE_VALUE   0xAA
#define N_CHECK_VALUE    0x5555
#define N_CHECK_BYTE_VALUE 0x55

/* Transmitter isolated pulse constants */
#define ISOLATED_PULSE_OUTPUT_MINUS3   0
#define ISOLATED_PULSE_OUTPUT_MINUS1   1
#define ISOLATED_PULSE_OUTPUT_PLUS3    2
#define ISOLATED_PULSE_OUTPUT_PLUS1    3

#define ISOLATED_PULSE_METER_INTERVAL       0x100
#define ALT_SYMBOL_METER_INTERVAL           0x001

/* VCXO fixed values */
#define VCXO_FREQ_NOMINAL          0
#define VCXO_FREQ_MIN         -30000
#define VCXO_FREQ_MAX          30000

/* Analog Loopack constants */
#define ALB_DISABLED        0
#define ALB_BYPASS_HYBRID   1
#define ALB_TRANSMITTING    2
#define ALB_SILENT          3

#define ALB_EC_TAP      3

#define TM_ALB_INTERVAL     (BP_U_16BIT)6000
#define TM_ERLE_INTERVAL    (BP_U_16BIT)6000  /* 10.5s at 1168k */
#define TM_AAGC_INTERVAL    (BP_U_16BIT)6000

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

BP_U_8BIT _AnalogLoopback(BP_U_8BIT no);

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

/***********************************************************/
/*    _TestMode()                                          */
/*    Run Bitpump testmodes and loopbacks.                 */
/*                                                         */
/*    returns: _PASS/_FAIL                                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_8BIT mode                      */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _TestMode(no, _NEAR_LOOPBACK);       */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                                         */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

BP_U_8BIT _TestMode (BP_U_8BIT no, BP_U_8BIT mode)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    user_setup_low_type user_setup_low;
    user_setup_high_type user_setup_high;

#ifdef TDEBUG
    PREFIX;
    printf("Test Mode \n");
#endif

    /*-------------------------------------------*/
    /* Initiate bitpump pointers                 */
    /*-------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    RD_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup);
    switch (mode)
        {
        case _EXIT_TEST_MODE:
            {

#ifdef TDEBUG
            PREFIX;
            printf("Exit Test Mode \n");
#endif

            switch (user_setup_high.bits.test_mode) /* Previous mode */
                {
                case _NEAR_LOOPBACK:
                    if (user_setup_low.bits.rx_descr) /* Activate receive descrambler */
                        BP_WRITE_BIT(bp_mode_ptr, detector_modes, output_mux_control, SELECT_DESCRAMBLER_OUTPUT);
                    else /* Bypass receive descrambler */
                        BP_WRITE_BIT(bp_mode_ptr, detector_modes, output_mux_control, SELECT_DETECTOR_OUTPUT);
                    break;

                case _FAR_LOOPBACK:
                    if (user_setup_low.bits.seq_source == _EXTERNAL) /* External startup sequence source */
                        if (user_setup_low.bits.tx_scr) /* Activate transmit scrambler */
                            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA);
                        else /* Bypass transmit scrambler */
                            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA);
                    else /* Internal startup sequence source */
                        BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
                    break;

                case _ANALOG_LOOPBACK:
                case _ISOLATED_PULSE_PLUS3:
                case _ISOLATED_PULSE_PLUS1:
                case _ISOLATED_PULSE_MINUS1:
                case _ISOLATED_PULSE_MINUS3:
                case _FOUR_LEVEL_SCR:
                case _TWO_LEVEL_SCR:
                case _VCXO_NOMINAL:
                case _VCXO_MIN:
                case _VCXO_MAX:
                case _INTERNAL_ANALOG_LOOPBACK:
                case _ISOLATED_ANALOG_LOOPBACK:
                case _ERLE_TEST:
                case _MEASURE_AAGC:
                case _ALTERNATING_SYMBOLS_3:
                case _ALTERNATING_SYMBOLS_1:
                    _BtInitialize(no);
                    _SetMeterTimer(no, ALT_METER);
                    WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
                    WR_WORD(no, STAGE2, TEMP_ENV_IDLE, 0);
                    break;

                } /* END-SWITCH previous mode */
            } /* END-CASE EXIT_TEST_MODE */
            break;

        case _ANALOG_LOOPBACK:
        case _ISOLATED_ANALOG_LOOPBACK:
        case _INTERNAL_ANALOG_LOOPBACK:
            WR_BYTE(no, STATUS, INIT_STATUS_REG); /* Reset Normal Op flag */
            WR_WORD(no, STAGE, IDLE, TM_ALB_INITIALIZE);
            break;

        case _NEAR_LOOPBACK:
#ifdef TDEBUG
            PREFIX;
            printf("Near Loopback \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, detector_modes, output_mux_control, SELECT_TRANSMITTER_INPUT);
            break;

        case _FAR_LOOPBACK:
#ifdef TDEBUG
            PREFIX;
            printf("Far Loopback \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, DETECTOR_LOOPBACK);
            break;

        case _ISOLATED_PULSE_PLUS3:
        case _ISOLATED_PULSE_PLUS1:
        case _ISOLATED_PULSE_MINUS1:
        case _ISOLATED_PULSE_MINUS3:
            /* combined to save code space */
            SET_WORD(bp_ptr, meter_low, meter_high, ISOLATED_PULSE_METER_INTERVAL);
            BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, ON);
            BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON);
            RESTART(bp_mode_ptr, meter);
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, ISOLATED_PULSE); /* Isolated pulse mode */

            if ( mode == _ISOLATED_PULSE_PLUS3 )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, isolated_pulse, ISOLATED_PULSE_OUTPUT_PLUS3);
#ifdef TDEBUG
                PREFIX;
                printf("Isolated pulse +3 \n");
#endif
                }
            else if ( mode == _ISOLATED_PULSE_PLUS1 )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, isolated_pulse, ISOLATED_PULSE_OUTPUT_PLUS1);
#ifdef TDEBUG
                PREFIX;
                printf("Isolated pulse +1 \n");
#endif
                }
            else if ( mode == _ISOLATED_PULSE_MINUS1 )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, isolated_pulse, ISOLATED_PULSE_OUTPUT_MINUS1);
#ifdef TDEBUG
                PREFIX;
                printf("Isolated pulse -1 \n");
#endif
                }
            else 
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, isolated_pulse, ISOLATED_PULSE_OUTPUT_MINUS3);
#ifdef TDEBUG
                PREFIX;
                printf("Isolated pulse -3 \n");
#endif
                }


            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Operate transmitter */
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        case _FOUR_LEVEL_SCR:
#ifdef TDEBUG
            PREFIX;
            printf("Four level scrambled 1's \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Operate transmitter */
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        case _TWO_LEVEL_SCR:
#ifdef TDEBUG
            PREFIX;
            printf("Two level scrambled 1's \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Operate transmitter */
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        case _VCXO_NOMINAL:
#ifdef TDEBUG
            PREFIX;
            printf("VCXO control voltage at nominal value \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Use fixed mode */
            SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_FREQ_NOMINAL);
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        case _VCXO_MIN:
#ifdef TDEBUG
            PREFIX;
            printf("VCXO control voltage at minimum value \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Use fixed mode */
            SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_FREQ_MIN);
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        case _VCXO_MAX:
#ifdef TDEBUG
            PREFIX;
            printf("VCXO control voltage at maximum value \n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Use fixed mode */
            SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_FREQ_MAX);
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        case _ERLE_TEST:
#ifdef ERLE

#ifdef TDEBUG
            PREFIX;
            printf("ERLE\n");
#endif
            /* resets Normal Operation bit */
            WR_BYTE(no, STATUS, RESET);

            WR_WORD(no, STAGE, IDLE, TM_ERLE_INIT);
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, 0);
#endif /* ERLE */
            break;

        case _MEASURE_AAGC:
#ifdef ERLE

#ifdef TDEBUG
            PREFIX;
            printf("Measure AAGC\n");
#endif
            /* resets Normal Operation bit */
            WR_BYTE(no, STATUS, RESET);

            WR_WORD(no, STAGE, IDLE, TM_AAGC_INIT);
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, 0);

#endif /* ERLE */
            break;

        case _ALTERNATING_SYMBOLS_3:
        case _ALTERNATING_SYMBOLS_1:
            /* combined to save code space */
            if ( GET_BITPUMP_TYPE() <= BT8970 )
                {
                return _PASS;
                }

            SET_WORD(bp_ptr, meter_low, meter_high, ALT_SYMBOL_METER_INTERVAL);
            BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, ON);
            BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON);
            RESTART(bp_mode_ptr, meter);
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, ALTERNATING_SYMBOL);

            if ( mode == _ALTERNATING_SYMBOLS_3 )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, isolated_pulse, ISOLATED_PULSE_OUTPUT_PLUS3);
#ifdef TDEBUG
                PREFIX;
                printf("Alternating Pulses +/-3 \n");
#endif
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, isolated_pulse, ISOLATED_PULSE_OUTPUT_PLUS1);
#ifdef TDEBUG
                PREFIX;
                printf("Alternating Pulses +/-1 \n");
#endif
                }

            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Operate transmitter */
            WR_WORD(no, STAGE, IDLE, TEST_MODE_IDLE);
            break;

        default:
            return (_FAIL);

        } /* END-SWITCH */
    user_setup_high.bits.test_mode = mode;
    WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup);
    return (_PASS);

} /* END _TestMode() */


/***********************************************************/
/*    _SelfTest()                                          */
/*    Run Bitpump self test.                               */
/*                                                         */
/*    returns: _PASS/_FAIL                                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _SelfTest(no);                       */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                                         */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

BP_U_8BIT _SelfTest (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
     
    /*-------------------------------------------*/
    /* Initiate bitpump pointers                 */
    /*-------------------------------------------*/    
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    /*-------------------------------------------------------------------*/
    /* _SelfTest routine tests Bitpump.                                  */
    /*-------------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, linear_ec_modes, RESET);
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze Linear EC */

    SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, CHECK_VALUE);
    SET_WORD(bp_ptr, access_data_byte2, access_data_byte3, CHECK_VALUE);
    BP_WRITE_REG(bp_ptr, linear_ec_tap_select_write, 0);
    DELAY2SYMBOL;
    SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, N_CHECK_VALUE);
    SET_WORD(bp_ptr, access_data_byte2, access_data_byte3, N_CHECK_VALUE);
    BP_WRITE_REG(bp_ptr, linear_ec_tap_select_read, 0);
    DELAY2SYMBOL;

    _ReadAccessByteRAM(no, LEC_ACCESS_RAM, 0);
    if ( (BP_U_8BIT)_bp_vars[no].access_data_byte[0] != CHECK_BYTE_VALUE )
        {
#ifdef TDEBUG
        PREFIX;
        if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[0]) == N_CHECK_BYTE_VALUE)
            printf("RAM present instead of Bitpump\n");
        else
            printf("Bitpump not present\n");
#endif
        return (_FAIL);
        }

    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[1]) != CHECK_BYTE_VALUE)
        return (_FAIL);
    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[2]) != CHECK_BYTE_VALUE)
        return (_FAIL);
    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[3]) != CHECK_BYTE_VALUE)
        return (_FAIL);

    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF);
    SET_RESET(bp_mode_ptr, linear_ec_modes, zero_coefficients);

    _ReadAccessByteRAM(no, LEC_ACCESS_RAM, 0);
    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[0]) != 0)
        return (_FAIL);
    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[1]) != 0)
        return (_FAIL);
    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[2]) != 0)
        return (_FAIL);
    if ((BP_U_8BIT)(_bp_vars[no].access_data_byte[3]) != 0)
        return (_FAIL);

#ifdef TDEBUG
    printf("Bitpump present\n");
#endif
    return (_PASS);

} /* end _SelfTest() */



/***********************************************************/
/*    _HandleTestMode()                                    */
/*                                                         */
/*    returns: nothing                                     */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*                                                         */
/* Programmer:                                             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _HandleTestMode (BP_U_8BIT no)
{

    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    user_setup_high_type user_setup_high;
    user_setup_low_type user_setup_low;
    status_reg_type status_reg;           /* Status variable */
    BP_S_8BIT temp;
    BP_U_8BIT su_stage, tm_stage;
    BP_S_16BIT value;

#ifdef ERLE
    ERLE_MODE erle_mode;
    BP_S_16BIT slm, felm;
#endif

#ifdef INT_BUG
    BP_U_16BIT u16_value;
#endif

    /*-------------------------------------------*/
    /* Initiate bitpump pointers                 */
    /*-------------------------------------------*/   
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    /*------------------------------------------------------------*/
    /* Read Test Mode stage number from Bitpump.                  */
    /*------------------------------------------------------------*/
    RD_WORD(no, STAGE, su_stage, tm_stage);

    switch ( tm_stage )
        {
        case TEST_MODE_IDLE:
            break;

        /*------------------------------------------------------------*/
        /* Wait Meter Intervals                                       */
        /* The next stage after waiting the meter intervals is set    */
        /* by the upper byte of the STAGE2 EQ Reg File.               */
        /*------------------------------------------------------------*/

        case TM_WAIT_METER_INTERVAL1:
            RESTART(bp_mode_ptr, meter);
#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif

            tm_stage = TM_WAIT_METER_INTERVAL2;
            break;

    
        case TM_WAIT_METER_INTERVAL2:
            TIMER_BREAK(meter);

#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif

            /* Set Stage to the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            break;


        /*------------------------------------------------------------*/
        /* Analog Loop Back Testmodes                                 */
        /*------------------------------------------------------------*/
        case TM_ALB_INITIALIZE:
            /* Need to save 'testmode' setup - found in high setup
             * it is cleared in BtInit
             */
            RD_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup);

            _BtInitialize(no);
            _SetMeterTimer(no, ALT_METER);

            if ( user_setup_high.bits.test_mode == _ANALOG_LOOPBACK )
                {
                BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, ALB_DISABLED);
#ifdef TDEBUG
                PREFIX;
                printf("Analog Loopback\n");
#endif
                }
            else if ( user_setup_high.bits.test_mode == _INTERNAL_ANALOG_LOOPBACK )
                {
                BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, ALB_TRANSMITTING);
#ifdef TDEBUG
                PREFIX;
                printf("Internal Analog Loopback (Transmitting)\n");
#endif
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, ALB_SILENT);
#ifdef TDEBUG
                PREFIX;
                printf("Isolated Analog Loopback (Tx Silent)\n");
#endif
                }

            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_low.setup);

            /* Set Activation Interval */
            SET_WORD(bp_ptr, sut4_low, sut4_high, TM_ALB_INTERVAL);
#ifdef INT_BUG
            _RestartVirtualTimer(no, _SUT4, TM_ALB_INTERVAL);
#endif
            RESTART(bp_mode_ptr, sut4);

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_ALB_ACTIVATE);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;


        case TM_ALB_ACTIVATE:
            TIMER_BREAK(meter);
            _SetFelmMask(no);

            /*------------------------------------------------------------*/
            /* Trasmit 2-Level Scrambled Ones                             */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */

            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0); /* Init receive sampling phase */

            _SetAdcControlAgain(no, AGAIN0DB);

            tm_stage = TM_ALB_ADAPT_EC;
            break;

        case TM_ALB_ADAPT_EC:

            /*------------------------------------------------------------*/
            /* Adapt EC.                                                  */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON);

            INIT_TIMER(t3, _T3, t3_low, t3_high, 30000);
            tm_stage = TM_ALB_ADAPT_DAGC;
            break;


        case TM_ALB_ADAPT_DAGC:
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze EC coefficient */

            _ReadAccessByteRAM(no, LEC_ACCESS_RAM, 3);
            value = BYTE2WORD(_bp_vars[no].access_data_byte[3], _bp_vars[no].access_data_byte[2]);

            value -= 2500;
            SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, RESET);
            SET_WORD(bp_ptr, access_data_byte2, access_data_byte3, value);
            BP_WRITE_REG(bp_ptr, linear_ec_tap_select_write, ALB_EC_TAP);
            DELAY2SYMBOL;

            /*------------------------------------------------------------*/
            /* Adapt DAGC.                                                */
            /*------------------------------------------------------------*/
            SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, DAGC_TARGET_INIT_VALUE); /* Set DAGC target init value */
            INIT_DAGC;

            _AdaptDagcSelfMode(no);
            INIT_TIMER(t3, _T3, t3_low, t3_high, 20000);

            tm_stage = TM_ALB_OPEN_EYE1;
            break;

        case TM_ALB_OPEN_EYE1:
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

#ifdef TDEBUG
            printf("After Self Mode - ");
            _PrintfDAGC(no);
#endif

            /*------------------------------------------------------------*/
            /* Open eye process.                                          */
            /*------------------------------------------------------------*/
    
            INIT_DFE;

            _OpenEyeLastFfe(no); /* PKD, adapt last FFE */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 4501);
            tm_stage = TM_ALB_OPEN_EYE2;
            break;

        case TM_ALB_OPEN_EYE2:
            TIMER_BREAK(t3);

            ADAPT_DFE;

            INIT_TIMER(t3, _T3, t3_low, t3_high, 6700);

            tm_stage = TM_ALB_OPEN_EYE3;
            break;

        case TM_ALB_OPEN_EYE3:
            TIMER_BREAK(t3);
            
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, detector_modes, enable_peak_detector, OFF); /* Disable peak detector */ 

            INIT_TIMER(t3, _T3, t3_low, t3_high, 55000);

            tm_stage = TM_ALB_OPEN_EYE4;
            break;

        case TM_ALB_OPEN_EYE4:
            TIMER_BREAK(t3);
            
            FREEZE_DFE_FFE;

            RESTART(bp_mode_ptr, meter); 
#ifdef INT_BUG
            _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << ALT_METER)); 
#endif

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_ALB_OPEN_EYE5);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;

        case TM_ALB_OPEN_EYE5:
            TIMER_BREAK(meter);

            if (_IsEyeOpen(no) == _FAIL) /* Open eye failed, continue to next phase */
                {
                temp = BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs);
                temp++; /* move to next phase */
                if (temp > 15)
                    {
                    /* If pass through all phases, set to IDLE */

                    /* Set Fail Bit */
                    RD_BYTE(no, STATUS, status_reg.status);
                    status_reg.bits.activation_interval = ON;
                    WR_BYTE(no, STATUS, status_reg.status);

                    BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, ON);

                    tm_stage = TEST_MODE_IDLE;
                    break;
                    }
#ifdef TDEBUG
                PREFIX;
                printf("Failed Eye Openend, trying next phase = %d.\n", (int)temp);
#endif

                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, temp); /* Init receive sampling phase */
                tm_stage = TM_ALB_ADAPT_EC;
                break;
                } /* END-ELSE continue to the next phase */

            BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, ON);

#ifdef TDEBUG
            PREFIX;
            printf("Eye Opened\n");
#endif
            /*------------------------------------------------------------*/
            /* Adapt equalizer.                                           */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN); /* DFE at high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN); /* FFE at high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_last_coeff, OFF); /* FFE adapt all mode */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 60000);

            tm_stage = TM_ALB_TX_4LEVEL;
            break;

        case TM_ALB_TX_4LEVEL:
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */

#ifdef TDEBUG
            _PrintfDAGC(no);
#endif

            /*------------------------------------------------------------*/
            /* Move to 4 level external data.                             */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, OFF); /* 4-level Slicing */

            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read Bitpump user setup */
            if (user_setup_low.bits.seq_source) /* Internal startup sequence source */ 
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES); /* Internal 4 level scrambled 1's */
            else /* External startup sequence source */
                if (user_setup_low.bits.tx_scr) /* Scramble transmitted symbols */
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA); /* External (scarmbled) data */
                else /* No scrambling */
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA); /* External (unscarmbled) data */

#ifdef TDEBUG
            PREFIX;
            printf("Loopback Complete...\n");
#endif

            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.normal_operation = ON;
            WR_BYTE(no, STATUS, status_reg.status);

            tm_stage = TEST_MODE_IDLE;
            break;


#ifdef ERLE

        /*------------------------------------------------------------*/
        /* ERLE Test Mode                                             */
        /*------------------------------------------------------------*/
        case TM_ERLE_INIT:
            /* Need to save 'testmode' setup - found in high setup
             * it is cleared in BtInit
             */
            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read user setup */

            _BtInitialize(no);

            /* Set Activation Interval */
            SET_WORD(bp_ptr, sut4_low, sut4_high, TM_ERLE_INTERVAL);
#ifdef INT_BUG
            _RestartVirtualTimer(no, _SUT4, TM_ERLE_INTERVAL);
#endif
            RESTART(bp_mode_ptr, sut4);


            WR_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Write user setup */

            RD_WORD(no, ERLE_SETUP, temp, erle_mode.erle);


            /*---------------------------------------*/
            /* Configure Transmitter                 */
            /*---------------------------------------*/
            if ( erle_mode.bits.transmit_level )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
                }

            if ( erle_mode.bits.transmit_state )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, ON); /* Transmit OFF */
                }

#ifdef INT_BUG
            BP_WRITE_REG(bp_ptr, mask_low_reg, 0xFF); /* Disable All Ints */
#else
            BP_WRITE_REG(bp_ptr, mask_low_reg, 0xBE); /* Just Enable T3 & SUT1 */
#endif

            /*---------------------------------------*/
            /* Configure Analog Gain Control         */
            /*---------------------------------------*/
            _SetAdcControlAgain(no, erle_mode.bits.again);
            SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , 0x00); /* Set DC offset register */

#ifdef TDEBUG
            PREFIX;
            printf("AAGC %ddB\n", (int) (erle_mode.bits.again * 3) );
#endif
 
            /*---------------------------------------*/
            /* Configure Meter Interval              */
            /*---------------------------------------*/

            _SetMeterTimer(no, NORMAL_METER);  /* sets up masks */
            INIT_TIMER(meter, _METER, meter_low, meter_high, 0xFFFF);

            BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON); /* Meter timer continous mode */

            RESTART(bp_mode_ptr, meter);


            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_ERLE_DC_CANCEL);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;

        case TM_ERLE_DC_CANCEL:
            TIMER_BREAK(meter);

            /*----------------------------------------------------*/
            /* Read DC meter value                                */
            /*----------------------------------------------------*/
            READ_METER_REG(dc_meter_low, dc_meter_high, value, 0);

            /*----------------------------------------------------------*/
            /* Set DC offset register to compensate for measured offset */
            /*----------------------------------------------------------*/
            SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , value); /* Set DC offset register */

            WR_ERLE_RESULTS(ERLE_DC_OFFSET_RESULT, value);

#ifdef TDEBUG
            PREFIX;
            printf("DC Offset = %d\n", (int)value);
#endif

            tm_stage = TM_ERLE_ADAPT_EC1;
            break;

        case TM_ERLE_ADAPT_EC1:
#ifdef TDEBUG
            PREFIX;
            printf("EC Highest, ");
#endif

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt EC */
            _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)500); /* Adapt Ec at highest gain */

            INIT_TIMER(sut1, _SUT1, sut1_low, sut1_high, 500);  /* 500,000 symbols */

            tm_stage = TM_ERLE_ADAPT_EC2;
            break;

        case TM_ERLE_ADAPT_EC2:
            TIMER_BREAK(sut1);

#ifdef TDEBUG
            printf("Higher, ");
#endif

            _AdaptEc(no, HIGHER_GAIN, (BP_U_16BIT)500); /* Adapt EC at higher gain */

            INIT_TIMER(sut1, _SUT1, sut1_low, sut1_high, 500); /* 500,000 symbols */

            tm_stage = TM_ERLE_ADAPT_EC3;
            break;

        case TM_ERLE_ADAPT_EC3:
            TIMER_BREAK(sut1);

#ifdef TDEBUG
            printf("High  ");
#endif
            _AdaptEc(no, HIGH_GAIN, (BP_U_16BIT)500); /* Adapt EC at high gain */

            INIT_TIMER(sut1, _SUT1, sut1_low, sut1_high, 500); /* 500,000 symbols */
            
            RD_WORD(no, ERLE_SETUP, temp, erle_mode.erle);
            if ( erle_mode.bits.nl_ec )
                {
                tm_stage = TM_ERLE_ADAPT_NLEC1;
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze LEC */
                tm_stage = TM_ERLE_BYPASS_NLEC;
                }
            break;

        case TM_ERLE_BYPASS_NLEC:
            TIMER_BREAK(sut1);

#ifdef TDEBUG
            PREFIX;
            printf("Bypass NLEC.\n");
#endif

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_ERLE_MEASURE);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;


        case TM_ERLE_ADAPT_NLEC1:
            TIMER_BREAK(sut1);

#ifdef TDEBUG
            printf(":  NLEC High, ");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* NL EC high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */

            INIT_TIMER(sut1, _SUT1, sut1_low, sut1_high, 1000);
            tm_stage = TM_ERLE_ADAPT_NLEC2;
            break;

        case TM_ERLE_ADAPT_NLEC2:
            TIMER_BREAK(sut1);

#ifdef TDEBUG
            printf("Normal\n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN); /* NL EC normal adaptation gain */
            INIT_TIMER(sut1, _SUT1, sut1_low, sut1_high, 1000);

            tm_stage = TM_ERLE_ADAPT_NLEC3;
            break;

        case TM_ERLE_ADAPT_NLEC3:
            TIMER_BREAK(sut1);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze EC */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NLEC */

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_ERLE_MEASURE);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;

        case TM_ERLE_MEASURE:
            TIMER_BREAK(meter);

#ifdef TDEBUG
            PREFIX;
            printf("Measure ... \n");
#endif

            /* Read SLM meter */
            READ_METER_REG(slm_low, slm_high, slm, 0);

            WR_ERLE_RESULTS(ERLE_SLM_RESULT, slm);

            /* Read FELM meter */
            READ_METER_REG(felm_low, felm_high, felm, 0);

            WR_ERLE_RESULTS(ERLE_FELM_RESULT, felm);

            BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, ALB_BYPASS_HYBRID);

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_AERLE_MEASURE);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;

        case TM_AERLE_MEASURE:
            TIMER_BREAK(meter);

            RD_ERLE_RESULTS(ERLE_SLM_RESULT, slm);

            /* Read SLM meter - 'value = SLM2' */
            READ_METER_REG(slm_low, slm_high, value, 0);

            WR_ERLE_RESULTS(ERLE_SLM2_RESULT, value);

            BP_WRITE_BIT(bp_mode_ptr, adc_control, lb ,  ALB_DISABLED);

#ifndef INT_BUG
            BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, ON); /* Mask meter timer interrupt */
#endif

            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.normal_operation = ON;
            WR_BYTE(no, STATUS, status_reg.status);

            tm_stage = TEST_MODE_IDLE;
            break;



        /*------------------------------------------------------------*/
        /* Measure AAGC                                               */
        /*------------------------------------------------------------*/
        case TM_AAGC_INIT:
            /* Need to save 'testmode' setup - found in high setup
             * it is cleared in BtInit
             */
            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read user setup */

            _BtInitialize(no);

            /* Set Activation Interval */
            SET_WORD(bp_ptr, sut4_low, sut4_high, TM_AAGC_INTERVAL);
#ifdef INT_BUG
            _RestartVirtualTimer(no, _SUT4, TM_AAGC_INTERVAL);
#endif
            RESTART(bp_mode_ptr, sut4);



            WR_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Write user setup */

            RD_WORD(no, ERLE_SETUP, temp, erle_mode.erle);

            /* Enable 2 or 4-Level Scrambled 1's */
            if ( erle_mode.bits.transmit_level )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
                }
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */

            _SetMeterTimer(no, NORMAL_METER);  /* sets up masks */
            INIT_TIMER(meter, _METER, meter_low, meter_high, 0xFFFF);

            BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON); /* Meter timer continous mode */

            RESTART(bp_mode_ptr, meter);

            _SetAdcControlAgain(no, AGAIN0DB);

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_AAGC_DC_CANCEL);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;


        case TM_AAGC_DC_CANCEL:
            TIMER_BREAK(meter);

            /*----------------------------------------------------*/
            /* Read DC meter value                                */
            /*----------------------------------------------------*/
            READ_METER_REG(dc_meter_low, dc_meter_high, value, 0);

            /*----------------------------------------------------------*/
            /* Set DC offset register to compensate for measured offset */
            /*----------------------------------------------------------*/
            SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , value); /* Set DC offset register */

            /* Set the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, tm_stage);
            WR_WORD(no, STAGE2, temp, TM_AAGC_MEASURE);

            tm_stage = TM_WAIT_METER_INTERVAL1;
            break;



        case TM_AAGC_MEASURE:
            TIMER_BREAK(meter);

            temp = _GetAdcControlAgain(no);

            /* Read SLM meter */
            READ_METER_REG(slm_low, slm_high, slm, 0);

            WR_ERLE_RESULTS(temp, slm);

            if ( temp < AGAIN15DB )
                {
                _SetAdcControlAgain(no, (BP_U_8BIT)(temp + 1));

                /* Set the next stage after wait meter intervals */
                RD_WORD(no, STAGE2, temp, tm_stage);
                WR_WORD(no, STAGE2, temp, TM_AAGC_DC_CANCEL);

                tm_stage = TM_WAIT_METER_INTERVAL1;
                break;
                }
            else
                {
#ifndef INT_BUG
                BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, ON); /* Mask meter timer interrupt */
#endif

                RD_BYTE(no, STATUS, status_reg.status);
                status_reg.bits.normal_operation = ON;
                WR_BYTE(no, STATUS, status_reg.status);

                tm_stage = TEST_MODE_IDLE;
                break;
                }

            break;

#endif /* ERLE */

        default:
            tm_stage = TEST_MODE_IDLE;
            break;

        } /* end switch tm_stage */

    WR_WORD(no, STAGE, su_stage, tm_stage);

} /* END _HandleTestMode() */

