/************************************************************/
/*    SUUTIL.C                                              */
/*    Bitpump startup utility routines.                     */
/*    (C) Copyright 1993 by Rockwell Corporation            */
/*                                                          */
/*    This program is copyrighted by Rockwell Corporation   */
/*                                                          */
/* Description:                                             */
/*    This file contains several startup utility routines   */
/*    that are common to HTU-C and HTU-R.                   */
/*                                                          */
/* User Modifiable Code:                                    */
/*    _SetActivationInterval()                              */
/*                                                          */
/* List of functions included in this module:               */
/*   _AdaptEc()                                             */
/*   _OpenEyeLastFfe()                                      */
/*   _IsEyeOpen()                                           */
/*   _IsFirstEyeOpen()                                      */
/*   _IsPhaseLocked()                                       */
/*   _CalculateOptimalPhase()                               */
/*   _SetMeterTimer()                                       */
/*   _SetFelmMask()                                         */
/*   _FfeReverse()                                          */
/*   _ShiftFfe()                                            */
/*   _CalculatePhaseQuality()                               */
/*   _ReOpenOptimalPhase()                                  */
/*   _InitDagc()                                            */
/*   _InitDfe()                                             */
/*   _InitFfe()                                             */
/*   _InitDetector()                                        */
/*   _AdaptDagcSelfMode()                                   */
/*   _AdaptDagcEqErrorMode()                                */
/*   _AdaptAllFfe()                                         */
/*   _AdaptDfe()                                            */
/*   _FreezeDfeFfe()                                        */
/*   _Aagc()                                                */
/*   _SetLosThreshold()                                     */
/*   _SymbolRateLut()                                       */
/*   _SetAdcControlAgain()                                  */
/*   _GetAdcControlAgain()                                  */
/*   _SetFfeScale()                                         */
/*   _ScaleDagcFfe()                                        */
/*   _ScaleFfe125()                                         */
/*   _CheckAagc()                                           */
/*   _PrintfLECCoefs()                                      */
/*   _PrintfDAGC()                                          */
/*   _PrintfFFECoefs()                                      */
/*   _PrintfDFECoefs()                                      */
/*   _PrintfEPCoefs()                                       */
/*                                                          */
/*                                                          */
/* Programmer:                                              */
/*     Dean Rasmussen             March - 1998              */
/*     Iris Shuker                21-Oct-1993               */
/*                                                          */
/* Revision History:                                        */
/*                                                          */
/************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define LOW_RATE_FFE_INIT      70   /* 580kbps */

#define FFE_TH                 1000
#define NLM_TH                 700
#define AVERAGE_TH             44
#define ABS_AVERAGE_TH         44
#define ZIP_ABS_AVERAGE_TH     72         
#define PHASE_LOCK_TRY_NO      4
#define LOG2_TRY_NO            2
#define LONG_LOOP_TH           600

#define FFE_LAST_COEFF_INIT_VALUE     8000
#define DAGC_GAIN_INIT_VALUE          0x40000000

#define NUM_CANCEL_FALSE_PHASES     3
#define NUM_CANCEL_NOISE_PHASES     5

#define MINIMUM_FELM_LOW_TH   100

/* Far End Signal Thresholds */
#define LONG_LOOP_BT8970                        1500
#define MEDIUM_LOOP_BT8970                      3000
#define SHORT_LOOP_BT8970                       7700

/*---------------------------------------------------------*/
/*  External Functions                                     */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Constants                                              */
/*---------------------------------------------------------*/

static BP_U_8BIT BP_CONSTANT phase_delta[4] = {2, 2, 2, 3};
static BP_U_8BIT BP_CONSTANT phase_quality_scale[4] = {7, 6, 3, 2};

BP_U_8BIT BP_CONSTANT dagc_th[4] = {12, 12, 6, 5};
extern BP_S_16BIT BP_CONSTANT  ffe_init[2][4][8];

/***********************************************************/
/*    _AdaptEc()                                           */
/*    Adapt Echo Canceller.                                */
/*                                                         */
/*    returns: none.                                       */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_8BIT gain                      */
/*                     BP_U_16BIT no_symbols               */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _AdaptEc(no, HIGHER_GAIN, 30000);             */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _AdaptEc (BP_U_8BIT no, BP_U_8BIT gain, BP_U_16BIT no_symbols)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    /*----------------------------------------*/
    /* Set adaptation gain, start Timer3.     */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, gain); /* EC at highest adaptation gain */

    INIT_TIMER(t3, _T3, t3_low, t3_high, no_symbols);

} /* END _AdaptEc() */


/***********************************************************/
/*    _OpenEyeLastFfe()                                    */
/*    Open eye process last FFE coefficient adaptation.    */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _OpenEyeLastFfe(no);                          */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _OpenEyeLastFfe (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    /*----------------------------------------*/
    /* Use PKD                                */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, ON); /* 2-level slicing */
    BP_WRITE_BIT(bp_mode_ptr, detector_modes, enable_peak_detector, ON); /* Use PKD */

    /*----------------------------------------*/
    /* Clear  FFE coefficients                */
    /*----------------------------------------*/

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, ON); /* Clear FFE coeffs */
    DELAY2SYMBOL;
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, OFF);

    /*----------------------------------------*/
    /* Initialize last FFE coefficient        */
    /*----------------------------------------*/
    WR_WORD(no, LAST_FFE, LOW(FFE_LAST_COEFF_INIT_VALUE), HIGH(FFE_LAST_COEFF_INIT_VALUE)); /* Set FFE last coefficient value */

    /*----------------------------------------*/
    /* Set adaptation modes                   */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, HIGH_GAIN); /* FFE at high adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_last_coeff, ON); /* FFE adapt last */

    /*----------------------------------------*/
    /* Start FFE adaptation                   */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

} /* END _OpenEyeLastFfe() */

/***********************************************************/
/*    _IsEyeOPen()                                         */
/*    Check if eye is open.                                */
/*                                                         */
/*    returns: _PASS/_FAIL                                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _IsEyeOpen(no);                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _IsEyeOpen (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT meter;
    BP_S_16BIT nlm_value, ffe_last_coeff_value;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/    
    INIT_BP_PTR;

    NORM(meter) /* Calculate meter normalization factor */

    /*----------------------------------------------------*/
    /* Read Noise Level Meter value                       */
    /*----------------------------------------------------*/
    READ_METER_REG(nlm_low, nlm_high, nlm_value, meter);

    /*----------------------------------------------------*/
    /* Read last FFE coefficient value                    */
    /*----------------------------------------------------*/
    _ReadAccessByteRAM(no, EQ_ACCESS_RAM, LAST_FFE);
    ffe_last_coeff_value = BYTE2WORD(_bp_vars[no].access_data_byte[1], _bp_vars[no].access_data_byte[0]);

    /*----------------------------------------------------*/
    /* Eye is considered to be open if the noise level is */
    /* lower than a threshold, and the last FFE coeff     */
    /* is higher than a threshold.                        */
    /*----------------------------------------------------*/

#ifdef TDEBUG
    printf("    NLM=%d, Last=%d\n", (int)nlm_value, (int)ffe_last_coeff_value);
#endif

    if ((nlm_value <= NLM_TH) && (ffe_last_coeff_value > FFE_TH)) /* Eye is open */
        return (_PASS);
    else /* Eye not open */
        return (_FAIL);

} /* END _IsEyeOpen() */


/***********************************************************/
/*    _IsFirstEyeOPen()                                    */
/*    Check if eye is open.                                */
/*                                                         */
/*    returns: _PASS/_FAIL                                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _IsFirstEyeOpen(no);                 */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

BP_U_8BIT _IsFirstEyeOpen (BP_U_8BIT no)
{

    DECLARE_PTR;

    BP_S_8BIT i;
    BP_S_16BIT ffe_coeff;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/    
    INIT_BP_PTR;

    /*----------------------------------------------------*/
    /* If any FFE coeff is high enough, return "_PASS",   */
    /* Otherwise - return "_Fail".                        */
    /*----------------------------------------------------*/

    for (i=0; i<8; i++) 
        {   
        /* Read FFE coefficient */
        _ReadAccessByteRAM(no, EQ_ACCESS_RAM, i);
        ffe_coeff = BYTE2WORD(_bp_vars[no].access_data_byte[1], _bp_vars[no].access_data_byte[0]);

        if (ffe_coeff > FFE_TH) /* Eye is open */
            return (_PASS);
        } /* End if */
    return (_FAIL);

} /* END _IsFirstEyeOpen() */

/***********************************************************/
/*    _IsPhaseLocked                                       */
/*    Check if VCXO phase is locked to incoming signal     */
/*                                                         */
/*    returns: _PASS/_FAIL                                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _IsPhaseLocked(no);                  */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
#ifdef HTUR
BP_U_8BIT _IsPhaseLocked (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    BP_S_8BIT temp;
    BP_S_16BIT value, average, abs_average;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    /* sometimes the VCXO goes to the limits. this indicates error */
    READ_METER_REG(vcxo_frequency_low, vcxo_frequency_high, value, 0);

#ifdef TDEBUG
    printf("VCXO = %d\n", (int)value);
#endif

    if ( abs(value) > MAX_PLL_VALUE )
        {
        return(_FAIL);          /* Phase is not locked */
        }

    average = abs_average = 0;

#ifdef TDEBUG
    PREFIX;
#endif

    for ( temp = 0; temp < PHASE_LOCK_TRY_NO; temp++)
        {
#ifdef INT_BUG
        _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << PHASE_LOCK_METER));
#endif

        WAIT_FOR_TIMER(meter);

        /*----------------------------------------------------*/
        /* Read phase detector meter value                    */
        /*----------------------------------------------------*/
        READ_METER_REG(pdm_low, pdm_high, value, 0);

        /*----------------------------------------------------*/
        /* Accumulate and normalize PDM and |PDM| values       */
        /*----------------------------------------------------*/
    
        average += value;
        abs_average += abs(value);

#ifdef TDEBUG
        printf("PDM(%d) = %d ; ", (int)temp, (int)value);
#endif

        } /* END-FOR */

#ifdef TDEBUG
        printf("\n");
#endif

    int_ptr -> bits.timer_source.meter = OFF; /* Clear meter timer interrupt flag */

    /*----------------------------------------------------*/
    /* Normalize accumulated results                       */
    /*----------------------------------------------------*/

    average >>= LOG2_TRY_NO; /* Divide by log 2 of try no */
    abs_average >>= LOG2_TRY_NO; /* Divide by log 2 of try no */
    temp = BP_READ_BIT(bp_mode_ptr, pll_modes, phase_detector_gain); /* PD Gain Value */
    switch (temp) /* Divide by phase detector gain */
        {
        case NORMAL_GAIN: /* Divide by 1/2 */
            average <<= 1;
            abs_average <<= 1;
            break;

        case HIGH_GAIN: /* Divide by 1 */
            break;

        case HIGHER_GAIN: /* Divide by 2 */
            average >>= 1;
            abs_average >>= 1;
            break;

        case HIGHEST_GAIN: /* Divide by 4 */
            average >>= 2;
            abs_average >>= 2;
            break;

        } /* END-SWITCH divide by phase detector gain */

#ifdef TDEBUG
    PREFIX;
    printf("Avg PDM = %d ; |Avg PDM| = %d\n ", (int)average, (int)abs_average);
#endif

    /*-----------------------------------------------------------------*/
    /* Phase is locked if PDM and |PDM| are below specified thresholds */
    /*-----------------------------------------------------------------*/
    if ( ZIP_START_MODE )
        {
        if (abs_average <= ZIP_ABS_AVERAGE_TH ) /* Phase is locked */
            return(_PASS);
        else /* Phase is not locked */
            return(_FAIL);
        }
    else
        {
        if ((average <= AVERAGE_TH) && (abs_average <= ABS_AVERAGE_TH)) /* Phase is locked */
            return(_PASS);
        else /* Phase is not locked */
            return(_FAIL);
        }

} /* END _IsPhaseLocked() */
#endif

/***********************************************************/
/*    _CalculateOptimalPhase()                             */
/*    Calculate optimal phase, based on eye opening        */
/*    results at 16 sampling phases.                       */
/*                                                         */
/*    returns: _PASS/_FAIL                                 */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*          status = _CalculateOptimalPhase(no);           */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _CalculateOptimalPhase1 (BP_U_8BIT no)
{
    BP_U_16BIT temp_value, min_value;
    BP_U_8BIT min_index;
    BP_S_8BIT index, j;


    /*
     * Write 3 (n) lowest phase quality values to 0, must mark them
     * as 0xFFFF first.
     */
#ifdef TDEBUG
    printf("Selected as BAD Phase Quality: ");
#endif

    for ( j = 0 ; j < NUM_CANCEL_FALSE_PHASES ; j++ )
        {
        min_value = 0xFFFF;
        min_index = 0;
        for (index = 0; index < 16; index++)
            {
            RD_PHASE_QUALITY(index, temp_value);
            if ( temp_value < min_value )
                {
                min_index = index;
                min_value = temp_value;
                }
            }  /* end for index */
    
        WR_PHASE_QUALITY(min_index, 0xFFFF);
#ifdef TDEBUG
        printf("%d(%d) ", (int)min_index, (int)min_value);
#endif
        }  /* end for j */

      return ;
} /* END _CalculateOptimalPhase1() */

void _CalculateOptimalPhase2 (BP_U_8BIT no)
{
    BP_U_16BIT temp_value;
    BP_S_8BIT index;

    /* Now set 3 lowest to 0 */
    for (index = 0; index < 16; index++)
        {
        RD_PHASE_QUALITY(index, temp_value);
        if ( temp_value == 0xFFFF )
            {
            WR_PHASE_QUALITY(index, 0);
            }
        }  /* end for index */


      return ;

} /* END _CalculateOptimalPhase2() */

void _CalculateOptimalPhase3 (BP_U_8BIT no)
{
    SU_FLAGS_TYPE su_flags;
    BP_U_16BIT temp_value;
    BP_U_8BIT scale;
    BP_S_8BIT index, j;

    RD_WORD(no, FCELL__SU_FLAGS, index, su_flags.reg);
    scale = su_flags.bits.ffe_init_scale;

    /*------------------------------------------------*/
    /* Remove Phases near 0's (possible false phases) */
    /*------------------------------------------------*/
    for (index = 0; index < 16; index++)
        {
        RD_PHASE_QUALITY(index, temp_value); /* Read phase quality */

        if (temp_value != 0)
            {
            for (j = -phase_delta[scale]; j <= phase_delta[scale]; j++)
                {
                RD_PHASE_QUALITY((BP_U_8BIT)((BP_S_8BIT)(16 + index + j) % 16), temp_value)
                if (temp_value == 0)
                    {
#ifdef TDEBUG
                    printf("%d(1) ", (int)index);
#endif
                    WR_PHASE_QUALITY(index, 1);
                    break;
                    } /* END-IF */              
                } /* END-FOR */
            } /* END-IF */
        } /* END-FOR */

#ifdef TDEBUG    
    printf("\n");
#endif

      return ;

} /* END _CalculateOptimalPhase3() */


void _CalculateOptimalPhase4 (BP_U_8BIT no)
{
    BP_U_16BIT temp_value, max_value;
    BP_U_8BIT max_index;
    BP_S_8BIT index, j;

    /*--------------------------------*/
    /* Remove High Noise (NLM) Phases */
    /*--------------------------------*/
#ifdef TDEBUG
    PREFIX;
    printf("Selected as High Noise Phases: ");
#endif

    for ( j = 0 ; j < NUM_CANCEL_NOISE_PHASES ; j++ )
        {
        max_value = 0;
        max_index = 0;
        for (index = 0; index < 16; index++)
            {
            RD_NOISE_QUALITY(index, temp_value);
            if ( temp_value > max_value )
                {
                max_index = index;
                max_value = temp_value;
                }
            }  /* end for index */
    
        WR_NOISE_QUALITY(max_index, 1);
        WR_PHASE_QUALITY(max_index, 1);
#ifdef TDEBUG
        printf("%d(%d) ", (int)max_index, (int)max_value);
#endif
        }  /* end for j */
      return ;

} /* END _CalculateOptimalPhase4() */


BP_U_8BIT _CalculateOptimalPhase (BP_U_8BIT no)
{
    BP_U_16BIT temp_value, max_value;
    BP_U_8BIT opt_phase;
    BP_S_8BIT index, j;
    BP_U_8BIT opt_value, value;

#ifdef TDEBUG
    printf("\nPhases = [");
#endif
    
    /*----------------------------------------------*/
    /* Find optimal phase, peak phase quality left  */
    /*----------------------------------------------*/
    max_value = 0;
    opt_phase = 0;
    for (index = 0; index < 16; index++)
        {
        RD_PHASE_QUALITY(index, temp_value); /* Read phase quality */

#ifdef TDEBUG
        printf("%d ", (int)temp_value);
#endif

        if (temp_value > max_value)
            {
            max_value =  temp_value;
            opt_phase = index;
            } /* END-IF */
        } /* END-FOR */

#ifdef TDEBUG
    printf("]\n");
#endif

    /*------------------------------------------*/
    /* If no good (peak) Phase Quality is found */
    /* use the center of island approach        */
    /*------------------------------------------*/
    if (max_value == 0x1)
        {
        value = opt_value = 0;

        for (j=0;  j<=1; j++) /* Check the phases success twice due to rewind phases */
            {
            for (index=15; index>=0; index--) /* Check sixteen available phases */
                {
                RD_PHASE_QUALITY(index, temp_value);
                if (temp_value == 0x1) /* Good phase */
                    {
                    value++;
                    }
                else /* Bad phase */
                    {
                    if (value > opt_value)
                        {
                        opt_value = value; /* Save current value */
                        opt_phase = index + 1;
                        } /* END-IF */
                    value = 0; /* Look for next flow of success phases */
                    } /* END-ELSE bad phase */
                } /* END-FOR check sixteen available phases */
            } /* END-FOR check phase success twice */
    
        /*-------------------*/
        /* Set optimal phase */
        /*-------------------*/
        opt_phase = opt_phase + (opt_value >> 1); /* Calculate optimal phase */
        if (opt_phase > 15)
            {
            opt_phase -= 16;
            }
        } /* END-IF  */

#ifdef TDEBUG
    PREFIX;
    printf("Optimal phase = %d\n", (int)opt_phase);
#endif

    return (opt_phase);

} /* END _CalculateOptimalPhase() */



/************************************************************/
/*    _SetMeterTimer                                        */
/*    Set Meter Timer value                                 */
/*                                                          */
/*    returns: void                                         */
/*                                                          */
/*    Input variables: BP_U_8BIT no, BP_U_8BIT value        */
/*                                                          */
/*    Output variables: None                                */
/*                                                          */
/*    example:                                              */
/*           _SetMeterTimer(no, value);                     */
/*                                                          */
/* Programmer:                                              */
/*     Iris Shuker                29-Oct-1993               */
/*                                                          */
/* revision history:                                        */
/*     No longer wait for the meter intervals to elapse     */
/*                                                          */
/************************************************************/

void _SetMeterTimer (BP_U_8BIT no, BP_U_8BIT value)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    BP_U_8BIT low, high;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON); /* Mask low FELM int. */
    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON); /* Mask high FELM int. */

    SET_WORD(bp_ptr, meter_low, meter_high, (DEFAULT_METER_VALUE << value));

    RD_WORD(no, FELM_THRESHOLD, low, high);

    SET_WORD(bp_ptr, far_end_high_alarm_th_low, far_end_high_alarm_th_high, (high << value) );
    SET_WORD(bp_ptr, far_end_low_alarm_th_low, far_end_low_alarm_th_high, (low << value) );

    RESTART(bp_mode_ptr, meter);
#ifndef INT_BUG
    BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, meter, OFF);
#endif

#ifdef INT_BUG
    _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << value));
#endif

    
} /* END _SetMeterTimer() */

/***********************************************************/
/*    _SetFelmMask()                                       */
/*                                                         */
/*    Description:  This function set the HIGH or LOW      */
/*                  FELM mask off depending on the LOS bit.*/
/*                                                         */
/*                  Both HIGH/LOW masks are ON before      */
/*                  entering this function.                */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             04-June-1997             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _SetFelmMask (BP_U_8BIT no)
{
    status_reg_type status_reg; 
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    RD_BYTE(no, STATUS, status_reg.status);

    if (status_reg.bits.los)
        {
        BP_WRITE_BIT(bp_mode_ptr, irq_source, high_felm, OFF);
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, OFF);
        } /* END-IF */
    else
        {
        BP_WRITE_BIT(bp_mode_ptr, irq_source, low_felm, OFF);
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, OFF);
        } /* END-ELSE */
}



/***********************************************************/
/*    _FfeReverse()                                        */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _FfeReverse (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_S_8BIT temp, temp1;   
    BP_S_16BIT temp_value, max_value;
    BP_U_8BIT index;
    BP_BIT_FIELD tmpIndication;
    status_reg_type status_reg;
    user_setup_low_type user_setup_low;


    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    /*---------------------*/
    /* Find FFE main tap   */
    /*---------------------*/
    max_value = 0;
    for (index = 4; index < 8; index++)
        {
        RD_WORD(no, index, temp1, temp);
        temp_value = (BP_S_16BIT) BYTE2WORD(temp, temp1);
        if (abs(temp_value) > abs(max_value))
            {
            max_value = temp_value;
            } /* END IF */
        } /* END FOR */

    /*---------------------*/
    /* FFE reverse         */
    /*---------------------*/
    if (max_value < 0) /* FFE main tap negative */
        {
#ifdef TDEBUG
        PREFIX;
        printf("Reverse FFE Coefs\n");
#endif
        for (index = 0; index < 8; index++)
            {
            RD_WORD(no, index, temp1, temp);
            temp_value = (BP_S_16BIT) BYTE2WORD(temp, temp1);
            temp_value = -temp_value;
            WR_WORD(no, index, LOW(temp_value), HIGH(temp_value));
            } /* END FOR */


        /*------------------------------------------------*/
        /* Need to Reverse Tip/Ring                       */
        /*------------------------------------------------*/
        RD_WORD(no, USER_SETUP, user_setup_low.setup, temp);
        if ( user_setup_low.bits.auto_tip_ring )
            {
            RD_BYTE(no, STATUS, status_reg.status);
            tmpIndication = !status_reg.bits.tip_ring_indication;

            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, negate_symbol, tmpIndication);  
            status_reg.bits.tip_ring_indication = tmpIndication;  
            WR_BYTE(no, STATUS, status_reg.status);

#ifdef TDEBUG
            PREFIX;
            printf("Tip/Ring toggled.\n");
#endif
            } /* end if auto tip/ring */
        } /* end if ffe < 0 */

} /* END _FfeReverse() */


#ifdef HTUC

/***********************************************************/
/*    _ShiftFfe()                                          */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _ShiftFfe (BP_U_8BIT no)
{
    BP_U_8BIT index, main_tap;
    BP_S_8BIT temp, temp1;
    BP_S_16BIT max_value, temp_value;
    DECLARE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;

    /*-------------------*/
    /* Find FFE main tap */
    /*-------------------*/
    max_value = 0;
    main_tap = 0;
    for (index = 0; index < 8; index++)
        {
        RD_WORD(no, index, temp1, temp);
        temp_value = (BP_S_16BIT) BYTE2WORD(temp, temp1);
        if (abs(temp_value) > abs(max_value))
            {
            max_value = temp_value;
            main_tap = index;
            } /* END IF */
        } /* END FOR */

    /*-----------*/
    /* Shift FFE */
    /*-----------*/
    if (main_tap == 7)
        {
        for (index=2; index<8; index++)
            {
            RD_WORD(no, index, temp, temp1);
            WR_WORD(no, (BP_U_8BIT)(index-2), temp, temp1);
            } /* End for */
        for (index=6; index<8; index++)
            {
            WR_WORD(no, index, 0, 0);
            } /* End for */
        } /* End if */
    
} /* End _ShiftFfe() */
#endif

/***********************************************************/
/*    _CalulatePhaseQuality()                              */
/*                                                         */
/*    returns: BP_S_16BIT temp_p (phase quality)           */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                16-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

BP_S_16BIT _CalculatePhaseQuality (BP_U_8BIT no)
{

    DECLARE_PTR;
    SU_FLAGS_TYPE su_flags;
    BP_U_8BIT temp_u;
    BP_S_8BIT i, scale;
    BP_S_8BIT temp, temp1;
    BP_S_16BIT temp_p,max_ffe;
    BP_S_32BIT dfe_p, ffe_p;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;

    RD_WORD(no, FCELL__SU_FLAGS, i, su_flags.reg);
    scale = su_flags.bits.ffe_init_scale;
   
    _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
    temp_u = (BP_U_8BIT) _bp_vars[no].access_data_byte[1]; /* High Byte Only */

    max_ffe = 0;
    for ( i = 4; i < 8; i++)
        {
        RD_WORD(no, i , temp1, temp);
        temp_p = (BP_S_16BIT) BYTE2WORD(temp, temp1);
        if ( abs(temp_p) > max_ffe)
            {
            max_ffe = abs(temp_p);
            }
        }

#ifdef TDEBUG
    printf("F = %d, D = %d ",max_ffe,((int)temp_u)<<4);
#endif

    if ( max_ffe < (BP_S_16BIT)10000 && ((BP_S_16BIT)temp_u)<<4 > 1800 )
        {
        return 0;
        }



    if ((abs(max_ffe) > 1500) && (temp_u > dagc_th[scale])) /* FFE OK */
        {
        /* ----------- */
        /* FFE Measure */
        /* ----------- */
        ffe_p=0;
        for (temp_u=0; temp_u<8; temp_u+=2)
            {
            RD_WORD(no, temp_u, temp1, temp);
            ffe_p += (BP_S_32BIT)( (BP_S_16BIT)BYTE2WORD(temp, temp1) >> 4);
            }
        for (temp_u=1; temp_u<8; temp_u+=2)
            {
            RD_WORD(no, temp_u, temp1, temp);
            ffe_p -= (BP_S_32BIT)( (BP_S_16BIT)BYTE2WORD(temp, temp1) >> 4);
            }
        if (ffe_p<0)
            {
            ffe_p = (-ffe_p);
            }

        /* ----------- */
        /* DFE Measure */
        /* ----------- */
        dfe_p=0;
        for (temp_u=0; temp_u<20; temp_u+=2)
            {
            _ReadAccessByteRAM(no, DFE_ACCESS_RAM, temp_u);
            temp = _bp_vars[no].access_data_byte[1];
            temp1 = _bp_vars[no].access_data_byte[0];

            dfe_p += (BP_S_32BIT)( (BP_S_16BIT)BYTE2WORD(temp, temp1) >> 4);
            }
        for (temp_u=1; temp_u<20; temp_u+=2)
            {
            _ReadAccessByteRAM(no, DFE_ACCESS_RAM, temp_u);
            temp = _bp_vars[no].access_data_byte[1];
            temp1 = _bp_vars[no].access_data_byte[0];

            dfe_p -= (BP_S_32BIT)( (BP_S_16BIT)BYTE2WORD(temp, temp1) >> 4);
            }
        dfe_p -= 1024;      /* consider cursor level */
        if (dfe_p<0)
            {
            dfe_p = (-dfe_p);
            }


        /* ------------- */
        /* Total Measure */
        /* ------------- */

        _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
        temp = _bp_vars[no].access_data_byte[1];  /* High Byte Only */

        dfe_p <<= 8;
        ffe_p *= temp;
        ffe_p >>= phase_quality_scale[scale];
        if (ffe_p == 0)
            {
            ffe_p = 1;
            }
        dfe_p /= ffe_p;
        temp_p =  (BP_S_16BIT) dfe_p;
        }
     else
        {
        temp_p = 0;
        }

     return (temp_p); /* Return phase */

} /* END _CalculatePhaseQuality */


#ifdef HTUR
/***********************************************************/
/*    _ReOpenOptimalPhase()                                */
/*    Uses same algorithm as optimal phase search          */
/*                                                         */
/*    returns: BP_U_8BIT _PASS/_FAIL                       */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables:                                    */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                20-June-1995             */
/*     Dean Rasmussen             01-Oct-1998              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _ReOpenOptimalPhase (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    /* ------------------------------ */
    /* Initialize DAGC, DFE, and FFE: */
    /* ------------------------------ */
    INIT_DAGC;
    INIT_DFE;
    INIT_FFE;

    /* ---------------------------------- */
    /* Adapt DAGC @ HIGH Gain, Use Slicer */
    /* ---------------------------------- */
    INIT_DETECTOR;
    ADAPT_DAGC_EQ_ERROR_MODE;

    INIT_TIMER(t3, _T3, t3_low, t3_high, 5000);
    WAIT_FOR_TIMER(t3);

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

    /* ---------- */
    /* Adapt FFE: */
    /* ---------- */

    ADAPT_ALL_FFE;
    INIT_TIMER(t3, _T3, t3_low, t3_high, 10000);
    WAIT_FOR_TIMER(t3);

    /* -------------- */
    /* Adapt FFE+DFE: */
    /* -------------- */

    ADAPT_DFE;
    INIT_TIMER(t3, _T3, t3_low, t3_high, 10000);
    WAIT_FOR_TIMER(t3);

    FREEZE_DFE_FFE;

#ifdef TDEBUG
    printf("ReOpen Eye\n");
#endif

    return (_PASS);

} /* End _ReOpenOptimalPhase() */

#endif /* HTUR */


/***********************************************************/
/*    _InitDagc()                                          */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitDagc (BP_U_8BIT no)
{

    DECLARE_PTR;
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers             */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
    SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, LOW_WORD(DAGC_GAIN_INIT_VALUE));
    BP_WRITE_REG(bp_ptr, eq_add_write, DAGC_LOW); /* Write DAGC coefficient low word */
    DELAY2SYMBOL;
    SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, HIGH_WORD(DAGC_GAIN_INIT_VALUE));
    BP_WRITE_REG(bp_ptr, eq_add_write, DAGC_HIGH); /* Write DAGC coefficient high word */
    DELAY2SYMBOL;

} /* END _InitDagc() */


/***********************************************************/
/*    _InitDfe()                                           */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitDfe (BP_U_8BIT no)
{

    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers             */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */
    SET_RESET(bp_mode_ptr, dfe_modes, zero_coefficients); /* Clear DFE coefficients */

} /* END _InitDfe() */


/***********************************************************/
/*    _InitFfe()                                           */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitFfe (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    SU_FLAGS_TYPE su_flags;
    BP_U_8BIT index;

    /*----------------------------------------*/
    /* Initiate bitpump pointers             */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    RD_WORD(no, FCELL__SU_FLAGS, index, su_flags.reg);

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */
    for ( index = 0 ; index < 8; index++ )
        {
        WR_WORD(no, index,
            LOW(ffe_init[su_flags.bits.ffe_data_scale][su_flags.bits.ffe_init_scale][index]),
            HIGH(ffe_init[su_flags.bits.ffe_data_scale][su_flags.bits.ffe_init_scale][index]));
        }

} /* END _InitFfe() */


/***********************************************************/
/*    _InitDetector()                                      */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitDetector (BP_U_8BIT no)
{

    DECLARE_MODE_PTR;
 
    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, ON);             /* 2-Level slicing */ 
    BP_WRITE_BIT(bp_mode_ptr, detector_modes, enable_peak_detector, OFF); /* Use Slicer */

} /* END _InitDetector() */


/***********************************************************/
/*    _AdaptDagcSelfMode()                                 */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptDagcSelfMode (BP_U_8BIT no)
{
    DECLARE_MODE_PTR;
 
    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;
 
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, OFF); /* DAGC at self adaptation mode */
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN); /* DAGC at high adaptation gain */

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON);   /* Adapt DAGC */

} /* END _AdaptDagcSelfMode() */


/***********************************************************/
/*    _AdaptDagcEqErrorMode()                              */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptDagcEqErrorMode (BP_U_8BIT no)
{
    DECLARE_MODE_PTR;
 
    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;
 
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON); /* DAGC at EQ_Error adaptation mode */ 
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, HIGH_GAIN);   /* DAGC at HIGH adaptation gain */ 
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON);   /* Adapt DAGC */

} /* END _AdaptDagcEqErrorMode() */

 
/***********************************************************/
/*    _AdaptAllFfe()                                       */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptAllFfe (BP_U_8BIT no)
{

    DECLARE_MODE_PTR;
 
    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;
  
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_last_coeff, OFF); /* FFE adapt all */ 
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, HIGH_GAIN); /* FFE HIGH adaptation gain */ 
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

} /* END _AdaptAllFfe() */ 


/***********************************************************/
/*    _AdaptDfe()                                          */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptDfe (BP_U_8BIT no)
{

    DECLARE_MODE_PTR;
 
    /*----------------------------------------*/
    /* initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, HIGH_GAIN); /* DFE HIGH adaptation gain */ 
    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */

} /* END _AdaptDfe() */


/***********************************************************/
/*    _FreezeDfeFfe()                                      */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _FreezeDfeFfe (BP_U_8BIT no)
{

    DECLARE_MODE_PTR;
 
    /*----------------------------------------*/
    /* initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;
 
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */ 
    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */

} /* END _FreezeDfeFfe() */

/***********************************************************/
/*    _Aagc()                                              */
/*                                                         */
/*    returns: none                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                     BP_U_8BIT gain;                     */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _Aagc (BP_U_8BIT no, BP_U_8BIT gain)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    BP_S_8BIT low, high;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON); /* Mask low FELM int. */
    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON); /* Mask high FELM int. */

    high = HIGH_FELM_ALARM_TH;
    low = LOW_FELM_ALARM_TH;
    switch (gain)
        {
        case AGAIN0DB:
            break;
        case AGAIN3DB:
            high = high + (high >> 1);
            low = low  + (low >> 1);
            break;
        case AGAIN6DB:
            high <<= 1;
            low <<= 1;
            break;
        case AGAIN9DB:
        case AGAIN12DB:
        case AGAIN15DB:
            high = (high << 1) + high;
            low = (low << 1) + low;
            break;
        } /* End switch */

    /* Use BP_WRITE_REG instead of SET_WORD because "high" and "low" are bytes and
     * not words. BP_WRITE_REG also saves ROM and eliminates compiler warnings.
     */
    BP_WRITE_REG(bp_ptr, far_end_high_alarm_th_low, high);
    BP_WRITE_REG(bp_ptr, far_end_high_alarm_th_high, 0x00);
    BP_WRITE_REG(bp_ptr, far_end_low_alarm_th_low, low);
    BP_WRITE_REG(bp_ptr, far_end_low_alarm_th_high, 0x00);

    WR_WORD(no, FELM_THRESHOLD, low, high);

    _SetAdcControlAgain(no, gain);
   
} /* END _Aagc() */



void _SetLosThreshold (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    BP_U_8BIT meter;
    BP_S_16BIT value;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    READ_METER_REG(felm_low, felm_high, value, 0);

    if ( value < MINIMUM_FELM_LOW_TH )
        {
        value = MINIMUM_FELM_LOW_TH;
        }

#ifdef TDEBUG
    printf("Non normlized FELM: %d\n", (int)value);
#endif  

    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON); /* Mask low FELM int. */
    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON); /* Mask high FELM int. */

    SET_WORD(bp_ptr, far_end_high_alarm_th_low, far_end_high_alarm_th_high, (value>>1));
    SET_WORD(bp_ptr, far_end_low_alarm_th_low, far_end_low_alarm_th_high, (value>>2));

    NORM(meter);
    meter = 6 - meter;

    value >>= meter;

    WR_WORD(no, FELM_THRESHOLD, (BP_U_8BIT)(value>>2), (BP_U_8BIT)(value>>1));

    RESTART(bp_mode_ptr, meter);

#ifdef INT_BUG
    _RestartVirtualTimer(no, _METER, DEFAULT_METER_VALUE<<DEFAULT_METER);
#endif

}  /* END _SetLosThreshold() */


/***********************************************************/
/*    _SetAdcControlAgain()                                */
/*                                                         */
/*    returns: none                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                     BP_U_8BIT gain;                     */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    This function sets the 'adc_control.again' bits      */
/*    to the specified 'gain' setting for the desired      */
/*    bitpump.                                             */
/*                                                         */
/*    This function is needed because the original Bt8952  */
/*    LICs and the new generation LICs have different      */
/*    AGAIN pinouts.                                       */
/*                                                         */
/*    Note: The default is to reference the new generation */
/*    again description.  The original Bt8952 is then a    */
/*    special case.                                        */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                11-Jan-1996              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _SetAdcControlAgain (BP_U_8BIT no, BP_U_8BIT gain)
{
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, adc_control, again, gain);

} /* End _SetAdcControlAgain() */


/***********************************************************/
/*    _GetAdcControlAgain()                                */
/*                                                         */
/*    returns: again setting based on new generation       */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    This function gets the 'adc_control.again' bits      */
/*    for the desired bitpump                              */
/*                                                         */
/*    This function is needed because the original Bt8952  */
/*    LICs and the new generation LICs have different      */
/*    AGAIN pinouts.                                       */
/*                                                         */
/*    Note: The default is to reference the new generation */
/*    again description.  The original Bt8952 is then a    */
/*    special case.                                        */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                11-Jan-1996              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _GetAdcControlAgain  (BP_U_8BIT no)
{
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_MODE_PTR;

    return ( BP_READ_BIT(bp_mode_ptr, adc_control, again) );
    
} /* End _GetAdcControlAgain() */


/***********************************************************/
/*    _SetFfeScale()                                       */
/*                                                         */
/*    returns: FFE Scale Init Value:  0 - 3                */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                16-Feb-1996              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _SetFfeScale (BP_U_8BIT no)
{
    DECLARE_PTR;
    SU_FLAGS_TYPE su_flags;
    BP_S_8BIT meter, temp;
    BP_S_16BIT value;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;

    RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

    NORM(meter);
    READ_METER_REG(felm_low, felm_high, value, meter);

    if (value < LONG_LOOP_BT8970)
        su_flags.bits.ffe_init_scale = 0;
    else if (value < MEDIUM_LOOP_BT8970)
        su_flags.bits.ffe_init_scale = 1;
    else if (value < SHORT_LOOP_BT8970)
        su_flags.bits.ffe_init_scale = 2;
    else
        su_flags.bits.ffe_init_scale = 3;

    if ( _bp_vars[no].symbol_rate < LOW_RATE_FFE_INIT )
        {
        su_flags.bits.ffe_data_scale = 1;
        }
    else
        {
        su_flags.bits.ffe_data_scale = 0;
        }

    WR_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

#ifdef TDEBUG
    PREFIX;
    printf("FELM = %d\n", (int)value);
    printf("FFE Init Scale=%d for %s Data Rates\n", (int) su_flags.bits.ffe_init_scale,
        su_flags.bits.ffe_data_scale==0?"High":"Low");
#endif


} /* End _SetFfeScale() */

      
#ifdef HTUC
/***********************************************************/
/*    _ScaleDagcFffe()                                     */
/*                                                         */
/*    returns: nothing                                     */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             18-Feb-1997              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _ScaleDagcFfe (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT index, temp, temp1;
    BP_S_16BIT temp_p;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/

    INIT_BP_PTR;

    /* Scale DAGC & FFE Coefficient values by ~1.25 */ 

    RD_WORD(no, DAGC_HIGH, temp, temp1);
    temp_p = BYTE2WORD(temp1, temp);
    temp_p = temp_p - (temp_p>>3);  /* divide by x.875 */ 
    WR_WORD(no, DAGC_HIGH, LOW(temp_p), HIGH(temp_p));

    for (index=0; index<8; index++)
        {
        RD_WORD(no, index, temp, temp1);
        temp_p = BYTE2WORD(temp1, temp);
        temp_p = temp_p + (temp_p>>2); /* x 1.25 */
        WR_WORD(no, index, LOW(temp_p), HIGH(temp_p));
        }

    return ;
}
#endif


#ifdef HTUR
/***********************************************************/
/*    _ScaleFfe125()                                       */
/*                                                         */
/*    returns: nothing                                     */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             18-Feb-1997              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _ScaleFfe125 (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT index, temp, temp1;
    BP_S_16BIT temp_p;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;

    /* Scale FFE Coefficient values by 1.25 */ 

    for (index=0; index<8; index++)
        {
        RD_WORD(no, index, temp, temp1);
        temp_p = BYTE2WORD(temp1, temp);
        temp_p += (temp_p >> 2);
        WR_WORD(no, index, LOW(temp_p), HIGH(temp_p));
        }

    return ;
}
#endif


#ifdef HTUC
/***********************************************************/
/*    _CheckAagc()                                         */
/*                                                         */
/*    returns: 1 - AAGC incremented                        */
/*             0 - AAGC not incremented                    */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-June-1995             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _CheckAagc (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT gain;
    BP_U_8BIT meter;
    BP_S_16BIT value;

    /*----------------------------------------*/
    /* Initiate bitpump pointers              */
    /*----------------------------------------*/
    INIT_BP_PTR;

    gain = _GetAdcControlAgain(no);
#ifdef TDEBUG
    printf("current gain = %d dB\n", (int)(gain * 3));
#endif

    NORM(meter); /* Calculate meter normalization factor */
    READ_METER_REG(felm_low, felm_high, value, meter);

#ifdef TDEBUG
    PREFIX;
    printf("FELM = %d\n", (int)value);
#endif

    if ( (value < LONG_LOOP_TH) && (gain < MAX_AGAIN) )
        {
        gain++;
        AAGC(gain);

#ifdef TDEBUG
        printf("gain incremented to %d dB\n", (int)(gain * 3));

        NORM(meter); /* Calculate meter normalization factor */
        READ_METER_REG(slm_low, slm_high, value, meter);

        PREFIX;
        printf("SLM = %d\n", (int)value);
#endif
        return 1;
        } /* End if */ 
    else
        {
        return 0;
        }
} /* End _CheckAagc() */
#endif



#ifdef TDEBUG

/*
 * FUNCTION:   _PrintfDAGC
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the DAGC High.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfDAGC (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT temp;

    INIT_BP_PTR;

    _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
    temp = _bp_vars[no].access_data_byte[1];  /* High Byte only */

    PREFIX;
    printf("DAGC = %d\n", ((int)temp<<4));

    return ;
}

/*
 * FUNCTION:   _PrintfFFECoefs
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the FFE Coefs.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfFFECoefs (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT j, temp, temp1;

    INIT_BP_PTR;

    PREFIX;
    printf("FFE = [\n");
    for (j=0; j<8; j++) 
        {       
        RD_WORD(no, j, temp1, temp);

        printf("%d ;", (int) BYTE2WORD(temp, temp1));
        }

    printf("]\n");

    return ;
}


/*
 * FUNCTION:   _PrintfDFECoefs
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the DFE Coefs.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfDFECoefs (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT j, temp, temp1;

    INIT_BP_PTR;

    PREFIX;
    printf("DFE =[\n ");
    for (j=0; j< 8; j++)      /* was 116 */
        {
        BP_WRITE_REG(bp_ptr, dfe_tap_select_read, j);
        DELAY2SYMBOL;
        temp1 = BP_READ_REG(bp_ptr, access_data_byte0);
        temp = BP_READ_REG(bp_ptr, access_data_byte1);

        printf("%d ;", (int) BYTE2WORD(temp, temp1));
        }
    printf("];\n");

    return ;
}

#ifdef DEBUG_MENU

/*
 * FUNCTION:   _PrintfLECCoefs
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the LEC Coefs.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfLECCoefs (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT j, temp1, temp;

    INIT_BP_PTR;

    PREFIX;
    printf("LEC = [\n ");
    for (j=0; j<119; j++)
        {
        BP_WRITE_REG(bp_ptr, linear_ec_tap_select_read, j);
        DELAY2SYMBOL;
        temp1 = BP_READ_REG(bp_ptr, access_data_byte2);
        temp = BP_READ_REG(bp_ptr, access_data_byte3);

        printf("%d ;", (int) BYTE2WORD(temp, temp1));
        }
    printf("\n]; LEC End\n");

    return ;
}


/*
 * FUNCTION:   _PrintfEPCoefs
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the EP Coefs.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfEPCoefs (BP_U_8BIT no)
{
    DECLARE_PTR;
    BP_U_8BIT j, temp, temp1;

    INIT_BP_PTR;

    PREFIX;
    printf("EP =[\n ");
    for (j=16; j<=20; j++) 
        {       
        RD_WORD(no, j, temp1, temp);

        printf("%d ;", (int) BYTE2WORD(temp, temp1));
        }
    printf("];\n");

    return ;
}
#endif /* DEBUG_MENU */

#endif
