/*************************************************************/
/*    SUR.C                                                  */
/*    Startup control process for HTU-R terminal             */
/*    (C) Copyright 1993 by Rockwell Corporation             */
/*                                                           */
/*    This program is copyrighted by Rockwell Corporation    */
/*                                                           */
/* Description:                                              */
/*    Controlling HTU_R startup process, normal              */
/*    operation, idle mode, test modes etc.                  */
/*                                                           */
/* Notes:                                                    */
/*                                                           */
/* User Modifiable Code:                                     */
/*    None                                                   */
/*                                                           */
/* List of functions included in this module:                */
/*  _HturControlProcess()                                    */
/*                                                           */
/* Programmer:                                               */
/*     Dean Rasmussen             March - 1998               */
/*     Iris Shuker                26-Sept-1993               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define FREQUENCY_CELL_TRY_NO   4
#define FREQ_CELLS              9
#define VCXO_ACQ_INIT_VALUE     -10000    /* frequency acquisition */
#define ZIP_VCXO_ACQ_INIT_VALUE -2000

#define SUR_WATCH_DOG_INTERVAL   0xFFFF

#ifdef ZIP_START
/* 1500 * 1024 symbols - which is 2.6sec at 1168kbps */
#define ZIP_START_FREQ_INTERVAL    1500

/* 12000 * 1024 symbols - which is 21.0sec at 1168kbps */
#define ZIP_START_2LEVEL_INTERVAL    12000

#endif

/* 1000 * 1024 symbols - which is 1.75sec at 1168kbps */
#define ZIP_START_WAIT_4LEVEL_INTERVAL    1000

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Global Constants                                       */
/*---------------------------------------------------------*/

extern BP_U_8BIT BP_CONSTANT dagc_th[4];

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  External Functions                                     */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Constants                                              */
/*---------------------------------------------------------*/
#ifdef HTUR
static BP_S_16BIT BP_CONSTANT vcxo_freq[FREQ_CELLS] =
    {0,0,-21000,21000,14000,-14000,-7000,7000,-25000};
#endif

/***********************************************************/
/*    _HturControlProcess()                                */
/*    HTU-R terminal control process.                      */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _HturControlProcess(no);                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                14-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void _HturControlProcess (BP_U_8BIT no)
{
#ifdef HTUR

    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    SU_FLAGS_TYPE su_flags;
    user_setup_low_type user_setup_low;   /* User setup variable */
    status_reg_type status_reg;           /* Status variable */
    BP_U_8BIT i, stage, tm_stage;         /* Local variable */
    BP_U_8BIT prev_aagc;
    BP_S_8BIT temp, meter;                /* Local temporary buffers */
    BP_S_16BIT temp_s16;                  /* Local temporary buffer */
    BP_U_16BIT temp_u16;                  /* Local temporary buffer */

    /*------------------------------------------------------------*/
    /* Initiate bitpump pointers                                  */
    /*------------------------------------------------------------*/    
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;


    /*------------------------------------------------------------*/
    /* Read stage number from Bitpump, move to this stage.        */
    /*------------------------------------------------------------*/
    RD_WORD(no, STAGE, stage, tm_stage); /* Read Bitpump operational stage */
    switch (stage) /* Go to stage */
        {

        case ACTIVATE_SYSTEM: /* Activate startup process */

            /*------------------------------------------------------------*/
            /* Initialization.                                            */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Activate HTU-R \n");
#endif
            
            _BtInitialize(no); /* Initialize bit-pump status for startup */
            _SetMeterTimer(no, NORMAL_METER);
            /*Changed ALT_METER no NORMAL_METER to get a more accurate DC Offset*/
            
            if ( _bp_vars[no].symbol_rate <= 32 )
                {
                _SetAdcControlAgain(no, AGAIN12DB);
                }

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, DC_CANCELLATION);
            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case DC_CANCELLATION:
            TIMER_BREAK(meter);

            _DcCancel(no);

            /*
             *  LOS (WAIT_FOR_SIGNAL)meter interval requires the ALT_METER
             *  to be used.
             */
            _SetMeterTimer(no, ALT_METER);

            _SetFelmMask(no);

            /*
             * start Watch Dog Timer, 10-15 times longer than typical
             * startup times
             */
            INIT_TIMER(sut4, _SUT4, sut4_low, sut4_high, SUR_WATCH_DOG_INTERVAL);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, WAIT_FOR_SIGNAL);
            stage = SU_WAIT_METER_INTERVAL1;

#ifdef TDEBUG
            PREFIX;
            printf("Wait for signal...\n");
#endif
            break;


            /*------------------------------------------------------------*/
            /* If far-end signal power detected, start measuring startup  */
            /* time intervals. Otherwise - continue waiting for far-end   */
            /* signal.                                                    */
            /*------------------------------------------------------------*/

        case WAIT_FOR_SIGNAL: /* Wait for far end signal */
            RD_BYTE(no, STATUS, status_reg.status);

            if (status_reg.bits.los) /* Wait for far end signal */
                {
                break;
                }

            /*------------------------------------------------------------*/
            /* Far-end signal detected, Initialize and run startup timers */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("Signal detected\n");
#endif

            _SetMeterTimer(no, DEFAULT_METER);


            if ( !ZIP_START_MODE )
                {
                _SetActivationInterval(no);
                }
#ifdef ZIP_START
            else
                {
                SET_WORD(bp_ptr, sut4_low, sut4_high, ZIP_START_FREQ_INTERVAL);
#ifdef INT_BUG
                _RestartVirtualTimer(no, _SUT4, ZIP_START_FREQ_INTERVAL);
#endif
                RESTART(bp_mode_ptr, sut4);
                }
#endif


            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, OPEN_EYE1);
            stage = SU_WAIT_METER_INTERVAL1;

            break;


        /*------------------------------------------------------------*/
        /* Open eye process                                           */
        /*------------------------------------------------------------*/

        case OPEN_EYE1: /* Initialize open eye process */

            TIMER_BREAK(meter);

            NORM(meter);
            READ_METER_REG(felm_low, felm_high, temp_s16, meter);

#ifdef TDEBUG
            PREFIX;
            printf("felm = %d\n", (int)temp_s16);
#endif
            /*
             * When HTU-C first powers ON, it could cause a glitch
             */
            if ( temp_s16 < 100 )
                {
#ifdef TDEBUG
                PREFIX;
                printf("Detected Glitch - return to ACTIVATE_STATE\n");
#endif

                stage = ACTIVATE_SYSTEM;
                break;
                }

#ifdef TDEBUG
            PREFIX;
            printf("Open eye\n");
            PREFIX;
            printf("Freq cell 0\n");
#endif
            
            if ( !ZIP_START_MODE )
                {
                /* calculate scale */
                AAGC(AGAIN9DB);    /* Need for Set FFE Scale */
                _SetFfeScale(no);

                /* Need to extract FFE Init Scale info and set FCELL = 0 */
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                WR_WORD(no, FCELL__SU_FLAGS, 0, su_flags.reg);
                }
#ifdef ZIP_START
            else
                {
                /* Need to extract FFE Init Scale info and set FCELL = 0 */
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                su_flags.bits.ffe_init_scale = zip_start_global_var[no].ffe_init_scale;
                su_flags.bits.ffe_data_scale = zip_start_global_var[no].ffe_data_scale;
                WR_WORD(no, FCELL__SU_FLAGS, 0, su_flags.reg);
                }
#endif


            WR_BYTE(no, OPEN_ATTEMPTS, 0); /* Init open attempts counter */

            /*
             * Adjust AAGC based on FFE Init Scale (FELM)
             * For scale=1, already at 9dB
             */
            if ( su_flags.bits.ffe_init_scale == 0 )
                {
                AAGC(AGAIN12DB);
                }
            else if ( su_flags.bits.ffe_init_scale == 2 )
                {
                AAGC(AGAIN6DB);
                }
            else if ( su_flags.bits.ffe_init_scale == 3 )
                {
                AAGC(AGAIN3DB);
                }


            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, OPEN_EYE2);

            stage = SU_WAIT_METER_INTERVAL1;
            break;


        case OPEN_EYE2: /* set frequency cell, Init DFE+FFE, Adapt DAGC */

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Freeze PLL */
            RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

#ifdef ZIP_START
            if ( ZIP_START_MODE )
                {
                SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, (zip_start_global_var[no].vcxo_frequency + vcxo_freq[(BP_U_8BIT)temp] + ZIP_VCXO_ACQ_INIT_VALUE) );

                _ZipStart_HturOpenEye(no);
                stage = ACQUIRE_FREQ1;
                break;
                }

#endif /* ZIP_START */

            /* ------------------- */
            /* Set Frequency Cell: */
            /* ------------------- */
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0); /* Init receive sampling phase */

            SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, vcxo_freq[(BP_U_8BIT)temp]);
        
            /* ------------------------ */
            /* Initialize DAGC+DFE+FFE: */
            /* ------------------------ */
        
            INIT_DAGC;
            INIT_DFE;
            INIT_FFE;

            /* ---------------------------- */
            /* Adapt DAGC at EQ_Error mode: */
            /* ---------------------------- */

            INIT_DETECTOR;
            ADAPT_DAGC_EQ_ERROR_MODE;

            INIT_TIMER(t3, _T3, t3_low, t3_high, 17700);

            stage = OPEN_EYE3;
            break;

        case OPEN_EYE3: /* Adapt FFE */
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

            /* ---------- */
            /* Adapt FFE: */
            /* ---------- */

            ADAPT_ALL_FFE;

            INIT_TIMER(t3, _T3, t3_low, t3_high, 8700);
            
            stage = OPEN_EYE4;
            break;


        case OPEN_EYE4: /* Adapt FFE, DFE */
            TIMER_BREAK(t3);
                       
            /* -------------- */
            /* Adapt FFE+DFE: */
            /* -------------- */

            ADAPT_DFE;

            INIT_TIMER(t3, _T3, t3_low, t3_high, 6700);

            stage = OPEN_EYE5;
            break;

        case OPEN_EYE5: /* Freeze FFE, DFE */
            TIMER_BREAK(t3);

            FREEZE_DFE_FFE;

            stage = OPEN_EYE6;
            break;

        case OPEN_EYE6: /* Verify eye opening with Slicer */
            if (_IsFirstEyeOpen(no) == _PASS) /* Open eye succeeded */
                {
                stage = ACQUIRE_FREQ1; /* Move to frequency acquisition */

#ifdef TDEBUG
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                printf("Freq Cell: %d", (int)temp);
                RD_BYTE(no, OPEN_ATTEMPTS, temp);
                printf(", Attempts: %d\n", (int)temp);
#endif 
                }
            else /* Open eye failed */
                {
                stage = OPEN_EYE7; /* Try again at this frequency cell */
                }
            break;


        case OPEN_EYE7: /* Frequency cells/open attempts loops control */
            RD_BYTE(no, OPEN_ATTEMPTS, temp);
            temp++; /* Increment open attempts counter */
            WR_BYTE(no, OPEN_ATTEMPTS, temp); /* Update open attempts counter */
            if (temp >= FREQUENCY_CELL_TRY_NO) /* Ran out of attempts, move to next frequency cell */
                {
                WR_BYTE(no, OPEN_ATTEMPTS, 0); /* Init open attempts counter */
                
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                temp++; /* Increment frequency cell counter */
                WR_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

                if (temp >= FREQ_CELLS) /* Checked all frequency cells, SU failed */
                    {
#ifdef TDEBUG
                    PREFIX;
                    printf("NO Freq cell, SU failed, Retry SU...\n");
#endif
                    /* Set the next stage after wait meter intervals */
                    WR_WORD(no, STAGE2, TEMP_ENV_IDLE, OPEN_EYE1);
                    stage = SU_WAIT_METER_INTERVAL1;
                    break;
                    } /* END-IF all frequencies were checked */
#ifdef TDEBUG
                PREFIX;
                printf("Freq cell %d\n", (int) temp);
#endif
                } /* END-IF Ran out of attempts */

            stage = OPEN_EYE2; /* Move to the first stage */
            break;

        /*------------------------------------------------------------*/
        /* Frequency Acquisition                                      */
        /*------------------------------------------------------------*/

        case ACQUIRE_FREQ1: /* Frequency acquisition: highest gain */

#ifdef TDEBUG
            PREFIX;
            printf("Freq Acqu\n");
#endif


            /*
             * if this is a zip startup, the frequency is already
             * locked on the saved frequency, so don't need to set freq.
             */
            if ( !ZIP_START_MODE )
                {
                BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON);
                SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_ACQ_INIT_VALUE);
                }

            /*------------------------------------------------------------*/
            /* Set PLL parameters, activate PLL (highest gain)            */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, phase_detector_gain, HIGHER_GAIN); /* Phase detector at high gain */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHEST_GAIN); /* Wide PLL */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Adapt PLL */

            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, ZIP_START_MODE ? 120 : 536);

            stage = ACQUIRE_FREQ2;
            break;

        case ACQUIRE_FREQ2: /* Frequency acquisition: higher gain */
            TIMER_BREAK(sut2);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHER_GAIN); /* Mid-width PLL */

            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, ZIP_START_MODE ? 30 : 115);

            stage = ACQUIRE_FREQ3;
            break;

        case ACQUIRE_FREQ3: /* Acquire frequency stage */
            TIMER_BREAK(sut2);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGH_GAIN); /* Narrow PLL */

            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, ZIP_START_MODE ? 20 : 90);

            stage = ACQUIRE_FREQ4;
            break;

        case ACQUIRE_FREQ4: /* Acquire frequency stage */
            TIMER_BREAK(sut2);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, phase_detector_gain, NORMAL_GAIN); /* Phase detector at normal gain */

            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, ZIP_START_MODE ? 30 : 100);

            stage = ACQUIRE_FREQ5;
            break;

        case ACQUIRE_FREQ5: /* Acquire frequency stage */
            TIMER_BREAK(sut2);

            _SetMeterTimer(no, PHASE_LOCK_METER);
            
            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, PHASE_LOCKED);
            stage = SU_WAIT_METER_INTERVAL1;
            break;


        /*------------------------------------------------------------*/
        /* Verify that phase is locked                                */
        /*------------------------------------------------------------*/
        case PHASE_LOCKED: /* Checking phase lock */

#ifdef TDEBUG
            PREFIX;
            printf("Verify phase locking \n");
#endif
            
            if (_IsPhaseLocked(no) == _PASS) /* Phase is locked */
                {
                stage = OPTIMAL_PHASE;
                }
            else /* Phase is not locked */
                {

#ifdef TDEBUG
                PREFIX;
                printf("Phase not locked, try again\n");
#endif

                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                temp++; /* Increment frequency cell counter */
                if (temp >= FREQ_CELLS)
                    {
                    temp = 0;
                    }
                WR_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

                _SetMeterTimer(no, DEFAULT_METER);

                /* Set the next stage after wait meter intervals */
                WR_WORD(no, STAGE2, TEMP_ENV_IDLE, OPEN_EYE2);

                stage = SU_WAIT_METER_INTERVAL1;
                } /* END-ELSE phase is not locked */
            break;

        case OPTIMAL_PHASE: /* Search for optimal sampling phase */

#ifdef ZIP_START
            if ( ZIP_START_MODE )
                {
                _ZipStart_HturOptimalPhase1(no);
                stage = WS_OPTIMAL_PHASE;
                break;
                }
#endif /* ZIP_START */

#ifdef TDEBUG
            PREFIX;
            printf("Search opt phase\n");
#endif

            _DisableUserInterrupts(); /* Disable all user interrupts */

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Freeze PLL */

            /* -------------------------------------- */
            /* Begin - Search Optimal Phase Algorithm */
            /* -------------------------------------- */

            temp = 0; /* Count no. of phases opened */

            _SetMeterTimer(no, PHASE_QUALITY_METER);
            BP_WRITE_BIT(bp_mode_ptr, timer_source, meter, OFF);

            for ( i = 0 ; i < 16 ; i++) /* Sampling phase loop */
                {
                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, i); /* set receive sampling phase */

                /* ------------------------------ */
                /* Initialize DAGC, DFE, and FFE: */
                /* ------------------------------ */
                INIT_DAGC;
                INIT_DFE;
                INIT_FFE;

                /* ---------------------------------- */
                /* Adapt DAGC @ HIGH Gain, Use Slicer */
                /* ---------------------------------- */
                INIT_DETECTOR;
                ADAPT_DAGC_EQ_ERROR_MODE;

                INIT_TIMER(t3, _T3, t3_low, t3_high, 5000 /* 15000 */ );   /* 5000 */
                WAIT_FOR_TIMER(t3);

                BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

                /* ---------- */
                /* Adapt FFE: */
                /* ---------- */

                ADAPT_ALL_FFE;
                INIT_TIMER(t3, _T3, t3_low, t3_high, 10000 /* 20000 */);  /* 6800 */
                WAIT_FOR_TIMER(t3);

                /* -------------- */
                /* Adapt FFE+DFE: */
                /* -------------- */

                ADAPT_DFE;
                INIT_TIMER(t3, _T3, t3_low, t3_high, 10000 /* 20000 */); /* 6700 */
                WAIT_FOR_TIMER(t3);

                FREEZE_DFE_FFE;

#ifdef TDEBUG
                printf("(%d) ", (int)i); 
#endif

                /* -------------------------------- */
                /* Calculate phase quality measure: */
                /* -------------------------------- */
                temp_s16 = _CalculatePhaseQuality(no);

#ifdef TDEBUG
                printf("P=%d ", (int)temp_s16); 
#endif

                WR_PHASE_QUALITY(i, temp_s16);
                if (temp_s16 != 0)
                    {
                    temp++; /* Increment phase counter */
                    } /* END-IF */

                /* ---------------------------------------*/
                /* Read NLM (SNR):                        */
                /* Used during the Optimal Phase Search.  */
                /* ---------------------------------------*/

                /*
                 * Need to wait 2 meter intervals to ensure proper reading
                 */
                RESTART(bp_mode_ptr, meter);

#ifdef INT_BUG
                _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << PHASE_QUALITY_METER));
#endif
                WAIT_FOR_TIMER(meter);

                RESTART(bp_mode_ptr, meter);

#ifdef INT_BUG
                _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << PHASE_QUALITY_METER));
#endif
                WAIT_FOR_TIMER(meter);

                NORM(meter);
                READ_METER_REG(nlm_low, nlm_high, temp_s16, meter);
                WR_NOISE_QUALITY(i, temp_s16);

#ifdef TDEBUG
                printf("NLM=%d\n", (int)temp_s16);
#endif
                        
                }   /* for Sampling Phase */

            /*
             * Don't need to wait since ReOpenOptimalPhase() has a
             * large amount of symbol delays adapting equalizer
             */
            _SetMeterTimer(no, DEFAULT_METER);

            if ( temp < NO_OF_OPENED_PHASES ) /* Attenuator recognition */
                {
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                if ( su_flags.bits.ffe_init_scale < 3 )
                    {
                    su_flags.bits.ffe_init_scale++;
                    WR_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
                    BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Reactivate PLL */
                    _EnableUserInterrupts(); /* Enable all user interrupts */

#ifdef TDEBUG
                    PREFIX;
                    printf("FFE Scale changed to %d\n", (int)su_flags.bits.ffe_init_scale);

#endif
                    /*
                     * don't change stage number, re-do OPTIMAL_PHASE
                     * with new FFE Init Scale.
                     */
                    break;
                    } /* END-IF */

                } /* END-IF Attenuator recognition */

            /* -------------------------- */
            /* Set Optimal Sampling Phase */
            /* -------------------------- */

            _CalculateOptimalPhase1(no);
            _CalculateOptimalPhase2(no);
            _CalculateOptimalPhase3(no);
            _CalculateOptimalPhase4(no);

            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, _CalculateOptimalPhase(no) );

            /* ----------------------- */
            /* Begin Re-Open Process   */
            /* ----------------------- */
            _ReOpenOptimalPhase(no);

            /* ---------------------------------- */
            /* Adapt DFE @ HIGH Gain, Use Slicer: */
            /* ---------------------------------- */

            ADAPT_DFE;
            INIT_TIMER(t3, _T3, t3_low, t3_high, 20000);
            WAIT_FOR_TIMER(t3);

            /* ---------------------------------------- */
            /* Adapt DFE + FFE @ HIGH Gain, Use Slicer: */
            /* ---------------------------------------- */

            ADAPT_ALL_FFE;
            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, 90);
            WAIT_FOR_TIMER(sut2);

            /* --------------------------- */
            /* Adapt FFE+DFE @ NORMAL Gain */
            /* --------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);          /* FFE NORMAL adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN);          /* DFE NORMAL adaptation gain */


            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, 90);
            WAIT_FOR_TIMER(sut2);

            FREEZE_DFE_FFE;

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHER_GAIN); /* Select narrow width PLL filter */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Reactivate PLL */

            _EnableUserInterrupts(); /* Enable all user interrupts */
            stage = TRANSMIT_2LEVEL;
            break;

#ifdef ZIP_START
        case WS_OPTIMAL_PHASE:
            TIMER_BREAK(sut2);
            _ZipStart_HturOptimalPhase2(no);
            stage = TRANSMIT_2LEVEL;
            break;
#endif /* ZIP_START */

        case TRANSMIT_2LEVEL: /* Transmit RS0 startup sequence */

            /* --------------------------------------------- */
            /* Set DAGC Target value for future readaptation */
            /* --------------------------------------------- */

            NORM(meter); /* Calculate meter normalization factor */
            READ_METER_REG(felm_low, felm_high, temp_s16, 0);

            _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
            temp = _bp_vars[no].access_data_byte[1];  /* High Byte only */


            temp >>= 1;
            temp_s16 *= temp;
            temp_s16 = (BP_S_16BIT) (temp_s16 >> (6 - meter));
            temp_s16 = temp_s16 + (temp_s16 >> 1) + (temp_s16 >> 3); /* 62.5% scale */

            SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, temp_s16);


#ifdef TDEBUG
            printf("DAGC Target = %d\n", (int)temp_s16);
#endif

#ifdef TDEBUG
            PREFIX;
            printf("TX RS0\n");
#endif

            /* Restart Activation Interval */
            _SetActivationInterval(no);


            _DisableUserInterrupts(); /* Disable all user interrupts */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Freeze PLL */
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */


            if ( ZIP_START_MODE )
                {
#ifdef ZIP_START
                _SetAdcControlAgain(no,zip_start_global_var[no].aagc_control_again);
#endif
                }
            else
                {
#ifdef ZIP_START
                _ZipStart_SaveFreqEQDFE(no);
#endif
                     
                prev_aagc = _GetAdcControlAgain(no); /* Read AAGC Setting */
             
        
                /* --------------------------------------------- */
                /* Determine maximum AAGC setting.               */
                /* --------------------------------------------- */
                temp = MAX_AGAIN;
                do
                    {
                    AAGC(temp);

                    /*
                     * Need to wait 2 meter intervals to ensure proper reading
                     */

                    RESTART(bp_mode_ptr, meter);
#ifdef INT_BUG
                    READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
                    _RestartVirtualTimer(no, _METER, s16_value);
#endif

                    WAIT_FOR_TIMER(meter);

                    RESTART(bp_mode_ptr, meter);

#ifdef INT_BUG
                    READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
                    _RestartVirtualTimer(no, _METER, s16_value);
#endif

                    WAIT_FOR_TIMER(meter);

                    _SetFelmMask(no);

                    /* 'meter' has already been normalized */
                    READ_METER_REG(slm_low, slm_high, temp_u16, meter);

#ifdef TDEBUG
                    PREFIX;
                    printf("SLM @ %ddB = %d\n", (int)(temp*3), (int)temp_u16);
#endif

                    if ( temp_u16 < SIGNAL_LEVEL_TH )
                        {
                        break;
                        }

                    temp--;    /* decrement AAGC to next lower setting */

                    }
                while ( temp >= AGAIN0DB );


                /* -------------------------------- */
                /* Scaling DAGC, FFE:               */
                /* -------------------------------- */

#ifdef TDEBUG
                printf("FFE Coefficients (Before Scaling):\n");
                PRINTF_FFE_COEFS(no);
#endif

                temp = _GetAdcControlAgain(no); /* Read AAGC Setting */
        
                _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
                temp_s16 = ((BP_S_16BIT)_bp_vars[no].access_data_byte[1] << 4);  /* High Byte only */

#ifdef TDEBUG
                PREFIX;
                printf("DAGC (Before Scaling) = %d\n", (int)temp_s16);
#endif

                /*
                 * If AAGC dropped from default setting, need to scale
                 * FFE and/or DAGC
                 */
                switch (prev_aagc - temp)
                    {
                    default:
                        /* don't do anything */
                        break;

                    case 1:  /* AAGC Dropped 3dB */
                
                        if (temp_s16 > 1130)
                            {
                            /*
                            * DAGC Too High to compensate for 3dB
                            * Must use FFE Scale
                            */
                    
                            SCALE_FFE_125();  /* scale FFE Coeffs by 1.25 */                            
#ifdef TDEBUG
                            PREFIX;
                            printf("FFE X 1.25\n");
#endif
                            }
                        break;

                    case 2:  /* AAGC Dropped 6dB */
                        /*
                         * If DAGC < 1200
                         *      Use FFE Scale & DAGC to compensate
                         *
                         * If DAGC > 1200
                         *      Use AAGC, FFE Scale, & DAGC to compensate
                         */
                        if (temp_s16 > 1200)  /* was 1000 */
                            {
                            AAGC((BP_U_8BIT)(temp + 1));  /* increment AAGC by 3dB */
                
#ifdef TDEBUG
                            PREFIX;
                            printf("AAGC set to %ddB\n", (int)(temp+1) * 3 );
#endif
                            }

                        SCALE_FFE_125();  /* scale FFE Coeffs by 1.25 */

#ifdef TDEBUG
                        PREFIX;
                        printf("FFE X 1.25\n");
#endif

                        break;

                    case 3:  
                    case 4: /* AAGC Dropped 9dB */
                    case 5: /* See this on ETSI Loop 7 */

                        /* Always increase AAGC by 3dB */
                        AAGC((BP_U_8BIT)(temp + 1));
                
                        SCALE_FFE_125();  /* scale FFE Coeffs by 1.25 */

#ifdef TDEBUG
                        PREFIX;
                        printf("AAGC set to %ddB\n", (int)(temp+1) * 3 );
#endif

                        if (temp_s16 > 1130)
                            {
                            /*
                             * DAGC Too High to compensate
                             * Must use FFE Scale
                             */
                    
                            SCALE_FFE_125();  /* scale FFE Coeffs by 1.25 */                            
#ifdef TDEBUG
                            PREFIX;
                            printf("FFE X 1.25\n");
#endif
                            }

                        break;

                    }   /* end switch AGAIN */

                /*
                 * if DAGC > 1700, then scale FFE. 
                 * FFE may be scaled twice
                 */
#ifdef TDEBUG
                PREFIX;
                printf("DAGC (Before 2nd Scaling) = %d\n", (int)temp_s16);
#endif

                if (temp_s16 > 1700)
                    {
                    SCALE_FFE_125();  /* scale FFE Coeffs by 1.25 */                            
#ifdef TDEBUG
                    PREFIX;
                    printf("DAGC>1700:  FFE X 1.25\n");
#endif
                    }


                } /* end else not zip_startup_mode */


            /*----------------------------------------------------*/
            /* Linear EC adaptation                               */
            /*----------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Adapt EC\n");
#endif
            if ( !ZIP_START_MODE )
                {
                _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)49401); /* Adapt EC at highest gain */
                BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt linear EC */
                WAIT_FOR_TIMER(t3);

                _AdaptEc(no, HIGHER_GAIN, (BP_U_16BIT)63000); /* Adapt EC at higher gain */
                WAIT_FOR_TIMER(t3);

                _AdaptEc(no, HIGH_GAIN, (BP_U_16BIT)63000); /* Adapt EC at high gain */
                WAIT_FOR_TIMER(t3);
                }
#ifdef ZIP_START
            else 
                {
                _ZipStart_Htur2Level(no);
                INIT_TIMER(sut4, _SUT4, sut4_low, sut4_high, ZIP_START_2LEVEL_INTERVAL);
                }
#endif /* ZIP_START */


#ifdef TDEBUG
            PREFIX;
            printf("Before Self:  ");
            PRINTF_DAGC(no);
#endif

            /* ---------------------------- */
            /* Adapt DAGC - Self adaptation */
            /* ---------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, OFF); /* DAGC at self adaptation mode */
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 15000);

            WAIT_FOR_TIMER(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */


#ifdef TDEBUG
            PREFIX;
            printf("After Self:  ");
            PRINTF_DAGC(no);
#endif

            /* -------------------------------- */
            /* Adapt DAGC - Eq Error adaptation */
            /* -------------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON); /* DAGC at eq error adaptation mode */
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 10000);

            WAIT_FOR_TIMER(t3);

#ifdef TDEBUG
            PREFIX;
            printf("After Readapt:  ");
            PRINTF_DAGC(no);
#endif

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Reactivate PLL */
            _EnableUserInterrupts(); /* Enable all user interrupts */
            stage = ADAPT_EQ1;
            break;

        case ADAPT_EQ1: /* Restart Equalizer adaptation */
#ifdef TDEBUG
            PREFIX;
            printf("Restart EQ:  ");
            PRINTF_DAGC(no);
#endif

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON); /* DAGC at eq error adaptation mode */
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 10000);

            stage = ADAPT_EQ2;
            break;

        case ADAPT_EQ2: /* Adapt Equalizer */
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */

            BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, OFF); /* 4-level Slicing */
            _SetMeterTimer(no, FOUR_LEVEL_METER);


#ifdef INT_BUG
            _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << FOUR_LEVEL_METER));
#endif

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, WAIT_FOR_4LEVEL);
            stage = SU_WAIT_METER_INTERVAL1;
#ifdef TDEBUG
            PREFIX;
            printf("Waiting for 4 level...\n");
#endif
            break;


        case WAIT_FOR_4LEVEL: /* Wait for far end four level signal, CS1 */
            TIMER_BREAK(meter);

            i = BP_READ_REG(bp_ptr, symbol_histogram);

            /* If number of +/-1 symbols exceeds threshold: 4-level signal is received */
            if (i < FOUR_LEVEL_SIGNAL_TH) /* Far end 4-level signal NOT detected */
                {
#ifdef INT_BUG
                _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << FOUR_LEVEL_METER));
#endif
                break;
                }


            /*------------------------------------------------------------*/
            /* 4-level far-end signal detected                            */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("4 level detected\n");
#endif

            /* Read Bitpump user setup */
            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp);

            /*------------------------------------------------------------*/
            /* Check for Tip/Ring Reversal                                */
            /*------------------------------------------------------------*/
            if ( user_setup_low.bits.auto_tip_ring )
                {
                READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);

#ifdef TDEBUG
                PREFIX;
                printf("BER Meter = %d\n", (int)temp_u16);
#endif

                if ( temp_u16 > TIP_RING_TH )
                    {
                    /* Tip/Ring detected */
                    RD_BYTE(no, STATUS, status_reg.status);
                    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, negate_symbol, 1);
                    status_reg.bits.tip_ring_indication = 1;
                    WR_BYTE(no, STATUS, status_reg.status);

#ifdef TDEBUG
                    PREFIX;
                    printf("Tip/Ring Reversal detected.\n");
#endif
                    }
                }   /* end if check for Tip/Ring reversal */


            _SetMeterTimer(no, NORMAL_METER);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, WAIT_FOR_4LEVEL_2);

            stage = SU_WAIT_METER_INTERVAL1;

            break;

        case WAIT_FOR_4LEVEL_2:
            TIMER_BREAK(meter);
            
            _SetFelmMask(no);

            /*------------------------------------------------------------*/
            /* Transmit startup sequence RS1.                             */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("TX RS1\n");
#endif

            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp);

            if (user_setup_low.bits.seq_source) /* Internal startup sequence */
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES); /* Internal 4-level scrambled 1's */
            else /* External startup sequence */
                if (user_setup_low.bits.tx_scr) /* Scramble transmitted symbols */
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA); /* External scrambled data */
                else /* No scrambling */
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA); /* External unscrambled data */

            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.four_level_indication = ON;
            WR_BYTE(no, STATUS, status_reg.status);

#ifdef TDEBUG
            PRINTF_FFE_COEFS(no);
            PRINTF_DFE_COEFS(no);
#endif

            _FfeReverse(no);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 2001);

            stage = ZIP_START_MODE ? ADAPT_MODULES2 : ADAPT_MODULES1;
            break;

        case ADAPT_MODULES1: /* Adapt NL EC (high gain) */
            TIMER_BREAK(t3);
 
#ifdef TDEBUG
            PREFIX;
            printf("Adapt NL EC & EP\n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, OFF);
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* NL EC at high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 60000);

            stage = ADAPT_MODULES2;
            break;

        case ADAPT_MODULES2: /* Adapt NL EC (low gain) and EP */
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN); /* NL EC at normal adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NL EC */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */
            
            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_gain, NORMAL_GAIN); /* EP at normal adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, ON); /* Adapt EP */
            
            INIT_TIMER(t3, _T3, t3_low, t3_high, ZIP_START_MODE ? 35000 : 60000);
            stage = SET_LOS_THRESHOLD;
            break;

        case SET_LOS_THRESHOLD:
            TIMER_BREAK(t3);
        
            _SetLosThreshold(no);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, SET_NORMAL_CONDITIONS);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case SET_NORMAL_CONDITIONS: /* Set normal conditions */
            TIMER_BREAK(meter);

            _SetFelmMask(no);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */


#ifdef INT_BUG
            _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << NORMAL_METER));
#endif
            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.normal_operation = ON;
            WR_BYTE(no, STATUS, status_reg.status);

            RD_WORD(no, STAGE2, i, temp);
            WR_WORD(no, STAGE2, TE_WAIT_FOR_METER, temp);

#ifdef TDEBUG
            PREFIX;
            printf("Normal operation\n");
#endif

            stage = NORMAL_OPERATION;
            break;

        case NORMAL_OPERATION: /* Normal adaptation stage */
            break;

        case DEACTIVATE_MODE: /* Deactivate stage */
            break;

        case IDLE: /* Idle mode */
            break;


        case SU_WAIT_METER_INTERVAL1:
            RESTART(bp_mode_ptr, meter);
#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif

            stage = SU_WAIT_METER_INTERVAL2;
            break;

    
        case SU_WAIT_METER_INTERVAL2:
            TIMER_BREAK(meter);

#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif

            /* Set Stage to the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, stage);
            break;

        default:
            stage = IDLE;
            break;

        } /* END-SWITCH operational stage */

    WR_WORD(no, STAGE, stage, tm_stage);
#else
    no++;   /* just to resolve 'no' when HTUR not defined */
#endif  /* HTUR */
} /* END _HturControlProcess() */
