/*************************************************************/
/*    SUC.C                                                  */
/*    Startup control process for HTU-C terminal.            */
/*    (C) Copyright 1993 by Rockwell Corporation             */
/*                                                           */
/*    This program is copyrighted by Rockwell Corporation    */
/*                                                           */
/* Description:                                              */
/*    Controlling HTU-C startup process, normal operation,   */
/*    idle mode, test modes etc.                             */
/*                                                           */
/* Notes:                                                    */
/*                                                           */
/* User Modifiable Code:                                     */
/*    None                                                   */
/*                                                           */
/* List of functions included in this module:                */
/*  _HtucControlProcess()                                    */
/*                                                           */
/* Programmer:                                               */
/*     Dean Rasmussen             March - 1998               */
/*     Iris Shuker                26-Sept-1993               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

/* 5 Seconds */
#define T1MIN_INTERVAL(symbol_rate) \
((((BP_S_16BIT) symbol_rate <<2) + ((BP_S_16BIT) symbol_rate)) << 2)

/* 4 Seconds */
#define T4MAX_INTERVAL(symbol_rate) \
(((BP_S_16BIT) symbol_rate <<2) << 2)

#ifdef ZIP_START
/* 4500 * 1024 symbols - which is 8sec at 1168kbps */
#define ZIP_START_FREQ_INTERVAL    4500


/* 2000 * 1024 symbols - which is 3.5sec at 1168kbps */
#define ZIP_START_2LEVEL_INTERVAL    2000
#endif

#define MIN_ADJ_GLITCH_FELM_TH      (BP_U_16BIT)1000
#define MIN_NO_ADJ_GLITCH_FELM_TH   (BP_U_16BIT)100

#define ZERO_LEN_ADJUST_FELM  (BP_S_16BIT)512

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/


/*---------------------------------------------------------*/
/*  Global Constants                                       */
/*---------------------------------------------------------*/

extern BP_U_8BIT BP_CONSTANT dagc_th[4];

BP_S_16BIT BP_CONSTANT ffe_init[2][4][8] =

/* High Data Rates */
  {
   {{-1119,  1882, -3545, 5606,  -9171, 13685,  -6958,    97 },
    { -917,  1396, -2644,  3916, -6382,  9108,  -3345,  -460 },
    { -451,   500, -1139,  1624, -2988,  4492,  -1212,  -199 },
    { -224,   160,  -525,   678, -1497,  2362,   -387,    34 } },
/* Low Data Rates < 580kbps */
   {{-1028,  980, -1899,  3144, -7473, 15432, -11710,  2855 },
    { -635,   586, -1381,  1867, -3884,  6088,  -2744,    73 },
    { -557,   407, -1214,  1277, -2796,  3535,   -597,  -174 },
    {  -14,     8,   -78,   190,  -710,  2164,    825,   412 } } };

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/


/***********************************************************/
/*    _HtucControlProcess()                                */
/*    HTU-C terminal control process.                      */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _HtucControlProcess(no);                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                14-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void _HtucControlProcess (BP_U_8BIT no)
{
#ifdef HTUC

    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;
    SU_FLAGS_TYPE su_flags;
    user_setup_low_type user_setup_low;   /* User setup variable */
    status_reg_type status_reg;           /* Status variable */
    BP_U_8BIT temp_u;                     /* Local temporary buffer */
    BP_U_8BIT stage, tm_stage;            /* Startup stage variable */
    BP_S_8BIT temp, meter;                /* Local temporary buffer */
    BP_S_16BIT temp_s16;                  /* Local temporary buffer */
    BP_U_16BIT temp_u16;                  /* Local temporary buffer */

    /*------------------------------------------------------------*/
    /* Initiate bitpump pointers                                  */
    /*------------------------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;


    /*------------------------------------------------------------*/
    /* Read stage number from Bitpump, move to this stage.        */
    /*------------------------------------------------------------*/
    RD_WORD(no, STAGE, stage, tm_stage);

    switch (stage) /* Go to stage */
    {

        case ACTIVATE_SYSTEM: /* Activate startup process */

            /*------------------------------------------------------------*/
            /* Initialization.                                            */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Activate HTU-C\n");
#endif
        
            _BtInitialize(no); /* Initialize bit-pump status for startup */
            _SetMeterTimer(no, NORMAL_METER);
            /*Changed ALT_METER no NORMAL_METER to get a more accurate DC Offset*/

            /* At low data rate, use higher initial AAGC */ 
            if ( _bp_vars[no].symbol_rate <= 32 )
                {
                _SetAdcControlAgain(no, AGAIN12DB);
                }

            /*
             * Disable signal detection, turning on Tx before adapting EC
             * will cause false detection.
             */
            BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON);
            BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, DC_CANCELLATION);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case DC_CANCELLATION:
            TIMER_BREAK(meter);

            _DcCancel(no);

            /*
             *  LOS (INIT_STARTUP)meter interval requires the ALT_METER
             *  to be used.
             */
            _SetMeterTimer(no, ALT_METER);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, INIT_STARTUP);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case INIT_STARTUP: /* Start CS0 transmission */
            TIMER_BREAK(meter);


#ifdef TDEBUG
            PREFIX;
            printf("TX CS0 \n");
#endif
            /*------------------------------------------------------------*/
            /* Transmit startup sequence                                  */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */

            /*------------------------------------------------------------*/
            /* Initialize and operate startup timers (activation time)    */
            /*------------------------------------------------------------*/
            if ( !ZIP_START_MODE )
                {
                _SetActivationInterval(no);
                }
#ifdef ZIP_START
            else
                {
                SET_WORD(bp_ptr, sut4_low, sut4_high, ZIP_START_FREQ_INTERVAL);
#ifdef INT_BUG
                _RestartVirtualTimer(no, _SUT4, ZIP_START_FREQ_INTERVAL);
#endif
                RESTART(bp_mode_ptr, sut4);
                }
#endif


            /*------------------------------------------------------------*/
            /* Adapt EC                                                   */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Adapt EC \n");
#endif

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt! */

            _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)39201); /* Adapt Ec at highest gain */
            stage = ADAPT_EC_STAGE1;
            break;

        case ADAPT_EC_STAGE1: /* Linear EC adaptation stage */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGHER_GAIN, (BP_U_16BIT)58804); /* Adapt EC at higher gain */
            stage = ADAPT_EC_STAGE2;
            break;

        case ADAPT_EC_STAGE2: /* Linear EC adaptation stage */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGH_GAIN, (BP_U_16BIT)42202); /* Adapt EC at high gain */
            stage = ADAPT_EC_STAGE3;
            break;

        case ADAPT_EC_STAGE3: /* Linear Ec adaptation stage */
            TIMER_BREAK(t3);


            NORM(meter);
            READ_METER_REG(felm_low, felm_high, temp_s16, meter);
#ifdef TDEBUG
            PREFIX;
            printf("felm = %d\n", (int)temp_s16);
#endif

            /*
             * Special 0 length case: hybrid causes high FELM reading.
             * Therefore, just force high threshold to be (FELM + 512)
             *
             * Need to set 'high_felm_th' to non-normalized value since
             * SetMeterTimer() expects prescaled value.
             *
             * ((high_felm_th << meter) << (meter - 1))
             *   ==> Normalized High FELM Th / 2
             *
             * 10/98 - For lower data rates, need a different threshold
             *         
             */
            RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
            su_flags.bits.adjusted_felm = 0;
            WR_WORD(no, FCELL__SU_FLAGS, 0, su_flags.reg);

            RD_WORD(no, FELM_THRESHOLD, temp, temp_u);

            if ( temp_s16 >= (BP_S_16BIT)(((temp_u << meter) << (meter - 1) )) )
                {
                temp_u = ((temp_s16 + ZERO_LEN_ADJUST_FELM) >> meter) >> meter;

                WR_WORD(no, FELM_THRESHOLD, temp, temp_u);

                _SetMeterTimer(no, ALT_METER);
            
                su_flags.bits.adjusted_felm = 1;
                WR_WORD(no, FCELL__SU_FLAGS, 0, su_flags.reg);

#ifdef TDEBUG
                PREFIX;
                printf("0 length detected\n");
                printf("Adjust FELM High TH =  %d\n", (int)(temp_s16 + ZERO_LEN_ADJUST_FELM));
#endif

                /* Set the next stage after wait meter intervals */
                WR_WORD(no, STAGE2, TEMP_ENV_IDLE, ADJUST_FELM_HIGH_TH);

                stage = SU_WAIT_METER_INTERVAL1;
                break;

                }
            stage = GOTO_WAIT_FOR_SIGNAL;
            break;

        case ADJUST_FELM_HIGH_TH:
            TIMER_BREAK(meter);
            stage = GOTO_WAIT_FOR_SIGNAL;
            break;


        case GOTO_WAIT_FOR_SIGNAL:
            /*
             * Re-enable signal detection.
             */
            _SetFelmMask(no);

            stage = WAIT_FOR_SIGNAL;
            #ifdef TDEBUG
            PREFIX;
            printf("Wait for signal...\n");
#endif
            break;


        case WAIT_FOR_SIGNAL: /* Wait for far end signal */
            RD_BYTE(no, STATUS, status_reg.status);

            if (status_reg.bits.los) /* Wait for far end signal */
                {
                break;
                }

            /*------------------------------------------------------------*/
            /* Far-end signal detected: Initialize and run T1MIN, T4MAX   */
            /* timers.                                                    */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Signal detected \n");
#endif

            _SetMeterTimer(no, DEFAULT_METER);
            
            /* Set the next stage after wait meter intervals */
#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, AAGC1);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case AAGC1:
            TIMER_BREAK(meter);

            _SetFelmMask(no);

            temp_u16 = COMP_SYM_RATE();
            SET_WORD(bp_ptr, sut2_low, sut2_high, T4MAX_INTERVAL(temp_u16));
            SET_WORD(bp_ptr, sut3_low, sut3_high, T1MIN_INTERVAL(temp_u16));
            RESTART(bp_mode_ptr, sut2);
            RESTART(bp_mode_ptr, sut3);
#ifdef INT_BUG
            _RestartVirtualTimer(no, _SUT2, T4MAX_INTERVAL(temp_u16));
            _RestartVirtualTimer(no, _SUT3, T1MIN_INTERVAL(temp_u16));
#endif

            if ( !ZIP_START_MODE )
                {
                /* Restart Activation Interval when S0 detected */
                _SetActivationInterval(no);
                }
#ifdef ZIP_START
            else
                {
                SET_WORD(bp_ptr, sut4_low, sut4_high, ZIP_START_2LEVEL_INTERVAL);
                RESTART(bp_mode_ptr, sut4);

#ifdef INT_BUG
                _RestartVirtualTimer(no, _SUT4, ZIP_START_2LEVEL_INTERVAL);
#endif

                _ZipStart_HtucAagc(no);

#ifdef ZIP_START_EC_BUG
                /* Need Full EC Adapt */
                stage = ADAPT_EC_AFTER_AAGC1;
#else
                /* Check for Glitch */
                stage = ADAPT_EC_AFTER_AAGC4;
#endif /* ZIP_START_EC_BUG */
                break;                

                }
#endif

            _SetFfeScale(no);

            AAGC(MAX_AGAIN);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, AAGC2);

            stage = SU_WAIT_METER_INTERVAL1;
            break;


        case AAGC2:
            TIMER_BREAK(meter);

            _SetFelmMask(no);

            /*-----------------------------------------------------------*/
            /* Read the average absolute level of the input signal.      */
            /*-----------------------------------------------------------*/

            NORM(meter); /* Calculate meter normalization factor */
            READ_METER_REG(slm_low, slm_high, temp_u16, meter);

            temp_u = _GetAdcControlAgain(no); /* Read AAGC Setting */

#ifdef TDEBUG
            PREFIX;
            printf("SLM @ %ddB = %d\n", (int)(temp_u*3), (int)temp_u16);
#endif
            if ( temp_u16 < SIGNAL_LEVEL_TH || temp_u == AGAIN0DB )
                {
                stage = ADAPT_EC_AFTER_AAGC1;
                break;
                }

            /* decrement AAGC and re-measure SLM */
            temp_u--;
            AAGC(temp_u);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, AAGC2);

            stage = SU_WAIT_METER_INTERVAL1;
            break;


        case ADAPT_EC_AFTER_AAGC1: /* Adapt EC highest gain */
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Restart linear EC adaptation */

#ifdef TDEBUG
            printf("Adapt EC after AAGC\n");
#endif

            _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)39123); /* Adapt Ec at highest gain */
            stage = ADAPT_EC_AFTER_AAGC2;
            break;

        case ADAPT_EC_AFTER_AAGC2: /* Adapt EC higher gain */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGHER_GAIN, (BP_U_16BIT)58747); /* Adapt EC at higher gain */
            stage = ADAPT_EC_AFTER_AAGC3;
            break;

        case ADAPT_EC_AFTER_AAGC3: /* Adapt EC high gain */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGH_GAIN, (BP_U_16BIT)39019); /* Adapt EC at high gain */
            stage = ADAPT_EC_AFTER_AAGC4;
            break;

        case ADAPT_EC_AFTER_AAGC4: /* Adapt EC high gain */
            TIMER_BREAK(t3);

            NORM(meter);

#ifdef TDEBUG
            READ_METER_REG(slm_low, slm_high, temp_u16, meter);
            PREFIX;
            printf("SLM after AAGC = %d\n", (int)temp_u16);
#endif

            READ_METER_REG(felm_low, felm_high, temp_u16, meter);
#ifdef TDEBUG
            PREFIX;
            printf("FELM after AAGC = %d\n", (int)temp_u16);
#endif

            /*
             * Look at the FELM to see if the signal is present, could get
             * false glitch if cable disonnected then reconnected.
             *
             * If (FELM < 1000 and Adjusted FELM for Zero Length)
             *
             * Zero Length is determined if the 'high_felm_th' was adjusted.
             */
            RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
            if ( (su_flags.bits.adjusted_felm == 1 && temp_u16 < MIN_ADJ_GLITCH_FELM_TH) ||
                 (su_flags.bits.adjusted_felm == 0 && temp_u16 < MIN_NO_ADJ_GLITCH_FELM_TH) )
                {
                /* Reset critical variables */

                /* At low data rate, use higher initial AAGC */ 
                if ( _bp_vars[no].symbol_rate <= 32 )
                    {
                    _SetAdcControlAgain(no, AGAIN12DB);
                    }
                else
                    {
                    _SetAdcControlAgain(no, AGAIN9DB);
                    }


                WR_WORD(no, FELM_THRESHOLD, LOW_FELM_ALARM_TH, HIGH_FELM_ALARM_TH);

                _SetMeterTimer(no, ALT_METER);
                BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON);
                BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON);

                RD_BYTE(no, STATUS, status_reg.status);
                status_reg.bits.los = 1;
                WR_BYTE(no, STATUS, status_reg.status);

#ifdef TDEBUG
               PREFIX;
               printf("Detected glitch, return to wait for signal.\n");
#endif

                BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt! */
                _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)39201); /* Adapt Ec at highest gain */

               stage = ADAPT_EC_STAGE1;
               break;
               }


#ifdef ZIP_START
            if ( ZIP_START_MODE )
                {
                stage = START_OPEN_EYE;
                break;
                }
#endif /* ZIP_START */



            temp_u = _CheckAagc(no);
        
            stage = REMOTE_PHASE_LOCK;
            break;

            /*------------------------------------------------------------*/
            /* Wait for HTU-R phase lock (T4MAX), unless the other side is*/
            /* Bitpump - in which case continue without delay.            */
            /*------------------------------------------------------------*/
        case REMOTE_PHASE_LOCK: /* Check remote side configuration */

            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read Bitpump user setup */
                  
            if (user_setup_low.bits.bt_other_side)
                {
                stage = START_OPEN_EYE;
                }
            else
                {
                stage = WAIT_FOR_REMOTE_PHASE_LOCK;
                }
            break;

        case WAIT_FOR_REMOTE_PHASE_LOCK: /* Wait for remote phase lock (T4MAX) */
            TIMER_BREAK(sut2);
            stage = START_OPEN_EYE;
            break;

            /*-----------------------------------------------------------------*/
            /* Open eye:                                                       */
            /* The eye opening process includes operating the Equalizer        */
            /* (DAGC, DFE, FFE) and Detector modules.                          */
            /* The process is repeated at 16 different sampling phases,        */
            /* the best phase (according to phase quality measure) is selected */
            /*-----------------------------------------------------------------*/

        case START_OPEN_EYE: /* Start HTU-C search optimal phase process */

#ifdef ZIP_START
            if ( ZIP_START_MODE )
                {
                _ZipStart_HtucOpenEye(no);

                stage = OPEN_EYE_AT_OPTIMAL_PHASE6;
                break;
                }
#endif /* ZIP_START */

            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0); /* Init receive sampling phase */

            WR_BYTE(no, OPEN_ATTEMPTS, 0);


#ifdef TDEBUG
            PREFIX;
            printf("Search opt phase\n");
#endif

            _SetMeterTimer(no, PHASE_QUALITY_METER);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, START_OPEN_EYE2);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case START_OPEN_EYE2:
            TIMER_BREAK(meter);
            _SetFelmMask(no);
            stage = PHASE_ADAPT_EC1; 
            break;

        case PHASE_ADAPT_EC1: /* EC adaptation at current sampling phase */
            _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)10000); /* Adapt EC at highest gain */
            stage = PHASE_ADAPT_EC2;
            break;

        case PHASE_ADAPT_EC2: /* EC adaptation at current sampling phase */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGHER_GAIN, (BP_U_16BIT)30000); /* Adapt EC at higher gain */
            stage = PHASE_ADAPT_EC3;
            break;

        case PHASE_ADAPT_EC3: /* EC adaptation at current sampling phase */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGH_GAIN, (BP_U_16BIT)30000); /* Adapt EC at high gain */
            stage = PHASE_ADAPT_EC4;
            break;

        case PHASE_ADAPT_EC4: /* EC adaptation at current sampling phase */
            TIMER_BREAK(t3);
            stage = OPEN_EYE1;
            break;

        case OPEN_EYE1: /* Open eye process */

            /* ------------------------------ */
            /* Initialize DAGC, DFE, and FFE: */
            /* ------------------------------ */

            INIT_DAGC;
            INIT_DFE;
            INIT_FFE;

            /* ---------------------------------- */
            /* Adapt DAGC @ HIGH Gain, Use Slicer */
            /* ---------------------------------- */
            INIT_DETECTOR;
               
            ADAPT_DAGC_EQ_ERROR_MODE;                                                                                                                 

            INIT_TIMER(t3, _T3, t3_low, t3_high, 5000);
            stage = OPEN_EYE2;
            break;

        case OPEN_EYE2: /* Open eye process */
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

            /* ---------- */
            /* Adapt FFE: */
            /* ---------- */

            ADAPT_ALL_FFE;
            INIT_TIMER(t3, _T3, t3_low, t3_high, 10000 /* 20000 */);    /* 6800 */
            stage = OPEN_EYE3;
            break;

        case OPEN_EYE3:
            TIMER_BREAK(t3);
            /* -------------- */
            /* Adapt FFE+DFE: */
            /* -------------- */

            ADAPT_DFE;
            INIT_TIMER(t3, _T3, t3_low, t3_high, 10000 /* 20000 */);    /* 6700 */
            stage = OPEN_EYE4;
            break;


        case OPEN_EYE4:
            TIMER_BREAK(t3);


            FREEZE_DFE_FFE;

            /* -------------------------------- */
            /* Calculate phase quality measure: */
            /* -------------------------------- */

#ifdef TDEBUG
            printf("(%d) ", (int)BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs));
#endif

            temp_s16 = _CalculatePhaseQuality(no);
            temp = BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs);
            WR_PHASE_QUALITY(temp, temp_s16);

            if (temp_s16 != 0) 
                {
                RD_BYTE(no, OPEN_ATTEMPTS, temp_u);
                temp_u++;
                WR_BYTE(no, OPEN_ATTEMPTS, temp_u);
                }

#ifdef TDEBUG
            RD_PHASE_QUALITY(temp, temp_u16);
            printf("P=%d ", (int)temp_u16);
#endif


            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, OPEN_EYE5);

            stage = SU_WAIT_METER_INTERVAL1;
            break;


        case OPEN_EYE5:
            TIMER_BREAK(meter);

            temp = BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs);

            /* ---------------------------------------*/
            /* Read NLM (SNR):                        */
            /* Used during the Optimal Phase Search.  */
            /* ---------------------------------------*/
            NORM(meter);
            READ_METER_REG(nlm_low, nlm_high, temp_s16, meter);
            WR_NOISE_QUALITY(temp, temp_s16);

#ifdef TDEBUG
            printf("NLM=%d\n", (int)temp_s16); 
#endif


            /*------------------------------------------------------------*/
            /* If done (all 16 sampling phases) go to finding best phase, */
            /* otherwise - continue with next phase.                      */
            /*------------------------------------------------------------*/
            if ( temp < 15)  /* Not done yet - continue with next phase */
                {
                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, (temp + 1));
                stage = PHASE_ADAPT_EC1;
                } /* END-IF continue to the next phase */
            else /* This was the last phase */
                {
#ifdef TDEBUG
                printf("\n");
#endif
                stage = OPEN_EYE6;
                } /* END-ELSE last phase */
            break;

        case OPEN_EYE6:
            RD_BYTE(no, OPEN_ATTEMPTS, temp_u);
#ifdef TDEBUG
            printf("Number of Opened Phases = %d\n", (int)temp_u);
#endif
            
            if (temp_u < NO_OF_OPENED_PHASES) /* Attenuator Recognition */
                {
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

                /* limit to 0 - 3 */
                su_flags.bits.ffe_init_scale = (su_flags.bits.ffe_init_scale + 1) % 4;

                WR_WORD(no, FCELL__SU_FLAGS, 0, su_flags.reg);  /* FCELL__SU_FLAGS is don't care on HTU-C */
#ifdef TDEBUG
                printf("New FFE scale: %d\n", (int)su_flags.bits.ffe_init_scale);
#endif

                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0);

                WR_BYTE(no, OPEN_ATTEMPTS, 0);

                stage = PHASE_ADAPT_EC1;
                break;
                } /* end-if */

            /*------------------------------------------------------------*/
            /* Find and set optimal sampling phase.                       */
            /*------------------------------------------------------------*/
            _CalculateOptimalPhase1(no);
            stage = CALC_OPT_PHASE2;
            break;

        case CALC_OPT_PHASE2:
            _CalculateOptimalPhase2(no);
            stage = CALC_OPT_PHASE3;
            break;

        case CALC_OPT_PHASE3:
            _CalculateOptimalPhase3(no);
            stage = CALC_OPT_PHASE4;
            break;

        case CALC_OPT_PHASE4:
            _CalculateOptimalPhase4(no);
            stage = CALC_OPT_PHASE;
            break;

        case CALC_OPT_PHASE:
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, _CalculateOptimalPhase(no) );

            _SetMeterTimer(no, DEFAULT_METER);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, OPEN_EYE7);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case OPEN_EYE7:

            /*------------------------------------------------------------*/
            /* Adapt EC and re-open eye at the optimal phase              */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Adapt EC @ opt phase\n");
#endif
            _AdaptEc(no, HIGHEST_GAIN, (BP_U_16BIT)39444); /* Adapt EC at highest gain */
            stage = ADAPT_EC1;

            break;

        case ADAPT_EC1: /* EC adaptation at optimal phase */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGHER_GAIN, (BP_U_16BIT)63000); /* Adapt EC at higher gain */
            stage = ADAPT_EC2;
            break;

        case ADAPT_EC2: /* EC adaptation at optimal phase */
            TIMER_BREAK(t3);
            _AdaptEc(no, HIGH_GAIN, (BP_U_16BIT)63000); /* Adapt EC at high gain */
            stage = ADAPT_EC3;
            break;

        case ADAPT_EC3: /* EC adaptation at optimal phase */
            TIMER_BREAK(t3);
            stage = OPEN_EYE_AT_OPTIMAL_PHASE1;
            break;



        case OPEN_EYE_AT_OPTIMAL_PHASE1:
            /* ------------------------------ */
            /* Initialize DAGC, DFE, and FFE: */
            /* ------------------------------ */
            INIT_DAGC;
            INIT_DFE;
            INIT_FFE;

            /* ---------------------------------- */
            /* Adapt DAGC @ HIGH Gain, Use Slicer */
            /* ---------------------------------- */
            INIT_DETECTOR;
            ADAPT_DAGC_EQ_ERROR_MODE;
            INIT_TIMER(t3, _T3, t3_low, t3_high, 5000);
        
            stage = OPEN_EYE_AT_OPTIMAL_PHASE2;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE2:
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF);
            
            /* ---------- */
            /* Adapt FFE: */
            /* ---------- */
            
            ADAPT_ALL_FFE;
            INIT_TIMER  (t3, _T3, t3_low, t3_high, 10000);

            stage = OPEN_EYE_AT_OPTIMAL_PHASE3;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE3:
            TIMER_BREAK(t3);
        
            /* -------------- */
            /* Adapt FFE+DFE: */
            /* -------------- */
            
            ADAPT_DFE;
            INIT_TIMER  (t3, _T3, t3_low, t3_high, 10000);

            stage = OPEN_EYE_AT_OPTIMAL_PHASE4;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE4:
            TIMER_BREAK(t3);
        
            FREEZE_DFE_FFE;
#ifdef TDEBUG
            printf("ReOpen Eye At Optimal Phase Completed\n");
#endif
            _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
            temp_u = (BP_U_8BIT)_bp_vars[no].access_data_byte[1];  /* High Byte only */

            RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);
            if ( temp_u < dagc_th[su_flags.bits.ffe_init_scale] ) /* DAGC NOT OK */
                {
                RD_WORD(no, FCELL__SU_FLAGS, temp, su_flags.reg);

                /* limit to 0 - 3 */
                su_flags.bits.ffe_init_scale = (su_flags.bits.ffe_init_scale + 1) % 4;
                
                WR_WORD(no, FCELL__SU_FLAGS, 0, su_flags.reg); /* FCELL__SU_FLAGS is don't care on HTU-C */

#ifdef TDEBUG
                PREFIX;
                printf("Reopen Opt. phase failed!\n");
                printf("FFE Scale changed to %d\n", (int)su_flags.bits.ffe_init_scale);
#endif
            
                stage = OPEN_EYE_AT_OPTIMAL_PHASE1;
                break;
                }

            ADAPT_ALL_FFE;
            ADAPT_DFE;

            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, 90);
            stage = OPEN_EYE_AT_OPTIMAL_PHASE5;
            break;


        case OPEN_EYE_AT_OPTIMAL_PHASE5: /* Open eye at optimal phase */
            TIMER_BREAK(sut2); 
        
            /* --------------------------- */
            /* Adapt FFE+DFE @ NORMAL Gain */
            /* --------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);          /* FFE NORMAL adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN);          /* DFE NORMAL adaptation gain */


            INIT_TIMER(sut2, _SUT2, sut2_low, sut2_high, 90);
            stage = OPEN_EYE_AT_OPTIMAL_PHASE6;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE6: /* Open eye at optimal phase */
            TIMER_BREAK(sut2);
         
            FREEZE_DFE_FFE;

            _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
            temp_s16 = ((BP_S_16BIT)_bp_vars[no].access_data_byte[1] << 4);  /* High Byte only */

#ifdef TDEBUG

            PREFIX;
            printf("DAGC (After Reopen Eye) = %d\n", (int)temp_s16);

            PRINTF_FFE_COEFS(no);
#endif

            if (temp_s16 > 1700)
                {
                /* scale DAGC/FFE Coeffs */
                SCALE_DAGC_FFE();

#ifdef TDEBUG
                PREFIX;
                printf("DAGC>1700:  DAGC/0.875 & FFEx1.25\n");
#endif
                }


            /* -------------------------------- */
            /* Adapt DAGC Error Mode adaptation */
            /* -------------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            INIT_TIMER(t3, _T3, t3_low, t3_high, ZIP_START_MODE ? 10000 : 20000);

            stage = OPEN_EYE_AT_OPTIMAL_PHASE7;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE7:
            TIMER_BREAK(t3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF);

#ifdef ZIP_START
#ifdef ZIP_START_EC_BUG

            /* Add enough delay to compensate for SUR doing a full EC Adapt */
            if ( ZIP_START_MODE )
                {

                INIT_TIMER(t3, _T3, t3_low, t3_high, 63000);
                stage = OPEN_EYE_AT_OPTIMAL_PHASE8;

#if 0
                INIT_TIMER(t3, _T3, t3_low, t3_high, 50000);
                WAIT_FOR_TIMER(t3);
                INIT_TIMER(t3, _T3, t3_low, t3_high, 63000);
                WAIT_FOR_TIMER(t3);
                INIT_TIMER(t3, _T3, t3_low, t3_high, 63000);
                WAIT_FOR_TIMER(t3);
#endif
                }
#endif /* ZIP_START_EC_BUG */
#endif

            stage = WAIT_FOR_T1MIN;
            break;


#ifdef ZIP_START
#ifdef ZIP_START_EC_BUG
        case OPEN_EYE_AT_OPTIMAL_PHASE8:
            TIMER_BREAK(t3);
            stage = WAIT_FOR_T1MIN;
            break;
#endif /* ZIP_START_EC_BUG */
#endif


        /*------------------------------------------------------------*/
        /* Wait for T1MIN timeout before transmitting CS1             */
        /*------------------------------------------------------------*/
        case WAIT_FOR_T1MIN: /* Wait for T1 min */

            /*
             * If Bt Other Side or Zip Restart, then don't wait for
             * T1 Min, otherwise wait.
             */
            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read Bitpump user setup */
            if ( !user_setup_low.bits.bt_other_side && !ZIP_START_MODE )
                {
                TIMER_BREAK(sut3);
                }


#ifdef TDEBUG
            PREFIX;
            PRINTF_DAGC(no);
            PRINTF_FFE_COEFS(no);
#endif


            BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, OFF); /* 4-level Slicing */

            /*------------------------------------------------------------*/
            /* Transmit startup sequense CS1.                             */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("TX CS1 \n");
#endif

            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read Bitpump user setup */
            if (user_setup_low.bits.seq_source) /* Internal startup sequence source */
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES); /* Internal 4-level scrambled 1's */
            else /* External startup sequence source */
                if (user_setup_low.bits.tx_scr) /* Scramble transmitted symbols */
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA); /* External (scrambled) data */
                else /* No scrambling */
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA); /* External (unscrambled) data */

            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.four_level_indication = ON;
            WR_BYTE(no, STATUS, status_reg.status);

#ifdef TDEBUG
            PREFIX;
            printf("Wait for 4level...\n");
#endif

            _SetMeterTimer(no, FOUR_LEVEL_METER);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, WAIT_FOR_4LEVEL);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case WAIT_FOR_4LEVEL: /* Wait for 4 level far-end signal */
            TIMER_BREAK(meter);

            temp = BP_READ_REG(bp_ptr, symbol_histogram);

            /* If number of +/-1 symbols exceeds threshold: 4-level signal is received */
            if (temp < FOUR_LEVEL_SIGNAL_TH) /* Far end 4-level signal not detected */
                {
#ifdef INT_BUG
                _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE <<FOUR_LEVEL_METER));
#endif
                break;
                }

#ifdef TDEBUG
            PREFIX;
            PRINTF_DAGC(no);
            PRINTF_FFE_COEFS(no);
#endif

            /*------------------------------------------------------------*/
            /* 4-level far-end signal detected                            */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("4-level detected\n");
#endif

            /* Read Bitpump user setup */
            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp);

            /*------------------------------------------------------------*/
            /* Check for Tip/Ring Reversal                                */
            /*------------------------------------------------------------*/
            if ( user_setup_low.bits.auto_tip_ring )
                {
                READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);

#ifdef TDEBUG
                PREFIX;
                printf("BER Meter = %d\n", (int)temp_u16);
#endif

                if ( temp_u16 > TIP_RING_TH )
                    {
                    /* Tip/Ring detected */
                    RD_BYTE(no, STATUS, status_reg.status);
                    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, negate_symbol, 1);
                    status_reg.bits.tip_ring_indication = 1;
                    WR_BYTE(no, STATUS, status_reg.status);

#ifdef TDEBUG
                    PREFIX;
                    printf("Tip/Ring Reversal detected.\n");
#endif
                    }
                }   /* end if check for Tip/Ring reversal */


            _ShiftFfe(no); 

            INIT_TIMER(t3, _T3, t3_low, t3_high, 2200);
            stage = WAIT_FOR_4LEVEL_2;
            break;

        case WAIT_FOR_4LEVEL_2:
            TIMER_BREAK(t3);

            _SetMeterTimer(no, NORMAL_METER);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, WAIT_FOR_4LEVEL3);

            stage = SU_WAIT_METER_INTERVAL1;

            break;

        case WAIT_FOR_4LEVEL3:
            TIMER_BREAK(meter);
            
            _SetFelmMask(no);

            /* --------------------- */
            /* Set DAGC Target value */
            /* --------------------- */

            NORM(meter); /* Calculate meter normalization factor */
            READ_METER_REG(felm_low, felm_high, temp_s16, 0);

            _ReadAccessByteRAM(no, EQ_ACCESS_RAM, DAGC_HIGH);
            temp = _bp_vars[no].access_data_byte[1];  /* High Byte only */

            temp >>= 1;
            temp_s16 *= temp;
            temp_s16 = (BP_S_16BIT) (temp_s16 >> (6 - meter));
            temp_s16 = temp_s16 + (temp_s16 >> 1) + (temp_s16 >> 4); /* 58.333% scale */

            SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, temp_s16);

#ifdef TDEBUG
            PREFIX;
            printf("DAGC Target = %d\n", (int)temp_s16);

            PREFIX;
            PRINTF_DAGC(no);
            PRINTF_FFE_COEFS(no);
            PRINTF_DFE_COEFS(no);
#endif

            _FfeReverse(no);

            /* --------------------------------------- */
            /* DAGC self adaptation mode @ Normal gain */
            /* --------------------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, OFF);                
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN);

            /* ------------------ */
            /* FFE  @ Normal gain */
            /* ------------------ */

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);

            /* ---------------------- */
            /* Adapt DAGC + FFE + DFE */
            /* ---------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON);
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON);
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON);
        
            INIT_TIMER(t3, _T3, t3_low, t3_high, 10000);
            stage = ZIP_START_MODE ? ADAPT_MODULES2 : ADAPT_MODULES1;
            break;

        case ADAPT_MODULES1: /* NL EC (high gain) adaptation */
            TIMER_BREAK(t3);

#ifdef TDEBUG
            PREFIX;
            printf("Adapt NLEC & EP\n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, OFF);
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* NL EC high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 59111);
            stage = ADAPT_MODULES2;
            break;

        case ADAPT_MODULES2: /* NL EC (low gain) & EP adaptation */
            TIMER_BREAK(t3);
        
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN); /* NL EC normal adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NL EC */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */

            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_gain, NORMAL_GAIN); /* EP normal adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, ON); /* Adapt EP */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 8000);
            stage = SET_LOS_THRESHOLD;
            break;

        case SET_LOS_THRESHOLD:
            TIMER_BREAK(t3);
        
            _SetLosThreshold(no);

            /* Set the next stage after wait meter intervals */
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, SET_NORMAL_CONDITIONS);

            stage = SU_WAIT_METER_INTERVAL1;
            break;

        case SET_NORMAL_CONDITIONS: /* Set normal conditions */
            TIMER_BREAK(meter);

            _SetFelmMask(no);
        
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, NORMAL_GAIN);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */


#ifdef INT_BUG
            _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << NORMAL_METER));
#endif

#ifdef TDEBUG
            PREFIX;
            printf("Final ");
            PRINTF_DAGC(no);
            PRINTF_FFE_COEFS(no);
#endif

            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.normal_operation = ON;
            WR_BYTE(no, STATUS, status_reg.status);

            /* Set Temp/Environment Stage */
            WR_WORD(no, STAGE2, TE_WAIT_FOR_METER, IDLE);

#ifdef TDEBUG
            PREFIX;
            printf("Normal operation\n");
#endif

            stage = NORMAL_OPERATION;
            break;

        case NORMAL_OPERATION: /* Normal adaptation stage */
            break;


        case DEACTIVATE_MODE: /* Deactivate stage */
            break;

        case IDLE: /* Idle mode */
            break;


        case SU_WAIT_METER_INTERVAL1:
            RESTART(bp_mode_ptr, meter);
#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif

            stage = SU_WAIT_METER_INTERVAL2;
            break;

    
        case SU_WAIT_METER_INTERVAL2:
            TIMER_BREAK(meter);

#ifdef INT_BUG
            READ_METER_REG(meter_low, meter_high, (BP_U_16BIT)s16_value, 0);
            _RestartVirtualTimer(no, _METER, s16_value);
#endif

            /* Set Stage to the next stage after wait meter intervals */
            RD_WORD(no, STAGE2, temp, stage);
            break;

        default:
            stage = IDLE;
            break;

        } /* END-SWITCH operational stage */

    WR_WORD(no, STAGE, stage, tm_stage);

#else
    no++;   /* just to resolve 'no' when HTUC not defined */
#endif /* HTUC */

} /* END _HtucControlProcess() */
