/***********************************************************/
/*    MAIL.C                                               */
/*    Mail box management.                                 */
/*    (C) Copyright 1993 by Brooktree Corporation          */
/*                                                         */
/*    This program is copyrighted by Brooktree Corporation */
/*                                                         */
/* Description:                                            */
/*    Handle receive messages, transmit messages.          */
/*    Manage read/write to mail boxes.                     */
/*                                                         */
/* Notes:                                                  */
/*    All file is conditionally compiled on SER_COM        */
/*    definition.                                          */
/*                                                         */
/* User Modifiable Code:                                   */
/*    None                                                 */
/*                                                         */
/* List of C files included in this module:                */
/*                                                         */
/* List of H files included in this module:                */
/* bthomer.h                                               */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                31-August-1993           */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

#ifdef SER_COM

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define HOST_MAIL_BOX_LENGTH   16
#define ACK0                 0xFF
#define ACK1                 0xFF
#define ACK2                 0xFF
#define ACK3                 0x55

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/

BP_U_8BIT BP_MSPACE _message_buffer[4]; /* Message buffer, 4 bytes length,
                                     1 message deep */

BP_MAIL_FLAGS BP_MSPACE bp_mail_flags;

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

#define MAIL_MSPACE BP_XDATA

static BP_U_8BIT MAIL_MSPACE host_mail_box[HOST_MAIL_BOX_LENGTH]; /* Host mail box array */
static BP_U_8BIT * MAIL_MSPACE host_write_pointer; /* Write pointer */
static BP_U_8BIT * MAIL_MSPACE host_read_pointer;  /* Read pointer */
static BP_U_8BIT MAIL_MSPACE fifo_counter; /* Count number of full cells in the host mail box */


/***********************************************************/
/*    _InitHostMailBox(void)                               */
/*    Initialize host mail box pointers.                   */
/*                                                         */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: none.                               */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _InitHostMailBox()                            */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                06-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void _InitHostMailBox (void)
{

    host_read_pointer = &host_mail_box[0]; /* Init host mail box read pointer */
    host_write_pointer = &host_mail_box[0]; /* Init host mail box write pointer */
    bp_mail_flags.ffh = OFF;
    bp_mail_flags.efh = ON;
    bp_mail_flags.message_flag = OFF;
    fifo_counter = RESET;
#ifdef TDEBUG
    bp_mail_flags.tx_flag = OFF;
#else
    bp_mail_flags.tx_flag = ON;
#endif

} /* END _InitHostMailBox */

/*************************************************************/
/*    _WriteHostMailBox(source, opcode, parameter, checksum) */
/*    Write message (four bytes long) to host mail box.*/
/*    Host mail box is a small fifo with a read pointer      */
/*    and a write pointer.                                   */
/*                                                           */
/*    Returns: void.                                         */
/*                                                           */
/*    Input Variables:                                       */
/*                                                           */
/*    Output Variables: none.                                */
/*                                                           */
/*    Example:                                               */
/*     _WriteHostMailBox(BIT_PUMP0, _message_buffer[1], info */
/*     CHECKSUM(BIT_PUMP0, _message_buffer[1], info))        */
/*                                                           */
/* Programmer:                                               */
/*     Iris Shuker                06-Sept-1993               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

void _WriteHostMailBox (BP_U_8BIT source, BP_U_8BIT opcode, \
    BP_S_8BIT parameter, BP_U_8BIT checksum)
{

    /*--------------------------------------------------------------------*/
    /* Write message source and header byte, opcode byte, and parameter   */
    /* byte to host mail box. Clear empty flag and set (if necessary)     */
    /* full flag.                                                         */
    /*--------------------------------------------------------------------*/

    *host_write_pointer++ = source;    /* Write message source and header to host mail box at current write address. */
    *host_write_pointer++ = opcode;    /* Write message opcode to host mail box at current write address. */
    *host_write_pointer++ = parameter; /* Write message parameter to host mail box at current write address. */
    *host_write_pointer = checksum;    /* Write message checksum to host mail box at current write address. */
    fifo_counter += 4;                 /* Increment fifo counter by 4 - the length of a message */

    bp_mail_flags.efh = OFF; /* Clear empty flag */

    /*--------------------------------------------------------------------*/
    /* Wrap arround write pointer.                                        */
    /*--------------------------------------------------------------------*/

    if (host_write_pointer == (&host_mail_box[HOST_MAIL_BOX_LENGTH-1])) /* Wrap around write pointer */
        host_write_pointer = &host_mail_box[0]; /* Set write pointer to first address of array. */
    else host_write_pointer++;

    /*--------------------------------------------------------------------*/
    /* Set full flag only if there is not enough free space for two       */
    /* (8 byte BP_S_32BIT) messages.                                            */
    /*--------------------------------------------------------------------*/

    if (fifo_counter > (HOST_MAIL_BOX_LENGTH-8))
        bp_mail_flags.ffh = ON; /* Set host mail box full flag ON */


} /* END _WriteHostMailBox() */


/***********************************************************/
/*    _ReadHostMailBox()                                   */
/*    Read byte from host mail box.                        */
/*                                                         */
/*    Returns: "First Out" byte in the host mail box.      */
/*                                                         */
/*    Input Variables: none.                               */
/*                                                         */
/*    Output Variables: BP_U_8BIT temp.                    */
/*                                                         */
/*    Example:                                             */
/*           SBUF = _ReadHostMailBox()                     */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                06-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _ReadHostMailBox (void)
{
    BP_U_8BIT temp;

    /*-------------------------------------------------------------*/
    /* Return value at the current read address of host mail box.  */
    /* Update host mail box full flag and empty flag.              */
    /*-------------------------------------------------------------*/

    temp = *host_read_pointer; /* Read byte from host mail box to temporary buffer */
    fifo_counter--;            /* Decrement fifo counter */

    /*----------------------------------------------------------------------*/
    /* Wrap arround read pointer.                                            */
    /*----------------------------------------------------------------------*/

    if (host_read_pointer == (&host_mail_box[HOST_MAIL_BOX_LENGTH-1])) /* Wrap around read pointer */
        host_read_pointer = &host_mail_box[0]; /* Set read pointer to first address of array. */
    else host_read_pointer++;

    /*----------------------------------------------------------------------*/
    /* If the write pointer equals the read pointer => host mail box empty. */
    /* Set host mail box empty flag.                                        */
    /*----------------------------------------------------------------------*/

    if (host_read_pointer == host_write_pointer) /* Host mail box empty */
        bp_mail_flags.efh = ON; /* Set host mail box empty flag ON */

    /*------------------------------------------------------------------*/
    /* Clear full flag only if there is enough free space for two       */
    /* (8 byte BP_S_32BIT) messages.                                          */
    /*------------------------------------------------------------------*/

    if (bp_mail_flags.ffh) /* Host mail box full */
        if (fifo_counter <= (HOST_MAIL_BOX_LENGTH-8))
            bp_mail_flags.ffh = OFF; /* Clear host mail box full flag */

    return(temp); /* return the value of temporary buffer. */

} /*END _ReadHostMailBox() */

/***********************************************************/
/*    _HandleReceiveMessages()                             */
/*    Handle messages received from the host processor.    */
/*    Message is read from the message buffer.             */
/*    Control commands are executed using _BtControl().    */
/*    Status request commands are executed using           */
/*    _BtStatus().                                         */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: none.                               */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _HandleReceiveMessages()                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                31-August-1993           */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void _HandleReceiveMessages (void)
{

    BP_U_8BIT status;
    BP_U_8BIT no;
    BP_S_8BIT info;

    /*--------------------------------------------------------------------*/
    /* If there is a waiting message, and host (outgoing) mailbox is not  */
    /* full - handle the incoming message using _BtContorl() for control  */
    /* commands and _BtStatus() for status request commands.              */
    /*--------------------------------------------------------------------*/

    if (!bp_mail_flags.message_flag) /* No messages */
        return;

    if (bp_mail_flags.ffh) /* Host (outgoing) mail box full, can't handle the message
                because will not be able to acknowldege it. */
        {
        bp_mail_flags.message_flag = OFF; /* Clear message */
        return;
        } /* END-IF host mail box is full */

    no = DEST(_message_buffer[0]); /* Find destination of message */

    if (INFO_REQUEST(_message_buffer[1])) /* Status request command */
        {
        status = _BtStatus(no, _message_buffer[1], _message_buffer[2], &info); /* Read status information */
        if (status == _PASS) /* Information valid */
            {
            _WriteHostMailBox(ACK0, ACK1, ACK2, ACK3); /* Write acknowledge message to host mail box */
            _WriteHostMailBox((no | HEADER), _message_buffer[1], info,CHECKSUM((no | HEADER), _message_buffer[1], info)); /* Write status information to host mail box */
            } /* END-IF Information valid */
        } /* END-IF Status request command */
    else /* Control command */
        {
        status = _BtControl(no, _message_buffer[1], _message_buffer[2]);
        if (status == _PASS)
            _WriteHostMailBox(ACK0, ACK1, ACK2, ACK3); /* Write acknowledge message to host mail box */
        } /* END-ELSE control command */

    bp_mail_flags.message_flag = OFF; /* Set message flag OFF */

} /* END _HandleReceiveMessages() */

/***********************************************************/
/*    _HandleTransmitMessages()                            */
/*    Handle transmitted messages.                         */
/*    Handle messages transmitted from the host mail box   */
/*    to the host processor.                               */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: none.                               */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _HandleTransmitMessages()                     */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                06-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void _HandleTransmitMessages (void)
{
    /*--------------------------------------------------------------------*/
    /* Transmit host mail box                                             */
    /*--------------------------------------------------------------------*/
    /* Empty host mail box to SBUF, the transmit buffer. */
    while (!bp_mail_flags.efh)  /* Host mail box is not empty */
        {
        while (!bp_mail_flags.tx_flag);
        bp_mail_flags.tx_flag = OFF; /* Set transmit flag OFF*/
        SBUF = _ReadHostMailBox(); /* Read next byte from host mail box to SBUF. */
        }
} /* END _HandleTransmitMessages() */

#endif
