/***********************************************************/
/*    BITPUMP.C                                            */
/*    Bit pump operations.                                 */
/*    (C) Copyright 1993 by Rockwell Corporation           */
/*                                                         */
/*    This program is copyrighted by Rockwell Corporation  */
/*                                                         */
/* Description:                                            */
/*    Handle bit_pump interrupts status.                   */
/*                                                         */
/* Notes:                                                  */
/*                                                         */
/* User Modifiable Code:                                   */
/*    None                                                 */
/*                                                         */
/* List of functions included in this module:              */
/*   _HandleFlags()                                        */
/*   _HandleTempEnv()                                      */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             March - 1998             */
/*     Iris Shuker                31-August-1993           */
/*                                                         */
/* Revision History:                                       */
/*     date/name of reviser                                */
/*                                                         */
/***********************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define REBUILD_DAGC_TH         450
#define REBUILD_FFE_TH          140

#define CALC_LOST_TIME          ((COMP_SYM_RATE()<<2)/10*((BP_U_16BIT)t_los))

/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

/***********************************************************/
/*    _HandleFlags()                                       */
/*    Check interrupt flags status (set by interrupt       */
/*    routine) and act accordingly.                        */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output Variables: None.                              */
/*                                                         */
/* Example:                                                */
/*     _Handle_Flags(no);                                  */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                14-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void _HandleFlags (BP_U_8BIT no)
{

    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    status_reg_type status_reg;
    BP_U_8BIT stage, t_los;
    BP_S_8BIT temp;

    /*------------------------------------------------------------------*/
    /* Initiate bitpump pointers                                        */
    /*------------------------------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;


    RD_BYTE(no, STATUS, status_reg.status);
    RD_WORD(no, STAGE, stage, temp);

    /*------------------------------------------------------------------*/
    /* Handle LOS condition.                                            */
    /*------------------------------------------------------------------*/
    if (int_ptr -> bits.irq_source.low_felm) /* Low FELM interrupt flag ON */
        {
        int_ptr -> bits.irq_source.low_felm = OFF; /* Clear low felm interrupt flag */
        status_reg.bits.los = ON; /* Set status LOS */
#ifdef TDEBUG           
        printf(" LOS ON\n");
#endif
        BP_WRITE_BIT(bp_mode_ptr, irq_source, high_felm, OFF); /* Clear high FELM interrupt status bit */
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, OFF); /* Remove high FELM interrupt mask */
        } /* END-IF Low FELM interrupt flag ON */

    /*------------------------------------------------------------------*/
    /* Handle LOS condition at deactivate mode, operate lost timer.     */
    /*------------------------------------------------------------------*/
    if ((stage == DEACTIVATE_MODE) && status_reg.bits.los && !status_reg.bits.run_lost_timer)
        {
        /* Use SUT3, as timeout timer for LOST time period */
        status_reg.bits.run_lost_timer = ON;
        RD_WORD(no, PARAMETERS, temp, t_los);
        SET_WORD(bp_ptr, sut3_low, sut3_high, CALC_LOST_TIME);
        BP_WRITE_BIT(bp_mode_ptr, timer_enable, sut3, ON);
        status_reg.bits.lost = OFF;
        RESTART(bp_mode_ptr, sut3);
#ifdef INT_BUG
        _RestartVirtualTimer(no, _SUT3, CALC_LOST_TIME);
#endif
#ifdef TDEBUG   
        printf(" Activate LOST timer\n");
#endif
        } /* END-IF LOS condition at deactivate_mode stage */

    /*-------------------------------------------------------------------*/
    /* Signal is back.                                                   */
    /*-------------------------------------------------------------------*/
    if (int_ptr -> bits.irq_source.high_felm) /* High FELM interrupt flag ON */
        {
        int_ptr -> bits.irq_source.high_felm = OFF; /* Clear high felm interrupt flag */
        status_reg.bits.los = OFF; /* Set status LOS */
        status_reg.bits.lost = OFF; /* Set status LOST */
#ifdef TDEBUG           
        printf(" LOS OFF\n");
#endif
        BP_WRITE_BIT(bp_mode_ptr, irq_source, low_felm, OFF); /* Clear low FELM interrupt status bit */
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, OFF); /* Remove low FELM interrupt mask */
        if (status_reg.bits.run_lost_timer)  /* Lost timer is running */
            {
            status_reg.bits.run_lost_timer = OFF;
            BP_WRITE_BIT(bp_mode_ptr, timer_enable, sut3, OFF); /* Disable sut3 */
#ifdef INT_BUG
            _ResetVirtualTimer(no, _SUT3);
#endif
            int_ptr -> bits.timer_source.sut3 = OFF; /* Clear sut3 interrupt flag */
#ifdef TDEBUG   
            printf(" Disable LOST timer\n");
#endif
            } /* END-IF LOst timer is running */
        } /* END-IF High FELM interrupt flag ON */

    /*-------------------------------------------------------------------*/
    /* If LOST timer expired - set LOST status bit.                      */
    /*-------------------------------------------------------------------*/
    if (status_reg.bits.run_lost_timer)
        if (int_ptr -> bits.timer_source.sut3) /* Sut3 interrupt flag ON */
            {
            int_ptr -> bits.timer_source.sut3 = OFF; /* Clear sut3 interrupt flag */
            status_reg.bits.lost = ON; /* Set status LOST */
#ifdef TDEBUG   
            printf(" LOST ON\n");
#endif
            /* must preserve upper byte (Test Mode/Normal Op. Stage # */
            RD_WORD(no, STAGE, stage, temp);
            WR_WORD(no, STAGE, IDLE, temp); /* Move to idle STAGE */
            } /* END-IF sut3 interrupt flag ON */

    /*-----------------------------------------------------------------------*/
    /* If startup timer4 expired, set activation interval status bit.        */
    /*-----------------------------------------------------------------------*/
    if (int_ptr -> bits.timer_source.sut4) /* Startup timer1 interrupt flag ON, startup interval (30sec) ended. */
        {
        status_reg.bits.activation_interval = ON; /* Set status startup time limit exceded */
        int_ptr -> bits.timer_source.sut4 = OFF;
        } /* END-IF sut4 interrupt flag ON */


    WR_BYTE(no, STATUS, status_reg.status);

} /* END _HandleFlags() */


/***********************************************************/
/*    _HandleTempEnv()                                     */
/*    During bit-pump normal operation, periodically       */
/*    adapt the equalizers to handle any temperature/      */
/*    environmental changes.                               */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output Variables: None.                              */
/*                                                         */
/* Example:                                                */
/*     _HandleTempEnv(no);                                 */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             18-Sept-1997             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/
#ifdef TEMP_ENV
void _HandleTempEnv (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_INT_PTR;

    user_setup_low_type user_setup_low;
    BP_U_8BIT rebuild=0;
    BP_S_8BIT temp, temp1;
    BP_S_8BIT te_stage, meter_stage;
    BP_S_16BIT value, new_value;
#ifdef HTUC
    BP_U_8BIT i;
#endif

    /*------------------------------------------------------------------*/
    /* Initiate bitpump pointers                                        */
    /*------------------------------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;
    INIT_INT_PTR;

    /*-----------------------------------------------------------------------*/
    /* Adapt DAGC & FFE                                                      */
    /*-----------------------------------------------------------------------*/

    RD_WORD(no, STAGE2, te_stage, meter_stage);

    switch ( te_stage )
        {
        case TEMP_ENV_IDLE:
            break;

        case TE_WAIT_FOR_METER:
            TIMER_BREAK(meter);

            /*
             * Raise a flag to indicate to higher layers that a
             * meter interval has elapsed.  This can then be used to
             * trigger some higher level event, i.e. Auto Baud Line Quality
             * polling.
             */
            _bp_vars[no].bp_flags.bits.meter_interval_elapsed = 1;


            /*
             * Wait for multiple meter intervals before processing
             * temp/environment handler.  This will ease the bit-pump
             * microprocessor demands while making the bit-pump more
             * sensitive to temp/environments.
             */
            _bp_vars[no].te_meter_interval_cntr++;
            if ( _bp_vars[no].te_meter_interval_cntr < _bp_vars[no].te_num_meter_intervals )
                {
                break;
                }
            _bp_vars[no].te_meter_interval_cntr = 0;

            _bp_vars[no].bp_flags.bits.te_meter_interval_elapsed = 1;

            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read HTU-C/R configuration */

            RD_WORD(no, DAGC_HIGH, temp, temp1); /* Read DAGC High word value */
            WR_WORD(no, DAGC_VALUE, temp, temp1);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            if (user_setup_low.bits.terminal_flag == _HTUC)
                {
                BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */
                _bp_vars[no].bp_flags.bits.temp_env_adapt = TEMP_ENV_ADAPT_DAGC_DFE;
                }
            else
                {
                _bp_vars[no].bp_flags.bits.temp_env_adapt = TEMP_ENV_ADAPT_DAGC;
                }

            INIT_TIMER(t3, _T3, t3_low, t3_high, 500);

            te_stage = TE_HTUC_ADAPT_FFE;
            break;

        case TE_HTUC_ADAPT_FFE:
            TIMER_BREAK(t3);

            RD_WORD(no, USER_SETUP, user_setup_low.setup, temp); /* Read HTU-C/R configuration */

            /* Recall Previous DAGC Value, before adapting */
            RD_WORD(no, DAGC_VALUE, temp, temp1);
            value = BYTE2WORD(temp1, temp);

            RD_WORD(no, DAGC_HIGH, temp, temp1); /* Read DAGC new value */
            new_value = BYTE2WORD(temp1, temp);

            if ((abs(new_value - value)) > REBUILD_DAGC_TH)
                {
                WR_WORD(no, DAGC_HIGH, LOW(value), HIGH(value));
                }

#ifdef HTUR
            if ( user_setup_low.bits.terminal_flag == _HTUR )
                {
#ifdef INT_BUG
                _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << NORMAL_METER));
#endif
                te_stage = TE_WAIT_FOR_METER;
                break;
                }
#endif /* HTUR */

#ifndef HTUC
            te_stage = TE_WAIT_FOR_METER;
            break;
#endif

#ifdef HTUC
            /*-------------------------------------------------------*/
            /* HTU-C Only, adapt FFE                                 */
            /* First store away FFE Coefs, then after adapting       */
            /* check to see if FFE Coefs changed drastically.  If so */
            /* assume a micro-interruption on the line occured       */
            /* then recall previous known good FFE Coef valus.       */
            /*-------------------------------------------------------*/

            for ( i = 0 ; i < 8 ; i++ )
                {
                RD_WORD(no, i, temp, temp1);
                WR_FFE_COEFF(no, i, temp, temp1);
                }

            _bp_vars[no].bp_flags.bits.temp_env_adapt = TEMP_ENV_ADAPT_FFE;

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

            INIT_TIMER(t3, _T3, t3_low, t3_high, 500);

            te_stage = TE_HTUC_REBUILD;
            break;

        case TE_HTUC_REBUILD:
            TIMER_BREAK(t3);

            /* Read FFE Coefficient #5 (main tap) */
            value = RD_FFE_COEFF(no, 5);
            RD_WORD(no, 5, temp, temp1);
            new_value = BYTE2WORD(temp1, temp);

            if ((abs(new_value - value)) < REBUILD_FFE_TH)
                {
                /* Read FFE Coefficient #6 */
                value = RD_FFE_COEFF(no, 6);
                RD_WORD(no, 6, temp, temp1);
                new_value = BYTE2WORD(temp1, temp);

                if ((abs(new_value - value)) < REBUILD_FFE_TH)
                    {
                    /* Read FFE Coefficient #4 */
                    value = RD_FFE_COEFF(no, 4);
                    RD_WORD(no, 4, temp, temp1);
                    new_value = BYTE2WORD(temp1, temp);
                    if ((abs(new_value - value)) >= REBUILD_FFE_TH)
                        {
                        rebuild = ON;
                        }
                    }
                else
                    {
                    rebuild = ON;
                    }
                }
            else
                {
                rebuild = ON;
                }

            if (rebuild == ON)
                {
                /*
                 * Reload previous good FFE Coef values
                 */
                for ( i = 0 ; i < 8 ; i++ )
                    {
                    value = RD_FFE_COEFF(no, i);
                    WR_WORD(no, i, LOW(value), HIGH(value));
                    }

                }  /* end if rebuild */

#ifdef INT_BUG
            _RestartVirtualTimer(no, _METER, (DEFAULT_METER_VALUE << NORMAL_METER));
#endif

            te_stage = TE_WAIT_FOR_METER;
            break;
#endif  /* HTUC */

        }   /* end switch te_stage */

    WR_WORD(no, STAGE2, te_stage, meter_stage);


   return ;
}

/***********************************************************/
/*    _GetMeterIntervalElapsed                             */
/*    This function returns whether or not a meter interval*/
/*    has elapsed.  The meter interval can be used to      */
/*    trigger some higher layer event, i.e. Auto Baud      */
/*                                                         */
/*    Returns: 0 - not elapsed                             */
/*             1 - elapsed                                 */
/*                                                         */
/*    Input Variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output Variables: None.                              */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             15-March-1999            */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _GetMeterIntervalElapsed (BP_U_8BIT no)
{
    if ( _bp_vars[no].bp_flags.bits.meter_interval_elapsed )
        {

        _bp_vars[no].bp_flags.bits.meter_interval_elapsed = 0;
        return 1;
        }
    else
        {
        return 0;
        }
}

/***********************************************************/
/*    _GetTeMeterIntervalElapsed                           */
/*    This function returns whether or not a temp/evn      */
/*    meter interval has elapsed.  The meter interval can  */
/*    be used to trigger some higher layer event.          */
/*                                                         */
/*    This TeMeterIntervalElapsed flag is different        */
/*    than the MeterInterval Elapsed flag.  The TE         */
/*    triggers when multiple meter intervals have elapsed  */
/*    as determined by the _TE_METER_INTERVALS API command.*/
/*                                                         */
/*    Returns: 0 - not elapsed                             */
/*             1 - elapsed                                 */
/*                                                         */
/*    Input Variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output Variables: None.                              */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             15-March-1999            */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _GetTeMeterIntervalElapsed (BP_U_8BIT no)
{
    if ( _bp_vars[no].bp_flags.bits.te_meter_interval_elapsed )
        {

        _bp_vars[no].bp_flags.bits.te_meter_interval_elapsed = 0;
        return 1;
        }
    else
        {
        return 0;
        }
}


#endif /* TEMP_ENV */


