/************************************************************/
/*    API.C                                                 */
/*    Application Programmer Interface.                     */
/*    (C) Copyright 1993 by Rockwell Corporation            */
/*                                                          */
/*    This program is copyrighted by Rockwell Corporation   */
/*                                                          */
/* Description:                                             */
/*    This file contains code for implementing the Bitpump  */
/*    software Application Programmer Interface (API).      */
/*                                                          */
/* Notes:                                                   */
/*                                                          */
/* User Modifiable Code:                                    */
/*    Channel Unit API Hooks                                */
/*                                                          */
/* List of functions included in this module:               */
/*   _BtControl()                                           */
/*   _BtStatus()                                            */
/*                                                          */
/* Programmer:                                              */
/*     Iris Shuker                21-Oct-1993               */
/*     Dean Rasmussen             March - 1998              */
/*                                                          */
/* Revision History:                                        */
/*                                                          */
/************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "bthomer.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define SW_VERSION     13
#define MAJOR_VERSION  4
#define MINOR_VERSION  4

/* Far-End in 1dB increments for V4.2+, V4.1- was 0.5dB increments */
#define FAR_END_SIGNAL_DB(x)     (x)

#define DC_TAP_BT8970             119
#define DC_TAP_BT8960             60

#define NOISE_MARGIN_TH (BP_S_8BIT)-10

#define SYMBOL_RATE_DEFAULT_VALUE   (BP_U_16BIT)98


#define MINIMUM_BT8973_SYM_RATE 18  /* 144kbps */

/*---------------------------------------------------------*/
/*  Local Macros                                           */
/*---------------------------------------------------------*/


/* Global modes registers is active low */
#define _BITPUMP_POWER_UP     0
#define _BITPUMP_POWER_DOWN   1


/*---------------------------------------------------------*/
/*  Static Functions                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Variables                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Static Variables                                       */
/*---------------------------------------------------------*/

/*---------------------------------------------------------*/
/*  Global Functions                                       */
/*---------------------------------------------------------*/



/**************************************************************/
/*    _BitpumpControl()                                       */
/*    Execute control command.                                */
/*                                                            */
/*    Returns: _PASS/_FAIL status indication:                 */
/*          _PASS - Command successfuly interpreted, and will */
/*                 be executed.                               */
/*          _FAIL - Illegal command, command not executed.    */
/*                 Cause may be illegal control command       */
/*                 opcode, destination, or parameter.         */
/*                                                            */
/*    Input Variables:                                        */
/*        BP_U_8BIT no - which bit_pump                       */
/*        BP_U_8BIT opcode                                    */
/*        BP_U_8BIT parameter                                 */
/*                                                            */
/*    Output Variables:                                       */
/*        BP_U_8BIT status                                    */
/*                                                            */
/*    Example:                                                */
/*    status = _BitpumpControl(_BIT_PUMP0, _TERMINAL_TYPE,    */
/*                                                    _HTUR)  */
/*                                                            */
/* Programmer:                                                */
/*     Iris Shuker                21-Oct-1993                 */
/*                                                            */
/* Revision History:                                          */
/*                                                            */
/**************************************************************/
BP_U_8BIT _BitpumpControl (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter)
{
    user_setup_low_type user_setup_low;
    user_setup_high_type user_setup_high;
    user_param_low_type user_param_low;
    status_reg_type status_reg;
    BP_U_8BIT temp;
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    if ( no >= _NO_OF_LOOPS )
        {
        return _FAIL;
        }

    /*--------------------------------------------------------------------*/
    /* Initiate bit_pump pointers.                                        */
    /*--------------------------------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    /*-------------------------------------------------------------------*/
    /* Handle _SYSTEM_CONFIG opcode.                                     */
    /*-------------------------------------------------------------------*/

    if ( opcode == _SYSTEM_CONFIG )
        {
        if ( parameter == _NOT_PRESENT )
            {
            if ( GET_BITPUMP_TYPE() == BT8973 )
                {
                /* For Bt8973, set clock to lowest value */
                _BitpumpSetSymbolRate(no, MINIMUM_BT8973_SYM_RATE);
                }

            /* power down AFE section - middle 2 bits of 4-bit reserve field*/
            BP_WRITE_BIT(bp_mode_ptr, misc_test, reserved2, 0x6);

            BP_WRITE_REG(bp_ptr, mask_low_reg, 0xFF);
            BP_WRITE_REG(bp_ptr, mask_high_reg, 0xFF);

            _bp_vars[no].bp_flags.bits.operational = _NOT_PRESENT;
            BP_WRITE_BIT(bp_mode_ptr, global_modes, mode, _BITPUMP_POWER_DOWN);

            }
        else if (parameter == _PRESENT) /* Operate bit_pump */
            {
            _bp_vars[no].bp_flags.bits.operational = _PRESENT;
            BP_WRITE_BIT(bp_mode_ptr, global_modes, mode, _BITPUMP_POWER_UP);

            /* power up AFE section - middle 2 bits of 4-bit reserve field*/
            BP_WRITE_BIT(bp_mode_ptr, misc_test, reserved2, 0x0);

            _bp_vars[no].symbol_rate = SYMBOL_RATE_DEFAULT_VALUE;
            _BtReset(no);
            }
        else 
            {
            return _FAIL;
            }

        return (_PASS);
        } /* END-IF set system configuration */

    /*-------------------------------------------------------------------*/
    /* Verify correct opcodes and parameters. IF not - return FAIL       */
    /* status, otherwise execute the command.                            */
    /*-------------------------------------------------------------------*/
    if (!_bp_vars[no].bp_flags.bits.operational) /* Bit pump not operational */
        return(_FAIL);

    RD_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Read user setup */

    /*-------------------------------------------------------------------*/
    /* Identify the received command opcode and execute the command.     */
    /*-------------------------------------------------------------------*/

    switch(opcode) /* Identify the command opcode */
        {

        case _TERMINAL_TYPE: /* Set terminal type */

            /*-----------------------------------------------------------*/
            /* If the message parameter is 1 the terminal type is HTU_R, */
            /* 0 - the terminal type is HTU_C                            */
            /*-----------------------------------------------------------*/

            if (parameter == _HTUR)
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, htur_lfsr, _HTUR); /* Transmitter HTU_R lfsr */
                BP_WRITE_BIT(bp_mode_ptr, detector_modes, htur_lfsr, _HTUR); /* Detector HTU_R lfsr */
                user_setup_low.bits.terminal_flag = _HTUR; /* Terminal flag ON */
                } /* END-IF message parameter is _HTUR */
            else if (parameter == _HTUC)
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, htur_lfsr, _HTUC); /* Transmitter HTU_C lfsr */
                BP_WRITE_BIT(bp_mode_ptr, detector_modes, htur_lfsr, _HTUC); /* Detector HTU_C lfsr */
                user_setup_low.bits.terminal_flag = _HTUC; /* Terminal flag OFF */
                } /* END-IF message parameter is _HTUC */
                else return (_FAIL); /* unrecognized parameter */
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _ANALOG_AGC_CONFIG: /* Set analog agc configuration */
            /*
             * Do nothing, always uses 6-Level AGC
             * (kept hook to maintain backwards compatibility)
             */
            break;

        case _STARTUP_SEQ_SOURCE: /* Set startup sequence source */

            /*---------------------------------------------------------------*/
            /* Set startup sequence source to be either external or internal.*/
            /*---------------------------------------------------------------*/

            if (parameter == _INTERNAL)
                user_setup_low.bits.seq_source = ON; /* User setup sequence source bit */
            else if (parameter == _EXTERNAL)
                user_setup_low.bits.seq_source = OFF; /* User setup sequence source bit */
                else return (_FAIL); /* unrecognized parameter */
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _TRANSMIT_SCR: /* Transmitter scrambler */

            /*----------------------------------------------------------*/
            /* Activate/Bypass transmitter scrambler. Has effect only   */
            /* during normal operation.                                 */
            /*----------------------------------------------------------*/

            if (parameter == _ACTIVATE_SCR) /* Scramble transmitted symbols */
                {   
                user_setup_low.bits.tx_scr = ON; /* User setup tx scrambler bit */

                if ( BP_READ_BIT(bp_mode_ptr, transmitter_modes, data_source) == UNSCRAMBLED_FOUR_LEVEL_DATA )
                    {
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA);
                    }
                } /* END-IF scramble transmitted symbols */
            else if (parameter == _BYPASS) /* No scrambling */
                {
                if ( BP_READ_BIT(bp_mode_ptr, transmitter_modes, data_source) == SCRAMBLED_FOUR_LEVEL_DATA)
                    {
                    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA);
                    }
                user_setup_low.bits.tx_scr = OFF; /* User setup tx scrambler bit */
                } /* END-ELSE no scrambling */
                else return (_FAIL); /* Unrecognized parameter */
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _RECEIVE_DESCR: /* Receiver Descrambler */

            /*----------------------------------------------------------*/
            /* Activate/Bypass Receiver Descrambler. Has effect only    */
            /* during normal operation.                                 */
            /*----------------------------------------------------------*/

            if (parameter == _ACTIVATE_DESCR) /* Descramble receive symbols */
                {
                user_setup_low.bits.rx_descr = ON; /* User setup rx descrambler bit */
                BP_WRITE_BIT(bp_mode_ptr, detector_modes, descr_on, ON);
                BP_WRITE_BIT(bp_mode_ptr, detector_modes, output_mux_control, SELECT_DESCRAMBLER_OUTPUT);
                } /* END-IF descramble receive symbols */
            else if (parameter == _BYPASS) /* No descrambling */
                {
                user_setup_low.bits.rx_descr = OFF; /* User setup rx descrambler bit */
                BP_WRITE_BIT(bp_mode_ptr, detector_modes, descr_on, OFF);
                BP_WRITE_BIT(bp_mode_ptr, detector_modes, output_mux_control, SELECT_DETECTOR_OUTPUT);
                } /* END-IF no descrambling */
                else return (_FAIL); /* unrecognized parameter */
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _FRAMER_FORMAT: /* Data transfer format */

            /*---------------------------------------------------------*/
            /* Select Rx/Tx data transfer format: parallel with clock  */
            /* inputs/outputs or serial.                               */
            /*---------------------------------------------------------*/

            switch (parameter) /* Check message parameter */
                {
                case _PARALLEL_MASTER:
                    BP_WRITE_BIT(bp_mode_ptr, cu_interface_modes, interface_mode, _PARALLEL_MASTER); /* Channel Unit parallel interface with clock outputs */
                    break;

                case _PARALLEL_SLAVE:
                    BP_WRITE_BIT(bp_mode_ptr, cu_interface_modes, interface_mode, _PARALLEL_SLAVE); /* Channel Unit parallel interface with clock inputs */
#ifdef REPEATER
                    /* The Rs8953B work around for repeater requires the polarity be changed */
                    BP_WRITE_BIT(bp_mode_ptr, cu_interface_modes, tbclk_pol, _FALLING_EDGE); /* Channel Unit parallel interface with clock inputs */
                    BP_WRITE_BIT(bp_mode_ptr, cu_interface_modes, rbclk_pol, _FALLING_EDGE); /* Channel Unit parallel interface with clock inputs */
#endif /* REPEATER */
                    SET_RESET(bp_mode_ptr, cu_interface_modes, fifos_mode);
                    break;

                case _SERIAL:
                    BP_WRITE_BIT(bp_mode_ptr, cu_interface_modes, interface_mode, _SERIAL); /* Channel Unit serial interface with clock outputs */
                    break;

                case _SERIAL_SWAP:
                    /* ------------------------------------------------- */
                    /* Channel Unit serial interface with clock outputs. */
                    /* Magnitude & Sign bits swapped.                    */
                    /* ------------------------------------------------- */
                    BP_WRITE_BIT(bp_mode_ptr, cu_interface_modes, interface_mode, _SERIAL_SWAP);
                    break;

                default:
                    return (_FAIL); /* unrecognized parameter */

                } /* END-SWITCH  message parameter */
            user_setup_high.bits.framer_format = parameter;
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _BT_OTHER_SIDE: /* Set if the remote is Bitpump */

            if (parameter == _BT)
                user_setup_low.bits.bt_other_side = ON; /* User setup bt_other_side bit */
            else if (parameter == _NO_BT)
               user_setup_low.bits.bt_other_side = OFF; /* User setup bt_other_side bit */
            else return (_FAIL); /* unrecognized parameter */
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _SYM_RATE_HI:
            _bp_vars[no].symbol_rate = (parameter & 0x03) << 8;
            break;

        case _SYM_RATE: /* Set symbol rate */
            _bp_vars[no].symbol_rate |= parameter;  /* append to Sym Rate High */
            _BtReset(no);
            break;

        case _LOST_TIME_PERIOD: /* Set LOST time period */
            RD_WORD(no, PARAMETERS, user_param_low.reg, temp);
            WR_WORD(no, PARAMETERS, user_param_low.reg, parameter);
            break;

        case _TRANSMIT_EXT_DATA: /* Start transmitting external data */
            if (user_setup_low.bits.tx_scr) /* Scramble transmitted symbols */
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA);
            else /* No scrambling */
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA);
        break;

        case _RESET_SYSTEM: /* RESET Bitpump */
            BP_WRITE_BIT(bp_mode_ptr, global_modes, mode, _BITPUMP_POWER_UP);
            _bp_vars[no].symbol_rate = SYMBOL_RATE_DEFAULT_VALUE;
            _BtReset(no);
            break;

        case _ACTIVATE: /* Begin startup */

            WR_BYTE(no, STATUS, INIT_STATUS_REG); /* Reset Normal Op flag */
            WR_WORD(no, STAGE, ACTIVATE_SYSTEM, TEST_MODE_IDLE);
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, 0);

            ZIP_START_MODE = OFF;

#ifdef ZIP_START
            if ( parameter == _ACTIVATE_ZIP && zip_start_global_var[no].zip_start_enabled )
                {
                ZIP_START_MODE = ON;
                }
#endif

            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _DEACTIVATE: /* Deactivate bit-pump */
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze linear EC output */

            if ( GET_BITPUMP_TYPE() == BT8960 )
                {
                _ReadAccessByteRAM(no, LEC_ACCESS_RAM, DC_TAP_BT8960);
                }
            else
                {
                _ReadAccessByteRAM(no, LEC_ACCESS_RAM, DC_TAP_BT8970);
                }

            SET_RESET(bp_mode_ptr, linear_ec_modes, zero_coefficients); /* Zero linear EC output */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero non linear EC output */

            BP_WRITE_REG(bp_ptr, access_data_byte0, _bp_vars[no].access_data_byte[0]);
            BP_WRITE_REG(bp_ptr, access_data_byte1, _bp_vars[no].access_data_byte[1]);
            BP_WRITE_REG(bp_ptr, access_data_byte2, _bp_vars[no].access_data_byte[2]);
            BP_WRITE_REG(bp_ptr, access_data_byte3, _bp_vars[no].access_data_byte[3]);

            if ( GET_BITPUMP_TYPE() == BT8960 )
                {
                BP_WRITE_REG(bp_ptr, linear_ec_tap_select_write, DC_TAP_BT8960);
                }
            else
                {
                BP_WRITE_REG(bp_ptr, linear_ec_tap_select_write, DC_TAP_BT8970);
                }

            DELAY2SYMBOL; 
    
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, ON); /* Set transmitter OFF */

            RD_BYTE(no, STATUS, status_reg.status);
            status_reg.bits.four_level_indication = OFF;
            status_reg.bits.normal_operation = OFF;
            status_reg.bits.tip_ring_indication = 0;
            WR_BYTE(no, STATUS, status_reg.status);

            BP_WRITE_BIT(bp_mode_ptr, irq_source, low_felm, OFF); /* Clear low FELM interrupt status bit */
            BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, OFF); /* Remove low FELM interrupt mask */

            WR_WORD(no, STAGE, DEACTIVATE_MODE, TEST_MODE_IDLE);
            WR_WORD(no, STAGE2, TEMP_ENV_IDLE, 0);

#ifdef BER_METER            
            _bp_vars[no].bp_flags.bits.ber_meter_state = 0;
#endif
            break;

        case _TEST_MODE: /* Activate bit-pump test mode */
            temp = _TestMode(no, parameter); /* Operate test mode */
            if  (temp == _FAIL)
                return(_FAIL);
            break;

        case _OPERATE_NLEC: /* Operate Non Linear EC */
            RD_WORD(no, PARAMETERS, user_param_low.reg, temp);

            if (parameter == _NLEC_ON) 
                {
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, OFF); 
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN);
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */
                user_param_low.bits.operate_nlec = 1;
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NL EC */
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */
                user_param_low.bits.operate_nlec = 0;
                }

            WR_WORD(no, PARAMETERS, user_param_low.reg, temp);
            break;
            
        case _WRITE_TX_GAIN:
            BP_WRITE_BIT(bp_mode_ptr, tx_gain, tx_gain, parameter);
            break;

        case _REVERSE_TIP_RING:
            RD_BYTE(no, STATUS, status_reg.status);
            if ( parameter )
                {
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, negate_symbol, TIP_RING_REVERSE);
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, negate_symbol, TIP_RING_NORMAL);
                }
            status_reg.bits.tip_ring_indication = BP_READ_BIT(bp_mode_ptr, nonlinear_ec_modes, negate_symbol);
            WR_BYTE(no, STATUS, status_reg.status);
            break;

        case _BER_METER_START:
#ifdef BER_METER
            RD_BYTE(no, STATUS, status_reg.status);

            /* Only operational during normal operation */
            if ( !status_reg.bits.normal_operation )
                {
                break;
                }
    
            _bp_vars[no].bp_flags.bits.ber_meter_state = 1;

            _bp_vars[no].ber_meter.bit_errors = 0;
            _bp_vars[no].ber_meter.meter_intervals = 0;

            /* Turn on 4-Level Internal Scrambled One's */
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
#endif
            break;

        case _BER_METER_STOP:
#ifdef BER_METER
            _bp_vars[no].bp_flags.bits.ber_meter_state = 0;

            /* Turn on 4-Level (Un)Scrambled external data */
            if (user_setup_low.bits.tx_scr) /* Scramble transmitted symbols */
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_DATA); /* External (scrambled) data */
            else /* No scrambling */
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, UNSCRAMBLED_FOUR_LEVEL_DATA); /* External (unscrambled) data */

#endif
            break;

        case _AUTO_TIP_RING:
            if ( parameter )
                {
                user_setup_low.bits.auto_tip_ring = AUTO_TIP_RING_ON;
                }
            else
                {
                user_setup_low.bits.auto_tip_ring = AUTO_TIP_RING_OFF;
                }
            WR_WORD(no, USER_SETUP, user_setup_low.setup, user_setup_high.setup); /* Write user setup */
            break;

        case _ERLE_TEST_MODE:
#ifdef ERLE
            /* user_setup_high.setup is used a temporary variable */
            RD_WORD(no, ERLE_SETUP, temp, user_setup_high.setup);
            WR_WORD(no, ERLE_SETUP, temp, parameter);

            /* _TestMode() updates USER_SETUP */
            temp = _TestMode(no, _ERLE_TEST);
#endif
            break;

#ifdef ZIP_START
        case _ZIP_START_UPDATE:
            if ( zip_start_global_var[no].zip_start_enabled )
                {
                zip_start_global_var[no].update_state = ZIP_UPDATE_PARAMS;
                }

            break;

        case _ZIP_START_CONFIG:
            if ( parameter )
                {
                zip_start_global_var[no].zip_start_enabled = 1;
                }
            else 
                {
                zip_start_global_var[no].zip_start_enabled = 0;
                }
            break;

#endif

        case _HCLK_SELECT:

            if ( parameter > _HCLK_64_TIMES )
                {
                parameter = 0;
                }
            BP_WRITE_BIT(bp_mode_ptr, serial_monitor_source, hclk_freq, parameter);

            RD_WORD(no, PARAMETERS, user_param_low.reg, temp);
            user_param_low.bits.hclk_select = parameter;
            WR_WORD(no, PARAMETERS, user_param_low.reg, temp);
            break;

        case _ACTIVATION_TIMEOUT:

            if ( parameter > _ACT_TIME_VARIABLE )
                {
                parameter = _ACT_TIME_30SEC;
                }

            RD_WORD(no, PARAMETERS, user_param_low.reg, temp);
            user_param_low.bits.activation_timeout = parameter;
            WR_WORD(no, PARAMETERS, user_param_low.reg, temp);
            break;

        case _REGENERATOR_MODE:

            temp = GET_BITPUMP_TYPE();
            if ( temp <= BT8970 )
                {
                /* not supported (necessary) by earlier bit-pump versions */
                break;
                }

            RD_WORD(no, PARAMETERS, user_param_low.reg, temp);
            if ( parameter )
                {
                BP_WRITE_BIT(bp_mode_ptr, misc_test, reg_clk_en, _REGENERATOR_ON);
                user_param_low.bits.regenerator_mode = _REGENERATOR_ON;
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, misc_test, reg_clk_en, _REGENERATOR_OFF);
                user_param_low.bits.regenerator_mode = _REGENERATOR_OFF;
                }

            WR_WORD(no, PARAMETERS, user_param_low.reg, temp);
            break;

        case _TE_METER_INTERVALS:
            _bp_vars[no].te_num_meter_intervals = parameter;
            break;

        case _ACTIVATION_TIMER:
            if ( parameter )    /* Start or Restart the timer */
                {
                _SetActivationInterval(no);
                }
            else 
                {                /* Stop the timer */
                BP_WRITE_BIT(bp_mode_ptr, mask_low_reg, sut4, OFF);
                }
            break;
   
#ifdef _CHAR_TESTING

        case _SET_IMP_SHORT:
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, imp_short, parameter);
            break;

        case _SET_CONT_TIME:
            BP_WRITE_BIT(bp_mode_ptr, adc_control, cont_time, parameter);
            break;

        case _SET_SWITCH_CAP:
            BP_WRITE_BIT(bp_mode_ptr, adc_control, switch_cap_pole, parameter);
            break;

        case _HALF_DUPLEX_MODE:
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, ON); /* Turn OFF */
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF);
            SET_RESET(bp_mode_ptr, linear_ec_modes, zero_coefficients); /* Clear linear EC coefficients */
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt! */
            break;

#endif /* _CHAR_TESTING */

        default: /* Unrecognized opcode */
            return (_FAIL);

        } /* END-SWITCH message opcode */

    return (_PASS); /* Return _PASS status to the user */

} /* END _BitpumpControl */

/***********************************************************/
/*    _BitpumpStatus()                                     */
/*    Execute status request command. Return the requested */
/*    information, and a _PASS/_FAIL indication.           */
/*                                                         */
/*    Returns: _PASS/_FAIL status indication:              */
/*          _PASS - Command successfuly interpreted, and   */
/*                 will be executed.                       */
/*          _FAIL - Illegal command, command not executed. */
/*                 Cause may be illegal status request     */
/*                 opcode, destination, or parameter.      */
/*                                                         */
/*    Input Variables:                                     */
/*        BP_U_8BIT no - which bit_pum                     */
/*        BP_U_8BIT opcode                                 */
/*        BP_U_8BIT parameter                              */
/*                                                         */
/*    output Variables:                                    */
/*      BP_U_8BIT status                                   */
/*      BP_S_8BIT *indication                              */
/*                                                         */
/*    Example:                                             */
/*    status = _BitpumpStatus(_BITPUMP0, _CONFIGURATION,   */
/*                       _BIT_RATE, &indication);          */
/*                                                         */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                21-Oct-1993              */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/
BP_U_8BIT _BitpumpStatus (BP_U_8BIT no, BP_U_8BIT opcode, BP_U_8BIT parameter, BP_S_8BIT *indication)
{
    user_setup_high_type user_setup_high;
    status_reg_type status_reg;
    BP_U_8BIT meter, temp;
    BP_S_8BIT s8_temp;
    BP_S_8BIT *ptr;
    BP_S_16BIT value;

    DECLARE_PTR;
    DECLARE_MODE_PTR;


    if ( no >= _NO_OF_LOOPS )
        {
        return _FAIL;
        }

    /*--------------------------------------------------------------------*/
    /* Init bit_pump pointers.                                            */
    /*--------------------------------------------------------------------*/
    INIT_BP_PTR;
    INIT_BP_MODE_PTR;

    /*--------------------------------------------------------------------*/
    /* Handle _BIT_PUMP_PRESENT opcode.                                   */
    /*--------------------------------------------------------------------*/
    
    if (opcode == _BIT_PUMP_PRESENT) /* Check the presence of destinated bit pump */
        {
        BP_WRITE_BIT(bp_mode_ptr, global_modes, mode, OFF); /* Set Bitpump power on */
        if (_SelfTest(no) == _PASS)
            *indication = _PRESENT;
        else *indication = _NOT_PRESENT;
        BP_WRITE_BIT(bp_mode_ptr, global_modes, mode, ON); /* Set Bitpump power down */
        return (_PASS);
        } /* END-IF checking presence of destinated bit_pump */

    /*-------------------------------------------------------------------*/
    /* Verify correct opcodes and parameters. IF not - return FAIL       */
    /* status, otherwise execute the command.                            */
    /*-------------------------------------------------------------------*/

    if (!_bp_vars[no].bp_flags.bits.operational) /* Bit_pump not operational */
        return (_FAIL);

    /*-------------------------------------------------------------------*/
    /* Identify the received command opcode and execute the command.     */
    /*-------------------------------------------------------------------*/

    switch (opcode) /* Status opcodes identification */
        {
        case _SLM: /* Request signal level information */

            NORM(meter); /* Normlize to meter timer value */
            READ_METER_REG(slm_low, slm_high, value, meter);
            *indication = HIGH(value);
            break;

        case _DC_METER: /* Request average DC level information */

            NORM(meter); /* Normlize to meter timer value */
            READ_METER_REG(dc_meter_low, dc_meter_high, value, meter);
            *indication = HIGH(value);
            break;

        case _FELM: /* Request signal attenuation information */

            NORM(meter); /* Normlize to meter timer value */
            READ_METER_REG(felm_low, felm_high, value, meter);

            value = _ScaleByGain(no, value);
            *indication = _LookUpTable((BP_TABLE *)&_far_end_signal[_bp_vars[no].felm_lookup_index], value);
            *indication = FAR_END_SIGNAL_DB(*indication);
            break;

        case _NMR: /* Request SNR information */
            *indication = _ReadNmr(no);
            break;

        case _TIMING_RECOVERY_CONTROL: /* Request VCXO frequency deviation information */
            READ_METER_REG(vcxo_frequency_low, vcxo_frequency_high, value, 0);
            *indication = HIGH(value);
            break;

        case _AAGC_VALUE: /* Request AAGC bits value */
            *indication = BP_READ_BIT(bp_mode_ptr, adc_control, again);
            break;
        
        case _STARTUP_STATUS: /* Request startup status information */
            RD_BYTE(no, STATUS, status_reg.status); /* Read startup status */
            s8_temp = _ReadNmr(no);
            if (s8_temp >= NOISE_MARGIN_TH) /* -5dB Noise Margin */
                {
                status_reg.bits.nmr_ok = ON;
                }
            else
                {
                status_reg.bits.nmr_ok = OFF;
                }
            WR_BYTE(no, STATUS, status_reg.status);
            
            *indication = status_reg.status;
            break;

        case _LEC_COEFF: /* Request linear EC coefficient value */

            if (parameter >= NUM_LEC_COEFFS) /* Unrecognized parameter */
                return (_FAIL);

            _ReadAccessByteRAM(no, LEC_ACCESS_RAM, parameter);
            value = BYTE2WORD(_bp_vars[no].access_data_byte[3], _bp_vars[no].access_data_byte[2]);

            *indication = (BP_S_8BIT) (value >> 6); /* Linear EC coefficient 8 significant bits (bits 22-29) */
            break;

        case _NLEC_COEFF: /* Request non linear EC coefficient value */

            if (parameter >= NUM_NLEC_COEFFS) /* Unrecognized parameter */
                return (_FAIL);

            _ReadAccessByteRAM(no, NLEC_ACCESS_RAM, parameter);
            *indication = _bp_vars[no].access_data_byte[1];    /* High byte only */
            break;

        case _EQ_COEFF: /* Request EQ coefficient value */

            if (parameter >= NUM_EQ_COEFFS) /* Unrecognized parameter */
                return (_FAIL);

            _ReadAccessByteRAM(no, EQ_ACCESS_RAM, parameter);
            *indication = _bp_vars[no].access_data_byte[1];    /* High byte only */
            break;

        case _DFE_COEFF: /* Request DFE coefficient value */

            if (parameter >= NUM_DFE_COEFFS) /* Unrecognized parameter */
                return (_FAIL);

            _ReadAccessByteRAM(no, DFE_ACCESS_RAM, parameter);
            *indication = _bp_vars[no].access_data_byte[1];    /* High byte only */
            break;
    
        case _VERSION: /* Request version number information */

            switch (parameter)
                {
#ifdef ZIP_START
                case _ZIP_MAJOR_VERSION:
                    *indication = ZIP_START_MAJOR_VERSION;
                    break;

                case _ZIP_MINOR_VERSION:
                    *indication = ZIP_START_MINOR_VERSION;
                    break;
#else
                case _ZIP_MAJOR_VERSION:
                case _ZIP_MINOR_VERSION:
                    *indication = 0;
                    break;
#endif
                case _MAJOR_SW_VERSION: /* Major software version */
                    *indication = MAJOR_VERSION;
                    break;
                case _MINOR_SW_VERSION: /* Minor software version */
                    *indication = MINOR_VERSION;
                    break;
                case _HW_TYPE_VERSIONS: /* Bit-pump type and bit-pump version. */
                    temp = BP_READ_REG(bp_ptr, global_modes);
                    temp = ((temp & 0xF0) | ((temp & 0x0E) >> 1));
                    *indication = temp;
                    break;
                case _HW_SW_VERSIONS: /* Upper-Software version Lower-Chip version */
                default:
                    /* Read Bitpump version and software version */
                    *indication = (((BP_READ_REG(bp_ptr, global_modes)) & 0xF0) | SW_VERSION); 
                    break;
                } /* End switch */
            break;

        case _SELF_TEST: /* Run bit-pump self test */
            *indication = _SelfTest(no); /* Perform self test */
            break;

        case _REGISTER: /* Read Bitpump internal register */
            ptr = (BP_S_8BIT*) bp_ptr;
            *indication = *(ptr + parameter);
            break;

        case _CONFIGURATION: /* Read user configurations */

            switch(parameter) /* Parameter value */
                {
                case _USER_SETUP_LOW_BYTE:
                    RD_WORD(no, USER_SETUP, *indication, temp);
                    break;

                case _USER_SETUP_HIGH_BYTE:
                    RD_WORD(no, USER_SETUP, temp, user_setup_high.setup);
                    user_setup_high.bits.activation_mode = ZIP_START_MODE;
                    *indication = user_setup_high.setup;
                    break;

                case _LOST:
                    RD_WORD(no, PARAMETERS, temp, *indication);
                    break;

                case _BIT_RATE:
                    *indication = _bp_vars[no].symbol_rate & 0xFF;
                    break;

                case _ERLE_SETUP_CONFIG:
                    RD_WORD(no, ERLE_SETUP, temp, *indication);
                    break;

                case _BIT_RATE_HI:
                    *indication = (_bp_vars[no].symbol_rate & 0x300) >> 8;
                    break;

                case _USER_PARAM_LOW_BYTE:
                    RD_WORD(no, PARAMETERS, *indication, temp);
                    break;

                default: /* unrecognized parameter */
                    return (_FAIL);

                } /* END-SWITCH parameter value */
            break;

        case _STAGE_NUMBER: /* Request startup stage number */
            switch (parameter)
                {
                case _ACTIVATION_STAGE:
                    RD_WORD(no, STAGE, *indication, temp);
                    break;

                case _TEST_MODE_STAGE:
                    RD_WORD(no, STAGE, temp, *indication);
                    break;

                case _TEMP_ENV_STAGE:
                    RD_WORD(no, STAGE2, *indication, temp);
                    break;
                }

            break;

        case _READ_TX:

            switch (parameter)
                {
                case _CALIBRATION:
                    *indication = BP_READ_BIT(bp_mode_ptr, tx_calibrate, tx_gain);
                    break;
                case _GAIN:
                    *indication = BP_READ_BIT(bp_mode_ptr, tx_gain, tx_gain);
                    break;
                default:
                    return (_FAIL);    
                } /* End switch */
            break;
 
        case _BER_METER_STATUS: /* Request BER Meter information */

#ifdef BER_METER
            switch (parameter)
                {
                default:
                case _BER_STATUS:
                    *indication = _bp_vars[no].bp_flags.bits.ber_meter_state;
                    break;

                case _BER_BIT_ERRORS_LOW:
                    *indication = LOW(_bp_vars[no].ber_meter.bit_errors);
                    break;

                case _BER_BIT_ERRORS_HIGH:
                    *indication = HIGH(_bp_vars[no].ber_meter.bit_errors);
                    break;

                case _BER_METER_INTERVALS_LOW:
                    *indication = LOW(_bp_vars[no].ber_meter.meter_intervals);
                    break;

                case _BER_METER_INTERVALS_HIGH:
                    *indication = HIGH(_bp_vars[no].ber_meter.meter_intervals);
                    break;
                }
#else
            /*
             * If BER_METER compiler flag not defined, always return bypassed
             */
            *indication = 0;
#endif /* BER_METER */
            break;


        case _ERLE_RESULTS:

#ifdef ERLE
            switch (parameter)
                {
                default:
                case _ERLE_SLM_LOW:
                    RD_ERLE_RESULTS(ERLE_SLM_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _ERLE_SLM_HIGH:
                    RD_ERLE_RESULTS(ERLE_SLM_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _ERLE_FELM_LOW:
                    RD_ERLE_RESULTS(ERLE_FELM_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _ERLE_FELM_HIGH:
                    RD_ERLE_RESULTS(ERLE_FELM_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _ERLE_SLM2_LOW:
                    RD_ERLE_RESULTS(ERLE_SLM2_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _ERLE_SLM2_HIGH:
                    RD_ERLE_RESULTS(ERLE_SLM2_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _ERLE_DC_OFFSET_LOW:
                    RD_ERLE_RESULTS(ERLE_DC_OFFSET_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _ERLE_DC_OFFSET_HIGH:
                    RD_ERLE_RESULTS(ERLE_DC_OFFSET_RESULT, value);
                    *indication = HIGH(value);
                    break;

                }

#endif /* ERLE */            
            break;

        case _AAGC_RESULTS:
#ifdef ERLE
            switch (parameter)
                {
                default:
                case _AAGC_SLM_0_LOW:
                    RD_ERLE_RESULTS(AAGC_SLM_0_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _AAGC_SLM_0_HIGH:
                    RD_ERLE_RESULTS(AAGC_SLM_0_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _AAGC_SLM_3_LOW:
                    RD_ERLE_RESULTS(AAGC_SLM_3_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _AAGC_SLM_3_HIGH:
                    RD_ERLE_RESULTS(AAGC_SLM_3_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _AAGC_SLM_6_LOW:
                    RD_ERLE_RESULTS(AAGC_SLM_6_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _AAGC_SLM_6_HIGH:
                    RD_ERLE_RESULTS(AAGC_SLM_6_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _AAGC_SLM_9_LOW:
                    RD_ERLE_RESULTS(AAGC_SLM_9_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _AAGC_SLM_9_HIGH:
                    RD_ERLE_RESULTS(AAGC_SLM_9_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _AAGC_SLM_12_LOW:
                    RD_ERLE_RESULTS(AAGC_SLM_12_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _AAGC_SLM_12_HIGH:
                    RD_ERLE_RESULTS(AAGC_SLM_12_RESULT, value);
                    *indication = HIGH(value);
                    break;

                case _AAGC_SLM_15_LOW:
                    RD_ERLE_RESULTS(AAGC_SLM_15_RESULT, value);
                    *indication = LOW(value);
                    break;

                case _AAGC_SLM_15_HIGH:
                    RD_ERLE_RESULTS(AAGC_SLM_15_RESULT, value);
                    *indication = HIGH(value);
                    break;
                }

#endif /* ERLE */            
            break;


        case _METER_INTERVAL_STATUS:
#ifdef TEMP_ENV
            *indication = _GetMeterIntervalElapsed(no);
#endif
            break;

        case _TE_METER_INTERVAL_STATUS:
#ifdef TEMP_ENV
            *indication = _GetTeMeterIntervalElapsed(no);
#endif
            break;

        default:
            return (_FAIL);

        } /* END-SWITCH opcode identification */

    return (_PASS); /* Returm PASS status to the user */


} /* END _BitpumpStatus */

