#include "../libc/types.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/config.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/util.h"
#include "../lib8973/bpstate.h"
#include "../lib8973/timermac.h"
#include "state.h"
#include "privstate.h"
#include "stages.h"
#include "miscmac.h"

/***********************************************************/
/*    _HandleFlags()                                       */
/*    Check interrupt flags status (set by interrupt       */
/*    routine) and act accordingly.                        */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output Variables: None.                              */
/*                                                         */
/* Example:                                                */
/*     _Handle_Flags(no);                                  */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                14-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void
_HandleFlags(st)
	struct sdcore_state *st;
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_PST;
    u_short lost_timer;
    int s;

    /*------------------------------------------------------------------*/
    /* Handle LOS condition.                                            */
    /*------------------------------------------------------------------*/
    if (BP_global_state.latched_alarm_irq & IRQMASK_LOWFELM) {
	/* Low FELM interrupt flag ON */
	s = splup(1);
	BP_global_state.latched_alarm_irq &= ~IRQMASK_LOWFELM;
	splx(s);
        st->status.bits.los = ON; /* Set status LOS */
#ifdef TDEBUG           
	/* HACK: suppress LOS ON/OFF messages during CM pulse detection */
	if (pst->stage < CM_MULTWAIT1 || pst->stage > CM_BABBLE2)
	    printf(" LOS ON\r\n");
#endif
	bp_ptr->irq_source = ~IRQMASK_HIGHFELM;
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, OFF); /* Remove high FELM interrupt mask */
    } /* END-IF Low FELM interrupt flag ON */

    /*------------------------------------------------------------------*/
    /* Handle LOS condition at deactivate mode, operate lost timer.     */
    /*------------------------------------------------------------------*/
    if (pst->stage == DEACTIVATE_MODE && st->status.bits.los &&
	!st->status.bits.run_lost_timer) {

        /* Use SUT3, as timeout timer for LOST time period */
        st->status.bits.run_lost_timer = ON;

	lost_timer = SYMRATE_DECBIN_ADJUST((BP_global_state.symbol_rate << 2) *
			st->lost_time / 10);

        SET_WORD(bp_ptr, sut3_low, sut3_high, lost_timer);
        BP_WRITE_BIT(bp_mode_ptr, timer_enable, sut3, ON);
        st->status.bits.lost = OFF;
        RESTART_SUT3;

#ifdef TDEBUG   
        printf(" Activate LOST timer\r\n");
#endif
    } /* END-IF LOS condition at deactivate_mode stage */

    /*-------------------------------------------------------------------*/
    /* Signal is back.                                                   */
    /*-------------------------------------------------------------------*/
    if (BP_global_state.latched_alarm_irq & IRQMASK_HIGHFELM) {
	/* High FELM interrupt flag ON */
	s = splup(1);
	BP_global_state.latched_alarm_irq &= ~IRQMASK_HIGHFELM;
	splx(s);
        st->status.bits.los = OFF; /* Set status LOS */
        st->status.bits.lost = OFF; /* Set status LOST */
#ifdef TDEBUG           
	/* HACK: suppress LOS ON/OFF messages during CM pulse detection */
	if (pst->stage < CM_MULTWAIT1 || pst->stage > CM_BABBLE2)
	    printf(" LOS OFF\r\n");
#endif
	bp_ptr->irq_source = ~IRQMASK_LOWFELM;
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, OFF); /* Remove low FELM interrupt mask */
        if (st->status.bits.run_lost_timer)  /* Lost timer is running */
            {
            st->status.bits.run_lost_timer = OFF;
	    s = splup(1);
            BP_WRITE_BIT(bp_mode_ptr, timer_enable, sut3, OFF); /* Disable sut3 */
	    bp_ptr->timer_source = ~TIMERMASK_SUT3;
	    BP_global_state.latched_timer_irq &= ~TIMERMASK_SUT3;
	    splx(s);
#ifdef TDEBUG   
            printf(" Disable LOST timer\r\n");
#endif
            } /* END-IF LOst timer is running */
    } /* END-IF High FELM interrupt flag ON */

    /*-------------------------------------------------------------------*/
    /* If LOST timer expired - set LOST status bit.                      */
    /*-------------------------------------------------------------------*/
    if (st->status.bits.run_lost_timer &&
	bp_ptr->timer_source & TIMERMASK_SUT3) {

	s = splup(1);
	bp_ptr->timer_source = ~TIMERMASK_SUT3;
	BP_global_state.latched_timer_irq &= ~TIMERMASK_SUT3;
	splx(s);
	st->status.bits.lost = ON; /* Set status LOST */
#ifdef TDEBUG   
	printf(" LOST ON\r\n");
#endif
    } /* END-IF sut3 interrupt flag ON */

    /*-----------------------------------------------------------------------*/
    /* If startup timer4 expired, set activation interval status bit.        */
    /*-----------------------------------------------------------------------*/
    if (BP_global_state.latched_timer_irq & TIMERMASK_SUT4) {
	s = splup(1);
	BP_global_state.latched_timer_irq &= ~TIMERMASK_SUT4;
	splx(s);
        st->status.bits.activation_interval = ON; /* Set status startup time limit exceded */
    } /* END-IF sut4 interrupt flag ON */

} /* END _HandleFlags() */
