/*
 * Craft terminal functionality for VersaDSU operational programs
 */

#include "types.h"
#include "ctype.h"
#include "stdio.h"
#include "strings.h"
#include "monitor_api.h"
#include "craftcmd.h"

extern volatile long timebase;

extern char appname[];
extern char *config_search();

#define	MAX_COMMANDS	32
struct craftcmd craft_cmd_table[MAX_COMMANDS+1];
struct craftcmd *craft_cmd_table_endp = craft_cmd_table;

static int craft_timeout;
static char craftcmd_buf[80];
static int craftcmd_active, craftcmd_len;
static long craftcmd_time;

craft_init()
{
	char *s;
	int val;

	s = config_search("craft_timeout");
	if (!s) {
		val = 10;	/* 10 s default */
		goto set;
	}
	if (!is_string_num(s)) {
		monapi_error("craft_timeout=%s: invalid setting", s);
		return(-1);
	}
	val = atoi(s);
set:	craft_timeout = val * 20;
	bzero(craft_cmd_table, sizeof craft_cmd_table);
	init_show_subcmd_table();
	craftcmd_active = 0;
	return(0);
}

void
craft_terminal_process()
{
	int c;

	/* check for timeout of old command first */
	if (craftcmd_active && craft_timeout &&
	    (timebase - craftcmd_time) >= craft_timeout) {
		printf("<Command prompt timeout>\r\n");
		craftcmd_active = 0;
	}
	/* anything new? */
	c = getchar_poll();
	if (c == EOF)
		return;
	craftcmd_time = timebase;
	if (!craftcmd_active) {
		if (c == '\r' || c == '\n') {
			printf("\r\n%s>", appname);
			craftcmd_active = 1;
			craftcmd_len = 0;
		}
		return;
	}

	switch (c) {
	case '\r':
	case '\n':
		printf("\r\n");
		craftcmd_buf[craftcmd_len] = '\0';
		craftcmd_active = 0;
		craft_command();
		return;
	case 0x03:
		printf("^C\r\n");
		craftcmd_active = 0;
		return;
	case 0x15:
		printf("^U\r\n%s>", appname);
		craftcmd_len = 0;
		return;
	case 0x12:
		printf("^R\r\n%s>%.*s", appname, craftcmd_len, craftcmd_buf);
		return;
	case '\b':
	case 0x7F:
		if (craftcmd_len) {
			printf("\b \b");
			craftcmd_len--;
		}
		return;
	default:
		if (!isprint(c) || craftcmd_len >= 79) {
			putchar(0x07);
			return;
		}
		putchar(c);
		craftcmd_buf[craftcmd_len++] = c;
	}
}

craftcmd_cmpfunc(tw, uw, min)
	char *tw, *uw;
{
	char *p1 = uw, *p2 = tw;
	int cnt = 0;

	for (; *p1 && *p2; p1++, p2++, cnt++) {
		char ch;

		ch = islower(*p1) ? toupper(*p1) : *p1;
		if (ch != *p2)
			break;
	}
	return(!*p1 && cnt >= min);
}

craft_command()
{
	char *cp, *np;
	struct craftcmd *ctp;

	for (cp = craftcmd_buf; isspace(*cp); cp++)
		;
	if (!*cp)
		return;
	for (np = cp; *cp && !isspace(*cp); cp++)
		;
	if (*cp)
		*cp++ = '\0';
	for (ctp = craft_cmd_table; ctp < craft_cmd_table_endp; ctp++)
		if (craftcmd_cmpfunc(ctp->word, np, ctp->min))
			break;
	if (ctp->func)
		ctp->func(cp);
	else
		printf("Error: unknown or unimplemented command; see HELP\r\n");
}

register_craft_command(cmdname, min, func)
	char *cmdname;
	int min;
	void (*func)();
{
	if (craft_cmd_table_endp - craft_cmd_table >= MAX_COMMANDS) {
		printf("OPCORE error: trying to register too many craft terminal commands\r\n");
		return(-1);
	}
	craft_cmd_table_endp->word = cmdname;
	craft_cmd_table_endp->min = min;
	craft_cmd_table_endp->func = func;
	craft_cmd_table_endp++;
	return(0);
}
