/*
 * This code assists with getting some common parameters out of CONFIG.TXT
 * and plugging them into the right places.
 */

#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "monitor_api.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"
#include "../sdcore/api.h"
#include "../sdcore/state.h"

extern struct sdcore_api *sdcore_api;
extern struct sdcore_state *sdcore_state;

extern char *config_search();

is_string_num(s)
	char *s;
{
	char *cp;

	cp = s;
	if (!isdigit(*cp++))
		return(0);
	while (*cp)
		if (!isdigit(*cp++))
			return(0);
	return(1);
}

parse_boolean_param(s)
	char *s;
{
	if (!strcasecmp(s, "yes") || !strcasecmp(s, "true") ||
	    !strcasecmp(s, "on") || !strcmp(s, "1"))
		return(1);
	else if (!strcasecmp(s, "no") || !strcasecmp(s, "false") ||
	    !strcasecmp(s, "off") || !strcmp(s, "0"))
		return(0);
	else
		return(-1);
}

get_ttype()
{
	char *s;

	s = config_search("terminal_type");
	if (!s) {
		monapi_error("Required parameter terminal_type= is missing");
		return(-1);
	}
	if (*s == 'C' || *s == 'c')
		sdcore_state->terminal_type = _HTUC;
	else if (*s == 'R' || *s == 'r')
		sdcore_state->terminal_type = _HTUR;
	else {
		monapi_error("terminal_type=%s: invalid setting", s);
		return(-1);
	}
	return(0);
}

get_datarate()
{
	char *s;
	int val;

	s = config_search("data_rate");
	if (!s) {
		monapi_error("Required parameter data_rate= is missing");
		return(-1);
	}
	if (!is_string_num(s)) {
		monapi_error("data_rate=%s: invalid setting", s);
		return(-1);
	}
	val = atoi(s);
	if (!sdcore_api->datarate_issup(val)) {
		monapi_error("%d kbps is not a valid SDSL data rate", val);
		return(-1);
	}
	sdcore_state->user_data_rate = val >> 3;
	return(0);
}

get_preact_type()
{
	char *s;

	s = config_search("preact_type");
	if (!s)
		return(0);
	if (!strcasecmp(s, "none"))
		sdcore_state->preact_type = PREACT_TYPE_NONE;
	else if (!strcasecmp(s, "CM"))
		sdcore_state->preact_type = PREACT_TYPE_CM;
	else if (!strcasecmp(s, "AutoBaud"))
		sdcore_state->preact_type = PREACT_TYPE_AUTOBAUD;
	else if (!strcasecmp(s, "IFCTF"))
		sdcore_state->preact_type = PREACT_TYPE_IFCTF;
	else {
		monapi_error("preact_type=%s: invalid setting", s);
		return(-1);
	}
	if (sdcore_state->preact_type == PREACT_TYPE_AUTOBAUD ||
	    sdcore_state->preact_type == PREACT_TYPE_IFCTF) {
		if (sdcore_state->terminal_type)
			return(get_ab_liberal());
		else
			return(get_preact_fallback());
	}
	return(0);
}

get_preact_fallback()
{
	char *s;
	int val;

	s = config_search("preact_fallback");
	if (!s)
		return(0);
	if (!is_string_num(s)) {
		monapi_error("preact_fallback=%s: invalid setting", s);
		return(-1);
	}
	val = atoi(s);
	sdcore_state->preact_fallback = val;
	return(0);
}

get_ab_flavor()
{
	char *s;
	int i;
	u_long accum;

	s = config_search("autobaud_flavor");
	if (!s) {
		monapi_error("Required parameter autobaud_flavor= is missing");
		return(-1);
	}
	if (strlen(s) != 4 || !isxdigit(s[0]) || !isxdigit(s[1]) ||
	    !isxdigit(s[2]) || !isxdigit(s[3])) {
		monapi_error("autobaud_flavor=%s: invalid setting", s);
		return(-1);
	}
	for (i = 0, accum = 0; i < 4; i++, s++) {
		accum <<= 4;
		if (isdigit(*s))
			accum += *s - '0';
		else if (isupper(*s))
			accum += *s - 'A' + 10;
		else
			accum += *s - 'a' + 10;
	}
	sdcore_state->cnxab_htuc_flavor[0] = accum >> 8;
	sdcore_state->cnxab_htuc_flavor[1] = accum & 0xFF;
	return(0);
}

get_ab_liberal()
{
	char *s;
	int bool;

	s = config_search("autobaud_liberal");
	if (!s) {
		bool = 1;	/* default to liberal for now */
		goto act;
	}
	bool = parse_boolean_param(s);
	if (bool < 0) {
		monapi_error("autobaud_liberal=%s: invalid setting", s);
		return(-1);
	}
act:	if (bool)
		sdcore_state->special_modes |= SDCORE_MODE_AB_CNX_IFCTF;
	else
		sdcore_state->special_modes &= ~SDCORE_MODE_AB_CNX_IFCTF;
	return(0);
}

get_quat_orient()
{
	char *s;

	s = config_search("quat_orient");
	if (!s)
		return(0);
	if (!strcasecmp(s, "signfirst"))
		sdcore_state->quat_orient = _SERIAL_SIGNFIRST;
	else if (!strcasecmp(s, "magfirst"))
		sdcore_state->quat_orient = _SERIAL_MAGFIRST;
	else {
		monapi_error("quat_orient=%s: invalid setting", s);
		return(-1);
	}
	return(0);
}
