#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "console.h"
#include "mc68302.h"
#include "gpio.h"
#include "../sdcore/api.h"
#include "../sdcore/state.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/bpstate.h"
#include "../lib8973/readmeter.h"
#include "../lib8973/readfelm.h"

extern volatile struct mc68302_regs mc68302_regs;

extern struct sdcore_api *sdcore_api;
extern struct sdcore_state sdcore_state;

int show_meters_raw;

show_ipl()
{
	printf("%d\r\n", current_ipl());
	return(0);
}

show_aagc()
{
	DECLARE_MODE_PTR;
	u_char aagc;

	aagc = bp_mode_ptr->adc_control.again;
	if (aagc <= 5)
		printf("%d dB\r\n", aagc * 3);
	else
		printf("Invalid value in the AAGC register!\r\n");
	return(0);
}

show_abflavor()
{
	printf("HTU-C flavor: %02X %02X\r\n", sdcore_state.cnxab_htuc_flavor[0],
		sdcore_state.cnxab_htuc_flavor[1]);
	printf("HTU-R flavor: %02X %02X\r\n", sdcore_state.cnxab_htur_flavor[0],
		sdcore_state.cnxab_htur_flavor[1]);
}

show_config()
{
	char str[16];

	printf("Terminal type: STU-%c\r\n",
		sdcore_state.terminal_type ? 'R' : 'C');
	switch (sdcore_state.preact_type) {
	case PREACT_TYPE_NONE:
		strcpy(str, "None");
		break;
	case PREACT_TYPE_CM:
		strcpy(str, "CM");
		break;
	case PREACT_TYPE_AUTOBAUD:
		strcpy(str, "AutoBaud");
		break;
	case PREACT_TYPE_IFCTF:
		strcpy(str, "IFCTF");
		break;
	default:
		sprintf(str, "%02X", sdcore_state.preact_type);
	}
	printf("Pre-activation type: %s\r\n", str);
	printf("User data rate: %u kbps\r\n", sdcore_state.user_data_rate * 8);
	printf("Special modes: %04X\r\n", sdcore_state.special_modes);
	switch (sdcore_state.quat_orient) {
	case _SERIAL_SIGNFIRST:
		strcpy(str, "Sign first");
		break;
	case _SERIAL_MAGFIRST:
		strcpy(str, "Magnitude first");
		break;
	default:
		sprintf(str, "%02X", sdcore_state.quat_orient);
	}
	printf("Quat orientation: %s\r\n", str);
	printf("LOST timer: %u\r\n", sdcore_state.lost_time);
	printf("Meter intervals before temp/env adjustment: %u\r\n",
		sdcore_state.tempenv_num_intervals);
	if (!sdcore_state.terminal_type &&
	    (sdcore_state.preact_type == PREACT_TYPE_AUTOBAUD ||
	     sdcore_state.preact_type == PREACT_TYPE_IFCTF))
		printf("Pre-activation fallback: %u\r\n",
			sdcore_state.preact_fallback);
	if (sdcore_state.special_modes & SDCORE_MODE_TXGAIN_OVERRIDE)
		printf("Tx gain override: %d\r\n", sdcore_state.txgain_setting);
	return(0);
}

show_felm()
{
	struct felmdata fd;

	sdcore_api->read_felm(&sdcore_state, &fd);
	printf("FELM = %u, %d dB attenuation\r\n", fd.norm_felm, fd.db_atten);
	return(0);
}

show_mdisplay()
{
	printf("Meter readings are shown %s\r\n",
		show_meters_raw ? "raw" : "normalized");
	return(0);
}

show_meters()
{
	u_short value;
	short svalue;

	value = sdcore_api->read_meter(&sdcore_state, READMETER_INTERVAL);
	printf("Meter interval is %u symbols\r\n", value);
	printf("Meter readings are shown %s\r\n",
		show_meters_raw ? "raw" : "normalized");
	value = sdcore_api->read_meter(&sdcore_state, READMETER_DCOFFSET);
	svalue = (short) value;
	printf("DC adjustment: %d\r\n", svalue);
	value = sdcore_api->read_meter(&sdcore_state,
			show_meters_raw ? READMETER_DC_RAW : READMETER_DC_NORM);
	svalue = (short) value;
	printf("DC meter: %d\r\n", svalue);
	value = sdcore_api->read_meter(&sdcore_state,
		show_meters_raw ? READMETER_SLM_RAW : READMETER_SLM_NORM);
	printf("SLM: %u\r\n", value);
	value = sdcore_api->read_meter(&sdcore_state,
		show_meters_raw ? READMETER_FELM_RAW : READMETER_FELM_NORM);
	printf("FELM: %u\r\n", value);
	value = sdcore_api->read_meter(&sdcore_state, READMETER_OVERFLOW);
	printf("Overflow meter: %u\r\n", value);
	value = sdcore_api->read_meter(&sdcore_state,
		show_meters_raw ? READMETER_NLM_RAW : READMETER_NLM_NORM);
	printf("NLM: %u\r\n", value);
	value = sdcore_api->read_meter(&sdcore_state, READMETER_BER);
	printf("BER meter: %u\r\n", value);
	return(0);
}

show_mux()
{
	char *s;

	switch (mc68302_regs.padat & PORTA_TXMUX_MASK) {
	case PORTA_TXMUX_DCE:
		s = "TxD line";
		break;
	case PORTA_TXMUX_DCEINV:
		s = "inverted TxD line";
		break;
	case PORTA_TXMUX_SCC1:
		s = "SCC1";
		break;
	case PORTA_TXMUX_FPGA:
		s = "FPGA";
		break;
	}
	printf("SDSL bit stream is sourced from %s\r\n", s);

	switch (mc68302_regs.padat & PORTA_DCEMUX_MASK) {
	case PORTA_DCEMUX_SDSL:
		s = "bitpump RDAT";
		break;
	case PORTA_DCEMUX_SDSLINV:
		s = "inverted bitpump RDAT";
		break;
	case PORTA_DCEMUX_SCC2:
		s = "SCC2";
		break;
	case PORTA_DCEMUX_FPGA:
		s = "FPGA";
		break;
	}
	printf("RxD line is sourced from %s\r\n", s);

	switch (mc68302_regs.padat & PORTA_114MUX_MASK) {
	case PORTA_114MUX_BCLK:
		s = "BCLK";
		break;
	case PORTA_114MUX_BCLK180:
		s = "inverted BCLK";
		break;
	case PORTA_114MUX_SWCLOCK:
		s = "software clock";
		break;
	case PORTA_114MUX_FPGA:
		s = "FPGA-generated clock";
		break;
	}
	printf("CCITT circuit 114 is driven with %s\r\n", s);

	switch (mc68302_regs.padat & PORTA_115MUX_MASK) {
	case PORTA_115MUX_BCLK:
		s = "BCLK";
		break;
	case PORTA_115MUX_BCLK180:
		s = "inverted BCLK";
		break;
	case PORTA_115MUX_SWCLOCK:
		s = "software clock";
		break;
	case PORTA_115MUX_FPGA:
		s = "FPGA-generated clock";
		break;
	}
	printf("CCITT circuit 115 is driven with %s\r\n", s);

	return(0);
}

show_nmr()
{
	int num, minus;

	num = sdcore_api->read_nmr(&sdcore_state);
	if (num < 0) {
		num = -num;
		minus = 1;
	} else
		minus = 0;
	printf("NMR: %s%u.%c dB\r\n", minus ? "-" : "", num >> 1,
		num & 1 ? '5' : '0');
	return(0);
}

paren_bit_output(flg, text, latch)
	int flg, *latch;
	char *text;
{
	if (!flg)
		return;
	printf("%s%s", *latch ? ", " : " (", text);
	*latch = 1;
}

show_status()
{
	u_char stat;
	int latch = 0;

	stat = sdcore_state.status.byte;
	printf("%02X", stat);
	paren_bit_output(stat & 0x80, "Normal op", &latch);
	paren_bit_output(stat & 0x40, "Tx 4-level", &latch);
	paren_bit_output(stat & 0x20, "LOST running", &latch);
	paren_bit_output(stat & 0x10, "Preact done", &latch);
	paren_bit_output(stat & 0x08, "Timer expired", &latch);
	paren_bit_output(stat & 0x04, "T/R rev", &latch);
	paren_bit_output(stat & 0x02, "LOST", &latch);
	paren_bit_output(stat & 0x01, "LOS", &latch);
	printf("%s\r\n", latch ? ")" : "");
}

show_symrate()
{
	printf("Current symbol rate: %u kbaud (%u kbps)\r\n",
		BP_global_state.symbol_rate * 4,
		BP_global_state.symbol_rate * 8);
	printf("User data rate: %u kbps\r\n", sdcore_state.user_data_rate * 8);
	return(0);
}

show_vcxo()
{
	short value;

	value = (short) sdcore_api->read_meter(&sdcore_state, READMETER_VCXO);
	printf("%d\r\n", value);
	return(0);
}

static const struct keyword show_subcmds[] = {
	{"AAGC", 2, (long) &show_aagc},
	{"ABFLAVOR", 2, (long) &show_abflavor},
	{"BCLK", 1, (long) &show_symrate},
	{"CONFIG", 1, (long) &show_config},
	{"DATARATE", 1, (long) &show_symrate},
	{"FELM", 1, (long) &show_felm},
	{"IPL", 1, (long) &show_ipl},
	{"MDISPLAY", 2, (long) &show_mdisplay},
	{"METERS", 2, (long) &show_meters},
	{"MUX", 2, (long) &show_mux},
	{"NMR", 1, (long) &show_nmr},
	{"QCLK", 1, (long) &show_symrate},
	{"STATUS", 2, (long) &show_status},
	{"SYMRATE", 2, (long) &show_symrate},
	{"VCXO", 1, (long) &show_vcxo},
	{NULL, 0, -1}};

cmd_show(cmd)
	char *cmd;
{
	struct param params[1];
	void (*subcmdfunc)();
	int ambigsubcmd;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
		find_keyword(params[0].text, params[0].len, show_subcmds,
				&ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)();
	else {
		if (ambigsubcmd)
			error("Ambiguous parameter");
		else
			error("Illegal parameter");
	}
	return(0);
}
