/*
 * The black magic of implementing Nokia SDSL/ATM
 * using only an MC68302 SCC.
 */

#include "types.h"
#include "param.h"
#include "globals.h"
#include "nokia.h"
#include "nokia_impl.h"
#include "mc68302.h"
#include "cp.h"
#include "intr.h"
#include "gpio.h"
#include "sccsave.h"
#include "sdsl_asm.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"
#include "../libutil/hec.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile struct mc68302_scc_bd scc1_rx_bds[8];
extern volatile struct mc68302_scc_bd scc1_tx_bds[8];
extern volatile struct mc68302_scc_params scc1_params;
extern struct scc_saved_initstate scc1_saved_initstate;

extern volatile long timebase;

extern struct buffer *get_buffer_from_queue();
extern const u_char crc6_table[256];

struct nokia_frame_scc nokia_rxf_bufs[NFRX_BUFS] __attribute__ ((aligned (2)));
struct nokia_frame_scc tx_idle_frame __attribute__ ((aligned (2)));
int nokia_rxf_bufptr, nokia_txf_bufptr;

const u_char idle_cell_hdr[5] = {0x00, 0x00, 0x00, 0x01, 0x52};

/* one time init function */
nokia_init_idle_frame()
{
	int i, j;
	u_char crc, *cp;

	for (i = 0; i < 8; i++) {
		bcopy(idle_cell_hdr, tx_idle_frame.cells[i].cellhdr, 5);
		for (j = 0; j < 48; j++)
			tx_idle_frame.cells[i].payload[j] = 0x6A;
	}
	bzero(tx_idle_frame.hdrbytes, 6);
	tx_idle_frame.hdrbytes[1] = tx_eoc_octet;
	cp = (u_char *) &tx_idle_frame;
	for (i = 0, crc = 0; i < 430; i++)
		crc = crc6_table[crc ^ *cp++];
	tx_idle_frame.hdrbytes[6] = crc;
	tx_idle_frame.hdrbytes[7] = NOKIA_SYNC_OCTET;
	return(0);
}

/* This is where we prepare all our machinery */
nokia_prepare()
{
	int i;

	scc1_params.rx_int_state = scc1_saved_initstate.rx_int_state;
	scc1_params.rbdn = scc1_saved_initstate.rbdn;
	scc1_params.tx_int_state = scc1_saved_initstate.tx_int_state;
	scc1_params.tbdn = scc1_saved_initstate.tbdn;

	mc68302_regs.scc1.scm = SCM_SOFT_CTS_CD | SCM_BISYNC_REVD
				| SCM_BISYNC_NTSYN | SCM_MODE_BISYNC;
	mc68302_regs.scc1.dsr = NOKIA_SCC_SYNC_WORD;
	for (i = 0; i < NFRX_BUFS; i++) {
		scc1_rx_bds[i].bd_flags = 0xD000;
		scc1_rx_bds[i].bd_bufptr = (u_char *) &nokia_rxf_bufs[i];
	}
	scc1_rx_bds[NFRX_BUFS-1].bd_flags = 0xF000;
	nokia_rxf_bufptr = 0;

	scc1_tx_bds[0].bd_flags = 0xD000;
	scc1_tx_bds[0].bd_bufptr = (u_char *) &tx_idle_frame;
	scc1_tx_bds[0].bd_len = 432;
	scc1_tx_bds[1].bd_flags = 0xF000;
	scc1_tx_bds[1].bd_bufptr = (u_char *) &tx_idle_frame;
	scc1_tx_bds[1].bd_len = 432;
	nokia_txf_bufptr = 0;
	initidle_count = nokia_initial_idle;

	scc1_params.rfcr = 0x50;
	scc1_params.tfcr = 0x50;
	scc1_params.mrblr = 432;
	mc68302_regs.scc1.scce = 0xFF;
	mc68302_regs.scc1.sccm = 0x17;
	nokia_rx_state = NOKIARX_STATE_HUNT;
	nokia_tx_error = 0;
	mc68302_regs.imr |= INTMASK_SCC1;
	return(0);
}

nokia_scc_inthandler()
{
	u_char evt;

	/*
	 * We don't bother with the spl3 trick any more. There are no
	 * higher priority interrupt sources at IPL 4 than SCC1 that are
	 * used on this board, so those spl3() and spl4() calls would be
	 * just extra delay.
	 */
	evt = mc68302_regs.scc1.scce & mc68302_regs.scc1.sccm;
	mc68302_regs.scc1.scce = evt;
	if (evt & 0x10)
		nokia_tx_error = 1;
	if (evt & 0x04)
		nokia_rx_state = NOKIARX_STATE_OVERRUN;
	if (evt & 0x02)
		nokia_scc_tx_handler();
	if (evt & 0x01)
		nokia_scc_rx_handler();
}

nokia_scc_rx_handler()
{
	int i;
	u_short flags;

	for (i = nokia_rxf_bufptr; ; ) {
		flags = scc1_rx_bds[i].bd_flags;
		if (flags & 0x8000)
			break;
		switch (nokia_rx_state) {
		case NOKIARX_STATE_HUNT:
			/* receiving *anything* means the SCC synced */
			nokia_rx_state = NOKIARX_STATE_SYNC;
			nokia_rx_counter = 0;
			/* fall through */
		case NOKIARX_STATE_SYNC:
			if (flags & 0x0002) {
				nokia_rx_state = NOKIARX_STATE_OVERRUN;
				break;
			}
			nokia_pass_rxframe_to_atm(&nokia_rxf_bufs[i]);
			eocdump_input(nokia_rxf_bufs[i].hdrbytes[1]);
			if (nokia_rxf_bufs[i].hdrbytes[7] == NOKIA_SYNC_OCTET)
				nokia_rx_counter = 0;
			else
				nokia_rx_counter++;
			if (nokia_rx_counter >= nokia_losw_count)
				nokia_rx_state = NOKIARX_STATE_LOSW;
			break;
		default:
			/*
			 * Once we are in one of the error states,
			 * do nothing more.
			 */
		}
		scc1_rx_bds[i].bd_flags = (flags & 0x7000) | 0x8000;
		i++;
		if (i >= NFRX_BUFS)
			i = 0;
	}
	nokia_rxf_bufptr = i;
}

nokia_pass_rxframe_to_atm(frame)
	struct nokia_frame_scc *frame;
{
	int i, badhec;

	for (i = 0; i < 8; i++) {
		badhec = compute_hec(frame->cells[i].cellhdr) !=
				frame->cells[i].cellhdr[4];
		atm_cell_rx(&frame->cells[i], badhec);
		if (badhec)
			stats.atm_rx_badhec++;
	}
}

nokia_scc_tx_handler()
{
	int i;
	u_short flags;
	u_char *oldbuf;
	struct nokia_preptxf_buf *newbuf;

	for (i = nokia_txf_bufptr; ; i = !i) {
		flags = scc1_tx_bds[i].bd_flags;
		if (flags & 0x8000)
			break;
		oldbuf = scc1_tx_bds[i].bd_bufptr;
		if (oldbuf != (u_char *) &tx_idle_frame)
			p2f_free_txfb(oldbuf - 4);
		if (initidle_count) {
			initidle_count--;
			newbuf = NULL;
		} else
			newbuf = (struct nokia_preptxf_buf *)
				get_buffer_from_queue(&nokia_preptxf_queue);
		scc1_tx_bds[i].bd_bufptr = (u_char *)
				(newbuf ? &newbuf->frame : &tx_idle_frame);
		scc1_tx_bds[i].bd_flags = (flags & 0x7000) | 0x8000;
	}
	nokia_txf_bufptr = i;
}

sdsl_startup_success()
{
	sdcore_api->set_tx(&sdcore_state, 1, SCRAMBLED_FOUR_LEVEL_DATA);
	mc68302_regs.scc1.scm |= SCM_ENR | SCM_ENT;
	printf("Bitpump activation successful, hunting for Nokia frames\r\n");
	printf("Startup time %d s\r\n", (timebase - sdsl_asm_state_time) / 20);
	set_sdsl_state(FRAME_ACQU_STATE);
	mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
				| PORTA_LSLED_ORANGE;
}

sdsl_shutdown()
{
	struct buffer *buf;

	mc68302_regs.scc1.scm &= ~(SCM_ENR | SCM_ENT);
	while (buf = get_buffer_from_queue(&atm_outgoing_queue))
		l2conv_api->free_buffer(buf);
	p2f_flush();
	spl4();
	nokia_salvage_tx();
	spl0();
	spl4();
	eocdump_init();
	spl0();
}

nokia_salvage_tx()
{
	int i;
	u_char *oldbuf;

	for (i = 0; i < 2; i++) {
		oldbuf = scc1_tx_bds[i].bd_bufptr;
		if (oldbuf != (u_char *) &tx_idle_frame)
			p2f_free_txfb(oldbuf - 4);
		scc1_tx_bds[i].bd_bufptr = (u_char *) &tx_idle_frame;
	}
}

nokia_handle_rx_overrun()
{
	printf("Nokia frame Rx overrun error; fatal condition\r\n");
	stats.sdsl_frame_rx_overruns++;
	set_sdsl_state(DEACTIVATED_STATE);
}

nokia_handle_tx_error()
{
	printf("Nokia frame Tx underrun error; fatal condition\r\n");
	stats.sdsl_frame_tx_underruns++;
	set_sdsl_state(DEACTIVATED_STATE);
}
