#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "mon302.h"
#include "excstate.h"
#include "mc68302.h"
#include "gpio.h"

extern volatile struct mc68302_regs mc68302_regs;

cmd_show_exception(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	printf("Exception context at: %08X\r\n", excstate);
	printf("D0=%08X  D1=%08X  D2=%08X  D3=%08X\r\n",
		excstate[EXCSTATE_D0], excstate[EXCSTATE_D1],
		excstate[EXCSTATE_D2], excstate[EXCSTATE_D3]);
	printf("D4=%08X  D5=%08X  D6=%08X  D7=%08X\r\n",
		excstate[EXCSTATE_D4], excstate[EXCSTATE_D5],
		excstate[EXCSTATE_D6], excstate[EXCSTATE_D7]);
	printf("A0=%08X  A1=%08X  A2=%08X  A3=%08X\r\n",
		excstate[EXCSTATE_A0], excstate[EXCSTATE_A1],
		excstate[EXCSTATE_A2], excstate[EXCSTATE_A3]);
	printf("A4=%08X  A5=%08X  A6=%08X\r\n",
		excstate[EXCSTATE_A4], excstate[EXCSTATE_A5],
		excstate[EXCSTATE_A6]);
	printf("PC=%08X  SR=%04X\r\n", excstate[EXCSTATE_PC],
		excstate[EXCSTATE_SR] & 0xFFFF);
	return(0);
}

cmd_show_ipl(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	printf("%d\r\n", current_ipl());
	return(0);
}

cmd_show_mctl(cmd)
	char *cmd;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	printf("DTR is %s\r\n", mc68302_regs.pbdat & PORTB_DTR ?
		"asserted" : "negated");
	printf("DSR is %s\r\n", mc68302_regs.pbdat & PORTB_DSR ?
		"asserted" : "negated");
	printf("CD is %s\r\n", mc68302_regs.pbdat & PORTB_CD ?
		"asserted" : "negated");
	printf("RTS is %s\r\n", mc68302_regs.pbdat & PORTB_RTS ?
		"asserted" : "negated");
	printf("CTS is %s\r\n", mc68302_regs.pbdat & PORTB_CTS ?
		"asserted" : "negated");
	return(0);
}

cmd_show_mux(cmd)
	char *cmd;
{
	char *s;

	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);

	switch (mc68302_regs.padat & PORTA_TXMUX_MASK) {
	case PORTA_TXMUX_DCE:
		s = "TxD line";
		break;
	case PORTA_TXMUX_DCEINV:
		s = "inverted TxD line";
		break;
	case PORTA_TXMUX_SCC1:
		s = "SCC1";
		break;
	case PORTA_TXMUX_FPGA:
		s = "FPGA";
		break;
	}
	printf("SDSL bit stream is sourced from %s\r\n", s);

	switch (mc68302_regs.padat & PORTA_DCEMUX_MASK) {
	case PORTA_DCEMUX_SDSL:
		s = "bitpump RDAT";
		break;
	case PORTA_DCEMUX_SDSLINV:
		s = "inverted bitpump RDAT";
		break;
	case PORTA_DCEMUX_SCC2:
		s = "SCC2";
		break;
	case PORTA_DCEMUX_FPGA:
		s = "FPGA";
		break;
	}
	printf("RxD line is sourced from %s\r\n", s);

	switch (mc68302_regs.padat & PORTA_114MUX_MASK) {
	case PORTA_114MUX_BCLK:
		s = "BCLK";
		break;
	case PORTA_114MUX_BCLK180:
		s = "inverted BCLK";
		break;
	case PORTA_114MUX_SWCLOCK:
		s = "software clock";
		break;
	case PORTA_114MUX_FPGA:
		s = "FPGA-generated clock";
		break;
	}
	printf("CCITT circuit 114 is driven with %s\r\n", s);

	switch (mc68302_regs.padat & PORTA_115MUX_MASK) {
	case PORTA_115MUX_BCLK:
		s = "BCLK";
		break;
	case PORTA_115MUX_BCLK180:
		s = "inverted BCLK";
		break;
	case PORTA_115MUX_SWCLOCK:
		s = "software clock";
		break;
	case PORTA_115MUX_FPGA:
		s = "FPGA-generated clock";
		break;
	}
	printf("CCITT circuit 115 is driven with %s\r\n", s);

	return(0);
}

static const struct keyword show_subcmds[] = {
	{"EXCEPTION", 1, (long) &cmd_show_exception},
	{"IPL", 1, (long) &cmd_show_ipl},
	{"MCTL", 2, (long) &cmd_show_mctl},
	{"MUX", 2, (long) &cmd_show_mux},
	{NULL, 0, -1}};

cmd_show(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	char *cp = cmd, *subcmd;
	int count;
	void (*subcmdfunc)();
	int ambigsubcmd;

	while (*cp == ' ')
		cp++;
	if (!issym(*cp)) {
		error("Subcommand required");
		return(0);
	}
	for (subcmd = cp, count = 0; issym(*cp); cp++)
		count++;
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
			find_keyword(subcmd, count, show_subcmds, &ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)(cp, excstate);
	else {
		if (ambigsubcmd)
			error("Ambiguous subcommand");
		else
			error("Illegal subcommand");
	}
	return(0);
}
