#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "mc68302.h"
#include "gpio.h"
#include "mon302.h"
#include "rt11ffs.h"
#include "../libffs/ffsimpl.h"

extern volatile struct mc68302_regs mc68302_regs;

cmd_fpga_kill(cmd)
	char *cmd;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	mc68302_regs.pbdat &= ~PORTB_FPGA_nCONFIG;
	return(0);
}

fpga_load_doit(findstruct)
	struct find_file_results *findstruct;
{
	int seg, bytecount;
	u_char *cp, *endp;
	int shiftreg, j;

	/* nCONFIG low clears out whatever came before us */
	mc68302_regs.pbdat &= ~PORTB_FPGA_nCONFIG;
	mc68302_regs.pbdat &= ~PORTB_MISC6;
	/*
	 * Verify that the FPGA is there.  If the FPGA is present, nSTATUS
	 * and CONF_DONE ought to be low within 200 ns (according to datasheet)
	 * after nCONFIG low.  If the FPGA is not populated, the nets will be
	 * high from the pull-up resistors.
	 */
	cpu_delay_us(1);
	if (mc68302_regs.pbdat & (PORTB_FPGA_nSTATUS | PORTB_FPGA_CONFDONE)) {
		error("FPGA is not populated");
		return(-1);
	}
	/* nCONFIG low pulse width: 2 us min per datasheet */
	cpu_delay_us(2);
	mc68302_regs.pbdat |= PORTB_FPGA_nCONFIG;
	/* datasheet says nSTATUS comes up within 4 us */
	cpu_delay_us(5);
	if (!(mc68302_regs.pbdat & PORTB_FPGA_nSTATUS)) {
		error("FPGA nSTATUS failed to rise after nCONFIG high");
		mc68302_regs.pbdat |= PORTB_MISC7;
		return(-1);
	}
	if (mc68302_regs.pbdat & PORTB_FPGA_CONFDONE) {
		error("FPGA CONF_DONE is high when it shouldn't be");
		mc68302_regs.pbdat |= PORTB_MISC7;
		return(-1);
	}
	cpu_delay_us(5);
	/*
	 * Start pushing the data bits!
	 * No timing concerns because DCLK/DATA0 accept faster data rates
	 * than we can push with our CPU.
	 *
	 * We do spl7() to avoid any issues with the FPGA's IRQ output
	 * glitching low during configuration/initialization.
	 */
	spl7();
	for (bytecount = seg = 0; seg < findstruct->nfrags; seg++) {
		cp = findstruct->frags[seg].addr;
		endp = cp + findstruct->frags[seg].nblks * RT11FFS_AU_SIZE;
		while (cp < endp) {
			shiftreg = *cp++;
			for (j = 0; j < 8; shiftreg >>= 1, j++) {
				if (shiftreg & 1)
					mc68302_regs.pbdat |= PORTB_MISC7;
				else
					mc68302_regs.pbdat &= ~PORTB_MISC7;
				mc68302_regs.pbdat |= PORTB_MISC6;
				mc68302_regs.pbdat &= ~PORTB_MISC6;
			}
			bytecount++;
			if (!(mc68302_regs.pbdat & PORTB_FPGA_nSTATUS)) {
				error("FPGA nSTATUS low after %d bytes",
					bytecount);
				mc68302_regs.pbdat |= PORTB_MISC7;
				return(-1);
			}
		}
	}
	if (!(mc68302_regs.pbdat & PORTB_FPGA_CONFDONE)) {
	    error("FPGA CONF_DONE isn't asserted at the end of configuration");
		mc68302_regs.pbdat |= PORTB_MISC7;
		return(-1);
	}
	mc68302_regs.pbdat &= ~PORTB_MISC7;
	/*
	 * It's all done!
	 * Let's cross our fingers and allow IRQ6 in...
	 */
	spl0();
	return(0);
}

static const struct keyword fpga_load_quals[] = {
	{"FFS", 1, 1},
	{"MEMORY", 1, 2},
	{NULL, 0, -1}};

cmd_fpga_load(cmd)
	char *cmd;
{
	struct param params[1];
	struct qual quals[3], *q;
	int mode;
	u_short filename_rad50[3];
	struct find_file_results findstruct;
	u_long membase;

	bzero(quals, sizeof quals);
	if (parse_cmd(cmd, 1, 1, params, NULL, fpga_load_quals, quals))
		return(0);
	mode = 0;
	for (q = quals; q->id; q++) {
		if (q->value) {
			error("No value qualifier");
			return(0);
		}
		if (mode) {
			error("Qualifier conflict");
			return(0);
		}
		mode = q->id;
	}
	if (!mode)
		mode = 1;
	switch (mode) {
	case 1:
		if (ascii_to_rad50_filename(params[0].text, params[0].len,
					filename_rad50, RBF_DEFEXT) < 0) {
			error("Invalid filename");
			return(0);
		}
		if (ffs_find_file_wrapper(filename_rad50, &findstruct) < 0)
			return(0);
		break;
	case 2:
		if (parse_hexnum(params[0].text, params[0].len, &membase))
			return(0);
		findstruct.nfrags = 1;
		findstruct.frags[0].addr = membase;
		findstruct.frags[0].nblks = 100;
		break;
	}
	fpga_load_doit(&findstruct);
	return(0);
}

cmd_fpga_map(cmd)
	char *cmd;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	mc68302_regs.or3 = 0x1FFC;
	mc68302_regs.br3 = 0xB001;
	return(0);
}

static const struct keyword fpga_subcmds[] = {
	{"KILL", 1, (long) &cmd_fpga_kill},
	{"LOAD", 1, (long) &cmd_fpga_load},
	{"MAP", 1, (long) &cmd_fpga_map},
	{NULL, 0, -1}};

cmd_fpga(cmd)
	char *cmd;
{
	char *cp = cmd, *subcmd;
	int count;
	void (*subcmdfunc)();
	int ambigsubcmd;

	while (*cp == ' ')
		cp++;
	if (!issym(*cp)) {
		error("Subcommand required");
		return(0);
	}
	for (subcmd = cp, count = 0; issym(*cp); cp++)
		count++;
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
			find_keyword(subcmd, count, fpga_subcmds, &ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)(cp);
	else {
		if (ambigsubcmd)
			error("Ambiguous subcommand");
		else
			error("Illegal subcommand");
	}
	return(0);
}
