/*
 * FFS write commands executed by loading and calling FLASHW
 */

#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "mon302.h"
#include "rt11ffs.h"
#include "../libffs/ffsimpl.h"
#include "../flashw/api.h"

extern u_long last_load_addr;

struct flashw_api_table *flashw_api_table;

const u_short flashw_filename[3] __attribute__ ((section (".rodata")))
	= {0x2761, 0x7817, 0x0DF6};

load_flashw()
{
	struct program_header *hdr;

	if (load_from_ffs(flashw_filename, 0) < 0)
		return(-1);
	hdr = (struct program_header *)last_load_addr;
	flashw_api_table = (struct flashw_api_table *) hdr->api_table_ptr;
	return(0);
}

cmd_ffs_delete(cmd)
	char *cmd;
{
	struct param params[1];
	u_short filename_rad50[3];
	char filename_print[16];
	struct find_file_results findstruct;
	int stat;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (ascii_to_rad50_filename(params[0].text, params[0].len,
					filename_rad50, 0) < 0) {
		error("Invalid filename");
		return(0);
	}
	stat = ffs_find_file(filename_rad50, &findstruct);
	switch (stat) {
	case FIND_STAT_FOUND:
	case FIND_STAT_INVALID_FRAG:
		break;
	case FIND_STAT_NOTFOUND:
		rad50_filename_to_ascii_trimspaces(filename_rad50,
						   filename_print);
		error("%s not found", filename_print);
		return(0);
	case FIND_STAT_INVALID_DIR:
		error("RT11FFS appears to be corrupt");
		return(0);
	}
	if (!bcmp(filename_rad50, flashw_filename, 6)) {
		error("FLASHW.BIN may not be deleted");
		return(0);
	}
	if (load_flashw() < 0)
		return(0);
	flashw_api_table->ffs_delete_file(filename_rad50, 0);
	return(0);
}

cmd_ffs_erase(cmd)
	char *cmd;
{
	struct param params[1];
	u_long segno;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &segno))
		return(0);
	if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
		error("Invalid segment number");
		return(0);
	}
	if (load_flashw() < 0)
		return(0);
	flashw_api_table->ffs_erase_reinit(segno);
	return(0);
}

cmd_ffs_initialize(cmd)
	char *cmd;
{
	struct param params[1];
	u_long segno;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &segno))
		return(0);
	if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
		error("Invalid segment number");
		return(0);
	}
	if (load_flashw() < 0)
		return(0);
	flashw_api_table->ffs_init_segment(segno);
	return(0);
}

static const struct keyword store_quals[] = {
	{"CONTIGUOUS", 1, 'C'},
	{"SEGMENT", 1, 'S'},
	{NULL, 0, -1}};

cmd_ffs_store(cmd)
	char *cmd;
{
	struct param params[3];
	struct qual quals[3], *qp;
	u_short filename_rad50[3];
	struct ffs_create_options opts;
	u_long datasrc, nblocks, segno;

	bzero(quals, sizeof quals);
	if (parse_cmd(cmd, 3, 3, params, NULL, store_quals, quals))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &datasrc))
		return(0);
	if (datasrc & 1) {
		error("Odd address");
		return(0);
	}
	if (parse_hexnum(params[1].text, params[1].len, &nblocks))
		return(0);
	if (ascii_to_rad50_filename(params[2].text, params[2].len,
					filename_rad50, 0) < 0) {
		error("Invalid filename");
		return(0);
	}
	bzero(&opts, sizeof opts);
	for (qp = quals; qp->id; qp++)
		switch (qp->id) {
		case 'C':
			if (qp->value) {
				error("No value qualifier");
				return(0);
			}
			opts.contig = 1;
			continue;
		case 'S':
			if (!qp->value) {
				error("Value qualifier");
				return(0);
			}
			if (parse_hexnum(qp->value, qp->vallen, &segno))
				return(0);
			if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
				error("Invalid segment number");
				return(0);
			}
			opts.startseg = segno;
			continue;
		}
	if (load_flashw() < 0)
		return(0);
	flashw_api_table->ffs_create_file(filename_rad50, nblocks, datasrc,
					  &opts);
	return(0);
}
