#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "mon302.h"
#include "excstate.h"
#include "rt11ffs.h"
#include "../libffs/ffsimpl.h"

extern int cmd_call();
extern int cmd_continue();
extern int cmd_deposit();
extern int cmd_examine();
extern int cmd_ffs();
extern int cmd_fpga();
extern int cmd_help();
extern int cmd_hexdump();
extern int cmd_load();
extern int cmd_mux();
extern int cmd_reinit();
extern int cmd_restack();
extern int cmd_run();
extern int cmd_set();
extern int cmd_show();
extern int cmd_start();
extern int cmd_x();

extern int reboot();

int in_cmd_script;
struct cmdscript_nestlevel cmd_script_stack[MAX_CMDSCRIPT_NESTING];

const struct keyword commands[] = {
	{"CALL", 2, (long) &cmd_call},
	{"CONTINUE", 1, (long) &cmd_continue},
	{"DEPOSIT", 1, (long) &cmd_deposit},
	{"EXAMINE", 1, (long) &cmd_examine},
	{"FFS", 2, (long) &cmd_ffs},
	{"FPGA", 2, (long) &cmd_fpga},
	{"HELP", 3, (long) &cmd_help},
	{"HEXDUMP", 3, (long) &cmd_hexdump},
	{"LOAD", 1, (long) &cmd_load},
	{"MUX", 1, (long) &cmd_mux},
	{"REBOOT", 3, (long) &reboot},
	{"REINITIALIZE", 3, (long) &cmd_reinit},
	{"RESTACK", 3, (long) &cmd_restack},
	{"RUN", 2, (long) &cmd_run},
	{"SET", 2, (long) &cmd_set},
	{"SHOW", 2, (long) &cmd_show},
	{"START", 1, (long) &cmd_start},
	{"X", 1, (long) &cmd_x},
	{NULL, 0, -1}};

console(excstate)
	u_long *excstate;
{
	char command[80];
	int retmode;

	while (1) {
		char *cp;
		int count;

get_command:	if (!in_cmd_script) {
			/* Interactive command mode */
prompt:			printf("MON>");
			for (cp = command, count = 0; ; ) {
				char ch;

				ch = toascii(getchar());
				switch (ch) {
				case '\0':	/* ignore NULs */
					continue;
				case 0x03:	/* ^C */
					printf("^C\r\n");
					goto prompt;
				case 0x08:	/* BS */
				case 0x7F:	/* DEL */
					if (!count)
						continue;
					printf("\b \b");
					cp--;
					count--;
					continue;
				case '\r':	/* CR */
					printf("\r\n");
					goto process_command;
				case 0x15:	/* ^U */
					printf("^U\r\n");
					goto prompt;
				}
				if (iscntrl(ch) || count == 79) {
					putchar(0x07);
					continue;
				}
				putchar(ch);
				*cp++ = ch;
				count++;
			}
		} else {
			struct cmdscript_nestlevel *scr;

			/* Read command from script */
			scr = &cmd_script_stack[in_cmd_script - 1];
			while (scr->curptr < scr->endptr && !*scr->curptr)
				scr->curptr++;
			if (scr->curptr >= scr->endptr) {
				in_cmd_script--;
				goto get_command;
			}
			if (check_intr())
				goto get_command;
			for (cp = command, count = 0; ; ) {
				char ch;

				if (scr->curptr >= scr->endptr)
					goto process_command;
				ch = toascii(*scr->curptr++);
				if (!ch || ch == '\r')
					continue;
				if (ch == '\n')
					goto process_command;
				if (iscntrl(ch)) {
					error("Illegal character in command script");
					goto get_command;
				}
				if (count == 79) {
					error("Line too long in command script");
					goto get_command;
				}
				*cp++ = ch;
				count++;
			}
		}
process_command:
		*cp = '\0';
		if (in_cmd_script)
			printf("SCR>%s\r\n", command);
		/* Ignore comments */
		cp = index(command, '!');
		if (cp)
			*cp = '\0';
		for (cp = command; *cp == ' '; cp++)
			;
		if (!*cp)
			continue;
		if (exec_console_cmd(cp, excstate, &retmode))
			break;
	}
	return(retmode);
}

exec_console_cmd(cmd, excstate, retmode)
	char *cmd;
	u_long *excstate;
	int *retmode;
{
	char *cp;
	int count;
	int (*cmdfunc)();
	int ambigcmd;

	if (*cmd == '@') {
		exec_cmd_script(cmd + 1);
		return(0);
	}
	if (!issym(*cmd)) {
		error("Illegal command");
		return(0);
	}
	for (cp = cmd, count = 0; issym(*cp); cp++)
		count++;
	ambigcmd = 0;
	cmdfunc = (int (*)()) find_keyword(cmd, count, commands, &ambigcmd);
	if ((long) cmdfunc != -1)
		return (*cmdfunc)(cp, excstate, retmode);
	else {
		if (ambigcmd)
			error("Ambiguous command");
		else
			error("Illegal command");
		return(0);
	}
}

exec_cmd_script(cmd)
	char *cmd;
{
	char *cp;
	int count;
	u_short filename[3];
	struct find_file_results findstruct;

	if (in_cmd_script >= MAX_CMDSCRIPT_NESTING) {
		error("Command script nesting is too deep");
		return(0);
	}
	if (!issym(*cmd)) {
inv:		error("Illegal syntax");
		return(0);
	}
	for (cp = cmd, count = 0; issym(*cp); cp++)
		count++;
	while (*cp == ' ')
		cp++;
	if (*cp)
		goto inv;
	if (ascii_to_rad50_filename(cmd, count, filename, CMD_DEFEXT) < 0) {
		error("Invalid filename");
		return(0);
	}
	if (ffs_find_file_wrapper(filename, &findstruct) < 0)
		return(0);
	push_cmd_script(&findstruct);
	return(0);
}

push_cmd_script(findstruct)
	struct find_file_results *findstruct;
{
	if (findstruct->nfrags > 1) {
		error("Command scripts may not be fragmented in RT11FFS");
		return(-1);
	}
	cmd_script_stack[in_cmd_script].curptr = findstruct->frags[0].addr;
	cmd_script_stack[in_cmd_script].endptr = findstruct->frags[0].addr +
		findstruct->frags[0].nblks * RT11FFS_AU_SIZE;
	in_cmd_script++;
	return(0);
}
