/*
 * Generic parser for textual configuration files of name=value form
 *
 * The user must first copy the text file from FFS to a RAM buffer;
 * the text in that buffer will be munched up by this code.
 */

#include "types.h"
#include "ctype.h"
#include "stdio.h"
#include "strings.h"

struct pair {
	char	*name;
	char	*value;
};

parse_textconf(textbuf, textlen, ptrbuf, ptrbuflen, cfgname)
	char *textbuf, *cfgname;
	struct pair *ptrbuf;
{
	char *cp, *end, *err, savech;
	int lineno, pairsleft;

	pairsleft = ptrbuflen / sizeof(struct pair);
	cp = textbuf;
	end = cp + textlen;
	lineno = 1;
line:	while (cp < end && (isspace(*cp) || !*cp)) {
		if (*cp == '\n')
			lineno++;
		cp++;
	}
	if (cp >= end) {
out:		ptrbuf->name = NULL;
		ptrbuf->value = NULL;
		return(0);
	}
	if (!issym(*cp)) {
garbage:	err = "garbage character";
errout:		printf("%s parse error on line %d: %s\r\n", cfgname, lineno,
			err);
		return(-1);
	}
	ptrbuf->name = cp;
	while (cp < end && issym(*cp))
		cp++;
	if (cp >= end || *cp != '=') {
		err = "missing \'=\'";
		goto errout;
	}
	*cp++ = '\0';
	if (cp >= end || isspace(*cp)) {
		err = "missing value";
		goto errout;
	}
	if (!isgraph(*cp))
		goto garbage;
	ptrbuf->value = cp;
	while (cp < end && isgraph(*cp))
		cp++;
	if (cp >= end) {
unterm:		err = "missing newline";
		goto errout;
	}
	if (!isspace(*cp))
		goto garbage;
	savech = *cp;
	*cp++ = '\0';
	ptrbuf++;
	pairsleft--;
	if (!pairsleft) {
		printf("%s: too many parameter pairs (buffer overflow)\r\n",
			cfgname);
		return(-1);
	}
	if (savech == '\n') {
		lineno++;
		goto line;
	}
	while (cp < end && isspace(*cp) && *cp != '\n')
		cp++;
	if (cp >= end)
		goto unterm;
	if (*cp == '\n') {
		cp++;
		lineno++;
		goto line;
	} else
		goto garbage;
}

char *
textconf_search(ptrbuf, key)
	struct pair *ptrbuf;
	char *key;
{
	for (; ptrbuf->name; ptrbuf++)
		if (!strcasecmp(ptrbuf->name, key))
			break;
	return(ptrbuf->value);
}

void
textconf_dump(ptrbuf)
	struct pair *ptrbuf;
{
	for (; ptrbuf->name; ptrbuf++)
		printf("%s=%s\r\n", ptrbuf->name, ptrbuf->value);
}
