/*
 * This file is derived from ZipWire's suutil.c and contains
 * those "startup utility" functions which ought to be needed
 * both for SDCORE and for our analog test code, and which are
 * clean enough to live in lib8973.
 *
 * Also see ffeutils.c and openeye.c
 */

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "../libc/types.h"
#include "typedefs.h"
#include "config.h"
#include "bitpump.h"
#include "util.h"
#include "defmeter.h"
#include "bpstate.h"
#include "timermac.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define DAGC_GAIN_INIT_VALUE          0x40000000

/***********************************************************/
/*    _AdaptEc()                                           */
/*    Adapt Echo Canceller.                                */
/*                                                         */
/*    returns: none.                                       */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                     BP_U_8BIT gain                      */
/*                     BP_U_16BIT no_symbols               */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _AdaptEc(no, HIGHER_GAIN, 30000);             */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _AdaptEc (BP_U_8BIT no, BP_U_8BIT gain, BP_U_16BIT no_symbols)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Set adaptation gain, start Timer3.     */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, gain); /* EC at highest adaptation gain */

    SET_WORD(bp_ptr, t3_low, t3_high, no_symbols);
    RESTART_T3;

} /* END _AdaptEc() */


/***********************************************************/
/*    _InitDagc()                                          */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitDagc()
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
    SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, LOW_WORD(DAGC_GAIN_INIT_VALUE));
    BP_WRITE_REG(bp_ptr, eq_add_write, DAGC_LOW); /* Write DAGC coefficient low word */
    DELAY2SYMBOL;
    SET_WORD(bp_ptr, access_data_byte0, access_data_byte1, HIGH_WORD(DAGC_GAIN_INIT_VALUE));
    BP_WRITE_REG(bp_ptr, eq_add_write, DAGC_HIGH); /* Write DAGC coefficient high word */
    DELAY2SYMBOL;

} /* END _InitDagc() */


/***********************************************************/
/*    _InitDfe()                                           */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitDfe()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */
    SET_RESET(bp_mode_ptr, dfe_modes, zero_coefficients); /* Clear DFE coefficients */

} /* END _InitDfe() */


/***********************************************************/
/*    _InitDetector()                                      */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _InitDetector()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, ON);             /* 2-Level slicing */ 
    BP_WRITE_BIT(bp_mode_ptr, detector_modes, enable_peak_detector, OFF); /* Use Slicer */

} /* END _InitDetector() */


/***********************************************************/
/*    _AdaptDagcSelfMode()                                 */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptDagcSelfMode()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, OFF); /* DAGC at self adaptation mode */
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN); /* DAGC at high adaptation gain */

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON);   /* Adapt DAGC */

} /* END _AdaptDagcSelfMode() */


/***********************************************************/
/*    _AdaptDagcEqErrorMode()                              */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptDagcEqErrorMode()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON); /* DAGC at EQ_Error adaptation mode */ 
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, HIGH_GAIN);   /* DAGC at HIGH adaptation gain */ 
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON);   /* Adapt DAGC */

} /* END _AdaptDagcEqErrorMode() */

 
/***********************************************************/
/*    _AdaptAllFfe()                                       */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptAllFfe()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_last_coeff, OFF); /* FFE adapt all */ 
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, HIGH_GAIN); /* FFE HIGH adaptation gain */ 
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

} /* END _AdaptAllFfe() */ 


/***********************************************************/
/*    _AdaptDfe()                                          */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _AdaptDfe()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, HIGH_GAIN); /* DFE HIGH adaptation gain */ 
    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */

} /* END _AdaptDfe() */


/***********************************************************/
/*    _FreezeDfeFfe()                                      */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _FreezeDfeFfe()
{
    DECLARE_MODE_PTR;

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */ 
    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */

} /* END _FreezeDfeFfe() */

#ifdef TDEBUG

/*
 * FUNCTION:   _PrintfDAGC
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the DAGC High.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfDAGC()
{
    u_char data[2];

    bp_eq_read(DAGC_HIGH, data);

    PREFIX;
    printf("DAGC = %d\r\n", ((int)data[1]<<4));
}

/*
 * FUNCTION:   _PrintfFFECoefs
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the FFE Coefs.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfFFECoefs()
{
	int j;
	u_char data[2];

	PREFIX;
	printf("FFE = [\r\n");
	for (j = 0; j < 8; j++) {
		bp_eq_read(j, data);
		printf("%d ;", (BP_S_16BIT) BYTE2WORD(data[1], data[0]));
	}
	printf("]\r\n");
}


/*
 * FUNCTION:   _PrintfDFECoefs
 *
 * PARAMETERS: no - which bitpump; 
 *
 * PURPOSE:    This function prints out the DFE Coefs.  This is only
 *             intended to be used during development
 *             
 * RETURN:     SUCCESS - Successfully Completed Operation
 *
 * NOTES:      
 *
 * CHANGES:    February 4, 1997    V1.0  :  New
 */
void _PrintfDFECoefs()
{
    int j;
    u_char data[2];

    PREFIX;
    printf("DFE =[\r\n ");
    for (j = 0; j < 8; j++)      /* was 116 */
    {
	bp_dfe_read(j, data);
        printf("%d ;", (BP_S_16BIT) BYTE2WORD(data[1], data[0]));
    }
    printf("];\r\n");
}
#endif
