/*
 * Opening of the Eye of Ra.
 */

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "../libc/types.h"
#include "typedefs.h"
#include "config.h"
#include "bitpump.h"
#include "util.h"
#include "defmeter.h"
#include "openeye.h"

/***********************************************************/
/*    _OpenEyeLastFfe()                                    */
/*    Open eye process last FFE coefficient adaptation.    */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _OpenEyeLastFfe(no);                          */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void _OpenEyeLastFfe()
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    /*----------------------------------------*/
    /* Use PKD                                */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, ON); /* 2-level slicing */
    BP_WRITE_BIT(bp_mode_ptr, detector_modes, enable_peak_detector, ON); /* Use PKD */

    /*----------------------------------------*/
    /* Clear  FFE coefficients                */
    /*----------------------------------------*/

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, ON); /* Clear FFE coeffs */
    DELAY2SYMBOL;
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, OFF);

    /*----------------------------------------*/
    /* Initialize last FFE coefficient        */
    /*----------------------------------------*/
    WR_WORD(LAST_FFE, LOW(FFE_LAST_COEFF_INIT_VALUE), HIGH(FFE_LAST_COEFF_INIT_VALUE)); /* Set FFE last coefficient value */

    /*----------------------------------------*/
    /* Set adaptation modes                   */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, HIGH_GAIN); /* FFE at high adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_last_coeff, ON); /* FFE adapt last */

    /*----------------------------------------*/
    /* Start FFE adaptation                   */
    /*----------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

} /* END _OpenEyeLastFfe() */

/***********************************************************/
/*    _IsEyeOpen()                                         */
/*    Check if eye is open.                                */
/*                                                         */
/*    returns: true/false                                  */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _IsEyeOpen(no);                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
int _IsEyeOpen()
{
    DECLARE_PTR;
    BP_U_8BIT meter;
    BP_S_16BIT nlm_value, ffe_last_coeff_value;
    u_char data[2];

    NORM(meter) /* Calculate meter normalization factor */

    /*----------------------------------------------------*/
    /* Read Noise Level Meter value                       */
    /*----------------------------------------------------*/
    READ_METER_REG(nlm_low, nlm_high, nlm_value, meter);

    /*----------------------------------------------------*/
    /* Read last FFE coefficient value                    */
    /*----------------------------------------------------*/
    bp_eq_read(LAST_FFE, data);
    ffe_last_coeff_value = BYTE2WORD(data[1], data[0]);

    /*----------------------------------------------------*/
    /* Eye is considered to be open if the noise level is */
    /* lower than a threshold, and the last FFE coeff     */
    /* is higher than a threshold.                        */
    /*----------------------------------------------------*/

#ifdef TDEBUG
    printf("    NLM=%d, Last=%d\r\n",
		(int)nlm_value, (int)ffe_last_coeff_value);
#endif

    if ((nlm_value <= NLM_TH) && (ffe_last_coeff_value > FFE_TH)) /* Eye is open */
        return (1);
    else /* Eye not open */
        return (0);

} /* END _IsEyeOpen() */


/***********************************************************/
/*    _IsFirstEyeOPen()                                    */
/*    Check if eye is open.                                */
/*                                                         */
/*    returns: true/false                                  */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _IsFirstEyeOpen(no);                 */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

int _IsFirstEyeOpen()
{
    DECLARE_PTR;
    int i;
    BP_S_16BIT ffe_coeff;
    u_char data[2];

    /*----------------------------------------------------*/
    /* If any FFE coeff is high enough, return "_PASS",   */
    /* Otherwise - return "_Fail".                        */
    /*----------------------------------------------------*/

    for (i=0; i<8; i++) 
        {   
        /* Read FFE coefficient */
	bp_eq_read(i, data);
        ffe_coeff = BYTE2WORD(data[1], data[0]);

        if (ffe_coeff > FFE_TH) /* Eye is open */
            return (1);
        } /* End if */
    return (0);

} /* END _IsFirstEyeOpen() */
