/***********************************************************/
/*    BITPUMP.H                                            */
/*                                                         */
/* Description:                                            */
/*							   */
/*    Bitpump structure definitions.                       */
/*    This header file defines things which are		   */
/*    fixed by the bitpump hardware.			   */
/*                                                         */
/* Dependencies:                                           */
/*    typedefs.h					   */
/*                                                         */
/*                                                         */
/* Original author:                                        */
/*     Iris Shuker                30-August-1993           */
/*                                                         */
/* Revision History:                                       */
/*     Modified heavily from the ZipWire version	   */
/*                                                         */
/***********************************************************/

/*---------------------------------------------------------*/
/*  Global Defines Constants                               */
/*---------------------------------------------------------*/

/* Global Definitions */

/* Terminal type */
#define _HTUC            0
#define _HTUR            1


/*---------------------------------------------------------*/
/*  Global Defines Macros                                  */
/*---------------------------------------------------------*/

#define BP_WRITE_REG(ptr, reg, value)           ptr->reg = value
#define BP_WRITE_BIT(ptr, reg, bit, value)      ptr->reg.bit = value

#define BP_READ_REG(ptr, reg)                   ptr->reg
#define BP_READ_BIT(ptr, reg, bit)              ptr->reg.bit

/* The 27 us figure corresponds to 144 kbps */
#define DELAY2SYMBOL cpu_delay_us(27)

#define SET_RESET(ptr, reg, bit) \
    BP_WRITE_BIT(ptr, reg, bit, ON);\
    DELAY2SYMBOL;\
    BP_WRITE_BIT(ptr, reg, bit, OFF);

#define GET_BITPUMP_TYPE()      BP_READ_BIT(bp_mode_ptr, global_modes, id)


/*---------------------------------------------------------*/
/*  TYPEDEF Defines                                        */
/*---------------------------------------------------------*/


typedef BP_PACKED struct{
    BP_BIT_FIELD hw_revision:4;
    BP_BIT_FIELD id:3;
    BP_BIT_FIELD mode:1;
}BP_PACKED2 global_mode_type;

/* Bit-pump Types */
#define BT8952         0
#define BT8960         1
#define BT8970         2
#define BT8973         3
#define RS8973         3	/* the chip's actual designation */

/* Global modes registers is active low */
#define _BITPUMP_POWER_UP     0
#define _BITPUMP_POWER_DOWN   1

typedef BP_PACKED struct{
    BP_BIT_FIELD hclk_freq:2;
    BP_BIT_FIELD smon_source:6;
}BP_PACKED2 serial_monitor_type;

/* HCLK Select Options */
#define _HCLK_DEFAULT   0
#define _HCLK_16_TIMES  1
#define _HCLK_32_TIMES  2
#define _HCLK_64_TIMES  3

/* Define Smon source */
#define _ECHO_FREE_SIGNAL       0x1E
#define _EQ_ERROR               0x21
#define _SLICER_ERROR           0x22
#define _DAGC_OUTPUT            0x1C
#define _FFE_OUTPUT             0x1D

typedef BP_PACKED struct{
    BP_BIT_FIELD t4:1;
    BP_BIT_FIELD t3:1;
    BP_BIT_FIELD snr:1;
    BP_BIT_FIELD meter:1;
    BP_BIT_FIELD sut4:1;
    BP_BIT_FIELD sut3:1;
    BP_BIT_FIELD sut2:1;
    BP_BIT_FIELD sut1:1;
}BP_PACKED2 timer_type;

/*
 * We are in the process of migrating from
 * the bit fields above to the #defines below.
 */
#define	TIMERMASK_T4	0x80
#define	TIMERMASK_T3	0x40
#define	TIMERMASK_SNR	0x20
#define	TIMERMASK_METER	0x10
#define	TIMERMASK_SUT4	0x08
#define	TIMERMASK_SUT3	0x04
#define	TIMERMASK_SUT2	0x02
#define	TIMERMASK_SUT1	0x01

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:4;
    BP_BIT_FIELD sync:1;
    BP_BIT_FIELD high_felm:1;
    BP_BIT_FIELD low_felm:1;
    BP_BIT_FIELD low_snr:1;
}BP_PACKED2 irq_type;

#define	IRQMASK_SYNC		0x08
#define	IRQMASK_HIGHFELM	0x04
#define	IRQMASK_LOWFELM		0x02
#define	IRQMASK_LOWSNR		0x01

#define MASK_ALL                  0xFF

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:1;
    BP_BIT_FIELD imp_short:3;
    BP_BIT_FIELD rphs:4;
}BP_PACKED2 receive_select_type;

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:3;
    BP_BIT_FIELD tbclk_pol:1;
    BP_BIT_FIELD rbclk_pol:1;
    BP_BIT_FIELD fifos_mode:1;
    BP_BIT_FIELD interface_mode:2;
}BP_PACKED2 cu_interface_type;

/* Framer Format */
#define _PARALLEL_MASTER	0
#define _PARALLEL_SLAVE		1
#define	_SERIAL_MAGFIRST	2
#define	_SERIAL_SIGNFIRST	3

/* Clock Polarity for channel unit interface */
#define _RISING_EDGE	 0
#define _FALLING_EDGE	 1

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:2;
    BP_BIT_FIELD enable_dc_tap:1;
    BP_BIT_FIELD adapt_coefficients:1;
    BP_BIT_FIELD zero_coefficients:1;
    BP_BIT_FIELD zero_output:1;
    BP_BIT_FIELD adapt_gain:2;
}BP_PACKED2 linear_ec_mode_type;

/* Gain mode constants */
#define HIGHEST_GAIN 3
#define HIGHER_GAIN  2
#define HIGH_GAIN    1
#define NORMAL_GAIN  0

typedef BP_PACKED struct{
    BP_BIT_FIELD negate_symbol:1;
    BP_BIT_FIELD symbol_delay:3;
    BP_BIT_FIELD adapt_coefficients:1;
    BP_BIT_FIELD zero_coefficients:1;
    BP_BIT_FIELD zero_output:1;
    BP_BIT_FIELD adapt_gain:1;
}BP_PACKED2 nonlinear_ec_mode_type;

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:4;
    BP_BIT_FIELD adapt_coefficients:1;
    BP_BIT_FIELD zero_coefficients:1;
    BP_BIT_FIELD zero_output:1;
    BP_BIT_FIELD adapt_gain:1;
}BP_PACKED2 dfe_mode_type;

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:1;
    BP_BIT_FIELD isolated_pulse:2;
    BP_BIT_FIELD transmitter_off:1;
    BP_BIT_FIELD htur_lfsr:1;
    BP_BIT_FIELD data_source:3;
}BP_PACKED2 tx_mode_type;

/* Transmitter Mode Source */
#define ISOLATED_PULSE               0x0
#define DETECTOR_LOOPBACK            0x1
#define UNSCRAMBLED_FOUR_LEVEL_DATA  0x2
#define SCRAMBLED_FOUR_LEVEL_ONES    0x3
#define ALTERNATING_SYMBOL           0x4
#define SCRAMBLED_FOUR_LEVEL_DATA    0x5
#define UNSCRAMBLED_TWO_LEVEL_DATA   0x6
#define SCRAMBLED_TWO_LEVEL_ONES     0x7

/* Transmitter isolated pulse constants */
#define ISOLATED_PULSE_OUTPUT_MINUS3   0
#define ISOLATED_PULSE_OUTPUT_MINUS1   1
#define ISOLATED_PULSE_OUTPUT_PLUS3    2
#define ISOLATED_PULSE_OUTPUT_PLUS1    3

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved1:2;
    BP_BIT_FIELD reg_clk_en:1;
    BP_BIT_FIELD band_gap_test_mode:1;
    BP_BIT_FIELD reserved2:3;
    BP_BIT_FIELD async_mode:1;
}BP_PACKED2 misc_test_mode_type;

#define SYNCHRONOUS_MODE    0x00
#define ASYNCHRONOUS_MODE   0x01

typedef BP_PACKED struct{
    BP_BIT_FIELD cont_time:2;
    BP_BIT_FIELD lb:2;
    BP_BIT_FIELD switch_cap_pole:1;
    BP_BIT_FIELD again:3;
}BP_PACKED2 adc_control_type;

/* Continuous Real Time Filter Settings */
#define _1168K_FILTER     0
#define _784K_FILTER      1
#define _2320K_FILTER     2
#define _400K_FILTER      3

/* Analog Loopack constants */
#define ALB_DISABLED        0
#define ALB_BYPASS_HYBRID   1
#define ALB_TRANSMITTING    2
#define ALB_SILENT          3

/* Analog Gain Constants */
#define AGAIN15DB 0x5
#define AGAIN12DB 0x4
#define AGAIN9DB  0x3
#define AGAIN6DB  0x2
#define AGAIN3DB  0x1
#define AGAIN0DB  0x0

#ifdef AAGC15BUG
#define MAX_AGAIN   (BP_U_8BIT)AGAIN12DB
#else
#define MAX_AGAIN   (BP_U_8BIT)AGAIN15DB
#endif

typedef BP_PACKED struct{
    BP_BIT_FIELD clk_freq:2;
    BP_BIT_FIELD negate_symbols:1;
    BP_BIT_FIELD phase_detector_gain:2;
    BP_BIT_FIELD freeze_pll:1;
    BP_BIT_FIELD pll_gain:2;
}BP_PACKED2 pll_mode_type;

/* PLL Frequency */
#define FOUR_CHANNEL_60  0
#define SIX_CHANNEL_60   1
#define TWO_CHANNEL_60   2
#define EIGHT_CHANNEL_60 3

#define HALF_E1_70       0
#define HALF_T1_70       1
#define EIGHT_CHANNEL_70 2
#define FULL_T1_70       3

typedef BP_PACKED struct{
    BP_BIT_FIELD enable_peak_detector:1;
    BP_BIT_FIELD output_mux_control:2;
    BP_BIT_FIELD scr_out_to_dfe:1;
    BP_BIT_FIELD two_level:1;
    BP_BIT_FIELD lfsr_lock:1;
    BP_BIT_FIELD htur_lfsr:1;
    BP_BIT_FIELD descr_on:1;
}BP_PACKED2 det_mode_type;

#define SELECT_DETECTOR_OUTPUT         0x0
#define SELECT_TRANSMITTER_INPUT     0x1
#define SELECT_DESCRAMBLER_OUTPUT    0x2

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:5;
    BP_BIT_FIELD eq_error_adaptation:1;
    BP_BIT_FIELD adapt_coefficient:1;
    BP_BIT_FIELD adapt_gain:1;
}BP_PACKED2 dagc_mode_type;

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:4;
    BP_BIT_FIELD adapt_last_coeff:1;
    BP_BIT_FIELD zero_coefficients:1;
    BP_BIT_FIELD adapt_coefficients:1;
    BP_BIT_FIELD adapt_gain:1;
}BP_PACKED2 ffe_mode_type;

typedef BP_PACKED struct{
    BP_BIT_FIELD reserved:4;
    BP_BIT_FIELD zero_output:1;
    BP_BIT_FIELD zero_coefficients:1;
    BP_BIT_FIELD adapt_coefficients:1;
    BP_BIT_FIELD adapt_gain:1;
}BP_PACKED2 ep_mode_type;

typedef BP_PACKED struct{
    BP_BIT_FIELD band_gap_high:2;
    BP_BIT_FIELD tx_gain:4;
    BP_BIT_FIELD band_gap_low:2;
}BP_PACKED2 tx_gain_type;   

/*
 * OSDCU hardware specific: the bitpump is connected to the upper half of
 * the M68K data bus and all registers appear at even addresses.
 */

typedef BP_PACKED struct{
    global_mode_type global_modes;                /* Address 0 */
    BP_U_8BIT buspad0;
    serial_monitor_type serial_monitor_source;    /* Address 1 */
    BP_U_8BIT buspad1;
    timer_type mask_low_reg;                      /* Address 2 */
    BP_U_8BIT buspad2;
    irq_type mask_high_reg;                       /* Address 3 */
    BP_U_8BIT buspad3;
    timer_type timer_source;                      /* Address 4 */
    BP_U_8BIT buspad4;
    irq_type irq_source;                          /* Address 5 */
    BP_U_8BIT buspad5;
    cu_interface_type cu_interface_modes;         /* Address 6 */
    BP_U_8BIT buspad6;
    receive_select_type receive_phase_select;     /* Address 7 */
    BP_U_8BIT buspad7;
    linear_ec_mode_type linear_ec_modes;          /* Address 8 */
    BP_U_8BIT buspad8;
    nonlinear_ec_mode_type nonlinear_ec_modes;    /* Address 9 */
    BP_U_8BIT buspad9;
    dfe_mode_type dfe_modes;                      /* Address 10 */
    BP_U_8BIT buspad10;
    tx_mode_type transmitter_modes;               /* Address 11 */
    BP_U_8BIT buspad11;
    timer_type timer_restart;                     /* Address 12 */
    BP_U_8BIT buspad12;
    timer_type timer_enable;                      /* Address 13 */
    BP_U_8BIT buspad13;
    timer_type timer_continous;                   /* Address 14 */
    BP_U_8BIT buspad14;
    misc_test_mode_type misc_test;                /* Address 15 */
    BP_U_8BIT buspad15;
    BP_U_32BIT sut1_dontuse;                      /* Address 16 - 17 */
    BP_U_32BIT sut2_dontuse;                      /* Address 18 - 19 */
    BP_U_32BIT sut3_dontuse;                      /* Address 20 - 21 */
    BP_U_32BIT sut4_dontuse;                      /* Address 22 - 23 */
    BP_U_32BIT meter_timer_dontuse;               /* Address 24 - 25 */
    BP_U_32BIT snr_timer_dontuse;                 /* Address 26 - 27 */
    BP_U_32BIT t3_dontuse;                        /* Address 28 - 29 */
    BP_U_32BIT t4_dontuse;                        /* Address 30 - 31 */
    BP_U_8BIT register_32;                        /* Address 32 */
    BP_U_8BIT buspad32;
    adc_control_type adc_control;                 /* Address 33 */
    BP_U_8BIT buspad33;
    pll_mode_type pll_modes;                      /* Address 34 */
    BP_U_8BIT buspad34;
    BP_U_16BIT reserved6[5];                      /* Address 35 - 39 */
    tx_gain_type tx_calibrate;                    /* Address 40 */
    BP_U_8BIT buspad40;
    tx_gain_type tx_gain;                         /* Address 41 */
    BP_U_8BIT buspad41;
    BP_U_16BIT reserved2[16];                     /* Address 42 - 57 */
    det_mode_type detector_modes;                 /* Address 58 */
    BP_U_8BIT buspad58;
    BP_U_8BIT reserved7;                          /* Address 59 */
    BP_U_8BIT buspad59;
    dagc_mode_type dagc_modes;                    /* Address 60 */
    BP_U_8BIT buspad60;
    ffe_mode_type ffe_modes;                      /* Address 61 */
    BP_U_8BIT buspad61;
    ep_mode_type ep_modes;                        /* Address 62 */
    BP_U_8BIT buspad62;
    BP_U_16BIT reserved8[65];                     /* Address 63 - 127 */
}BP_PACKED2 transceiver_modes;

typedef BP_PACKED struct{
    BP_U_8BIT global_modes;                            /* Address 0 */
    BP_U_8BIT buspad0;
    BP_U_8BIT serial_monitor_source;                   /* Address 1 */
    BP_U_8BIT buspad1;
    BP_U_8BIT mask_low_reg;                            /* Address 2 */
    BP_U_8BIT buspad2;
    BP_U_8BIT mask_high_reg;                           /* Address 3 */
    BP_U_8BIT buspad3;
    BP_U_8BIT timer_source;                            /* Address 4 */
    BP_U_8BIT buspad4;
    BP_U_8BIT irq_source;                              /* Address 5 */
    BP_U_8BIT buspad5;
    BP_U_8BIT cu_interface_modes;                      /* Address 6 */
    BP_U_8BIT buspad6;
    BP_U_8BIT receive_phase_select;                    /* Address 7 */
    BP_U_8BIT buspad7;
    BP_U_8BIT linear_ec_modes;                         /* Address 8 */
    BP_U_8BIT buspad8;
    BP_U_8BIT nonlinear_ec_modes;                      /* Address 9 */
    BP_U_8BIT buspad9;
    BP_U_8BIT dfe_modes;                               /* Address 10 */
    BP_U_8BIT buspad10;
    BP_U_8BIT transmitter_modes;                       /* Address 11 */
    BP_U_8BIT buspad11;
    BP_U_8BIT timer_restart;                           /* Address 12 */
    BP_U_8BIT buspad12;
    BP_U_8BIT timer_enable;                            /* Address 13 */
    BP_U_8BIT buspad13;
    BP_U_8BIT timer_continous;                         /* Address 14 */
    BP_U_8BIT buspad14;
    BP_U_8BIT misc_test;                               /* Address 15 */
    BP_U_8BIT buspad15;
    BP_U_8BIT sut1_low;                                /* Address 16 */
    BP_U_8BIT buspad16;
    BP_U_8BIT sut1_high;                               /* Address 17 */
    BP_U_8BIT buspad17;
    BP_U_8BIT sut2_low;                                /* Address 18 */
    BP_U_8BIT buspad18;
    BP_U_8BIT sut2_high;                               /* Address 19 */
    BP_U_8BIT buspad19;
    BP_U_8BIT sut3_low;                                /* Address 20 */
    BP_U_8BIT buspad20;
    BP_U_8BIT sut3_high;                               /* Address 21 */
    BP_U_8BIT buspad21;
    BP_U_8BIT sut4_low;                                /* Address 22 */
    BP_U_8BIT buspad22;
    BP_U_8BIT sut4_high;                               /* Address 23 */
    BP_U_8BIT buspad23;
    BP_U_8BIT meter_low;                               /* Address 24 */
    BP_U_8BIT buspad24;
    BP_U_8BIT meter_high;                              /* Address 25 */
    BP_U_8BIT buspad25;
    BP_U_8BIT snr_timer_low;                           /* Address 26 */
    BP_U_8BIT buspad26;
    BP_U_8BIT snr_timer_high;                          /* Address 27 */
    BP_U_8BIT buspad27;
    BP_U_8BIT t3_low;                                  /* Address 28 */
    BP_U_8BIT buspad28;
    BP_U_8BIT t3_high;                                 /* Address 29 */
    BP_U_8BIT buspad29;
    BP_U_8BIT t4_low;                                  /* Address 30 */
    BP_U_8BIT buspad30;
    BP_U_8BIT t4_high;                                 /* Address 31 */
    BP_U_8BIT buspad31;
    BP_U_8BIT clock_freq;                              /* Address 32 */
    BP_U_8BIT buspad32;
    BP_U_8BIT adc_control;                             /* Address 33 */
    BP_U_8BIT buspad33;
    BP_U_8BIT pll_modes;                               /* Address 34 */
    BP_U_8BIT buspad34;
    BP_U_8BIT test_reg35;                              /* Address 35 */
    BP_U_8BIT buspad35;
    BP_S_8BIT pll_phase_offset_low;                    /* Address 36 */
    BP_U_8BIT buspad36;
    BP_S_8BIT pll_phase_offset_high;                   /* Address 37 */
    BP_U_8BIT buspad37;
    BP_S_8BIT dc_offset_low;                           /* Address 38 */
    BP_U_8BIT buspad38;
    BP_S_8BIT dc_offset_high;                          /* Address 39 */
    BP_U_8BIT buspad39;
    BP_S_8BIT tx_calibrate;                            /* Address 40 */
    BP_U_8BIT buspad40;
    BP_S_8BIT tx_gain;                                 /* Address 41 */
    BP_U_8BIT buspad41;
    BP_S_8BIT noise_histogram_th_low;                  /* Address 42 */
    BP_U_8BIT buspad42;
    BP_S_8BIT noise_histogram_th_high;                 /* Address 43 */
    BP_U_8BIT buspad43;
    BP_S_8BIT ep_pause_th_low;                         /* Address 44 */
    BP_U_8BIT buspad44;
    BP_S_8BIT ep_pause_th_high;                        /* Address 45 */
    BP_U_8BIT buspad45;
    BP_S_8BIT scr_sync_th;                             /* Address 46 */
    BP_U_8BIT buspad46;
    BP_U_8BIT reserved3;                               /* Address 47 */
    BP_U_8BIT buspad47;
    BP_S_8BIT far_end_high_alarm_th_low;               /* Address 48 */
    BP_U_8BIT buspad48;
    BP_S_8BIT far_end_high_alarm_th_high;              /* Address 49 */
    BP_U_8BIT buspad49;
    BP_S_8BIT far_end_low_alarm_th_low;                /* Address 50 */
    BP_U_8BIT buspad50;
    BP_S_8BIT far_end_low_alarm_th_high;               /* Address 51 */
    BP_U_8BIT buspad51;
    BP_S_8BIT snr_alarm_th_low;                        /* Address 52 */
    BP_U_8BIT buspad52;
    BP_S_8BIT snr_alarm_th_high;                       /* Address 53 */
    BP_U_8BIT buspad53;
    BP_S_8BIT cursor_level_low;                        /* Address 54 */
    BP_U_8BIT buspad54;
    BP_S_8BIT cursor_level_high;                       /* Address 55 */
    BP_U_8BIT buspad55;
    BP_S_8BIT dagc_target_low;                         /* Address 56 */
    BP_U_8BIT buspad56;
    BP_S_8BIT dagc_target_high;                        /* Address 57 */
    BP_U_8BIT buspad57;
    BP_U_8BIT detector_modes;                          /* Address 58 */
    BP_U_8BIT buspad58;
    BP_U_8BIT peak_detector_delay;                     /* Address 59 */
    BP_U_8BIT buspad59;
    BP_U_8BIT dagc_modes;                              /* Address 60 */
    BP_U_8BIT buspad60;
    BP_U_8BIT ffe_modes;                               /* Address 61 */
    BP_U_8BIT buspad61;
    BP_U_8BIT ep_modes;                                /* Address 62 */
    BP_U_8BIT buspad62;
    BP_U_8BIT reserved4;                               /* Address 63 */
    BP_U_8BIT buspad63;
    BP_S_8BIT pdm_low;                                 /* Address 64 */
    BP_U_8BIT buspad64;
    BP_S_8BIT pdm_high;                                /* Address 65 */
    BP_U_8BIT buspad65;
    BP_U_8BIT overflow_meter;                          /* Address 66 */
    BP_U_8BIT buspad66;
    BP_U_8BIT reserved5;                               /* Address 67 */
    BP_U_8BIT buspad67;
    BP_S_8BIT dc_meter_low;                            /* Address 68 */
    BP_U_8BIT buspad68;
    BP_S_8BIT dc_meter_high;                           /* Address 69 */
    BP_U_8BIT buspad69;
    BP_S_8BIT slm_low;                                 /* Address 70 */
    BP_U_8BIT buspad70;
    BP_S_8BIT slm_high;                                /* Address 71 */
    BP_U_8BIT buspad71;
    BP_S_8BIT felm_low;                                /* Address 72 */
    BP_U_8BIT buspad72;
    BP_S_8BIT felm_high;                               /* Address 73 */
    BP_U_8BIT buspad73;
    BP_S_8BIT noise_histogram_low;                     /* Address 74 */
    BP_U_8BIT buspad74;
    BP_S_8BIT noise_histogram_high;                    /* Address 75 */
    BP_U_8BIT buspad75;
    BP_S_8BIT ber_meter_low;                           /* Address 76 */
    BP_U_8BIT buspad76;
    BP_S_8BIT ber_meter_high;                          /* Address 77 */
    BP_U_8BIT buspad77;
    BP_U_8BIT symbol_histogram;                        /* Address 78 */
    BP_U_8BIT buspad78;
    BP_S_8BIT reserved6;                               /* Address 79 */
    BP_U_8BIT buspad79;
    BP_S_8BIT nlm_low;                                 /* Address 80 */
    BP_U_8BIT buspad80;
    BP_S_8BIT nlm_high;                                /* Address 81 */
    BP_U_8BIT buspad81;
    BP_U_16BIT reserved7[12];                          /* Address 82-93 */
    BP_S_8BIT vcxo_frequency_low;                      /* Address 94 */
    BP_U_8BIT buspad94;
    BP_S_8BIT vcxo_frequency_high;                     /* Address 95 */
    BP_U_8BIT buspad95;
    BP_U_16BIT reserved8[16];                          /* Address 96 - 111 */
    BP_U_8BIT linear_ec_tap_select_read;               /* Address 112 */
    BP_U_8BIT buspad112;
    BP_U_8BIT linear_ec_tap_select_write;              /* Address 113 */
    BP_U_8BIT buspad113;
    BP_U_8BIT nonlinear_ec_tap_select_read;            /* Address 114 */
    BP_U_8BIT buspad114;
    BP_U_8BIT nonlinear_ec_tap_select_write;           /* Address 115 */
    BP_U_8BIT buspad115;
    BP_U_8BIT dfe_tap_select_read;                     /* Address 116 */
    BP_U_8BIT buspad116;
    BP_U_8BIT dfe_tap_select_write;                    /* Address 117 */
    BP_U_8BIT buspad117;
    BP_U_8BIT scratch_pad_tap_select_read;             /* Address 118 */
    BP_U_8BIT buspad118;
    BP_U_8BIT scratch_pad_tap_select_write;            /* Address 119 */
    BP_U_8BIT buspad119;
    BP_U_8BIT eq_add_read;                             /* Address 120 */
    BP_U_8BIT buspad120;
    BP_U_8BIT eq_add_write;                            /* Address 121 */
    BP_U_8BIT buspad121;
    BP_U_8BIT eq_microcode_add_read;                   /* Address 122 */
    BP_U_8BIT buspad122;
    BP_U_8BIT eq_microcode_add_write;                  /* Address 123 */
    BP_U_8BIT buspad123;
    BP_S_8BIT access_data_byte0;                       /* Address 124 */
    BP_U_8BIT buspad124;
    BP_S_8BIT access_data_byte1;                       /* Address 125 */
    BP_U_8BIT buspad125;
    BP_S_8BIT access_data_byte2;                       /* Address 126 */
    BP_U_8BIT buspad126;
    BP_S_8BIT access_data_byte3;                       /* Address 127 */
    BP_U_8BIT buspad127;
}BP_PACKED2 transceiver;

extern volatile transceiver Bitpump_regs;

#define DECLARE_PTR transceiver BP_VOLATILE * const bp_ptr = &Bitpump_regs
#define DECLARE_MODE_PTR transceiver_modes BP_VOLATILE * const bp_mode_ptr \
			= (volatile transceiver_modes *) &Bitpump_regs

/*
 * The layout of the EQ register space is determined
 * by the microcode image in microcode.c
 *
 * We no longer store pure software state in the EQ register space
 * or even pretend to do so.
 */

/* Define EQ memory addresses */
#define DAGC_HIGH        27
#define DAGC_LOW         26
#define LAST_FFE          7
#define FFE               0
#define EP               16

/*
 * This one is also determined by the microcode
 */
#define PKD_DELAY_INIT_VALUE          4
