/*
 * Configuration and stats display functions
 */

#include "types.h"
#include "stdio.h"
#include "strings.h"
#include "struct.h"
#include "global.h"

extern struct vc vc_table[];
extern struct l2conv_global_config global_cfg;

show_vc_conf()
{
	struct vc *vc;
	u_long atmaddr;
	char *modestr, dlcistr[16], *llcstr;
	int i;

	printf("VPI\tVCI\tMode\tDLCI\tLLC header\r\n");
	for (vc = vc_table, i = 0; i < global_cfg.nvcs; vc++, i++) {
		atmaddr = *(u_long *)vc->cellhdr_normal;
		switch (vc->convmode) {
		case CONVMODE_FRF5:
			modestr = "FRF.5";
			break;
		case CONVMODE_FRF8:
			modestr = "FRF.8";
			break;
		case CONVMODE_FUNI:
			modestr = "FUNI";
			break;
		case CONVMODE_PPPOA_LLC:
		case CONVMODE_PPPOA_VC:
			modestr = "PPPoA";
			break;
		case CONVMODE_IPV4_VC:
			modestr = "FR/IPv4";
			break;
		default:
			modestr = "???";
		}
		if (global_cfg.hdlc_handling_mode == HDLC_MODE_Q922)
			sprintf(dlcistr, "%u",
				((vc->q922_hdr & 0xFC00) >> 6) |
				((vc->q922_hdr & 0x00F0) >> 4));
		else
			strcpy(dlcistr, "N/A");
		switch (vc->convmode) {
		case CONVMODE_FRF8:
		case CONVMODE_PPPOA_LLC:
			llcstr = "Yes";
			break;
		case CONVMODE_PPPOA_VC:
		case CONVMODE_IPV4_VC:
			llcstr = "No";
			break;
		default:
			llcstr = "N/A";
		}
		printf("%u\t%u\t%s\t%s\t%s\r\n", (atmaddr & 0x0FF00000) >> 20,
			(atmaddr & 0x000FFFF0) >> 4, modestr, dlcistr, llcstr);
	}
	printf("Total VCs: %u\r\n", global_cfg.nvcs);
	switch (global_cfg.hdlc_handling_mode) {
	case HDLC_MODE_Q922:
		modestr = "Q.922";
		break;
	case HDLC_MODE_FRF5:
		modestr = "FRF.5";
		break;
	case HDLC_MODE_PPP:
		modestr = "RFC 1662";
		break;
	default:
		modestr = "???";
	}
	printf("HDLC->ATM handling mode: %s\r\n", modestr);
	return(0);
}

show_vc_stats()
{
	struct vc *vc;
	u_long atmaddr;
	int i;

	for (vc = vc_table, i = 0; i < global_cfg.nvcs; vc++, i++) {
		atmaddr = *(u_long *)vc->cellhdr_normal;
		printf("VC %u*%u:\r\n", (atmaddr & 0x0FF00000) >> 20,
			(atmaddr & 0x000FFFF0) >> 4);
		printf("  AAL5 packets received: %lu\r\n", vc->atm_rx_packets);
		printf("  Oversize packets received: %lu\r\n",
			vc->atm_rx_giants);
		printf("  Bad AAL5 CRC-32: %lu\r\n", vc->atm_rx_badcrc32);
		printf("  AAL5 aborts received: %lu\r\n", vc->atm_rx_aborts);
		printf("  Bad length in AAL5 trailer: %lu\r\n",
			vc->atm_rx_badlen);
		if (vc->convmode == CONVMODE_FRF8 ||
		    vc->convmode == CONVMODE_PPPOA_LLC) {
			printf("  RFC 1483 packet types received:\r\n");
			printf("    IPv4: %lu\r\n", vc->rx_encap_ipv4);
			printf("    IPv6: %lu\r\n", vc->rx_encap_ipv6);
			printf("    Other routed: %lu\r\n",
				vc->rx_encap_routed);
			printf("    Bridged: %lu\r\n", vc->rx_encap_bridged);
			printf("    Other OUIs: %lu\r\n",
				vc->rx_encap_otheroui);
			printf("    PPPoA: %lu\r\n", vc->rx_encap_ppp);
			printf("    Other NLPIDs: %lu\r\n", vc->rx_encap_nlpid);
			printf("    Invalid: %lu\r\n", vc->rx_encap_invalid);
		}
		printf("  Tx packets converted: %lu\r\n",
			vc->tx_packets_converted);
	}
	return(0);
}

zero_vc_stats()
{
	struct vc *vc;
	int i;

	for (vc = vc_table, i = 0; i < global_cfg.nvcs; vc++, i++) {
		vc->atm_rx_packets = 0;
		vc->atm_rx_giants = 0;
		vc->atm_rx_badcrc32 = 0;
		vc->atm_rx_aborts = 0;
		vc->atm_rx_badlen = 0;
		vc->rx_encap_ipv4 = 0;
		vc->rx_encap_ipv6 = 0;
		vc->rx_encap_routed = 0;
		vc->rx_encap_bridged = 0;
		vc->rx_encap_otheroui = 0;
		vc->rx_encap_ppp = 0;
		vc->rx_encap_nlpid = 0;
		vc->rx_encap_invalid = 0;
		vc->tx_packets_converted = 0;
	}
	return(0);
}
