/*
 * Packet conversion in the HDLC->ATM direction
 */

#include "types.h"
#include "atm.h"
#include "l2encap.h"
#include "struct.h"
#include "global.h"
#include "frf8.h"

extern u_long crc32();

extern struct vc vc_table[];
extern struct l2conv_global_config global_cfg;

extern u_char llc_magic_snap[3];
extern u_char llc_magic_nlpid[3];
extern u_char oui_ethertype[3];
extern u_char oui_bridgetypes[3];

/*
 * Input assumptions:
 *   HDLC framing/FCS checks passed
 *   buf_datalen does NOT include FCS octets
 *
 * On successful return:
 *   buf_vc set
 *   AAL5 trailer filled in (with or without CRC-32)
 *   buf_datalen is a multiple of 48, ready for use as a cell count
 *
 * Return value 0:  converted successfully
 * Return value -1: some rule violated, discard packet
 * no stats counted
 */
pkt_hdlc2atm(buf, do_crc32)
	struct buffer *buf;
{
	struct vc *vc;
	u_char *headptr;
	u_short q922_addr;
	struct rfc1483_hdr_short *s1483;
	struct rfc1483_hdr_long *l1483;
	struct fr_hdr_short *frs;
	struct fr_hdr_long *frl;
	int frf8_mode, frf8_proto;
	struct aal5_trailer *tr;
	int i;

	headptr = buf->buf_dataspc + buf->buf_dataoff;

	/* first find the VC it's going to */
	switch (global_cfg.hdlc_handling_mode) {
	case HDLC_MODE_Q922:
		q922_addr = *(u_short *)headptr & 0xFFF1;
		for (vc = vc_table, i = 0; i < global_cfg.nvcs; vc++, i++)
			if (q922_addr == vc->q922_hdr)
				break;
		if (i >= global_cfg.nvcs)
			return(-1);
		break;
	case HDLC_MODE_PPP:
		if (*(u_short *)headptr != 0xFF03)
			return(-1);
		headptr += 2;
		buf->buf_dataoff += 2;
		buf->buf_datalen -= 2;
		/* FALL THRU */
	case HDLC_MODE_FRF5:
		/* only one VC exists in these modes */
		vc = vc_table;
		break;
	default:
		printf("L2CONV bug: unknown HDLC handling mode\r\n");
		asm("trap #0");
	}
	buf->buf_vc = vc;

	/* now handle the actual conversion modes */
	switch (vc->convmode) {
	case CONVMODE_FRF5:
		goto aal5_out;
	case CONVMODE_FUNI:
		headptr += 2;
		buf->buf_dataoff += 2;
		buf->buf_datalen -= 2;
		goto aal5_out;
	case CONVMODE_PPPOA_LLC:
		headptr -= sizeof(struct rfc1483_hdr_short);
		buf->buf_dataoff -= sizeof(struct rfc1483_hdr_short);
		buf->buf_datalen += sizeof(struct rfc1483_hdr_short);
		s1483 = (struct rfc1483_hdr_short *) headptr;
		bcopy(llc_magic_nlpid, s1483->llc, 3);
		s1483->nlpid = NLPID_PPP;
		goto aal5_out;
	case CONVMODE_PPPOA_VC:
		goto aal5_out;
	}

	/* gotta parse RFC 1490 header */
	if (buf->buf_datalen < sizeof(struct fr_hdr_short))
		return(-1);
	frs = (struct fr_hdr_short *) headptr;
	if (frs->fr_control != FR_CONTROL_UI)
		return(-1);
	switch (frs->fr_nlpid) {
	case NLPID_PAD:
		/* parse the long header */
		break;
	case NLPID_SNAP:
		/* invalid without pad! */
		return(-1);
	case NLPID_IPv4:
		frf8_mode = FRF8_MODE_SNAP_ROUTED;
		frf8_proto = ETHERTYPE_IP;
		goto strip_frs;
	case NLPID_IPv6:
		frf8_mode = FRF8_MODE_SNAP_ROUTED;
		frf8_proto = ETHERTYPE_IPv6;
		goto strip_frs;
	default:
		frf8_mode = FRF8_MODE_NLPID;
		frf8_proto = frs->fr_nlpid;
	strip_frs:
		headptr += sizeof(struct fr_hdr_short);
		buf->buf_dataoff += sizeof(struct fr_hdr_short);
		buf->buf_datalen -= sizeof(struct fr_hdr_short);
		goto frf8_logic;
	}
	/* do the long version */
	if (buf->buf_datalen < sizeof(struct fr_hdr_long))
		return(-1);
	frl = (struct fr_hdr_long *) headptr;
	headptr += sizeof(struct fr_hdr_long);
	buf->buf_dataoff += sizeof(struct fr_hdr_long);
	buf->buf_datalen -= sizeof(struct fr_hdr_long);
	if (frl->fr_nlpid2 != NLPID_SNAP)
		return(-1);
	if (!bcmp(frl->fr_snap_oui, oui_ethertype, 3)) {
		frf8_mode = FRF8_MODE_SNAP_ROUTED;
		frf8_proto = frl->fr_snap_type;
		goto frf8_logic;
	} else if (!bcmp(frl->fr_snap_oui, oui_bridgetypes, 3)) {
		frf8_mode = FRF8_MODE_SNAP_BRIDGED;
		frf8_proto = frl->fr_snap_type;
		goto frf8_logic;
	} else {
		headptr -= sizeof(struct rfc1483_hdr_long);
		buf->buf_dataoff -= sizeof(struct rfc1483_hdr_long);
		buf->buf_datalen += sizeof(struct rfc1483_hdr_long);
		l1483 = (struct rfc1483_hdr_long *) headptr;
		bcopy(llc_magic_snap, l1483->llc, 3);
		frf8_mode = FRF8_MODE_SNAP_UNKNOWN;
	}

frf8_logic:
	if (vc->convmode == CONVMODE_IPV4_VC) {
		if (frf8_mode == FRF8_MODE_SNAP_ROUTED &&
		    frf8_proto == ETHERTYPE_IP)
			goto aal5_out;
		else
			return(-1);
	}
	switch (frf8_mode) {
	case FRF8_MODE_NLPID:
		headptr -= sizeof(struct rfc1483_hdr_short);
		buf->buf_dataoff -= sizeof(struct rfc1483_hdr_short);
		buf->buf_datalen += sizeof(struct rfc1483_hdr_short);
		s1483 = (struct rfc1483_hdr_short *) headptr;
		bcopy(llc_magic_nlpid, s1483->llc, 3);
		s1483->nlpid = frf8_proto;
		goto aal5_out;
	case FRF8_MODE_SNAP_ROUTED:
		headptr -= sizeof(struct rfc1483_hdr_long);
		buf->buf_dataoff -= sizeof(struct rfc1483_hdr_long);
		buf->buf_datalen += sizeof(struct rfc1483_hdr_long);
		l1483 = (struct rfc1483_hdr_long *) headptr;
		bcopy(llc_magic_snap, l1483->llc, 3);
		bcopy(oui_ethertype, l1483->snap_oui, 3);
		l1483->snap_type = frf8_proto;
		goto aal5_out;
	case FRF8_MODE_SNAP_BRIDGED:
		switch (frf8_proto) {
		case 0x0001:
		case 0x0002:
		case 0x0003:
		case 0x0004:
		case 0x0007:
		case 0x0008:
		case 0x0009:
		case 0x000A:
			/* insert 2 pad octets */
			headptr -= 2;
			buf->buf_dataoff -= 2;
			buf->buf_datalen += 2;
			*(u_short *)headptr = 0;
			/* FALL THRU */
		case 0x000B:
		case 0x000E:
		case 0x000F:
			break;
		default:
			return(-1);
		}
		headptr -= sizeof(struct rfc1483_hdr_long);
		buf->buf_dataoff -= sizeof(struct rfc1483_hdr_long);
		buf->buf_datalen += sizeof(struct rfc1483_hdr_long);
		l1483 = (struct rfc1483_hdr_long *) headptr;
		bcopy(llc_magic_snap, l1483->llc, 3);
		bcopy(oui_bridgetypes, l1483->snap_oui, 3);
		l1483->snap_type = frf8_proto;
		goto aal5_out;
	case FRF8_MODE_SNAP_UNKNOWN:
		/* already done */
		goto aal5_out;
	}

aal5_out:
	if (buf->buf_datalen > AAL5_MAXPAYLOAD)
		return(-1);
	i = (buf->buf_datalen + sizeof(struct aal5_trailer) + 47) / 48;
	tr = (struct aal5_trailer *)
		(headptr + i*48 - sizeof(struct aal5_trailer));
	tr->cpcs_uu = 0;
	tr->cpcs_cpi = 0;
	tr->cpcs_length = buf->buf_datalen;
	buf->buf_datalen = i * 48;
	if (do_crc32)
		tr->cpcs_crc32 = ~crc32(AAL5_CRC32_INIT, headptr,
					buf->buf_datalen - 4);
	return(0);
}
