/*
 * craft terminal statistics display
 * based on the Hack-o-Rocket router version
 */

#include "types.h"
#include "ctype.h"
#include "stdio.h"
#include "strings.h"
#include "globals.h"

extern int hdlc_show_hw_stats();

struct keyword {
	char *word;
	int min;
	long id;
};

static long
find_keyword(word, len, table, ambig)
	char *word;
	int len;
	struct keyword table[];
	int *ambig;
{
	struct keyword *tp;

	for (tp = table; tp->word != NULL; tp++) {
		char *p1 = word, *p2 = tp->word;
		int cnt = 0;

		for (; cnt < len && *p2; p1++, p2++, cnt++) {
			char ch;

			ch = islower(*p1) ? toupper(*p1) : *p1;
			if (ch != *p2)
				break;
		}
		if (cnt < len)
			continue;
		if (cnt >= tp->min)
			return(tp->id);
		*ambig = 1;
	}
	return(-1);
}

cmd_stats_atm()
{
	printf("ATM layer statistics:\r\n");
	printf("  Good cells received: %lu\r\n", stats.atm_rx_good_cells);
	printf("  Non-understood cells received: %lu\r\n",
		stats.atm_rx_wrong_cells);
	printf("  HEC errors: %lu\r\n", stats.atm_rx_badhec);
	printf("  Nonzero GFC field received: %lu\r\n",
		stats.atm_rx_nonzero_gfc);
	printf("  Non-idle cells transmitted: %lu\r\n", stats.atm_tx_cells);
}

cmd_stats_a2h()
{
	printf("ATM->HDLC conversion statistics:\r\n");
	printf("  AAL5 packets received: %lu\r\n", stats.a2h_rx_packets);
	printf("  Packets sent to HDLC: %lu\r\n", stats.a2h_tx_packets);
	printf("  Packets dropped (congestion): %lu\r\n",
		stats.a2h_dropped_congest);
	printf("  Packets dropped (conversion failed): %lu\r\n",
		stats.a2h_dropped_convfail);
	printf("  HDLC Tx errors: %lu\r\n", stats.a2h_tx_errs);
}

cmd_stats_h2a()
{
	printf("HDLC->ATM conversion statistics:\r\n");
	printf("  Good frames received: %lu\r\n", stats.h2a_rx_packets);
	printf("  Rx framing errors: %lu\r\n", stats.h2a_rx_errors);
	printf("  AAL5 packets formed: %lu\r\n", stats.h2a_tx_packets);
	printf("  Packets dropped (SDSL down): %lu\r\n",
		stats.h2a_dropped_sdsldown);
	printf("  Packets dropped (congestion): %lu\r\n",
		stats.h2a_dropped_congest);
	printf("  Packets dropped (conversion failed): %lu\r\n",
		stats.h2a_dropped_invalid);
}

cmd_stats_sdsl()
{
	printf("SDSL layer statistics:\r\n");
	printf("  Successful startup cycles: %lu\r\n", stats.sdsl_activations);
	printf("  Shutdowns due to LOSW: %lu\r\n", stats.sdsl_losw_shutdowns);
	printf("  Frame Rx overruns: %lu\r\n", stats.sdsl_frame_rx_overruns);
	printf("  EOC messages received: %lu\r\n", stats.eoc_good_msgs_rcvd);
	printf("  EOC garbage received: %lu\r\n", stats.eoc_garbage_rcvd);
}

cmd_stats_pervc()
{
	l2conv_api->show_vc_stats();
}

cmd_stats_all()
{
	cmd_stats_sdsl();
	cmd_stats_atm();
	l2conv_api->show_vc_stats();
	cmd_stats_a2h();
	cmd_stats_h2a();
	hdlc_show_hw_stats();
}

cmd_zerostats()
{
	spl7();
	bzero(&stats, sizeof stats);
	l2conv_api->zero_vc_stats();
	hdlc_zero_hw_stats();
	spl0();
	printf("Statistics counters cleared\r\n");
}

static const struct keyword stats_subcmds[] = {
	{"A2H", 2, (long) &cmd_stats_a2h},
	{"ALL", 2, (long) &cmd_stats_all},
	{"ATM", 2, (long) &cmd_stats_atm},
	{"CLEAR", 1, (long) &cmd_zerostats},
	{"FULL", 1, (long) &cmd_stats_all},
	{"H2A", 1, (long) &cmd_stats_h2a},
	{"PERVC", 1, (long) &cmd_stats_pervc},
	{"SCC2", 2, (long) &hdlc_show_hw_stats},
	{"SDSL", 2, (long) &cmd_stats_sdsl},
	{"VC", 1, (long) &cmd_stats_pervc},
	{"ZERO", 1, (long) &cmd_zerostats},
	{NULL, 0, -1}};

cmd_stats(arg)
	char *arg;
{
	char *cp, *np;
	int (*funcp)();
	int ambig;

	for (cp = arg; isspace(*cp); cp++)
		;
	if (!*cp) {
help:		printf("Available statistics:\r\n");
		printf("  STATS A2H      -- ATM->HDLC conversion statistics\r\n");
		printf("  STATS H2A      -- HDLC->ATM conversion statistics\r\n");
		printf("  STATS ATM      -- ATM layer statistics\r\n");
		printf("  STATS VC       -- Per-VC statistics\r\n");
		printf("  STATS SDSL     -- SDSL layer statistics\r\n");
		printf("  STATS SCC2     -- SCC2 hardware statistics\r\n");
		printf("  STATS ALL      -- show all statistics\r\n");
		printf("  STATS FULL     -- ditto\r\n");
		printf("  STATS CLEAR    -- zero all statistics counters\r\n");
		printf("  STATS ZERO     -- ditto\r\n");
		return;
	}
	for (np = cp; *cp && !isspace(*cp); cp++)
		;
	funcp = (int (*)()) find_keyword(np, cp - np, stats_subcmds, &ambig);
	if ((long) funcp != -1)
		(*funcp)();
	else
		goto help;
}
