/*
 * The black magic of implementing Nokia SDSL/ATM
 * using only an MC68302 SCC.
 * Now reduced to just the Rx direction.
 */

#include "types.h"
#include "globals.h"
#include "nokia.h"
#include "nokia_impl.h"
#include "mc68302.h"
#include "cp.h"
#include "intr.h"
#include "gpio.h"
#include "sccsave.h"
#include "sdsl_asm.h"
#include "../libutil/hec.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile struct mc68302_scc_bd scc1_rx_bds[8];
extern volatile struct mc68302_scc_params scc1_params;
extern struct scc_saved_initstate scc1_saved_initstate;

extern volatile long timebase;

struct nokia_frame_scc nokia_rxf_bufs[NFRX_BUFS] __attribute__ ((aligned (2)));
int nokia_rxf_bufptr;

/* This is where we prepare all our machinery */
nokia_rx_prepare()
{
	int i;

	scc1_params.rx_int_state = scc1_saved_initstate.rx_int_state;
	scc1_params.rbdn = scc1_saved_initstate.rbdn;

	mc68302_regs.scc1.scm = SCM_SOFT_CTS_CD | SCM_BISYNC_REVD
				| SCM_BISYNC_NTSYN | SCM_MODE_BISYNC;
	mc68302_regs.scc1.dsr = NOKIA_SCC_SYNC_WORD;
	for (i = 0; i < NFRX_BUFS; i++) {
		scc1_rx_bds[i].bd_flags = 0xD000;
		scc1_rx_bds[i].bd_bufptr = (u_char *) &nokia_rxf_bufs[i];
	}
	scc1_rx_bds[NFRX_BUFS-1].bd_flags = 0xF000;
	nokia_rxf_bufptr = 0;

	scc1_params.rfcr = 0x50;
	scc1_params.mrblr = 432;
	mc68302_regs.scc1.scce = 0xFF;
	mc68302_regs.scc1.sccm = 0x05;
	nokia_rx_state = NOKIARX_STATE_HUNT;
	mc68302_regs.imr |= INTMASK_SCC1;
	return(0);
}

nokia_scc_inthandler()
{
	u_char evt;

	/*
	 * We don't bother with the spl3 trick any more. There are no
	 * higher priority interrupt sources at IPL 4 than SCC1 that are
	 * used on this board, so those spl3() and spl4() calls would be
	 * just extra delay.
	 */
	evt = mc68302_regs.scc1.scce & mc68302_regs.scc1.sccm;
	mc68302_regs.scc1.scce = evt;
	if (evt & 0x04)
		nokia_rx_state = NOKIARX_STATE_OVERRUN;
	nokia_scc_rx_handler();
}

nokia_scc_rx_handler()
{
	int i;
	u_short flags;

	for (i = nokia_rxf_bufptr; ; ) {
		flags = scc1_rx_bds[i].bd_flags;
		if (flags & 0x8000)
			break;
		switch (nokia_rx_state) {
		case NOKIARX_STATE_HUNT:
			/* receiving *anything* means the SCC synced */
			nokia_rx_state = NOKIARX_STATE_SYNC;
			nokia_rx_counter = 0;
			/* fall through */
		case NOKIARX_STATE_SYNC:
			if (flags & 0x0002) {
				nokia_rx_state = NOKIARX_STATE_OVERRUN;
				break;
			}
			nokia_pass_rxframe_to_atm(&nokia_rxf_bufs[i]);
			eocdump_input(nokia_rxf_bufs[i].hdrbytes[1]);
			if (nokia_rxf_bufs[i].hdrbytes[7] == NOKIA_SYNC_OCTET)
				nokia_rx_counter = 0;
			else
				nokia_rx_counter++;
			if (nokia_rx_counter >= nokia_losw_count)
				nokia_rx_state = NOKIARX_STATE_LOSW;
			break;
		default:
			/*
			 * Once we are in one of the error states,
			 * do nothing more.
			 */
		}
		scc1_rx_bds[i].bd_flags = (flags & 0x7000) | 0x8000;
		i++;
		if (i >= NFRX_BUFS)
			i = 0;
	}
	nokia_rxf_bufptr = i;
}

nokia_pass_rxframe_to_atm(frame)
	struct nokia_frame_scc *frame;
{
	int i, badhec;

	for (i = 0; i < 8; i++) {
		badhec = compute_hec(frame->cells[i].cellhdr) !=
				frame->cells[i].cellhdr[4];
		atm_cell_rx(&frame->cells[i], badhec);
		if (badhec)
			stats.atm_rx_badhec++;
	}
}

sdsl_startup_success()
{
	fpga_tx_enable();
	mc68302_regs.scc1.scm |= SCM_ENR;
	printf("Bitpump activation successful, hunting for Nokia frames\r\n");
	printf("Startup time %d s\r\n", (timebase - sdsl_asm_state_time) / 20);
	set_sdsl_state(FRAME_ACQU_STATE);
	mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
				| PORTA_LSLED_ORANGE;
}

sdsl_rx_shutdown()
{
	mc68302_regs.scc1.scm &= ~SCM_ENR;
	spl4();
	eocdump_init();
	spl0();
}

nokia_handle_rx_overrun()
{
	printf("Nokia frame Rx overrun error; fatal condition\r\n");
	stats.sdsl_frame_rx_overruns++;
	set_sdsl_state(DEACTIVATED_STATE);
}
