/*
 * Nokia SDSL Tx direction via the NOKFTX FPGA
 */

#include "types.h"
#include "globals.h"
#include "nokia.h"
#include "nokftx.h"
#include "monitor_api.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"

extern volatile struct nokftx_memblock FPGA_regs;

extern struct buffer *get_buffer_from_queue();

struct buffer *txpkt_buf;
u_char *txpkt_ptr, *txpkt_endptr;

fpga_regs_init()
{
	if (FPGA_regs.id_reg != NOKFTX_IDREG) {
		monapi_error("NOKFTX FPGA not found");
		return(-1);
	}
	FPGA_regs.csr = NOKFTX_CSR_CRC32_EN | NOKFTX_CSR_IDLE_EN;
	return(0);
}

fpga_init_frame_buf(buf)
	struct nokftx_frame *buf;
{
	int i;

	for (i = 0; i < 8; i++) {
		*(u_long *)buf->cells[i].cellhdr = 0x00000001;
		buf->cells[i].hec_coset = 0x55;
	}
	buf->trailer[0] = 0x00;
	buf->trailer[1] = tx_eoc_octet;
	buf->trailer[2] = 0x00;
	buf->trailer[3] = 0x00;
	buf->trailer[4] = 0x00;
	buf->trailer[5] = 0x00;
	buf->crc6 = 0;
	buf->sync_octet = NOKIA_SYNC_OCTET;
}

fpga_idleframes_init()
{
	fpga_init_frame_buf(&FPGA_regs.buffers[0]);
	fpga_init_frame_buf(&FPGA_regs.buffers[1]);
	txpkt_buf = NULL;
	initidle_count = nokia_initial_idle;
	tx_eoc_msg_count = 0;
}

fpga_reset_pulse()
{
	u_short csr;

	csr = FPGA_regs.csr;
	csr |= NOKFTX_CSR_RESET;
	FPGA_regs.csr = csr;
	/* 300 us is over 20 symbols at 144 kbps, so it should be enough */
	cpu_delay_us(300);
	csr &= ~NOKFTX_CSR_RESET;
	FPGA_regs.csr = csr;
}

fpga_tx_enable()
{
	sdcore_api->set_tx(&sdcore_state, 1, SCRAMBLED_FOUR_LEVEL_DATA);
	FPGA_regs.csr |= NOKFTX_CSR_INT_EN;
}

fill_tx_frame(frame)
	struct nokftx_frame *frame;
{
	struct vc *vc;
	int cellno;

	if (initidle_count) {
		initidle_count--;
		return;
	}
	for (cellno = 0; cellno < 8; cellno++) {
		if (!txpkt_buf) {
			txpkt_buf = get_buffer_from_queue(&atm_outgoing_queue);
			if (!txpkt_buf)
				break;
			txpkt_ptr = txpkt_buf->buf_dataspc +
					txpkt_buf->buf_dataoff;
			txpkt_endptr = txpkt_ptr + txpkt_buf->buf_datalen;
		}
		vc = txpkt_buf->buf_vc;
		bcopy_atmcell(txpkt_ptr, frame->cells[cellno].payload);
		txpkt_ptr += 48;
		if (txpkt_ptr < txpkt_endptr) {
			*(u_long *)frame->cells[cellno].cellhdr =
				*(u_long *)vc->cellhdr_normal;
		} else {
			*(u_long *)frame->cells[cellno].cellhdr =
				*(u_long *)vc->cellhdr_last;
			l2conv_api->free_buffer(txpkt_buf);
			txpkt_buf = NULL;
		}
		stats.atm_tx_cells++;
	}
	/* pad any remainder with idle cells */
	while (cellno < 8)
		*(u_long *)frame->cells[cellno++].cellhdr = 0x00000001;
	/* EOC output toy */
	if (tx_eoc_msg_count) {
		frame->trailer[1] = *tx_eoc_msg_ptr++;
		tx_eoc_msg_count--;
	}
}

nokftx_inthandler()
{
	if (FPGA_regs.csr & NOKFTX_CSR_BUF_LAST_WR) {
		FPGA_regs.csr &= ~NOKFTX_CSR_BUF_LAST_WR;
		fill_tx_frame(&FPGA_regs.buffers[0]);
	} else {
		FPGA_regs.csr |= NOKFTX_CSR_BUF_LAST_WR;
		fill_tx_frame(&FPGA_regs.buffers[1]);
	}
}

fpga_txbuf_clear()
{
	struct buffer *buf;

	FPGA_regs.csr &= ~NOKFTX_CSR_INT_EN;
	while (buf = get_buffer_from_queue(&atm_outgoing_queue))
		l2conv_api->free_buffer(buf);
	if (txpkt_buf) {
		l2conv_api->free_buffer(txpkt_buf);
		txpkt_buf = NULL;
	}
}
