/*
 * SCC2 HDLC code
 */

#include "types.h"
#include "mc68302.h"
#include "cp.h"
#include "intr.h"
#include "sdsl_asm.h"
#include "globals.h"
#include "../l2conv/bufmac.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile struct mc68302_scc_bd scc2_rx_bds[8];
extern volatile struct mc68302_scc_bd scc2_tx_bds[8];
extern volatile struct mc68302_scc_hdlcparams scc2_params;
extern struct scc_saved_initstate scc2_saved_initstate;

hdlc_init()
{
	struct buffer *buf;
	int i;

	mc68302_regs.scc2.scm = SCM_HDLC_FSE | SCM_HDLC_FLG | SCM_SOFT_CTS_CD
				| SCM_MODE_HDLC;

	for (i = 0; i < scc2_rxring_size; i++) {
		buf = l2conv_api->get_buffer();
		scc2_rx_bds[i].bd_bufptr = &buf->buf_dataspc[4];
		scc2_rx_bds[i].bd_flags = 0xC000;
	}
	scc2_rx_bds[scc2_rxring_size-1].bd_flags = 0xE000;
	hdlc_recv_ptr = 0;

	for (i = 0; i < 7; i++)
		scc2_tx_bds[i].bd_flags = 0;
	scc2_tx_bds[7].bd_flags = 0x2000;
	hdlc_txput_ptr = hdlc_txdone_ptr = hdlc_tx_outstanding = 0;

	scc2_params.rfcr = 0x50;
	scc2_params.tfcr = 0x50;
	scc2_params.mrblr = HDLC_RX_LIMIT;
	scc2_params.c_mask_l = 0xF0B8;
	scc2_params.disfc = 0;
	scc2_params.crcec = 0;
	scc2_params.abtsc = 0;
	scc2_params.nmarc = 0;
	scc2_params.retrc = 0;
	scc2_params.mflr = HDLC_RX_LIMIT;
	scc2_params.hmask = 0;

	mc68302_regs.scc2.scce = 0xFF;
	mc68302_regs.scc2.sccm = 0x12;
	mc68302_regs.imr |= INTMASK_SCC2;
	mc68302_regs.scc2.scm |= SCM_ENR | SCM_ENT;
	return(0);
}

hdlc2atm_process()
{
	volatile struct mc68302_scc_bd *bd;
	struct buffer *buf, *newbuf;
	u_short flags, len;

	bd = scc2_rx_bds + hdlc_recv_ptr;
	flags = bd->bd_flags;
	if (flags & 0x8000)
		return;
	len = bd->bd_len;
	buf = buf_from_midptr(bd->bd_bufptr);
	newbuf = l2conv_api->get_buffer();
	bd->bd_bufptr = &newbuf->buf_dataspc[4];
	bd->bd_flags = (flags & 0x7000) | 0x8000;
	hdlc_recv_ptr++;
	if (hdlc_recv_ptr >= scc2_rxring_size)
		hdlc_recv_ptr = 0;

	/* process the booty */
	if ((flags & 0x003F) || (len < 5)) {
		stats.h2a_rx_errors++;
		l2conv_api->free_buffer(buf);
		return;
	}
	buf->buf_dataoff = 4;
	buf->buf_datalen = len - 2;
	stats.h2a_rx_packets++;
	if (sdsl_asm_state != ACTIVE_STATE) {
		l2conv_api->free_buffer(buf);
		stats.h2a_dropped_sdsldown++;
		return;
	}
	if (l2conv_api->hdlc2atm(buf, 0) < 0) {
		l2conv_api->free_buffer(buf);
		stats.h2a_dropped_invalid++;
		return;
	}
	buf->buf_vc->tx_packets_converted++;
	put_buffer_on_queue(buf, &atm_outgoing_queue);
	stats.h2a_tx_packets++;
	if (queue_head_drop(&atm_outgoing_queue, atm_outq_limit))
		stats.h2a_dropped_congest++;
}

scc2_interrupt_handler()
{
	/*
	 * We *do* want to lower IPL to 3 in this case
	 * so the SCC1 interrupt handler remains at higher priority
	 * However, we've moved it into the assembly wrapper
	 */
	mc68302_regs.scc2.scce = 0x12;
	hdlc_reclaim_txbufs();
}

hdlc_post_tx(buf)
	struct buffer *buf;
{
	int i;

	i = hdlc_txput_ptr;
	scc2_tx_bds[i].bd_len = buf->buf_datalen;
	scc2_tx_bds[i].bd_bufptr = buf->buf_dataspc + buf->buf_dataoff;
	scc2_tx_bds[i].bd_flags = 0xDC00 | scc2_tx_bds[i].bd_flags & 0x2000;
	i++;
	if (i >= 8)
		i = 0;
	hdlc_txput_ptr = i;
	hdlc_tx_outstanding++;
}

hdlc_reclaim_txbufs()
{
	int i;
	u_short flags;
	struct buffer *buf;

	for (i = hdlc_txdone_ptr; hdlc_tx_outstanding; ) {
		flags = scc2_tx_bds[i].bd_flags;
		if (flags & 0x8000)
			break;
		buf = buf_from_midptr(scc2_tx_bds[i].bd_bufptr);
		/*
		 * We want to do hdlc_tx_outstanding--;
		 * But we need it to be atomic wrt higher-priority interrupts,
		 * which means a single instruction.
		 * gcc is too stupid to do it, so we use asm.
		 */
		asm("subq.l #1,hdlc_tx_outstanding");
		l2conv_api->free_buffer(buf);
		if (flags & 0x0003) {
			stats.a2h_tx_errs++;
			send_cp_command(CR_SCC2, CPCOM_RESTART_TX);
		}
		i++;
		if (i >= 8)
			i = 0;
	}
	hdlc_txdone_ptr = i;
}

hdlc_show_hw_stats()
{
	printf("SCC2 HDLC statistics (counted by the SCC):\r\n");
	printf("  Discarded frames (no buffers): %lu\r\n",
		(u_long) scc2_params.disfc);
	printf("  Rx CRC errors: %lu\r\n", (u_long) scc2_params.crcec);
	printf("  Abort sequence counter: %lu\r\n", (u_long) scc2_params.abtsc);
	printf("  NMARC (should be 0): %lu\r\n", (u_long) scc2_params.nmarc);
	printf("  RETRC (should be 0): %lu\r\n", (u_long) scc2_params.nmarc);
}

hdlc_zero_hw_stats()
{
	scc2_params.disfc = 0;
	scc2_params.crcec = 0;
	scc2_params.abtsc = 0;
	scc2_params.nmarc = 0;
	scc2_params.retrc = 0;
}
