/*
 * ATM Tx direction via the I.432.1 FPGA
 */

#include "types.h"
#include "globals.h"
#include "i4321_fpga.h"
#include "monitor_api.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"

extern volatile struct i4321_memblock FPGA_regs;

extern struct buffer *get_buffer_from_queue();

struct buffer *txpkt_buf;
u_char *txpkt_ptr, *txpkt_endptr;

fpga_regs_init()
{
	if (FPGA_regs.id_reg != I4321_IDREG) {
		monapi_error("I.432.1 FPGA not found");
		return(-1);
	}
	FPGA_regs.sss_ctrl = flavor_setting_sss;
	FPGA_regs.hec_coset = flavor_setting_coset ? 0x55 : 0;
	FPGA_regs.txcsr = I4321_TXCSR_CRC32_EN | I4321_TXCSR_IDLE_EN;
	FPGA_regs.rxcsr = I4321_RXCSR_RESET;
	cpu_delay_us(300);
	FPGA_regs.rx_overrun_count = 0;
	return(0);
}

fpga_idleframes_init()
{
	int i;

	for (i = 0; i < 8; i++)
		*(u_long *)FPGA_regs.tx_cells[i].cellhdr = 0x00000001;
	txpkt_buf = NULL;
	initidle_count = initial_idle_cells / 4;
}

fpga_tx_reset_pulse()
{
	u_short csr;

	csr = FPGA_regs.txcsr;
	csr |= I4321_TXCSR_RESET;
	FPGA_regs.txcsr = csr;
	/* 300 us is over 20 symbols at 144 kbps, so it should be enough */
	cpu_delay_us(300);
	csr &= ~I4321_TXCSR_RESET;
	FPGA_regs.txcsr = csr;
}

fpga_tx_enable()
{
	sdcore_api->set_tx(&sdcore_state, 1, SCRAMBLED_FOUR_LEVEL_DATA);
	FPGA_regs.txcsr |= I4321_TXCSR_INT_EN;
}

fill_tx_4cells(cells)
	struct atmcell64 *cells;
{
	struct vc *vc;
	int cellno;

	if (initidle_count) {
		initidle_count--;
		return;
	}
	for (cellno = 0; cellno < 4; cellno++) {
		if (!txpkt_buf) {
			txpkt_buf = get_buffer_from_queue(&atm_outgoing_queue);
			if (!txpkt_buf)
				break;
			txpkt_ptr = txpkt_buf->buf_dataspc +
					txpkt_buf->buf_dataoff;
			txpkt_endptr = txpkt_ptr + txpkt_buf->buf_datalen;
		}
		vc = txpkt_buf->buf_vc;
		bcopy_atmcell(txpkt_ptr, cells[cellno].payload);
		txpkt_ptr += 48;
		if (txpkt_ptr < txpkt_endptr) {
			*(u_long *)cells[cellno].cellhdr =
				*(u_long *)vc->cellhdr_normal;
		} else {
			*(u_long *)cells[cellno].cellhdr =
				*(u_long *)vc->cellhdr_last;
			l2conv_api->free_buffer(txpkt_buf);
			txpkt_buf = NULL;
		}
		stats.atm_tx_cells++;
	}
	/* pad any remainder with idle cells */
	while (cellno < 4)
		*(u_long *)cells[cellno++].cellhdr = 0x00000001;
}

fpga_tx_inthandler()
{
	if (FPGA_regs.txcsr & I4321_TXCSR_BUF_LAST_WR) {
		FPGA_regs.txcsr &= ~I4321_TXCSR_BUF_LAST_WR;
		fill_tx_4cells(&FPGA_regs.tx_cells[0]);
	} else {
		FPGA_regs.txcsr |= I4321_TXCSR_BUF_LAST_WR;
		fill_tx_4cells(&FPGA_regs.tx_cells[4]);
	}
}

fpga_txbuf_clear()
{
	struct buffer *buf;

	FPGA_regs.txcsr &= ~I4321_TXCSR_INT_EN;
	while (buf = get_buffer_from_queue(&atm_outgoing_queue))
		l2conv_api->free_buffer(buf);
	if (txpkt_buf) {
		l2conv_api->free_buffer(txpkt_buf);
		txpkt_buf = NULL;
	}
}
