/*
 * The Activation State Machine for Flavor A
 */

#include "types.h"
#include "stdio.h"
#include "mc68302.h"
#include "gpio.h"
#include "sdsl_asm.h"
#include "globals.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile long timebase;

char * const sdsl_asm_states[8] = {
	"INACTIVE_STATE",
	"WAIT_FOR_DTR",
	"ACTIVATING_STATE",
	"FRAME_ACQU_STATE",
	"ACTIVE_STATE",
	"DEACTIVATED_STATE",
	"WAIT_FOR_LOST",
	"WAIT_FOR_LOS"
};

sdsl_asm()
{
	int dtr;

	switch (sdsl_asm_state) {

	case INACTIVE_STATE:
		fpga_idleframes_init();
		dtr = opcore_api->dtrcheck(1, 1);
		if (dtr) {
			printf("Attempting SDSL startup\r\n");
			sdcore_api->activate(&sdcore_state);
			set_sdsl_state(ACTIVATING_STATE);
			if ((mc68302_regs.padat & PORTA_LSLED_MASK) !=
			    PORTA_LSLED_RED)
				mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		} else {
			sdcore_api->powerdown(&sdcore_state);
			printf("Waiting for DTR\r\n");
			set_sdsl_state(WAIT_FOR_DTR);
			mc68302_regs.padat &= ~PORTA_LSLED_MASK;
		}
		return;

	case WAIT_FOR_DTR:
		dtr = opcore_api->dtrcheck(1, 0);
		if (dtr) {
			printf("Attempting SDSL startup\r\n");
			sdcore_api->activate(&sdcore_state);
			set_sdsl_state(ACTIVATING_STATE);
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		}
		return;

	case ACTIVATING_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		if (sdcore_state.status.bits.normal_operation)
			sdsl_startup_success();
		else if (sdcore_state.status.bits.activation_interval) {
		  printf("Timer expired: SDSL startup failed, will retry\r\n");
		  set_sdsl_state(DEACTIVATED_STATE);
		} else if (sdcore_state.status.bits.preact_done)
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		return;

	case FRAME_ACQU_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		if (celldelim_sync) {
			/* yay! */
			printf("Cell delineation acquired\r\n");
			set_sdsl_state(ACTIVE_STATE);
			mc68302_regs.pbdat |= PORTB_CD | PORTB_CTS;
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_GREEN;
			stats.sdsl_activations++;
			pending_deact = 0;
			return;
		}
		if (timebase - sdsl_asm_state_time >= frame_acqu_time) {
			printf("Unable to acquire cell delineation\r\n");
			set_sdsl_state(DEACTIVATED_STATE);
		}
		return;

	case ACTIVE_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		/* ATM cell delineation is our link up criterion */
		if (do_pending_deact_logic())
			stats.sdsl_locd_shutdowns++;
		return;

	case DEACTIVATED_STATE:
		printf("Tearing SDSL link down\r\n");
		if (sdsl_asm_prev_state == FRAME_ACQU_STATE ||
		    sdsl_asm_prev_state == ACTIVE_STATE)
			sdsl_rx_shutdown();
		sdcore_api->deactivate(&sdcore_state);
		mc68302_regs.pbdat &= ~(PORTB_CD | PORTB_CTS);
		mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_RED;
		if (sdsl_asm_prev_state == FRAME_ACQU_STATE ||
		    sdsl_asm_prev_state == ACTIVE_STATE)
			fpga_txbuf_clear();
		if (sdcore_state.terminal_type) {
			printf("Now wait for LOS\r\n");
			set_sdsl_state(WAIT_FOR_LOS);
		} else {
			printf("Now wait for LOST\r\n");
			set_sdsl_state(WAIT_FOR_LOST);
		}
		return;

	case WAIT_FOR_LOST:
		if (sdcore_state.status.bits.lost) {
			set_sdsl_state(INACTIVE_STATE);
			return;
		}
		if (los_lost_timeout && timebase - sdsl_asm_state_time >=
		    los_lost_timeout) {
		    printf("Never got LOST, proceeding to retry anyway\r\n");
			set_sdsl_state(INACTIVE_STATE);
		}
		return;

	case WAIT_FOR_LOS:
		if (sdcore_state.status.bits.los) {
			set_sdsl_state(INACTIVE_STATE);
			return;
		}
		if (los_lost_timeout && timebase - sdsl_asm_state_time >=
		    los_lost_timeout) {
			printf("Never got LOS, proceeding to retry anyway\r\n");
			set_sdsl_state(INACTIVE_STATE);
		}
		return;

	default:
		printf("Fatal internal error: invalid state in SDSL ASM\r\n");
		asm("trap #0");
	}
}

set_sdsl_state(newstate)
	int newstate;
{
	sdsl_asm_prev_state = sdsl_asm_state;
	sdsl_asm_state = newstate;
	sdsl_asm_state_time = timebase;
}

do_pending_deact_logic()
{
	if (!celldelim_sync && !pending_deact) {
		printf("SDSL: Loss of cell delineation (LOCD)\r\n");
		if (pending_deact_gracetime) {
			pending_deact = 1;
			pending_deact_time = timebase;
			printf("Pending deactivation\r\n");
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
			return(0);
		} else {
			set_sdsl_state(DEACTIVATED_STATE);
			return(1);
		}
	}
	if (!pending_deact)
		return(0);
	if (celldelim_sync) {
		pending_deact = 0;
		printf("Good cell delineation regained\r\n");
		mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_GREEN;
		return(0);
	}
	if (timebase - pending_deact_time >= pending_deact_gracetime) {
		set_sdsl_state(DEACTIVATED_STATE);
		return(1);
	}
	return(0);
}
