/*
 * HEC RT11FFS write code (fairly generic)
 */

#include "types.h"
#include "stdio.h"
#include "rt11ffs.h"
#include "../libffs/ffsimpl.h"
#include "errors.h"
#include "api.h"

extern struct rt11ffs_dirent ffs_segment_header;

ffs_init_segment(segno)
	int segno;
{
	caddr_t segaddr;
	int stat;

	if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
		error("Invalid segment number");
		return(FLASHWR_ERR_INVARG);
	}
	segaddr = (caddr_t)FLASH_BASE + segno * FLASH_ERASEBLK_SIZE;
	if (!blank_check_region(segaddr, FLASH_ERASEBLK_SIZE)) {
		stat = FLASHWR_ERR_NOTBLANK;
		goto out;
	}
	stat = flash_program(&ffs_segment_header, segaddr,
				sizeof ffs_segment_header);
out:	if (stat)
		ffs_write_error(stat);
	return(stat);
}

ffs_create_file(filename, allocsize, datasrc, opts)
	u_short *filename;
	unsigned allocsize;
	caddr_t datasrc;
	struct ffs_create_options *opts;
{
	struct segment_scan_results scan[FFS_NSEGS];
	int seg, stat, fragsize, validseg = 0;
	unsigned left, fragbytes;
	struct rt11ffs_dirent dirent;

	if (allocsize < 1) {
		printf("FFS Create: invalid allocation size parameter\r\n");
		return(FLASHWR_ERR_INVARG);
	}
	if (opts->overwrite) {
		stat = ffs_delete_file(filename, 1);
		if (stat)
			return(stat);
	} else {
		stat = ffs_create_file_check_exist(filename);
		if (stat) {
			ffs_write_error(stat);
			return(stat);
		}
	}
	ffs_scan_flash(0, scan);

	/* pass 1: check everything we need */
	if (opts->startseg) {
		seg = opts->startseg;
		if (seg < FFS_MINSEG || seg > FFS_MAXSEG) {
		    printf("FFS Create: invalid explicit segment number\r\n");
		    return(FLASHWR_ERR_INVARG);
		}
		if (!scan[seg-1].valid_seg) {
			printf("FFS Create: segment %X doesn't exist\r\n",
				seg);
			return(FLASHWR_ERR_INVARG);
		}
	} else
		seg = FFS_MINSEG;
	for (left = allocsize; left && seg <= FFS_MAXSEG; seg++) {
		if (!scan[seg-1].valid_seg)
			continue;
		if (scan[seg-1].has_errs) {
			printf("FFS segment %X contains garbage, not used\r\n",
				seg);
			if (seg == opts->startseg)
				return(FLASHWR_ERR_BADFFS);
			else
				continue;
		}
		validseg = 1;
		if (opts->contig && scan[seg-1].free_blocks < allocsize) {
			if (opts->startseg) {
		printf("FFS Create: insufficient free space in segment %X\r\n",
					seg);
				return(FLASHWR_ERR_NOSPACE);
			} else
				continue;
		}
		if (!scan[seg-1].free_blocks)
			continue;
		fragsize = MIN(left, scan[seg-1].free_blocks);
		fragbytes = fragsize * RT11FFS_AU_SIZE;
		if (!blank_check_region(scan[seg-1].new_data_ptr, fragbytes)) {
			stat = FLASHWR_ERR_NOTBLANK;
			ffs_write_error(stat);
			return(stat);
		}
		left -= fragsize;
	}
	if (left) {
		if (validseg)
			printf("FFS Create: insufficient %sfree space\r\n",
				opts->contig ? "contiguous " : "");
		else
		    printf("FFS Create: no valid segment to write into!\r\n");
		return(FLASHWR_ERR_NOSPACE);
	}

	/* pass 2: write directory entries and data blocks */
	bcopy_words(filename, dirent.filename, 6);
	if (opts->startseg)
		seg = opts->startseg;
	else
		seg = FFS_MINSEG;
	for (left = allocsize; left; seg++) {
		if (!scan[seg-1].valid_seg || scan[seg-1].has_errs ||
		    !scan[seg-1].free_blocks)
			continue;
		if (opts->contig && scan[seg-1].free_blocks < allocsize)
			continue;
		fragsize = MIN(left, scan[seg-1].free_blocks);
		fragbytes = fragsize * RT11FFS_AU_SIZE;
		dirent.size_flags = RT11FFS_DIRENT_VALID | fragsize;
		if (left == allocsize)
			dirent.size_flags |= RT11FFS_DIRENT_FIRST;
		left -= fragsize;
		if (!left)
			dirent.size_flags |= RT11FFS_DIRENT_LAST;
		stat = flash_program(&dirent, scan[seg-1].new_dirent_ptr,
					sizeof(struct rt11ffs_dirent));
		if (stat) {
			ffs_write_error(stat);
			return(stat);
		}
		stat = flash_program(datasrc, scan[seg-1].new_data_ptr,
					fragbytes);
		if (stat) {
			ffs_write_error(stat);
			return(stat);
		}
		datasrc += fragbytes;
	}
	return(0);
}

ffs_create_file_check_exist(filename)
	u_short *filename;
{
	struct find_file_results findstruct;
	int stat;

	stat = ffs_find_file(filename, &findstruct);
	switch (stat) {
	case FIND_STAT_FOUND:
	case FIND_STAT_INVALID_FRAG:
		return(FLASHWR_ERR_FILEEXISTS);
	case FIND_STAT_NOTFOUND:
		return(0);
	case FIND_STAT_INVALID_DIR:
		return(FLASHWR_ERR_BADFFS);
	}
}

ffs_delete_file(filename, silent)
	u_short *filename;
{
	int segno, stat, err = 0, found = 0;
	char printname[16];

	for (segno = FFS_MINSEG; segno <= FFS_MAXSEG; segno++) {
		stat = ffs_delete_file_oneseg(segno, filename, &found);
		if (stat) {
			ffs_write_error(stat);
			if (!err)
				err = stat;
		}
	}
	if (!found && !silent && !err) {
		rad50_filename_to_ascii_trimspaces(filename, printname);
		printf("FFS Delete: %s not found\r\n", printname);
		err = FLASHWR_ERR_INVARG;
	}
	return(err);
}

ffs_delete_file_oneseg(segno, filename, found)
	int segno;
	u_short *filename;
	int *found;
{
	caddr_t segbase;
	int blkptr;
	struct rt11ffs_dirent *dirptr;
	int dirindex;
	int size;
	int stat;
	u_short new_size_flags;

	segbase = (caddr_t)FLASH_BASE + segno * FLASH_ERASEBLK_SIZE;
	dirptr = (struct rt11ffs_dirent *) segbase;
	if (bcmp(dirptr, &ffs_segment_header, sizeof ffs_segment_header))
		return(0);
	dirptr++;
	dirindex = 1;
	blkptr = 1;
	while (blkptr < FFS_AUs_PER_SEGMENT) {
		if (dirindex >= 64) {
			dirptr = (struct rt11ffs_dirent *)
					(segbase + blkptr++ * RT11FFS_AU_SIZE);
			dirindex = 0;
		}
		if (blank_check_region(dirptr, sizeof(struct rt11ffs_dirent)))
			break;
		if (dirptr->filename[0] > 63999 || dirptr->filename[1] > 63999
		    || dirptr->filename[2] > 63999)
			return(FLASHWR_ERR_BADFFS);
		size = dirptr->size_flags & RT11FFS_DIRENT_SIZEMASK;
		if (size < 1 || size > (FFS_AUs_PER_SEGMENT - blkptr))
			return(FLASHWR_ERR_BADFFS);
		if ((dirptr->size_flags & RT11FFS_DIRENT_VALID) &&
		    !bcmp(dirptr->filename, filename, 6)) {
			*found = 1;
			new_size_flags = dirptr->size_flags &
							~RT11FFS_DIRENT_VALID;
			stat = flash_program(&new_size_flags,
					     &dirptr->size_flags, 2);
			if (stat)
				return(stat);
		}
		blkptr += size;
		dirptr++;
		dirindex++;
	}
	return(0);
}

/*
 * The following "primitive" is logically funky, but is meant
 * to be "user-friendly". It's an illusion of an erase primitive
 * acting within an established FFS segment.
 */
ffs_erase_reinit(segno)
	unsigned segno;
{
	struct segment_scan_results scan;
	int stat;

	if (segno < FFS_MINSEG || segno > FFS_MAXSEG)
		return(FLASHWR_ERR_INVARG);
	ffs_scan_segment(segno, 0, &scan);
	if (!scan.valid_seg) {
		error("FFS segment %X doesn't exist", segno);
		return(FLASHWR_ERR_BADFFS);
	}
	if (!scan.has_errs && scan.active_blocks) {
		error("Segment %X contains active files; delete them first",
			segno);
		return(FLASHWR_ERR_NOTBLANK);
	}
	stat = flash_erase(1 << segno);
	if (stat)
		return(stat);
	stat = ffs_init_segment(segno);
	return(stat);
}

ffs_write_error(errcode)
{
	char *msg;

	switch (errcode) {
	case FLASHWR_ERR_HW:
		msg = "Flash hardware error";
		break;
	case FLASHWR_ERR_INVARG:
		msg = "Invalid argument";
		break;
	case FLASHWR_ERR_WRPROT:
		msg = "Flash sector write-protected in hardware";
		break;
	case FLASHWR_ERR_NOTBLANK:
		msg = "Flash is not blank";
		break;
	case FLASHWR_ERR_BADFFS:
		msg = "RT11FFS appears to be corrupt";
		break;
	case FLASHWR_ERR_FILEEXISTS:
		msg = "File already exists";
		break;
	case FLASHWR_ERR_NOSPACE:
		msg = "Insufficient free space";
		break;
	default:
		msg = "Unknown error";
	}
	printf("FFS write error: %s\r\n", msg);
}
