#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include <varargs.h>
#include "console.h"

extern u_long last_memacc_addr;
extern u_short last_memacc_size;

long
find_keyword(word, len, table, ambig)
	char *word;
	int len;
	struct keyword table[];
	int *ambig;
{
	struct keyword *tp;

	for (tp = table; tp->word != NULL; tp++) {
		char *p1 = word, *p2 = tp->word;
		int cnt = 0;

		for (; cnt < len && *p2; p1++, p2++, cnt++) {
			char ch;

			ch = islower(*p1) ? toupper(*p1) : *p1;
			if (ch != *p2)
				break;
		}
		if (cnt < len)
			continue;
		if (cnt >= tp->min)
			return(tp->id);
		*ambig = 1;
	}
	return(-1);
}

const struct keyword datactl_quals[] =
	{{"B", 1, DATASIZE_BYTE},
	 {"W", 1, DATASIZE_WORD},
	 {"L", 1, DATASIZE_LONG},
	 {"P", 1, ADDRSPC_MEM},
	 {"N", 1, COUNT_QUAL},
	 {"STEP", 1, STEP_QUAL},
	 {NULL, 0, -1}};

is_sym_string(text, len)
	char *text;
	int len;
{
	char *cp = text, *end = text + len;

	while (cp < end && issym(*cp))
		cp++;
	return(cp == end);
}

is_hex_string(text, len)
	char *text;
	int len;
{
	char *cp = text, *end = text + len;

	while (cp < end && isxdigit(*cp))
		cp++;
	return(cp == end);
}

parse_hexnum(text, len, num)
	char *text;
	int len;
	u_long *num;
{
	char *cp, *end, ch;
	u_long accum = 0;

	for (cp = text, end = text + len; cp < end; cp++) {
		ch = *cp;
		if (!isxdigit(ch)) {
			error("Invalid digit");
			return(1);
		}
		if (accum & 0xF0000000L) {
			error("Value too large");
			return(1);
		}
		accum <<= 4;
		if (isdigit(ch))
			accum |= ch - '0';
		else if (isupper(ch))
			accum |= ch - 'A' + 10;
		else
			accum |= ch - 'a' + 10;
	}
	*num = accum;
	return(0);
}

parse_decnum(text, len, num)
	char *text;
	int len;
	u_long *num;
{
	char *cp, *end;
	u_long accum = 0;
	int ch, sigfigs = 0;

	for (cp = text, end = text + len; cp < end; cp++) {
		ch = *cp;
		if (!isdigit(ch)) {
			error("Invalid digit");
			return(1);
		}
		ch -= '0';
		if (sigfigs >= 9) {
			error("Value too large");
			return(1);
		}
		accum *= 10;
		accum += ch;
		if (ch || sigfigs)
			sigfigs++;
	}
	*num = accum;
	return(0);
}

parse_signed_decnum(text, len, num)
	char *text;
	int len;
	long *num;
{
	char *cp, *end;
	long accum = 0;
	int ch, sigfigs = 0, sign = 0;

	cp = text;
	end = text + len;
	if (cp == end) {
inv:		error("Invalid number");
		return(1);
	}
	if (*cp == '-') {
		sign = 1;
		cp++;
	} else if (*cp == '+')
		cp++;
	if (cp == end)
		goto inv;
	for (; cp < end; cp++) {
		ch = *cp;
		if (!isdigit(ch))
			goto inv;
		ch -= '0';
		if (sigfigs >= 9) {
			error("Value too large");
			return(1);
		}
		accum *= 10;
		accum += ch;
		if (ch || sigfigs)
			sigfigs++;
	}
	if (sign)
		accum = -accum;
	*num = accum;
	return(0);
}

parse_cmd(cmd, minparam, maxparam, params, datactl, qualtbl, quals)
    char *cmd;
    int minparam, maxparam;
    struct param params[];
    struct datactl *datactl;
    struct keyword qualtbl[];
    struct qual quals[];
{
    char *cp = cmd;
    int paramcnt = 0;
    int qualcnt = 0;

    for (;;) {
	while (*cp == ' ')
		cp++;
	if (!*cp)
		break;
	else if (issym(*cp) || *cp == '\"' || *cp == '*' || *cp == '+'
		 || *cp == '-' || *cp == '@') {
		/* This is a parameter */
		if (paramcnt >= maxparam) {
			error("Too many arguments");
			return(1);
		}
		params[paramcnt].text = cp;
		if (*cp == '*' || *cp == '+' || *cp == '-' || *cp == '@')
			cp++;
		else if (*cp == '\"') {
			while (*++cp && *cp != '\"')
				;
			if (!*cp++) {
				error("Illegal syntax");
				return(1);
			}
		} else {
			while (issym(*cp) || *cp == ':' || *cp == '.')
				cp++;
		}
		params[paramcnt].len = cp - params[paramcnt].text;
		paramcnt++;
	} else if (*cp == '/') {
		char *qual, *val;
		int quallen, vallen;
		long qualid;
		int ambigqual = 0;

		/* This is a qualifier */
		cp++;
		if (!issym(*cp)) {
			error("Illegal syntax");
			return(1);
		}
		qual = cp;
		while (issym(*cp))
			cp++;
		quallen = cp - qual;
		if (*cp == ':') {
			/* Value qualifier */
			cp++;
			val = cp;
			if (*cp == '*' || *cp == '+' || *cp == '-' ||
			    *cp == '@')
				cp++;
			else if (*cp == '\"') {
				while (*++cp && *cp != '\"')
					;
				if (!*cp++) {
					error("Illegal syntax");
					return(1);
				}
			} else if (issym(*cp)) {
				while (issym(*cp))
					cp++;
			} else {
				error("Illegal syntax");
				return(1);
			}
			vallen = cp - val;
		} else	/* No value qualifier */
			val = NULL;
		if (datactl && (qualid = find_keyword(qual, quallen,
				datactl_quals, &ambigqual)) != -1) {
			u_long numval;

			switch (qualid) {
			case DATASIZE_BYTE:
			case DATASIZE_WORD:
			case DATASIZE_LONG:
				if (val) {
					error("No value qualifier");
					return(1);
				}
				if (datactl->datasize_set) {
					if (datactl->datasize == qualid)
						error("Duplicate qualifier");
					else
						error("Qualifier conflict");
					return(1);
				}
				datactl->datasize_set = 1;
				datactl->datasize = qualid;
				break;
			case ADDRSPC_MEM:
				if (val) {
					error("No value qualifier");
					return(1);
				}
				/* doesn't do anything */
				break;
			case COUNT_QUAL:
				if (!val) {
					error("Value qualifier");
					return(1);
				}
				if (parse_hexnum(val, vallen, &numval))
					return(1);
				if (datactl->count_set) {
					error("Duplicate qualifier");
					return(1);
				}
				datactl->count_set = 1;
				datactl->count = numval;
				break;
			case STEP_QUAL:
				if (!val) {
					error("Value qualifier");
					return(1);
				}
				if (parse_hexnum(val, vallen, &numval))
					return(1);
				if (datactl->step_set) {
					error("Duplicate qualifier");
					return(1);
				}
				datactl->step_set = 1;
				datactl->step = numval;
				break;
			}
		} else if (qualtbl && (qualid = find_keyword(qual, quallen,
					qualtbl, &ambigqual)) != -1) {
			int i;

			for (i = 0; i < qualcnt; i++)
				if (quals[i].id == qualid) {
					error("Duplicate qualifier");
					return(1);
				}
			quals[i].id = qualid;
			quals[i].value = val;
			quals[i].vallen = vallen;
			qualcnt++;
		} else {
			if (ambigqual)
				error("Ambiguous qualifier");
			else
				error("Unknown qualifier");
			return(1);
		}
	} else {
		error("Illegal syntax");
		return(1);
	}
    }
    if (paramcnt < minparam) {
	error("Too few arguments");
	return(1);
    }
    return(0);
}

parse_cmd_plusminus(cmd, minparam, maxparam, params)
    char *cmd;
    int minparam, maxparam;
    struct param params[];
{
    char *cp = cmd;
    int paramcnt = 0;

    for (;;) {
	while (*cp == ' ')
		cp++;
	if (!*cp)
		break;
	else if (issym(*cp) || *cp == '\"' || *cp == '+' || *cp == '-') {
		/* This is a parameter */
		if (paramcnt >= maxparam) {
			error("Too many arguments");
			return(1);
		}
		params[paramcnt].text = cp;
		if (*cp == '+' || *cp == '-') {
			cp++;
			while (isdigit(*cp))
				cp++;
		} else if (*cp == '\"') {
			while (*++cp && *cp != '\"')
				;
			if (!*cp++) {
				error("Illegal syntax");
				return(1);
			}
		} else {
			while (issym(*cp) || *cp == ':' || *cp == '.')
				cp++;
		}
		params[paramcnt].len = cp - params[paramcnt].text;
		paramcnt++;
	} else {
		error("Illegal syntax");
		return(1);
	}
    }
    if (paramcnt < minparam) {
	error("Too few arguments");
	return(1);
    }
    return(0);
}

parse_addr(addrtext, addrlen, datactl, addr)
	char *addrtext;
	int addrlen;
	struct datactl *datactl;
	u_long *addr;
{
	/* See what kind of address this is */
	if (is_hex_string(addrtext, addrlen)) {
		/* Pure numeric address */
		if (parse_hexnum(addrtext, addrlen, addr))
			return(1);
	} else if (addrlen == 1 && (addrtext[0] == '*' || addrtext[0] == '+'
				   || addrtext[0] == '-')) {
		*addr = last_memacc_addr;
		if (addrtext[0] == '+')
			*addr += last_memacc_size;
	} else {
		error("Illegal address");
		return(1);
	}
	if (!datactl->datasize_set) {
		datactl->datasize_set = 1;
		datactl->datasize = last_memacc_size;
	}
	if (!datactl->count_set) {
		datactl->count_set = 1;
		datactl->count = 0;
	}
	if (!datactl->step_set) {
		datactl->step_set = 1;
		datactl->step = datactl->datasize;
	}
	if (addrlen == 1 && addrtext[0] == '-')
		*addr -= datactl->step;
	return(0);
}

error(va_alist)
	va_dcl
{
	va_list ap;
	char *fmt;

	printf("ERROR: ");
	va_start(ap);
	fmt = va_arg(ap, char *);
	vprintf(fmt, ap);
	va_end(ap);
	printf("\r\n");
}
