/*
 * The SDSL Activation State Machine (ASM)
 * is arguably the core of a bit-transparent DSU
 */

#include "types.h"
#include "stdio.h"
#include "mc68302.h"
#include "gpio.h"
#include "sdsl_asm.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"
#include "../sdcore/api.h"
#include "../sdcore/state.h"
#include "../opcore/api.h"
#include "globals.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile long timebase;

char * const sdsl_asm_states[8] = {
	"INACTIVE_STATE",
	"WAIT_FOR_DTR",
	"ACTIVATING_STATE",
	"<invalid>",
	"ACTIVE_STATE",
	"DEACTIVATED_STATE",
	"WAIT_FOR_LOST",
	"WAIT_FOR_LOS"
};

sdsl_asm()
{
	int dtr;

	switch (sdsl_asm_state) {

	case INACTIVE_STATE:
		dtr = opcore_api->dtrcheck(1, 1);
		if (dtr) {
			printf("Attempting SDSL startup\r\n");
			sdcore_api->activate(&sdcore_state);
			set_sdsl_state(ACTIVATING_STATE);
			if ((mc68302_regs.padat & PORTA_LSLED_MASK) !=
			    PORTA_LSLED_RED)
				mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		} else {
			sdcore_api->powerdown(&sdcore_state);
			printf("Waiting for DTR\r\n");
			set_sdsl_state(WAIT_FOR_DTR);
			mc68302_regs.padat &= ~PORTA_LSLED_MASK;
		}
		return;

	case WAIT_FOR_DTR:
		dtr = opcore_api->dtrcheck(1, 0);
		if (dtr) {
			printf("Attempting SDSL startup\r\n");
			sdcore_api->activate(&sdcore_state);
			set_sdsl_state(ACTIVATING_STATE);
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		}
		return;

	case ACTIVATING_STATE:
		dtr = cisco_dtr_bug || opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		if (sdcore_state.status.bits.normal_operation) {
			printf("Startup time %d s\r\n",
				(timebase - sdsl_asm_state_time) / 20);
			set_sdsl_state(ACTIVE_STATE);
			mc68302_regs.pbdat |= PORTB_CD;
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_GREEN;
			pending_deact = 0;
			tx_data_on = 0;
		} else if (sdcore_state.status.bits.activation_interval) {
		  printf("Timer expired: SDSL startup failed, will retry\r\n");
		  set_sdsl_state(DEACTIVATED_STATE);
		} else if (sdcore_state.status.bits.preact_done)
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		return;

	case ACTIVE_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		if (do_los_nmr_logic())
			return;
		do_rts_logic();
		return;

	case DEACTIVATED_STATE:
		printf("Tearing SDSL link down\r\n");
		sdcore_api->deactivate(&sdcore_state);
		mc68302_regs.pbdat &= ~(PORTB_CD | PORTB_CTS);
		mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_RED;
		if (sdcore_state.terminal_type) {
			printf("Now wait for LOS\r\n");
			set_sdsl_state(WAIT_FOR_LOS);
		} else {
			printf("Now wait for LOST\r\n");
			set_sdsl_state(WAIT_FOR_LOST);
		}
		return;

	case WAIT_FOR_LOST:
		if (sdcore_state.status.bits.lost) {
			set_sdsl_state(INACTIVE_STATE);
			return;
		}
		if (los_lost_timeout && timebase - sdsl_asm_state_time >=
		    los_lost_timeout) {
		    printf("Never got LOST, proceeding to retry anyway\r\n");
			set_sdsl_state(INACTIVE_STATE);
		}
		return;

	case WAIT_FOR_LOS:
		if (sdcore_state.status.bits.los) {
			set_sdsl_state(INACTIVE_STATE);
			return;
		}
		if (los_lost_timeout && timebase - sdsl_asm_state_time >=
		    los_lost_timeout) {
			printf("Never got LOS, proceeding to retry anyway\r\n");
			set_sdsl_state(INACTIVE_STATE);
		}
		return;

	default:
		printf("Fatal internal error: invalid state in SDSL ASM\r\n");
		asm("trap #0");
	}
}

set_sdsl_state(newstate)
	int newstate;
{
	sdsl_asm_prev_state = sdsl_asm_state;
	sdsl_asm_state = newstate;
	sdsl_asm_state_time = timebase;
}

do_los_nmr_logic()
{
	int up, nmr;

	up = !sdcore_state.status.bits.los;
	if (up && nmr_threshold_set) {
		nmr = sdcore_api->read_nmr(&sdcore_state);
		if (nmr < nmr_threshold)
			up = 0;
	}
	if (!up && !pending_deact) {
		if (sdcore_state.status.bits.los)
                        printf("LOS detected\r\n");
		else
			printf("NMR fell below the threshold\r\n");
		if (pending_deact_gracetime) {
			pending_deact = 1;
			pending_deact_time = timebase;
			printf("Pending deactivation\r\n");
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
			return(0);
		} else {
			set_sdsl_state(DEACTIVATED_STATE);
			return(1);
		}
	}
	if (!pending_deact)
		return(0);
	if (up) {
		pending_deact = 0;
		printf("Good link returned\r\n");
		mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_GREEN;
		return(0);
	}
	if (timebase - pending_deact_time >= pending_deact_gracetime) {
		set_sdsl_state(DEACTIVATED_STATE);
		return(1);
	}
	return(0);
}

do_rts_logic()
{
	int rts;

	rts = mc68302_regs.pbdat & PORTB_RTS;
	if (!tx_data_on) {
		if (!rts && !rts_override)
			return;
		if (rts)
			printf("DSU: Detected RTS ON\r\n");
		else
			printf("RTS is off but overridden\r\n");
		printf("Enabling Tx bit stream\r\n");
		sdcore_api->set_tx(&sdcore_state, 1, SCRAMBLED_FOUR_LEVEL_DATA);
		mc68302_regs.pbdat |= PORTB_CTS;
		tx_data_on = 1;
	} else {
		if (rts || rts_override)
			return;
		printf("DSU: Detected RTS OFF\r\n");
		printf("Switching Tx bit stream to all 1s\r\n");
		sdcore_api->set_tx(&sdcore_state, 1, SCRAMBLED_FOUR_LEVEL_ONES);
		mc68302_regs.pbdat &= ~PORTB_CTS;
		tx_data_on = 0;
	}
}
