/*
 * This is our cool Nokia frame transmitter.  We operate in the BCLK clock
 * domain, running as a state machine in an endless loop, iterating over all
 * bit positions in the Nokia SDSL frame structure.  We read the data to be
 * transmitted from a buffer which is made to look like a simple asynchronous
 * continuous-read RAM.  We take an asynchronous reset_control signal,
 * pass it through a double synchronizer and use it to initialize our state
 * machine.  This reset logic also provides QCLK alignment.
 *
 * The Nokia frame read from the RAM buffer is transformed as follows:
 *
 * - The read address logic skips one byte after each HEC octet so that
 *   all cell payload is word-aligned on the M68K bus.
 * - Cell header CRC-8 and frame CRC-6 are computed (in the most classic
 *   manner as the bits are shifted out) and filled in.  The respective
 *   RAM values are still read and XORed together with the filled-in value,
 *   so the software provides the HEC coset and can transmit deliberately
 *   wrong CRC for testing.
 * - Cell payload: we have an optional feature to fill in 6A idle cell payload
 *   regardless of what's in RAM and to compute and fill in AAL5 CRC-32.
 *   See README.crc32 for the important restrictions.
 */

module xmitter (ram_addr, ram_data, BCLK, QCLK, SDSLout, reset_control,
		crc32_enable, idlecell_enable);

output [9:0] ram_addr;
input [7:0] ram_data;

input BCLK, QCLK;
output SDSLout;

input reset_control, crc32_enable, idlecell_enable;

/*
 * Note: all following registers describe the *next* octet to be
 * fetched from RAM, *not* the octet currently being shifted out!
 */
reg [2:0] bit_counter;
reg [9:0] state_counter;
wire last_frame_octet, last_frame_bit;
reg [8:0] ram_pointer;
reg buffer_select;

reg QCLK_latch;
reg reset_control_sync1, reset_control_sync2;

reg [7:0] shiftreg;
reg [7:2] crc6_accum;
wire [7:2] crc6_nextval;
reg [7:0] crc8_accum;
wire [7:0] crc8_nextval;

reg current_cell_idle, current_cell_endpkt;
reg [31:0] crc32_accum;
wire [31:0] crc32_nextval;
reg payload_outgoing_now, crc32_outgoing_now;

always @(negedge BCLK)
	QCLK_latch <= QCLK;

always @(posedge BCLK)
	reset_control_sync1 <= reset_control;

always @(posedge BCLK)
	reset_control_sync2 <= reset_control_sync1;

always @(posedge BCLK)
	if (reset_control_sync2 && QCLK_latch)
		bit_counter <= 3'b0;
	else
		bit_counter <= bit_counter + 3'd1;

assign last_frame_octet = state_counter[9] && state_counter[2:0] == 3'b111;
assign last_frame_bit = last_frame_octet && bit_counter == 3'b111;

always @(posedge BCLK)
	if (reset_control_sync2 || last_frame_bit)
		state_counter[5:0] <= 6'b0;
	else if (bit_counter == 3'b111)
	begin
		if (state_counter[5:0] == 6'd4 && !state_counter[9])
			state_counter[5:0] <= 6'd16;
		else
			state_counter[5:0] <= state_counter[5:0] + 6'd1;
	end

always @(posedge BCLK)
	if (reset_control_sync2 || last_frame_bit)
		state_counter[9:6] <= 4'b0;
	else if (bit_counter == 3'b111 && state_counter[5:0] == 6'b111111)
		state_counter[9:6] <= state_counter[9:6] + 4'd1;

always @(posedge BCLK)
	if (reset_control_sync2 || last_frame_bit)
		ram_pointer <= 9'b0;
	else if (bit_counter == 3'b111)
	begin
		if (state_counter[5:0] == 6'd4 && !state_counter[9])
			ram_pointer <= ram_pointer + 9'd2;
		else
			ram_pointer <= ram_pointer + 9'd1;
	end

always @(posedge BCLK)
	if (reset_control_sync2)
		buffer_select <= 1'b0;
	else if (last_frame_bit)
		buffer_select <= !buffer_select;

assign ram_addr = {buffer_select,ram_pointer};

always @(posedge BCLK)
	if (bit_counter == 3'b111)
	begin
		if (state_counter[9] && state_counter[2:0] == 3'b110)
			shiftreg <= ram_data ^ {crc6_nextval,2'b00};
		else if (!state_counter[9] && state_counter[5:0] == 6'd4)
			shiftreg <= ram_data ^ crc8_nextval;
		else if ((|state_counter[5:4]) && current_cell_idle &&
				idlecell_enable)
			shiftreg <= 8'h6A;
		else if ((&state_counter[5:2]) && current_cell_endpkt &&
				crc32_enable)
			shiftreg <= crc32_nextval[31:24];
		else
			shiftreg <= ram_data;
	end
	else
		shiftreg <= shiftreg << 1;

always @(posedge BCLK)
	if (bit_counter == 3'b111)
	begin
		if ((|state_counter[5:4]))
			payload_outgoing_now <= 1'b1;
		else
			payload_outgoing_now <= 1'b0;
	end

always @(posedge BCLK)
	if (bit_counter == 3'b111)
	begin
		if ((&state_counter[5:2]) && current_cell_endpkt &&
				crc32_enable)
			crc32_outgoing_now <= 1'b1;
		else
			crc32_outgoing_now <= 1'b0;
	end

always @(posedge BCLK)
	if (bit_counter == 3'b111 && !state_counter[9] &&
	    state_counter[5:0] == 6'd3)
	begin
		current_cell_endpkt <= ram_data[1];
		current_cell_idle   <= ram_data[0];
	end

assign SDSLout = shiftreg[7] ^ crc32_outgoing_now;

assign crc6_nextval = (crc6_accum << 1) ^
			(crc6_accum[7] ^ SDSLout ? 6'b000011 : 6'b000000);

always @(posedge BCLK)
	if (state_counter == 10'b0 && bit_counter == 3'b111)
		crc6_accum <= 6'b000000;
	else
		crc6_accum <= crc6_nextval;

assign crc8_nextval = (crc8_accum << 1) ^
			(crc8_accum[7] ^ shiftreg[7] ? 8'b00000111 : 8'b0);

always @(posedge BCLK)
	if (state_counter[5:0] == 6'b0 && bit_counter == 3'b111)
		crc8_accum <= 8'b00000000;
	else
		crc8_accum <= crc8_nextval;

assign crc32_nextval = (crc32_accum << 1) ^
			(crc32_accum[31] ^ shiftreg[7] ? 32'h04C11DB7 : 32'b0);

always @(posedge BCLK)
	if (payload_outgoing_now)
		crc32_accum <= crc32_nextval;
	else if (state_counter[5:0] == 6'd2 &&
			(current_cell_endpkt || current_cell_idle))
		crc32_accum <= 32'hFFFFFFFF;

endmodule
